/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file balloon_layer.cpp
 * \brief Implementation of the ptb::balloon_layer class.
 * \author Sebastien Angibaud
 */
#include "ptb/layer/balloon_layer.hpp"
#include "ptb/export.hpp"
#include "engine/game.hpp"

GUI_LAYER_EXPORT( balloon_layer, ptb )

/*--------------------------------------------------------------------------*/
const std::string ptb::balloon_layer::s_default_name( "balloon_layer" );
const unsigned int ptb::balloon_layer::s_border = 20;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor of a msg_add_balloon.
 */
ptb::balloon_layer::msg_add_speaker::msg_add_speaker()
  : m_speaker(NULL)
{

} // msg_add_speaker::msg_add_speaker()

/*--------------------------------------------------------------------------*/
/*
 * \brief Set the speaker.
 * \param speaker The speaker.
 */
void ptb::balloon_layer::msg_add_speaker::set_speaker
( ptb::speaker_item* speaker) 
{
  m_speaker = speaker;
} // msg_add_speaker::set_speaker()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to the balloon layer.
 * \param that The balloon layer to apply the message to.
 */
bool ptb::balloon_layer::msg_add_speaker::apply_to
( bear::communication::messageable& that )
{
  CLAW_PRECOND( dynamic_cast<balloon_layer*>(&that) != NULL );
  bool ok = false;

  if ( m_speaker != NULL )
    {
      balloon_layer* l = static_cast<balloon_layer*>(&that);
      ok = l->add_speaker(m_speaker);
    }

  return ok;
} // msg_add_speaker::apply_to




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor
 */
ptb::balloon_layer::balloon_layer()
  : bear::communication::messageable(default_name())
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();
  glob.register_item(*this);
} // balloon_layer::balloon_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the players balloon and update local values.
 * \param elapsed_time The time elapsed since the last call.
 */
void ptb::balloon_layer::progress( bear::universe::time_type elapsed_time )
{
  std::list<ptb::speaker_item*>::iterator it;
  for ( it = m_speakers.begin(); it != m_speakers.end(); ++it) 
    if ( !(*it)->has_finished_to_chat() )
      {
        (*it)->get_balloon()->progress(elapsed_time);
        adjust_balloon_position(*it);
      }
  
  for ( it = m_speakers.begin(); it != m_speakers.end(); ) 
    if ( (*it)->has_finished_to_chat() )
      {
        std::list<ptb::speaker_item*>::iterator it2 = it;
        ++it;
        m_speakers.erase(it2);
      }
    else
      ++it;
} // balloon_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render balloons.
 * \param screen The screen on which we will render.
 */
void ptb::balloon_layer::render( bear::visual::screen& screen )
{
  std::list<ptb::speaker_item*>::iterator it;
  for ( it = m_speakers.begin(); it != m_speakers.end(); ++it) 
    if (  !(*it)->has_finished_to_chat() )
      (*it)->get_balloon()->render(screen);
} // balloon_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a speaker.
 * \param speaker The speaker.
 * Return true if the speaker is added.
 */
bool ptb::balloon_layer::add_speaker( ptb::speaker_item* speaker)
{
  bool contain = false;
  std::list<ptb::speaker_item*>::iterator it;
  for ( it = m_speakers.begin(); 
        ( it != m_speakers.end() ) && ( ! contain ); ++it) 
    if ( *it == speaker )
      contain = true;

  if ( ! contain )
    {
      m_speakers.push_back(speaker);
      speaker->create_balloon();
    }

  return ( ! contain );
} // balloon_layer::add_balloon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the default name of the balloon_layer.
 */
const std::string& ptb::balloon_layer::default_name()
{
  return s_default_name;
} // balloon_layer::default_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the position of balloon of a speaker.
 */
void ptb::balloon_layer::adjust_balloon_position(ptb::speaker_item* speaker)
{
  bool top = true;
  bool right = true;

  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  bear::engine::camera::msg_get_focus msg_cam;
  glob.send_message( bear::engine::camera::default_name(), msg_cam );
   
  if ( ( speaker->get_speaker()->get_position().x >= 
         msg_cam.focus.position.x + msg_cam.focus.width ) || 
       ( speaker->get_speaker()->get_right() <= msg_cam.focus.position.x ) || 
       ( speaker->get_speaker()->get_position().y  >= 
         msg_cam.focus.position.y + msg_cam.focus.height ) || 
       ( speaker->get_speaker()->get_bottom() <= msg_cam.focus.position.y) )
    adjust_position_in_camera(speaker, msg_cam, top, right);
  else
    adjust_position(speaker, msg_cam, top, right);

  speaker->get_balloon()->set_spike_right(right);
  speaker->get_balloon()->set_spike_top(top);
} // balloon_layer::adjust_balloon_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the position of the frame when the balloon is in the focus.
 *
 * \param msg_cam The focus.
 */
void ptb::balloon_layer::adjust_position
( ptb::speaker_item* speaker, bear::engine::camera::msg_get_focus& msg_cam,
  bool& top, bool& right)
{
  bear::universe::position_type pos(0,0);
  bear::gui::frame* frame = speaker->get_balloon()->get_frame();

  if(  speaker->get_speaker()->get_right() + frame->width() + s_border + 20
       <= msg_cam.focus.position.x + msg_cam.focus.width )
    pos.x = speaker->get_speaker()->get_right() + 20;
  else
    {
      right = false;
      pos.x = speaker->get_speaker()->get_position().x - frame->width() - 20;
    }

  if(  speaker->get_speaker()->get_position().y <= 
       msg_cam.focus.position.y + frame->height() + s_border )
    {
      pos.y = speaker->get_speaker()->get_bottom();
      top = false;
    }
  else
    pos.y = speaker->get_speaker()->get_position().y - frame->height();
  
  pos -= msg_cam.focus.position;

  speaker->get_balloon()->set_position
    (pos.cast_value_type_to<unsigned int>());
} // balloon_layer::adjust_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the position of the frame when the balloon is out the focus.
 *
 * \param msg_cam The focus.
 */
void ptb::balloon_layer::adjust_position_in_camera
( ptb::speaker_item* speaker, bear::engine::camera::msg_get_focus& msg_cam,
  bool& top, bool& right)
{
  bear::universe::position_type pos(0,0);
  bear::gui::frame* frame = speaker->get_balloon()->get_frame();

  if( speaker->get_speaker()->get_position().x + s_border >= 
      msg_cam.focus.position.x + msg_cam.focus.width )
    {
      pos.x = msg_cam.focus.width - frame->width() - s_border;
      right = false;
    }
  else if ( speaker->get_speaker()->get_right() 
            <= msg_cam.focus.position.x + s_border ) 
    pos.x = s_border;
  else if ( speaker->get_speaker()->get_right() 
            + frame->width() + 20 + s_border <= 
            msg_cam.focus.position.x + msg_cam.focus.width )
    pos.x =  speaker->get_speaker()->get_right() + 20 - 
      msg_cam.focus.position.x;
  else
    {
      pos.x =  speaker->get_speaker()->get_position().x 
        - 20 - frame->width() - msg_cam.focus.position.x;
      right = false;
    }
      
  if ( speaker->get_speaker()->get_position().y  + s_border >= 
       msg_cam.focus.position.y + msg_cam.focus.height )
    pos.y = msg_cam.focus.height - frame->height() - s_border;
  else if ( speaker->get_speaker()->get_bottom() <= 
            msg_cam.focus.position.y + s_border )
    {
      top = false;
      pos.y = s_border;
    }
  else if ( speaker->get_speaker()->get_position().y >= 
            msg_cam.focus.position.y + s_border + frame->height() ) 
    pos.y = speaker->get_speaker()->get_position().y 
      - msg_cam.focus.position.y - frame->height();
  else
    { 
      top = false;
      pos.y = speaker->get_speaker()->get_bottom() - 
        msg_cam.focus.position.y;
    }

  speaker->get_balloon()->set_position
    ( pos.cast_value_type_to<unsigned int>() );
} // balloon_layer::adjust_position_in_camera()
