/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file item_information_layer.cpp
 * \brief Implementation of the ptb::item_information_layer class.
 * \author Julien Jorge
 */
#include "ptb/layer/item_information_layer.hpp"

#include "engine/font_factory.hpp"
#include "engine/camera.hpp"
#include "engine/game.hpp"
#include "engine/world.hpp"
#include "text/font.hpp"
#include "ptb/export.hpp"

#include <sstream>

GUI_LAYER_EXPORT( item_information_layer, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param item The item from which we want informations.
 * \param camera_box The box of the camera in the level.
 */
ptb::item_information_layer::info_box::info_box
( bear::engine::base_item& item,
  const bear::universe::rectangle_type& camera_box )
  : m_item(&item),
    m_delta( item.get_size().cast_value_type_to<unsigned int>() ),
    m_text( NULL,
            bear::engine::font_factory::create("font/fixed_white-7x12.tga") )
{
  m_text.set_auto_size(true);

  get_informations(camera_box);
} // item_information_layer::info_box::info_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::item_information_layer::info_box::~info_box()
{
  // nothing to do
} // item_information_layer::info_box::~info_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tells if the box includes a given point.
 * \param pos The point to check.
 */
bool ptb::item_information_layer::info_box::includes
( const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  claw::math::rectangle<unsigned int> box
    ( m_text.get_position().x, m_text.get_position().y,
      m_text.width(), m_text.height() );

  return box.includes(pos);
} // item_information_layer::info_box::includes()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if this info_box links to a given item.
 * \param item The item to check
 */
bool ptb::item_information_layer::info_box::has_item
( const bear::engine::base_item* item ) const
{
  return m_item == item;
} // item_information_layer::info_box::has_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the informations.
 * \param delta Vector to add to the current position.
 */
void ptb::item_information_layer::info_box::add_delta
( const claw::math::coordinate_2d<int>& delta )
{
  m_delta += delta;
} // item_information_layer::info_box::add_delta()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the text box.
 * \param screen The screen on which we draw.
 * \param camera_box The box of the camera in the level.
 */
void ptb::item_information_layer::info_box::render
( bear::visual::screen& screen,
  const bear::universe::rectangle_type& camera_box )
{
  if (m_item)
    {
      bear::universe::position_type pos =
        m_item->get_position() + m_delta - camera_box.position;
      m_text.set_position( pos.cast_value_type_to<unsigned int>() );
    }
  else
    m_text.set_text("Invalid handle.");

  m_text.render( screen );
} // item_information_layer::info_box::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the informations of the item.
 * \param item The item from which we want informations.
 * \param camera_box The box of the camera in the level.
 */
void ptb::item_information_layer::info_box::get_informations
( const bear::universe::rectangle_type& camera_box )
{
  std::ostringstream oss;

  oss << *m_item << std::endl;
  m_text.set_text( oss.str() );
} // item_information_layer::info_box::get_informations()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param item The item from which we want informations.
 * \param camera_box The box of the camera in the level.
 */
ptb::item_information_layer::linked_info_box::linked_info_box
( bear::engine::base_item& item,
  const bear::universe::rectangle_type& camera_box )
  : info_box(item, camera_box)
{

} // item_information_layer::linked_info_box::linked_info_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the text box.
 * \param screen The screen on which we draw.
 * \param camera_box The box of the camera in the level.
 */
void ptb::item_information_layer::linked_info_box::render
( bear::visual::screen& screen,
  const bear::universe::rectangle_type& camera_box )
{
  if (m_item)
    get_informations( camera_box );

  info_box::render(screen, camera_box);
} // item_information_layer::linked_info_box::render()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::item_information_layer::item_information_layer()
  : m_dragged_item(NULL)
{

} // item_information_layer::item_information_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::item_information_layer::~item_information_layer()
{
  clear();
} // item_information_layer::~item_information_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Clear the layer.
 */
void ptb::item_information_layer::clear()
{
  info_box_list::iterator it;

  for (it=m_info_box.begin(); it!=m_info_box.end(); ++it)
    delete *it;

  m_info_box.clear();
} // item_information_layer::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the layer on a screen.
 * \param screen The screen on which we draw.
 */
void ptb::item_information_layer::render( bear::visual::screen& screen )
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();
  bear::engine::camera::msg_get_focus msg;

  glob.send_message( bear::engine::camera::default_name(), msg );

  info_box_list::iterator it;

  for (it=m_info_box.begin(); it!=m_info_box.end(); ++it)
    (*it)->render(screen, msg.focus);
} // item_information_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been pressed.
 * \param key The value of the maintained button.
 * \param pos The current position of the cursor.
 */
bool ptb::item_information_layer::mouse_pressed
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;

  switch(key)
    {
    case bear::input::mouse::mc_left_button:
      result = grab_info_box(pos);
      break;
    case bear::input::mouse::mc_right_button:
      result = put_in_background(pos);
      break;
    case bear::input::mouse::mc_middle_button:
      result = close_info_box(pos);
      break;
    }

  if (!result)
    {
      bear::engine::level_globals& glob =
        bear::engine::game::get_instance().current_level_globals();

      bear::engine::camera::msg_get_focus msg;

      glob.send_message( bear::engine::camera::default_name(), msg );

      switch(key)
        {
        case bear::input::mouse::mc_left_button:
          result = show_item(pos, msg.focus);
          break;
        case bear::input::mouse::mc_right_button:
          result = follow_item(pos, msg.focus);
          break;
        case bear::input::mouse::mc_middle_button:
          if ( !m_info_box.empty() )
            {
              result = true;
              clear();
            }
          break;
        }
    }

  return result;
} // item_information_layer::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been released.
 * \param key The value of the released button.
 * \param pos The current position of the cursor.
 */
bool ptb::item_information_layer::mouse_released
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;

  if (key == bear::input::mouse::mc_left_button)
    if (m_dragged_item != NULL)
      {
        m_dragged_item->add_delta( pos - m_drag_reference );
        m_dragged_item = NULL;
        result = true;
      }

  return result;
} // item_information_layer::mouse_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button is maintained.
 * \param key The value of the pressed button.
 * \param pos The current position of the cursor.
 */
bool ptb::item_information_layer::mouse_maintained
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;

  if (key == bear::input::mouse::mc_left_button)
    if (m_dragged_item != NULL)
      {
        m_dragged_item->add_delta( pos - m_drag_reference );
        m_drag_reference = pos;
        result = true;
      }

  return result;
} // item_information_layer::mouse_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the informations of an item, given a position.
 * \param pos The position where the item is searched.
 * \param camera_box The box of the camera in the level.
 */
bool ptb::item_information_layer::show_item
( const bear::universe::position_type& pos,
  const bear::universe::rectangle_type& camera_box )
{
  bear::engine::base_item* item = find_item(pos + camera_box.position);

  if (item != NULL)
    m_info_box.push_front( new info_box(*item, camera_box) );

  return item != NULL;
} // item_information_layer::show_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Follow an item and show its informations, given a position.
 * \param pos The position where the item is searched.
 * \param camera_box The box of the camera in the level.
 */
bool ptb::item_information_layer::follow_item
( const bear::universe::position_type& pos,
  const bear::universe::rectangle_type& camera_box )
{
  bear::engine::base_item* item = find_item(pos + camera_box.position);

  if (item != NULL)
    m_info_box.push_front( new linked_info_box(*item, camera_box) );

  return item != NULL;
} // item_information_layer::follow_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close an information box, given a position.
 * \param pos The position where the box is searched.
 */
bool ptb::item_information_layer::close_info_box
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  info_box_list::iterator it = find_info_box(pos);

  if (it!=m_info_box.end())
    {
      delete *it;
      m_info_box.erase(it);
      return true;
    }
  else
    return false;
} // item_information_layer::close_info_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Grab an information box, given a position.
 * \param pos The position where the box is searched.
 */
bool ptb::item_information_layer::grab_info_box
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  info_box_list::iterator it = find_info_box(pos);

  if ( it!=m_info_box.end() )
    {
      m_dragged_item = *it;
      m_drag_reference = pos;
      return true;
    }
  else
    return false;
} // item_information_layer:grab_info_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Put an information box in the background, given a position.
 * \param pos The position where the box is searched.
 */
bool ptb::item_information_layer::put_in_background
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  info_box_list::iterator it = find_info_box(pos);

  if ( it!=m_info_box.end() )
    {
      info_box* item = *it;
      m_info_box.erase(it);
      m_info_box.push_front(item);
      return true;
    }
  else
    return false;
} // item_information_layer:put_in_background()

/*----------------------------------------------------------------------------*/
/**
 * \brief Find an information box, given a position.
 * \param pos The position where the box is searched.
 */
ptb::item_information_layer::info_box_list::iterator
ptb::item_information_layer::find_info_box
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool found = false;

  info_box_list::iterator it = m_info_box.end();

  while ( !found && (it!=m_info_box.begin()) )
    {
      --it;

      if ( (*it)->includes(pos) )
        found = true;
    }

  if (found)
    return it;
  else
    return m_info_box.end();
} // item_information_layer::find_info_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Find an item, given a position.
 * \param pos The position where the item is searched.
 */
bear::engine::base_item* ptb::item_information_layer::find_item
( const bear::universe::position_type& pos ) const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  bear::engine::world::msg_pick_items msg(pos);
  glob.send_message( "world", msg );

  bear::engine::base_item* result = NULL;
  std::list<bear::engine::base_item*>::const_iterator it;

  for (it=msg.items.begin(); (it!=msg.items.end()) && (result==NULL); ++it)
    if ( !is_handled(*it) )
      result = *it;

  return result;
} // item_information_layer::find_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if we have an info_box on a given item.
 * \param item The item to search.
 */
bool ptb::item_information_layer::is_handled
( const bear::engine::base_item* item ) const
{
  bool result = false;

  info_box_list::const_iterator it;

  for (it=m_info_box.begin(); !result && (it!=m_info_box.end()); ++it )
    result = (*it)->has_item(item);

  return result;
} // item_information_layer::is_handled()
