/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file level_starting_layer.cpp
 * \brief Implmentation of the ptb::level_starting_layer class.
 * \author Julien Jorge
 */
#include "ptb/layer/level_starting_layer.hpp"

#include "engine/font_factory.hpp"
#include "engine/game.hpp"
#include "engine/string_base.hpp"
#include "ptb/export.hpp"

GUI_LAYER_EXPORT( level_starting_layer, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::level_starting_layer::level_starting_layer()
  : m_visibility_duration(5), m_fade_time(3), m_elapsed_time(0),
    m_active(true),
    m_level_name(NULL,
                 bear::engine::font_factory::create
                 ("font/level_name-42x50.png")),
    m_act( NULL, m_level_name.get_font() ),
    m_fill_color
  ( bear::engine::game::get_instance().current_level_globals().get_image
    ("gfx/color/black.png") ),
    m_thumb(NULL), m_thumb_position(622, 178), m_thumb_zoom_max(1.1),
    m_thumb_zoom(1), m_thumb_zoom_direction(0.02), m_thumb_angle(0),
    m_strip( bear::engine::game::
             get_instance().current_level_globals().get_image
             ("gfx/color/darkred.png") ),
    m_strip_position(0, 78),
    m_game_name( NULL,
                 bear::engine::font_factory::create
                 ("font/fixed_white-7x12.tga") )
{
  m_fill_color.set_size( get_size() );
  m_strip.set_size( get_size().x, 200 );

  m_level_name.set_position( 50, 100 );
  m_level_name.set_size( 394, 100 );

  m_act.set_auto_size(true);
  m_game_name.set_auto_size(true);

  m_game_name.set_text( bear::engine::game::get_instance().get_name() );
  m_game_name.set_position( m_thumb_position.x - m_game_name.width() - 5 - 128,
                            m_strip_position.y + m_strip.height() + 5 );

  bear::engine::game::get_instance().set_pause( true );
} // level_starting_layer::level_starting_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::level_starting_layer::~level_starting_layer()
{
  if ( m_thumb != NULL )
    delete m_thumb;
} // level_starting_layer::~level_starting_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the layer.
 */
void ptb::level_starting_layer::start()
{
  std::string level_name =
    bear::engine::game::get_instance().get_current_level().get_name();
  std::string::size_type pos = level_name.find_last_of('_');

  std::string sub_act = "1";
  std::string act("act");

  bear::engine::string_base::get_instance().get_string( act );

  // extract the level name and the act number
  if ( pos != std::string::npos )
    {
      if ( pos + 1 != level_name.size() )  
        sub_act = level_name.substr( pos+1 );

      level_name = level_name.substr(0, pos);
    }

  m_act.set_text( act + ' ' + sub_act );
  m_act.set_position
    ( (get_size().x - m_act.width()) / 2,
      get_size().y * 3 / 4 - m_act.get_font()->get_size().y / 2 );
                      
  // get the thumbnail
  const std::string thumb_name = "gfx/thumb/" + level_name + ".png";

  bear::engine::string_base::get_instance().get_string( level_name );

  m_level_name.set_text( level_name );

  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  if ( glob.image_exists( thumb_name ) )
    m_thumb = new bear::visual::sprite( glob.get_image( thumb_name ) );
} // level_starting_layer::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the components of the layer.
 * \param elapsed_time Elapsed time since the last call.
 */
void
ptb::level_starting_layer::progress( bear::universe::time_type elapsed_time )
{
  if ( m_elapsed_time < m_visibility_duration )
    {
      if ( m_elapsed_time > m_fade_time )
        {
          m_fill_color.set_alpha_blend
            ( 1 - (m_elapsed_time - m_fade_time)
              / (m_visibility_duration - m_fade_time) );

          if ( m_thumb != NULL )
            m_thumb->set_size
              ( (unsigned int)
                (m_thumb->clip_rectangle().width * m_thumb_zoom),
                (unsigned int)
                (m_thumb->clip_rectangle().height * m_thumb_zoom) );

          if ( m_thumb_zoom > 0 )
            {
              m_thumb_zoom += m_thumb_zoom_direction;

              if ( m_thumb_zoom + m_thumb_zoom_direction > m_thumb_zoom_max )
                m_thumb_zoom_direction = -m_thumb_zoom_direction;
            }

          m_thumb_angle += 0.1;
          m_strip_position.x -= 10;
          m_level_name.set_position( m_level_name.get_position().x - 10,
                                     m_level_name.get_position().y );

          m_game_name.set_visible(false);
        }

      m_elapsed_time += elapsed_time;
    }
  else if ( m_active )
    {
      m_active = false;
      bear::engine::game::get_instance().set_pause( false );
    }
} // level_starting_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the components of the layer.
 * \param screen The screen on which we render the components.
 */
void ptb::level_starting_layer::render( bear::visual::screen& screen )
{
  if ( m_active )
    {
      screen.render( claw::math::coordinate_2d<int>(0, 0), m_fill_color );
      screen.render( m_strip_position, m_strip );

      if ( m_thumb != NULL )
        screen.render( m_thumb_position - m_thumb->get_size() / 2, *m_thumb,
                       m_thumb_angle );

      m_level_name.render(screen);
      m_act.render(screen);
      m_game_name.render(screen);
    }
} // level_starting_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a key had been pressed.
 * \param key The value of the pressed key.
 * \remark This layer pretends to treat all keys, so we can maintain the pause
 *         mode.
 */
bool
ptb::level_starting_layer::key_pressed( bear::input::keyboard::key_code key )
{
  return m_active;
} // level_starting_layer::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a joystick button had been pressed.
 * \param button The value of the pressed button.
 * \param joy_index The index of the joystick.
 * \remark This layer pretends to treat all keys, so we can maintain the pause
 *         mode.
 */
bool ptb::level_starting_layer::button_pressed
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  return m_active;
} // level_starting_layer::button_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been pressed.
 * \param pos The current position of the cursor.
 * \param key The value of the pressed button.
 * \remark This layer pretends to treat all keys, so we can maintain the pause
 *         mode.
 */
bool ptb::level_starting_layer::mouse_pressed
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  return m_active;
} // level_starting_layer::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button is maintained.
 * \param pos The current position of the cursor.
 * \param key The value of the maintained button.
 * \remark This layer pretends to treat all keys, so we can maintain the pause
 *         mode.
 */
bool ptb::level_starting_layer::mouse_maintained
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  return m_active;
} // level_starting_layer::mouse_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been released.
 * \param pos The current position of the cursor.
 * \param key The value of the released button.
 * \remark This layer pretends to treat all keys, so we can maintain the pause
 *         mode.
 */
bool ptb::level_starting_layer::mouse_released
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  return m_active;
} // level_starting_layer::mouse_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that the mouse has been moved.
 * \param pos The new position of the mouse.
 * \remark This layer pretends to treat all keys, so we can maintain the pause
 *         mode.
 */
bool ptb::level_starting_layer::mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  return m_active;
} // level_starting_layer::mouse_move()
