/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file log_layer.cpp
 * \brief Implementation of the ptb::log_layer class.
 * \author Julien Jorge
 */
#include "ptb/layer/log_layer.hpp"

#include "engine/font_factory.hpp"
#include "text/font.hpp"
#include "ptb/export.hpp"

#include <claw/logger.hpp>
#include <claw/string_algorithm.hpp>

GUI_LAYER_EXPORT( log_layer, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::log_layer::log_layer()
  : m_visible(false), m_lines(15), m_append(false)
{
  bear::gui::static_text::font_type font =
    bear::engine::font_factory::create( "font/fixed_white-7x12.tga" );

  claw::math::coordinate_2d<unsigned int> pos( font->get_size() );

  for (unsigned int i=0; i!=m_lines.size(); ++i)
    {
      m_lines[i] = new bear::gui::static_text( NULL, font );
      m_lines[i]->set_auto_size( true );
      m_lines[i]->set_position( pos );
      pos.y += font->get_size().y;
    }

  claw::logger.merge(this);
} // log_layer::log_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::log_layer::~log_layer()
{
  claw::logger.remove(this);

  for (unsigned int i=0; i!=m_lines.size(); ++i)
    delete m_lines[i];
} // log_layer::~log_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the layer on a screen.
 * \param screen The screen on which we draw.
 */
void ptb::log_layer::render( bear::visual::screen& screen )
{
  if (m_visible)
    for (unsigned int i=0; i!=m_lines.size(); ++i)
      m_lines[i]->render(screen);
} // log_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a keyboard key has been pressed.
 * \param key The value of the pressed key.
 */
bool ptb::log_layer::key_pressed( bear::input::keyboard::key_code key )
{
  bool result = true;

  if ( key == bear::input::keyboard::kc_F6 )
    m_visible = !m_visible;
  else
    result = false;

  return result;
} // log_layer::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write a string in the stream.
 * \param str The string to write.
 */
void ptb::log_layer::write( const std::string& str )
{
  std::list<std::string> lines;
  unsigned int i;
  std::list<std::string>::const_iterator it;

  claw::text::split(lines, str, '\n');

  if ( lines.size() > m_lines.size() )
    {
      i = m_lines.size();
      it=lines.end();

      do
	{
	  --i;
	  --it;
	  m_lines[i]->set_text(*it);
	}
      while( i>0 );
    }
  else if ( !lines.empty() )
    {
      unsigned int keep = m_lines.size() - lines.size();

      if ( m_append )
	++keep;

      unsigned int j = m_lines.size() - keep;

      for (i=0; i!=keep; ++i, ++j)
	m_lines[i]->set_text( m_lines[j]->get_text() );

      it = lines.begin();

      if (m_append)
	{
	  m_lines[i-1]->set_text( m_lines[i-1]->get_text() + *it );
	  ++it;
	}

      for ( ; it!=lines.end(); ++it, ++i )
	m_lines[i]->set_text(*it);
    }

  if ( str.size() > 0 )
    m_append = ( str[str.size() - 1] != '\n' );
} // log_layer::flush()
