/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file status_layer.cpp
 * \brief Implementation of the ptb::status_layer class.
 * \author Julien Jorge
 */
#include "ptb/layer/status_layer.hpp"

#include "engine/game.hpp"
#include "generic_items/timer_item.hpp"
#include "ptb/item/plee/plee.hpp"
#include "ptb/export.hpp"

GUI_LAYER_EXPORT( status_layer, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::status_layer::player_status::player_status()
  : score(0), lifes(0), energy(0), stones(0)
{

} // status_layer::player_status::player_status()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor
 */
ptb::status_layer::status_layer()
  : m_first_player( bear::engine::player::player_name(1) ),
    m_second_player( bear::engine::player::player_name(2) ),
    m_font_1(bear::engine::font_factory::create("font/fixed_yellow-14x28.png")),
    m_font_2(bear::engine::font_factory::create("font/fixed_red-14x28.png")),
    m_text_stones( NULL, m_font_1),
    m_text_time( NULL, m_font_1),
    m_sprite_energy(NULL), m_sprite_no_energy(NULL), m_sprite_stone(NULL)
{
  m_text_stones.set_auto_size(true);
  m_text_stones.set_text("0");
  m_text_stones.set_position( 0,0 );

  m_text_time.set_auto_size(true);
  m_text_time.set_text("00:00");
  m_text_time.set_position( (get_size().x - m_text_time.width()) / 2 ,10 );
  
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();
  const bear::visual::image& image_resource = glob.get_image("gfx/status.tga");

  m_sprite_no_energy =
    new bear::visual::sprite
    ( image_resource, claw::math::rectangle<unsigned int>( 0, 0, 100, 20 ) );

  m_sprite_energy =
    new bear::visual::sprite
    ( image_resource, claw::math::rectangle<unsigned int>( 0, 20, 100, 20 ) );

  m_sprite_stone =
    new bear::visual::sprite
    ( image_resource, claw::math::rectangle<unsigned int>( 0, 40, 24, 24 ) );
} // status_layer::status_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::status_layer::~status_layer()
{
  delete m_sprite_no_energy;
  delete m_sprite_energy;
  delete m_sprite_stone;
} // status_layer::~status_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the players status and update local values.
 */
void ptb::status_layer::progress( bear::universe::time_type elapsed_time )
{
  progress_time(elapsed_time);

  // get the player instance
  if (m_first_player)
    {
      ptb::plee* plee1 = dynamic_cast<ptb::plee*>(&(*m_first_player));

      if (plee1 != NULL)
        {
          m_first_status.energy = plee1->get_energy();
          m_first_status.stones = plee1->get_stones_count();	
        }
    }

  // the same with the second player
  if (m_second_player)
    {
      ptb::plee* plee2 = dynamic_cast<ptb::plee*>(&(*m_second_player));

      if (plee2 != NULL)
        {
          m_second_status.energy = plee2->get_energy();
          m_second_status.stones = plee2->get_stones_count();	
        }
    }
} // status_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the time text.
 */
void ptb::status_layer::progress_time( bear::universe::time_type elapsed_time )
{
  bear::timer_item::msg_get_value msg;
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  if ( glob.send_message( "level_timer", msg ) )
    {
      bear::universe::time_type time = msg.value;
      unsigned int sec = (unsigned int)time % 60;
      unsigned int min = (unsigned int)time / 60;
  
      std::ostringstream oss_time;
      if ( min < 10 )
        oss_time << " ";
      oss_time << min;
  
      oss_time << ":";
    
      if ( sec < 10 ) 
        oss_time << "0";
      oss_time << sec;
  
      if ( ( min == 0 ) && ( sec <= 30 ) && 
	   ( (unsigned int)((time - 60*min - sec )*4)%2 == 1) ) 
	{
	  m_text_time.set_font(m_font_2);
	  m_text_time.set_text(oss_time.str());
	}
      else
	{   
	  m_text_time.set_font(m_font_1);
	  m_text_time.set_text(oss_time.str());
	}
    }
} // status_layer::progress_time()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the players status.
 * \param screen The screen on which we will render.
 */
void ptb::status_layer::render( bear::visual::screen& screen )
{
  // render first player status
  if (m_first_player)
    render_player1( screen, m_first_status);

  // render second player status (if any)
  if (m_second_player)
    render_player2( screen, m_second_status);

  m_text_time.render( screen );
} // status_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the arrow of first player .
 * \param screen The screen on which we'll draw.
 * \param status Status of the player.
 */
void ptb::status_layer::render_player1
( bear::visual::screen& screen, player_status status)
{
  std::ostringstream oss_stones;
  oss_stones << status.stones;
  m_text_stones.set_text(oss_stones.str());
  m_text_stones.set_position( 40, 10);
  m_text_stones.render( screen );

  claw::math::coordinate_2d<unsigned int> pos(10,12);
  screen.render( pos, *m_sprite_stone );
  pos.y = 45;
  screen.render( pos, *m_sprite_no_energy );

  unsigned int width = 0;
  width = ( 100 * status.energy ) / plee::s_max_energy;
  m_sprite_energy->set_width(width);
  screen.render( pos, *m_sprite_energy );
} // status_layer::render_player1()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the arrow of second player .
 * \param screen The screen on which we'll draw.
 * \param status Status of the player.
 */
void ptb::status_layer::render_player2
( bear::visual::screen& screen, player_status status)
{
  std::ostringstream oss_stones;
  oss_stones << status.stones;
  m_text_stones.set_text(oss_stones.str());
  m_text_stones.set_position
    ( get_size().x - 40 - m_text_stones.get_size().x, 10);
  m_text_stones.render( screen );

  claw::math::coordinate_2d<unsigned int> pos(get_size().x - 35,12);
  screen.render( pos, *m_sprite_stone );
  pos.x = get_size().x - 110;
  pos.y = 45;
  screen.render( pos, *m_sprite_no_energy );

  unsigned int width = 0;
  width = ( 100 * status.energy ) / plee::s_max_energy;
  m_sprite_energy->set_width(width);
  screen.render( pos, *m_sprite_energy );
} // status_layer::render_player1()
