/*
    Plee The Bear - Image Cutter

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file application.cpp
 * \brief Implementation of the application class.
 * \author Julien Jorge
 */
#include "application.hpp"

#include <fstream>
#include <iostream>
#include <sstream>
#include <iomanip>

#include <claw/png.hpp>
#include <claw/jpeg.hpp>
#include <claw/targa.hpp>
#include <claw/bitmap.hpp>
#include <claw/assert.hpp>
#include <claw/string_algorithm.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
application::application( int& argc, char** &argv )
  : claw::application(argc, argv), m_quit(false), m_image(NULL)
{
  check_arguments( argc, argv );
} // application::application()

/*----------------------------------------------------------------------------*/
/**
 * \brief Run the application : split the image.
 */
int application::run()
{
  int result = 0;

  if ( !m_quit && (m_image != NULL) )
    {
      if ( m_script.empty() )
        process_script( std::cin );
      else
        {
          std::ifstream f( m_script.c_str() );

          if ( f )
            process_script(f);
          else
            {
              std::cerr << "Can't read '" << m_script << "'." << std::endl;
              result = -1;
            }
        }
    }

  return result;
} // application::run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Print program usage.
 */
void application::help() const
{
  m_arguments.help();
} // application::help()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check program arguments.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
void application::check_arguments( int& argc, char** &argv )
{
  m_arguments.add( "-d", "--output",
                   "Path to the directory where we save the images.", true );
  m_arguments.add( "-s", "--script",
                   "The script to read to create the textures."
                   " Default is standard input.", true );
  m_arguments.add("-i", "--image",
                  "The image from which we create the textures.", false);
  m_arguments.add("-h", "--help", "Print this message and exit.", true);

  m_arguments.parse( argc, argv );

  if ( m_arguments.get_bool("-h") || !m_arguments.required_fields_are_set() )
    {
      help();
      m_quit = true;
    }
  else
    {
      if ( m_arguments.has_value("--output") )
        m_output_folder = m_arguments.get_string("--output") + "/";
      else
        m_output_folder = "./";

      if ( m_arguments.has_value("--script") )
        m_script = m_arguments.get_string("--script");

      std::string image_name = m_arguments.get_string("--image");
      std::ifstream f( image_name.c_str() );

      if (f)
        try { m_image = new claw::graphic::image(f); }
        catch ( std::exception& e ) { std::cerr << e.what() << std::endl; }

      if ( m_image == NULL )
        std::cerr << "Can't read image '" << image_name << "'." << std::endl;
    }
} // application::check_arguments()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the script and create the textures.
 * \param is The script.
 */
void application::process_script( std::istream& is ) const
{
  std::string line;

  while ( claw::text::getline( is, line ) )
    if ( !line.empty() )
      if ( line[0] != '#')
        process_file( line, is );
} // application::process_script()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a texture.
 * \param target_name The name of the resulting texture.
 * \param is The script.
 */
void application::process_file
( const std::string& target_name, std::istream& is ) const
{
  std::string line;

  bool stop = false;

  while ( !stop )
    {
      if ( claw::text::getline( is, line )  )
        if ( !line.empty() )
          stop = (line[0] != '#');
    }

  if ( !line.empty() )
    {
      std::istringstream iss(line);
      unsigned int w, h;

      if ( iss >> w >> h )
        {
          claw::graphic::image texture(w, h);
          copy_texture_parts( texture, is );
          save_image( texture, target_name );
        }
    }
} // application::process_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy the parts of the source image in the resulting texture.
 * \param texture The texture where we copy the parts of the image.
 * \param is The script.
 */
void application::copy_texture_parts
( claw::graphic::image& texture, std::istream& is ) const
{
  std::string line;

  bool stop = false;

  while ( !stop )
    if ( !claw::text::getline( is, line ) )
      stop = true;
    else if ( !line.empty() )
      if (line[0] != '#')
        {
          if (line == "write")
            stop = true;
          else
            {
              std::istringstream iss(line);
              int x, y;
              unsigned int w, h;
              claw::math::coordinate_2d<unsigned int> pos;
              
              if ( iss >> x >> y >> w >> h >> pos.x >> pos.y )
                {
                  claw::graphic::image part(w, h);
                  part.partial_copy
                    ( *m_image, claw::math::coordinate_2d<int>(-x, -y) );
                  texture.partial_copy( part, pos );
                }
            }
        }
} // application::copy_texture_parts()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the resulting texture.
 * \param texture The texture to save.
 * \param target_name The name of the texture file.
 */
void application::save_image
( const claw::graphic::image& texture, const std::string& target_name ) const
{
  std::string::size_type pos = target_name.find_last_of('.');
  bool saved = true;

  if ( pos == std::string::npos )
    saved = false;
  else
    {
      std::ofstream f( (m_output_folder + target_name).c_str() );

      if ( !f )
        saved = false;
      else
        {
          std::string ext = target_name.substr(pos+1);

          if ( ext == "tga" )
            claw::graphic::targa::writer( texture, f, true );
          else if ( ext == "png" )
            claw::graphic::png::writer( texture, f );
          else if ( ext == "jpg" )
            claw::graphic::jpeg::writer( texture, f );
          else if ( ext == "bmp" )
            claw::graphic::bitmap::writer( texture, f );
          else
            saved = false;

          f.close();
        }
    }

  if ( !saved )
    std::cerr << "Can't save texture '" << target_name << "'." << std::endl;
} // application::save_image()
