/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file picture.cpp
 * \brief Implementation of the bear::gui::picture class.
 * \author Julien Jorge
 */
#include "gui/picture.hpp"

#include "visual/scene_sprite.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component containing this picture.
 * \param pict The sprite to display.
 * \remark The sprite will be deleted in the constructor.
 */
bear::gui::picture::picture( visual_component* owner, visual::sprite* pict )
  : visual_component(owner), m_sprite(pict), m_stretch(false)
{
  if ( m_sprite != NULL )
    set_size( m_sprite->width(), m_sprite->height() );
} // picture::picture()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::gui::picture::~picture()
{
  if (m_sprite != NULL)
    delete m_sprite;
} // picture::~picture()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the displayed picture.
 * \param pict The picture to display.
 */
void bear::gui::picture::set_picture( visual::sprite* pict )
{
  if (m_sprite != NULL)
    delete m_sprite;

  m_sprite = pict;
} // picture::set_picture()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set if the sprite is stretched to fill the control.
 * \param b The new value.
 */
void bear::gui::picture::set_stretch( bool b )
{
  m_stretch = b;

  if (m_sprite != NULL)
    {
      if (m_stretch)
        m_sprite->set_size( get_size() );
      else
        m_sprite->set_size( m_sprite->clip_rectangle().width,
                            m_sprite->clip_rectangle().height );
    }
} // picture::set_stretch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the scene elements of the component.
 * \param e (out) The scene elements.
 */
void bear::gui::picture::display
( std::list<visual::scene_element>& e ) const
{
  if ( m_sprite != NULL )
    e.push_back( visual::scene_sprite(left(), top(), *m_sprite) );
} // picture::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called after the component has been resized.
 */
void bear::gui::picture::on_resized()
{
  visual_component::on_resized();

  if (m_sprite != NULL)
    if (m_stretch)
      m_sprite->set_size( get_size() );
} // visual_component::on_resized()
