/*
  Bear Engine - Editor library

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/item_class_xml_parser.tpp
 * \brief Implementation of the template methods of bf::item_class_xml_parser.
 * \author Julien Jorge
 */
#include "bf/xml/exception.hpp"
#include "bf/wx_facilities.hpp"

#include "bf/type_field_set.hpp"
#include "bf/type_field_interval.hpp"

#include <sstream>
#include <claw/logger.hpp>
#include <claw/string_algorithm.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a field for the item.
 * \param name The name of the field.
 * \param ft The type of the field.
 * \param node The node to explore.
 *
 * \b Template \b parameters
 * - \a T The type of the bounds of the interval. Ignored if
 *   \a RangeAvailable == false.
 * - \a SetAvailable Tell if the field takes its value in a set of values.
 * - \a RangeAvailable Tell if the field takes its value in an interval.
 */
template<typename T, bool SetAvailable, bool RangeAvailable>
bf::type_field* bf::item_class_xml_parser::create_field
( const std::string& name, type_field::field_type ft,
  const wxXmlNode* node ) const
{
  type_field* result(NULL);
  std::list<std::string> prec;
  std::string desc;
  std::string default_value;

  for ( ; node!=NULL; node = node->GetNext() )
    if ( node->GetName() == wxT("before") )
      prec.push_front( read_before(node) );
    else if ( node->GetName() == wxT("description") )
      desc = wx_to_std_string( node->GetNodeContent().Trim() );
    else if ( node->GetName() == wxT("default_value") )
      default_value = wx_to_std_string( node->GetNodeContent().Trim() );
    else if ( result == NULL )
      {
        if ( node->GetName() == wxT("set") )
          {
            if ( SetAvailable )
              {
                std::list<std::string> set;
                read_set( node, set );
                result = new type_field_set( name, ft, set );
              }
            else
              claw::logger << claw::log_warning << "Ignored node '"
                           << wx_to_std_string(node->GetName()) << "' " << name
                           << "\n" << wx_to_std_string(node->GetNodeContent())
                           << std::endl;
          }
        else if ( node->GetName() == wxT("interval") )
          {
            if ( RangeAvailable )
              {
                std::pair<T, T> range = read_interval<T>( node );
                result =
                  new type_field_interval<T>( name, range.first, range.second );
              }
            else
              claw::logger << claw::log_warning << "Ignored node '"
                           << wx_to_std_string(node->GetName()) << "' " << name
                           << "\n" << wx_to_std_string(node->GetNodeContent())
                           << std::endl;
          }
        else if ( node->GetName() != wxT("comment") )
          claw::logger << claw::log_warning << "Ignored node '"
                       << wx_to_std_string(node->GetName()) << "' " << name
                       << "\n" << wx_to_std_string(node->GetNodeContent())
                       << std::endl;
      }
    else if ( node->GetName() != wxT("comment") )
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "' " << name << "\n"
                   << wx_to_std_string(node->GetNodeContent()) << std::endl;

  if ( result == NULL )
    result = new type_field( name, ft );

  for ( std::size_t p=desc.find_first_of("\t\n"); p!=std::string::npos;
        p=desc.find_first_of("\t\n", p) )
    desc[p] = ' ';

  claw::text::trim(desc, " \t\n");
  claw::text::squeeze(desc, " \t\n");

  result->set_preceding( prec );
  result->set_description( desc );
  result->set_default_value( default_value );

  return result;
} // item_class_xml_parser::create_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the "field" property of a node "before".
 * \param node The node to explore.
 */
template<typename T>
std::pair<T,T>
bf::item_class_xml_parser::read_interval( const wxXmlNode* node ) const
{
  std::pair<T,T> result;
  wxString val;

  if ( !node->GetPropVal( wxT("from"), &val ) )
    throw xml::missing_property("from");

  std::istringstream iss( wx_to_std_string(val) );

  if ( !(iss >> result.first) )
    throw xml::bad_value( wx_to_std_string(val) );

  if ( !node->GetPropVal( wxT("to"), &val ) )
    throw xml::missing_property("to");

  iss.clear();
  iss.str( wx_to_std_string(val) );

  if ( !(iss >> result.second) )
    throw xml::bad_value( wx_to_std_string(val) );

  return result;
} // item_class_xml_parser::read_interval()
