/*
    Bear Engine - Editor library

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/xml/code/item_instance_field_node.cpp
 * \brief Implementation of the bf::xml::item_instance_field_node class.
 * \author Julien Jorge
 */
#include "bf/xml/item_instance_field_node.hpp"

#include "bf/item_instance.hpp"

#include <claw/assert.hpp>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Read an xml node "field".
 * \param item (out) The item class for we read the hierarchy.
 * \param node The node.
 */
void bf::xml::item_instance_field_node::read
( item_instance& item, const wxXmlNode* node ) const
{
  CLAW_PRECOND( node!=NULL );
  CLAW_PRECOND( node->GetName() == wxT("field") );

  const std::string field_name( reader_tool::read_string(node, wxT("name")) );
  const item_class& the_class(item.get_class());

  if ( !the_class.has_field(field_name) )
    claw::logger << claw::log_warning << "Unknown field '" << field_name
                 << "' in '" << the_class.get_class_name() << "'" << std::endl;
  else
    {
      const type_field& field = the_class.get_field(field_name);
      load_field( item, field, node->GetChildren() );
    }
} // item_instance_field_node::read()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write an xml node "field".
 * \param item The item instance to write.
 * \param f The field to save.
 * \param os The stream in which we write.
 */
void bf::xml::item_instance_field_node::write
( const item_instance& item, const type_field& f, std::ostream& os ) const
{
  os << "<field name='" << f.get_name() << "'>\n";
  save_field( item, f, os );
  os << "</field>\n";
} // item_instance_field_node::write()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the value of a field.
 * \param item The item in which we set the field.
 * \param f The field to load.
 * \param node The node to parse.
 */
void bf::xml::item_instance_field_node::load_field
( item_instance& item, const type_field& f, const wxXmlNode* node ) const
{
  if ( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        load_value_list<integer_type>( item, f.get_name(), "integer", node );
        break;
      case type_field::u_integer_field_type:
        load_value_list<u_integer_type>
          ( item, f.get_name(), "u_integer", node );
        break;
      case type_field::real_field_type:
        load_value_list<real_type>( item, f.get_name(), "real", node );
        break;
      case type_field::boolean_field_type:
        load_value_list<bool_type>( item, f.get_name(), "bool", node );
        break;
      case type_field::string_field_type:
        load_value_list<string_type>( item, f.get_name(), "string", node );
        break;
      case type_field::sprite_field_type:
        load_value_list<sprite>( item, f.get_name(), "sprite", node );
        break;
      case type_field::animation_field_type:
        load_value_list<animation_file_type>
          ( item, f.get_name(), "animation_file", node );
        break;
      case type_field::item_reference_field_type:
        load_value_list<item_reference_type>
          ( item, f.get_name(), "item_reference", node );
        break;
      case type_field::font_field_type:
        load_value_list<font_file_type>
          ( item, f.get_name(), "font_file", node );
        break;
      case type_field::sample_field_type:
        load_value_list<sample_file_type>
          ( item, f.get_name(), "sample_file", node );
        break;
      }
  else
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        load_value<integer_type>( item, f.get_name(), "integer", node );
        break;
      case type_field::u_integer_field_type:
        load_value<u_integer_type>( item, f.get_name(), "u_integer", node );
        break;
      case type_field::real_field_type:
        load_value<real_type>( item, f.get_name(), "real", node );
        break;
      case type_field::boolean_field_type:
        load_value<bool_type>( item, f.get_name(), "bool", node );
        break;
      case type_field::string_field_type:
        load_value<string_type>( item, f.get_name(), "string", node );
        break;
      case type_field::sprite_field_type:
        load_value<sprite>( item, f.get_name(), "sprite", node );
        break;
      case type_field::animation_field_type:
        load_value<animation_file_type>
          ( item, f.get_name(), "animation_file", node );
        break;
      case type_field::item_reference_field_type:
        load_value<item_reference_type>
          ( item, f.get_name(), "item_reference", node );
        break;
      case type_field::font_field_type:
        load_value<font_file_type>
          ( item, f.get_name(), "font_file", node );
        break;
      case type_field::sample_field_type:
        load_value<sample_file_type>
          ( item, f.get_name(), "sample_file", node );
        break;
      }
} // item_instance_field_node::load_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field.
 * \param item The item in which we take the field.
 * \param f The field to save.
 * \param os The stream in which we write.
 */
void bf::xml::item_instance_field_node::save_field
( const item_instance& item, const type_field& f, std::ostream& os ) const
{
  if ( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        save_value_list<integer_type>( os, f.get_name(), item, "integer" );
        break;
      case type_field::u_integer_field_type:
        save_value_list<u_integer_type>( os, f.get_name(), item, "u_integer" );
        break;
      case type_field::real_field_type:
        save_value_list<real_type>( os, f.get_name(), item, "real" );
        break;
      case type_field::boolean_field_type:
        save_value_list<bool_type>( os, f.get_name(), item, "bool" );
        break;
      case type_field::string_field_type:
        save_string_list( os, f.get_name(), item );
        break;
      case type_field::sprite_field_type:
        save_sprite_list( os, f.get_name(), item );
        break;
      case type_field::animation_field_type:
        save_animation_list( os, f.get_name(), item );
        break;
      case type_field::item_reference_field_type:
        save_value_list<item_reference_type>
          ( os, f.get_name(), item, "item_reference" );
        break;
      case type_field::font_field_type:
        save_font_list( os, f.get_name(), item );
        break;
      case type_field::sample_field_type:
        save_sample_list( os, f.get_name(), item );
        break;
      }
  else
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        save_value<integer_type>( os, f.get_name(), item, "integer" );
        break;
      case type_field::u_integer_field_type:
        save_value<u_integer_type>( os, f.get_name(), item, "u_integer" );
        break;
      case type_field::real_field_type:
        save_value<real_type>( os, f.get_name(), item, "real" );
        break;
      case type_field::boolean_field_type:
        save_value<bool_type>( os, f.get_name(), item, "bool" );
        break;
      case type_field::string_field_type:
        save_string( os, f.get_name(), item );
        break;
      case type_field::sprite_field_type:
        save_sprite( os, f.get_name(), item );
        break;
      case type_field::animation_field_type:
        save_animation( os, f.get_name(), item );
        break;
      case type_field::item_reference_field_type:
        save_value<item_reference_type>
          ( os, f.get_name(), item, "item_reference" );
        break;
      case type_field::font_field_type:
        save_font( os, f.get_name(), item );
        break;
      case type_field::sample_field_type:
        save_sample( os, f.get_name(), item );
        break;
      }
} // item_instance_field_node::save_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'string'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_string
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  string_type v;
  item.get_value( field_name, v );
  string_to_xml( os, v );
} // item_instance_field_node::save_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'sprite'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_sprite
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  sprite spr;
  item.get_value( field_name, spr );
  xml::value_to_xml<sprite>::write(os, spr);
} // item_instance_field_node::save_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'animation'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_animation
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  animation_file_type anim;
  item.get_value( field_name, anim );
  xml::value_to_xml<animation_file_type>::write(os, "animation_file", anim);
} // item_instance_field_node::save_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'font'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_font
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  font_file_type anim;
  item.get_value( field_name, anim );
  xml::value_to_xml<font_file_type>::write(os, "font_file", anim);
} // item_instance_field_node::save_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'sample'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_sample
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  sample_file_type anim;
  item.get_value( field_name, anim );
  xml::value_to_xml<sample_file_type>::write(os, "sample_file", anim);
} // item_instance_field_node::save_sample()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of list of type 'string'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 * \param node_name The name of the xml node.
 */
void bf::xml::item_instance_field_node::save_string_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  std::list<string_type> v;
  std::list<string_type>::const_iterator it;

  item.get_value( field_name, v );

  for (it=v.begin(); it!=v.end(); ++it)
    string_to_xml( os, *it );
} // item_instance_field_node::save_string_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'list of sprite'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_sprite_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  std::list<sprite> spr;
  std::list<sprite>::const_iterator it;

  item.get_value( field_name, spr );

  for (it=spr.begin(); it!=spr.end(); ++it)
    xml::value_to_xml<sprite>::write(os, *it);
} // item_instance_field_node::save_sprite_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'list of animation'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_animation_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  std::list<animation_file_type> anim;
  std::list<animation_file_type>::const_iterator it;

  item.get_value( field_name, anim );

  for (it=anim.begin(); it!=anim.end(); ++it)
    xml::value_to_xml<animation_file_type>::write(os, "animation_file", *it);
} // item_instance_field_node::save_animation_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'list of font'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_font_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  std::list<font_file_type> f;
  std::list<font_file_type>::const_iterator it;

  item.get_value( field_name, f );

  for (it=f.begin(); it!=f.end(); ++it)
    xml::value_to_xml<font_file_type>::write(os, "font_file", *it);
} // item_instance_field_node::save_font_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'list of sample'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::xml::item_instance_field_node::save_sample_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  std::list<sample_file_type> f;
  std::list<sample_file_type>::const_iterator it;

  item.get_value( field_name, f );

  for (it=f.begin(); it!=f.end(); ++it)
    xml::value_to_xml<sample_file_type>::write(os, "sample_file", *it);
} // item_instance_field_node::save_sample_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a node for a string.
 * \param os The stream in which we save the value.
 * \param v The value to write.
 */
void bf::xml::item_instance_field_node::string_to_xml
( std::ostream& os, const string_type& v ) const
{
  const std::string quote("&apos;");
  std::string s( v.get_value() );
  std::string::size_type p( s.find_first_of('\'') );

  while ( p!=std::string::npos )
    {
      s.replace(p, 1, quote);
      p = s.find_first_of('\'', p + quote.size() );
    }

  os << "<string value='" << s << "'/>\n";
} // item_instance_field_node::string_to_xml()
