/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file base_bonus.cpp
 * \brief Implementation of the ptb::base_bonus class.
 * \author Sebastien Angibaud
 */
#include "ptb/base_bonus.hpp"

#include "universe/collision_info.hpp"
#include "generic_items/decorative_item.hpp"
#include "ptb/item/little_plee.hpp"
#include "engine/game.hpp"

#include "engine/export.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
ptb::base_bonus::base_bonus()
  : m_type(unknown_type), m_stock(0), m_bonus_given(false)
{

} // base_bonus::base_bonus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <unsigned int>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::base_bonus::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool result = true;

  if ( name == "base_bonus.stock" )
    m_stock = value;
  else
    result = super::set_u_integer_field( name, value );

  return result;
} // base_bonus::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <string>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::base_bonus::set_string_field
( const std::string& name, const std::string& value )
{
  bool result = true;

  if ( name == "base_bonus.type" )
    {
      if ( value == "air_power" )
        m_type = air_power;
      else if ( value == "fire_power" )
        m_type = fire_power;
      else if ( value == "water_power" )
        m_type = water_power;
      else if ( value == "invincibility_power" )
        m_type = invincibility_power;
      else if ( value == "stones_stock" )
        m_type = stones_stock;
      else if ( value == "stones_big_stock" )
        m_type = stones_big_stock;
      else if ( value == "one_more_life" )
        m_type = one_more_life;
      else if ( value == "increase_max_energy" )
        m_type = increase_max_energy;
    }
  else
    result = super::set_string_field( name, value );

  return result;
} // base_bonus::set_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is well initialized.
 */
bool ptb::base_bonus::is_valid() const
{
  return (m_type != unknown_type) && super::is_valid();
} // base_bonus::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void ptb::base_bonus::get_visual
( std::list<bear::engine::scene_visual>& visuals ) const
{
  add_visual( m_animation, visuals );
} // base_bonus::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::base_bonus::progress( bear::universe::time_type elapsed_time )
{
  m_animation.next(elapsed_time);
} // base_bonus::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if the bonus is given.
 */ 
bool ptb::base_bonus::get_bonus_given() const
{
  return m_bonus_given;
} // base_bonus::get_bonus_given

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the type of the bonus.
 */ 
ptb::base_bonus::base_bonus_type ptb::base_bonus::get_type() const
{
  return m_type;
} // base_bonus::get_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the type of the bonus.
 * \param t The new type of bonus.
 */
void ptb::base_bonus::set_type(base_bonus_type t)
{
  m_type =t;
} // base_bonus::set_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the stock.
 */ 
unsigned int ptb::base_bonus::get_stock() const
{
  return m_stock;
} // base_bonus::get_stock()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the stock.
 * \param stock The new stock.
 */ 
void ptb::base_bonus::set_stock(unsigned int stock)
{
  m_stock = stock;
} // base_bonus::set_stock()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the animation.
 * \param anim the animation.
 */ 
void ptb::base_bonus::set_animation(const bear::visual::animation& anim)
{
  m_animation = anim;
} // base_bonus::set_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void ptb::base_bonus::to_string( std::string& str ) const
{
  std::ostringstream oss;

  super::to_string(str);

  oss << "\ntype: ";
  if ( m_type == air_power )
    oss << "air_power";
  else if ( m_type == fire_power )
    oss << "fire_power";
  else if ( m_type == water_power )
    oss << "water_power";
  else if ( m_type == invincibility_power )
    oss << "invincibility_power";
  else if ( m_type == stones_stock )
    oss << "stones_stock";
  else if ( m_type == stones_big_stock )
    oss << "stones_big_stock";
  else if ( m_type == one_more_life )
    oss << "one_more_life";
  else if ( m_type == increase_max_energy )
    oss << "increase_max_energy";
  oss << "\n";
  oss << "stock: " << m_stock << "\n";
  str += oss.str();
} // base_bonus::to_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the bonus.
 * \param p Plee.
 */
void ptb::base_bonus::give_bonus(plee* p)
{
  if ( m_type == air_power )
    game_variables::set_air_power(p->get_index(), true);
  else if ( m_type == fire_power )
    game_variables::set_fire_power(p->get_index(), true);
  else if ( m_type == water_power )
    game_variables::set_water_power(p->get_index(), true);
  else if ( m_type == invincibility_power )
    p->set_invincible(true);
  else if ( ( m_type == stones_stock ) || ( m_type == stones_big_stock ) )
    game_variables::set_stones_count
      ( p->get_index(), 
        game_variables::get_stones_count(p->get_index()) + m_stock);
  else if( m_type == one_more_life )
    create_little_plee(p->get_rendering_attributes().is_mirrored());
  else if ( m_type == increase_max_energy )
    {
      game_variables::set_max_energy
        ( p->get_index(), 
          game_variables::get_max_energy(p->get_index()) + m_stock);
    
      p->receive_energy( game_variables::get_max_energy(p->get_index()) );
    }

  m_bonus_given = true;
  kill();
} // base_bonus::give_bonus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the bonus.
 * \param plee_orientation The orientation of Plee.
 */
void ptb::base_bonus::create_little_plee( bool plee_orientation )
{
  little_plee* new_little_plee = new little_plee;

  new_item( *new_little_plee );

  CLAW_ASSERT(new_little_plee->is_valid(), 
	      "The little_plee isn't correctly initialized" );

  new_little_plee->set_center_of_mass(get_center_of_mass());
  new_little_plee->set_positive_orientation(plee_orientation);
  new_little_plee->set_z_position( get_z_position()+1 );
  new_little_plee->start();
} // base_bonus::create_little_plee()
