/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file code/level_variables.hpp
 * \brief Implementation of the ptb::level_variables class.
 * \author Julien Jorge
 */
#include "ptb/level_variables.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief A function that gets a level variable.
 * \param lvl The level.
 * \param n The name of the variable.
 * \param def The default value to return if the variable is not set.
 */
template<typename T>
static T ptb_level_variables_get_value
( const bear::engine::level& lvl, const std::string& n, const T& def )
{
  if ( lvl.get_level_variables().template exists<T>(n) )
    return lvl.get_level_variables().template get<T>(n);
  else
    return def;
} // ptb_level_variables_get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of players on an exit.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_players_on_exit( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "players_on_exit", (unsigned int)0 );
} // level_variables::get_players_on_exit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of players on an exit.
 * \param lvl The level in which we take the variable.
 * \param c The number of players.
 */
void ptb::level_variables::set_players_on_exit
( bear::engine::level& lvl, unsigned int c )
{
  lvl.get_level_variables().set<unsigned int>( "players_on_exit", c );
} // level_variables::set_players_on_exit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of killed enemies.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_killed_enemies( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "killed_ennemies", (unsigned int)0 );
} // level_variables::get_killed_enemies()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of killed enemies.
 * \param lvl The level in which we take the variable.
 * \param c The number of killed enemies.
 */
void ptb::level_variables::set_killed_enemies
( bear::engine::level& lvl, unsigned int nb )
{
  lvl.get_level_variables().set<unsigned int>( "killed_enemies", nb );
} // level_variables::set_killed_enemies()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of enemies in the level.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_enemy_count( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "enemy_count", (unsigned int)0 );
} // level_variables::get_enemy_count()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of enemies in the level.
 * \param lvl The level in which we take the variable.
 * \param c The number of killed enemies.
 */
void ptb::level_variables::set_enemy_count
( bear::engine::level& lvl, unsigned int nb )
{
  lvl.get_level_variables().set<unsigned int>( "enemy_count", nb );
} // level_variables::set_enemy_count()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the azelnut has been found.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::get_azelnut( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value( lvl, "azelnut", false );
} // level_variables::get_azelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set if the azelnut has been found.
 * \param lvl The level in which we take the variable.
 * \param v Found or not.
 */
void ptb::level_variables::set_azelnut( bear::engine::level& lvl, bool v )
{
  lvl.get_level_variables().set<bool>( "azelnut", v );
} // level_variables::set_azelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of honeypot found.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_honeypots_found( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "honeypots_found", (unsigned int)0 );
} // level_variables::get_honeypots_found()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of honeypots found.
 * \param lvl The level in which we take the variable.
 * \param c The number of found honeypots.
 */
void ptb::level_variables::set_honeypots_found
( bear::engine::level& lvl, unsigned int nb )
{
  lvl.get_level_variables().set<unsigned int>( "honeypots_found", nb );
} // level_variables::set_honeypots_found()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if we are leaving the level.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::is_exiting( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value( lvl, "exiting", false );
} // level_variables::is_exiting()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set that we are leaving the level.
 * \param lvl The level in which we take the variable.
 */
void ptb::level_variables::set_exiting( bear::engine::level& lvl )
{
  lvl.get_level_variables().set<bool>( "exiting", true );
} // level_variables::set_exiting()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the owl was met.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::owl_is_met( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value( lvl, "met_the_owl", false );
} // level_variables::owl_is_met()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set that the owl is met.
 * \param lvl The level in which we take the variable.
 */
void ptb::level_variables::meet_the_owl( bear::engine::level& lvl )
{
  lvl.get_level_variables().set<bool>( "met_the_owl", true );
} // level_variables::meet_the_owl()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the time has been already exceeded.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::get_time_exceeded( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value( lvl, "time_exceeded", false );
} // level_variables::get_time_exceeded()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set if the time has been already exceeded.
 * \param lvl The level in which we take the variable.
 */
void ptb::level_variables::set_time_exceeded( bear::engine::level& lvl )
{
  lvl.get_level_variables().set<bool>( "time_exceeded", true );
} // level_variables::set_time_exceeded()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the time record for the level.
 * \param lvl The level in which we take the variable.
 */
bear::universe::time_type
ptb::level_variables::get_time_record( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "time_record", (bear::universe::time_type)0 );
} // level_variables::get_time_record()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the time record for the level.
 * \param lvl The level in which we take the variable.
 * \param t The record.
 */
void ptb::level_variables::set_time_record
( bear::engine::level& lvl, bear::universe::time_type t )
{
  lvl.get_level_variables().set<bear::universe::time_type>
    ( "time_exceeded", t );
} // level_variables::set_time_record()
