/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_password.cpp
 * \brief Implementation of the ptb::frame_password class.
 * \author Julien Jorge
 */
#include "ptb/frame/frame_password.hpp"

#include "ptb/game_variables.hpp"
#include "engine/game.hpp"

#include "ptb/frame/frame_start_menu.hpp"
#include "ptb/item/plee/plee.hpp"

#include <claw/string_algorithm.hpp>
#include <libintl.h>

/*----------------------------------------------------------------------------*/
const unsigned int ptb::frame_password::s_margin = 10;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer owning the window.
 */
ptb::frame_password::frame_password( windows_layer* owning_layer )
  : frame(owning_layer), m_history_index(0)
{
  create_controls();
} // frame_password::frame_password()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool ptb::frame_password::on_key_press( const bear::input::key_info& key )
{
  bool result = true;

  if ( key.is_up() )
    on_up();
  else if ( key.is_down() )
    on_down();
  else if ( key.is_escape() )
    on_back();
  else if ( key.is_enter() )
    validate();
  else
    result = false;

  return result;
} // frame_password::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool ptb::frame_password::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  switch( button )
    {
    case bear::input::joystick::jc_axis_left:
    case bear::input::joystick::jc_axis_right:
      result = false;
      break;
    case bear::input::joystick::jc_axis_up:
      on_up();
      break;
    case bear::input::joystick::jc_axis_down:
      on_down();
      break;
    case bear::input::joystick::jc_button_1:
      on_back();
      break;
    default:
      validate();
      break;
    }

  return result;
} // frame_password::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Validate the selected action.
 */
void ptb::frame_password::validate()
{
  std::vector<std::string> command;
  std::string text( m_password->get_text() );
  m_password->set_text("");

  claw::text::trim(text);
  claw::text::squeeze(text, " ");
  claw::text::split(command, text, ' ');

  if ( !command.empty() )
    {
      execute_command(command);

      if( m_history.empty() || (m_history.back() != text) )
        m_history.push_back(text);

      m_history_index = m_history.size();
    }
} // frame_password::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the static text components.
 */
void ptb::frame_password::create_controls()
{
  bear::gui::static_text::font_type font = frame::get_font();

  create_input( font );
  create_back( font );

  set_size( m_back->right() + s_margin, m_back->bottom() + s_margin );
} // frame_password::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the text input control.
 * \param f The font to use for the text.
 */
void ptb::frame_password::create_input( bear::gui::static_text::font_type f )
{
  m_password =
    new bear::gui::text_input( &get_frame(), f, claw::graphic::red_pixel );

  m_password->set_position( s_margin, s_margin );
  m_password->set_width(300);
  m_password->set_border_color( get_border_color() );
} // frame_password::create_input()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "back" buttons.
 * \param f The font to use for the texts.
 */
void ptb::frame_password::create_back( bear::gui::static_text::font_type f )
{
  m_back =
    new button_with_text( *this, std::mem_fun(&frame_password::on_back), this );

  m_back->set_font( f );
  m_back->set_auto_size( true );
  m_back->set_text( gettext("Back") );

  const unsigned int y = m_password->bottom() + s_margin;

  if ( s_margin + m_back->width() > m_password->right() )
    m_back->set_position( s_margin, y );
  else
    m_back->set_position( m_password->right() - m_back->width(), y );
} // frame_language::create_back()

/*----------------------------------------------------------------------------*/
/**
 * \brief Exit.
 */
bool ptb::frame_password::on_back()
{
  close_window();
  return true;
} // frame_password::on_back()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the previous command in the history.
 */
void ptb::frame_password::on_up()
{
  if (m_history_index > 0)
    {
      --m_history_index;
      m_password->set_text(m_history[m_history_index]);
    }
} // frame_password::on_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the next command in the history.
 */
void ptb::frame_password::on_down()
{
  if (m_history_index < m_history.size())
    {
      ++m_history_index;

      if ( m_history_index == m_history.size() )
        m_password->set_text("");
      else
        m_password->set_text(m_history[m_history_index]);
    }
} // frame_password::on_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Execute a command entered by the user.
 * \param command The tokens read on the command line.
 */
void ptb::frame_password::execute_command
( const std::vector<std::string>& command ) const
{
  CLAW_PRECOND( !command.empty() );

  if ( command[0] == "goto" )
    command_load_level(command);
  else if ( command[0] == "give" )
    command_give(command, 1);
  else if ( command[0] == "givetoo" )
    command_give(command, 2);
  else if ( command[0] == "gamevar" )
    command_game_variable(command);
  else
    claw::logger << claw::log_warning << "Unknow command '" << command[0]
                 << "'" << std::endl;
} // frame_password::execute_command()

/*----------------------------------------------------------------------------*/
/**
 * \brief Execute a 'goto' command.
 * \param command The tokens read on the command line.
 */
void ptb::frame_password::command_load_level
( const std::vector<std::string>& command ) const
{
  CLAW_PRECOND( command[0] == "goto" );

  if ( command.size() == 2 )
    {
      bear::engine::game& game = bear::engine::game::get_instance();

      if ( game.level_exists(command[1]) )
        {
          game_variables::set_next_level_name( command[1] );
          show_window( new frame_start_menu(m_owning_layer, true) );
        }
      else
        claw::logger << claw::log_warning << "goto: no level named '"
                     << command[1] << "'." << std::endl;
    }
  else
    claw::logger << claw::log_warning << "goto: one parameter required, "
                 << (command.size()-1) << " given." << std::endl;
} // frame_password::command_load_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Execute a 'give' command.
 * \param command The tokens read on the command line.
 * \param index The index of the player to which we give something.
 */
void ptb::frame_password::command_give
( const std::vector<std::string>& command, unsigned int index ) const
{
  CLAW_PRECOND( (command[0] == "give") || (command[0] == "givetoo") );

  if ( command.size() == 2 )
    {
      plee::get_instance_message msg;
      m_owning_layer->get_level_globals().send_message
        (plee::player_name(index),msg);
      plee* player = msg.get_instance();

      if ( player != NULL )
        {
          if ( command[1] == "stones" )
            game_variables::set_stones_count
              (index, game_variables::get_stones_count(index) + 50);
          else if ( command[1] == "air" )
            game_variables::set_air_power(index, true);
          else if ( command[1] == "fire" )
            game_variables::set_fire_power(index, true);
          else if ( command[1] == "water" )
            game_variables::set_water_power(index, true);
          else if ( command[1] == "energy" )
            player->receive_energy( game_variables::get_max_energy(index) );
          else if ( command[1] == "oxygen" )
            player->receive_oxygen( plee::s_max_oxygen );
          else if ( command[1] == "life" )
            game_variables::set_lives_count
              (index, game_variables::get_lives_count(index) + 1);
          else
            claw::logger << claw::log_warning << "give: unknown gift '"
                         << command[1] << "'." << std::endl;
        }
      else
        claw::logger << claw::log_warning << "give: can't find player #"
                     << index << '.' << std::endl;
    }
  else
    claw::logger << claw::log_warning << "give: one parameter required, "
                 << (command.size()-1) << " given." << std::endl;
} // frame_password::command_give()

/*----------------------------------------------------------------------------*/
/**
 * \brief Execute a 'gamevar' command.
 * \param command The tokens read on the command line.
 */
void ptb::frame_password::command_game_variable
( const std::vector<std::string>& command ) const
{
  CLAW_PRECOND( command[0] == "gamevar" );

  if ( command.size() == 3 )
    {
      if ( command[1] == "int" )
        command_game_variable<int>( command[2] );
      else if ( command[1] == "u_int" )
        command_game_variable<unsigned int>( command[2] );
      else if ( command[1] == "real" )
        command_game_variable<double>( command[2] );
      else if ( command[1] == "string" )
        command_game_variable<std::string>( command[2] );
      else if ( command[1] == "bool" )
        command_game_variable<bool>( command[2] );
      else
        claw::logger << claw::log_warning << "gamevar: '" << command[1]
                     << "' is not a valid type." << std::endl;
    }
  else
    claw::logger << claw::log_warning
                 << "gamevar: incorrect parameters."
                 << " Must be 'gamevar type name=value'."
                 << std::endl;
} // frame_password::command_game_variable()
