/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file plee.cpp
 * \brief Implementation of the ptb::plee class.
 * \author Julien Jorge
 */
#include "ptb/item/plee/plee.hpp"

#include <sstream>

#include "engine/game.hpp"
#include "engine/world.hpp"
#include "engine/message/transition_effect_message.hpp"

#include "ptb/defines.hpp"
#include "ptb/monster.hpp"
#include "ptb/player_action.hpp"

#include "ptb/item/shared_camera.hpp"
#include "ptb/item/small_honeypot.hpp"
#include "ptb/item/stone/air_fire_stone.hpp"
#include "ptb/item/stone/air_fire_water_stone.hpp"
#include "ptb/item/stone/air_water_stone.hpp"
#include "ptb/item/stone/air_stone.hpp"
#include "ptb/item/stone/water_fire_stone.hpp"
#include "ptb/item/stone/water_stone.hpp"
#include "ptb/item/stone/fire_stone.hpp"
#include "ptb/item/azelnut.hpp"
#include "ptb/item/plee/state_roar.hpp"
#include "ptb/item/plee/state_walk.hpp"
#include "ptb/item/plee/state_idle.hpp"
#include "ptb/item/plee/state_jump.hpp"
#include "ptb/item/plee/state_fall.hpp"
#include "ptb/item/plee/state_dead.hpp"
#include "ptb/item/plee/state_game_over.hpp"
#include "ptb/item/plee/state_run.hpp"
#include "ptb/item/plee/state_slap.hpp"
#include "ptb/item/plee/state_start_jump.hpp"
#include "ptb/item/plee/state_vertical_jump.hpp"
#include "ptb/item/plee/state_look_upward.hpp"
#include "ptb/item/plee/state_crouch.hpp"
#include "ptb/item/plee/state_captive.hpp"
#include "ptb/item/plee/state_throw.hpp"
#include "ptb/item/plee/state_maintain.hpp"
#include "ptb/item/plee/state_wait.hpp"
#include "ptb/item/plee/state_start_cling.hpp"
#include "ptb/item/plee/state_cling.hpp"
#include "ptb/item/plee/state_clung_jump.hpp"
#include "ptb/item/plee/state_start_hang.hpp"
#include "ptb/item/plee/state_hang.hpp"
#include "ptb/item/plee/state_sink.hpp"
#include "ptb/item/plee/state_swimming.hpp"
#include "ptb/item/plee/state_float.hpp"
#include "ptb/item/plee/state_injured.hpp"

#include "ptb/transition_effect/game_over_effect.hpp"
#include "ptb/transition_effect/invincibility_effect.hpp"

BASE_ITEM_IMPLEMENT( plee, ptb )

/*----------------------------------------------------------------------------*/
const double ptb::plee::s_max_oxygen = 10000;
const double ptb::plee::s_max_fire_gauge = 10000;
const double ptb::plee::s_max_ice_gauge = 10000;
const double ptb::plee::s_mass = 100;
const double ptb::plee::s_density = 2;

const unsigned int ptb::plee::s_states_cardinality = 26;
const unsigned int ptb::plee::s_gauges_cardinality = 3;
const int ptb::plee::s_speed_to_run = 580;
const double ptb::plee::s_oxygen_loss_speed = 200;
const double ptb::plee::s_oxygen_inspiration_speed = 1500;
const double ptb::plee::s_fire_loss_speed = 200;
const double ptb::plee::s_fire_increase_speed = 1500;
const double ptb::plee::s_ice_loss_speed = 200;
const double ptb::plee::s_ice_increase_speed = 1500;

const bear::universe::size_type ptb::plee::s_max_halo_height = 64;
const bear::universe::size_type ptb::plee::s_max_halo_hand_width = 35;

const bear::universe::time_type ptb::plee::s_time_to_crouch = 0.5;
const bear::universe::time_type ptb::plee::s_time_to_look_upward = 0.5;
const bear::universe::time_type ptb::plee::s_time_to_wait = 3;
const bear::universe::time_type ptb::plee::s_time_to_jump = 1;
const bear::universe::time_type ptb::plee::s_time_to_run = 1.2;
const bear::universe::time_type ptb::plee::s_time_to_start_throw = 0.34;
const bear::universe::time_type ptb::plee::s_max_time_to_cling = 0.3;
const bear::universe::time_type ptb::plee::s_max_time_to_hang = 1;
const bear::universe::time_type ptb::plee::s_max_time_air_float = 1;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::plee::get_instance_message::get_instance_message()
  : m_plee_instance(NULL)
{

} // get_instance_message::get_instance_message()

/*----------------------------------------------------------------------------*/
/**
 * \brief Re-initialise the message.
 */
void ptb::plee::get_instance_message::clear()
{
  m_plee_instance = NULL;
} // get_instance_message::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to a player.
 * \param that The player to apply the message to.
 */
bool ptb::plee::get_instance_message::apply_to( plee& that )
{
  m_plee_instance = &that;
  return true;
} // get_instance_message::apply_to()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the instance of the player.
 */
ptb::plee*
ptb::plee::get_instance_message::get_instance() const
{
  return m_plee_instance;
} // get_instance_message::get_instance()






/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::plee::plee()
  : m_current_state(roar_state), m_progress(&plee::progress_roar),
    m_last_visual_time(0),
    m_status_look_upward(false), m_status_crouch(false), m_can_cling(false),
    m_halo_animation(NULL), m_halo_hand_animation(NULL),  
    m_move_right(false), m_move_left(false),
    m_right_move_force(0),
    m_index(0), m_nb_bottom_contact(0),
    m_marionette(false), m_animation_to_throw(NULL),
    m_hot_spot_position(0, 0), m_hot_spot_minimum(0, 0),
    m_hot_spot_maximum(0, 0), m_hot_spot_balance_move(0, 0)
    
{
  set_speaker_item(this);
  set_mass(s_mass);
  set_density(s_density);
  set_size( 50, 110 );
  
  set_z_fixed(false);

  set_spot_minimum(-150, -250);
  set_spot_maximum(150, 300);
  set_spot_balance_move(3, 15);

  m_offensive_force = 10;
  m_energy = 100;
  if ( get_index() == 1 )
    m_monster_type = monster::player_1_monster;
  else
    m_monster_type = monster::player_2_monster;
  m_offensive_phase = false;
  m_offensive_coefficients[normal_attack] = 1;
  
  m_gauges.resize(s_gauges_cardinality);
  m_gauges[oxygen_gauge] = new gauge(s_max_oxygen);
  m_gauges[fire_gauge] = new gauge(s_max_fire_gauge);
  m_gauges[ice_gauge] = new gauge(s_max_ice_gauge);

  save_position( get_center_of_mass() );
  m_last_bottom_left = bear::universe::position_type(0,0);

  m_invincible_time = 0;
  m_run_time = 0;

  m_states.resize(s_states_cardinality);
  m_states[walk_state] = new state_walk(this);
  m_states[idle_state] = new state_idle(this);
  m_states[jump_state] = new state_jump(this);
  m_states[fall_state] = new state_fall(this);
  m_states[dead_state] = new state_dead(this);
  m_states[game_over_state] = new state_game_over(this);
  m_states[roar_state] = new state_roar(this);
  m_states[run_state] = new state_run(this);
  m_states[slap_state] = new state_slap(this);
  m_states[start_jump_state] = new state_start_jump(this);
  m_states[vertical_jump_state] = new state_vertical_jump(this);
  m_states[look_upward_state] = new state_look_upward(this);
  m_states[crouch_state] = new state_crouch(this);
  m_states[captive_state] = new state_captive(this);
  m_states[throw_state] = new state_throw(this);
  m_states[wait_state] = new state_wait(this);
  m_states[start_cling_state] = new state_start_cling(this);
  m_states[cling_state] = new state_cling(this);
  m_states[clung_jump_state] = new state_clung_jump(this);
  m_states[start_hang_state] = new state_start_hang(this);
  m_states[hang_state] = new state_hang(this);
  m_states[swimming_state] = new state_swimming(this);
  m_states[sink_state] = new state_sink(this);
  m_states[float_state] = new state_float(this);
  m_states[maintain_state] = new state_maintain(this);
  m_states[injured_state] = new state_injured(this);
} // plee::plee()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::plee::~plee()
{
  unsigned int i;

  for (i=0; i!=s_states_cardinality; ++i)
    delete m_states[i];

  for (i=0; i!=s_gauges_cardinality; ++i)
    delete m_gauges[i];

  if ( m_halo_animation )
    delete m_halo_animation;

  if ( m_halo_hand_animation )
    delete m_halo_hand_animation;

  if ( m_animation_to_throw ) 
    {
      delete m_animation_to_throw;
      m_animation_to_throw = NULL;
    }
} // plee::~plee()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress( bear::universe::time_type elapsed_time )
{
  // THIS CONDITION MUST BE DONE IN FIRST
  if ( ! m_marionette ) 
    progress_input_reader(elapsed_time);
  
  super::progress(elapsed_time);

  m_state_time += elapsed_time;
  m_run_time += elapsed_time;

  if ( m_air_float )
    progress_air_float(elapsed_time);
  
  progress_invincibility(elapsed_time);

  if ( is_crushed() )
    apply_die();
  else
    {
      if ( m_progress != NULL )
        (this->*m_progress)(elapsed_time);

      update_powers();
      progress_spot( elapsed_time );
      progress_gauges( elapsed_time );
      update_orientation();
      m_can_cling = false;

      if ( ( m_current_state == plee::maintain_state ) ||
           ( m_current_state == plee::throw_state ) )
        m_halo_hand_animation->next(elapsed_time);
      
      if ( m_animation_to_throw != NULL )
        m_animation_to_throw->next(elapsed_time);
    }

  m_last_bottom_left = get_bottom_left();
  m_can_throw_power[air_attack] = true;
  m_can_throw_power[fire_attack] = true;
  m_can_throw_power[water_attack] = true;
} // plee::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void
ptb::plee::get_visual( std::list<bear::engine::scene_visual>& visuals ) const
{
  get_visuals_without_invincibility(visuals);

  std::list< std::list<bear::engine::scene_visual> >::const_iterator it;
  
  for ( it = m_last_visuals.begin(); it != m_last_visuals.end(); ++it )
    {
      std::list<bear::engine::scene_visual>::const_iterator it2;
      for (it2 = it->begin(); it2 != it->end(); ++it2 )
        visuals.push_front(*it2);
    }  
} // plee::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the media required by this class.
 */
void ptb::plee::pre_cache()
{
  // plee
  get_level_globals().load_model("model/plee/plee.cm");
    
  // halo for vertical jump
  get_level_globals().load_animation("animation/plee/halo.canim");
  get_level_globals().load_animation("animation/plee/halo_hand.canim");

  // little_plee
  get_level_globals().load_animation("animation/powerup/life_bonus/run.canim");
  get_level_globals().load_animation
    ("animation/powerup/life_bonus/jump.canim");

  // small honeypot
  get_level_globals().load_animation("animation/powerup/small_fire.canim");
  get_level_globals().load_animation("animation/powerup/small_air.canim");
  get_level_globals().load_animation("animation/powerup/small_water.canim");
  
  // stones
  get_level_globals().load_model("model/stones/stone.cm");
  get_level_globals().load_model("model/stones/air_stone.cm");
  get_level_globals().load_model("model/stones/water_stone.cm");
  get_level_globals().load_model("model/stones/fire_stone.cm");
  get_level_globals().load_model("model/stones/air_fire_stone.cm");
  get_level_globals().load_model("model/stones/air_water_stone.cm");
  get_level_globals().load_model("model/stones/water_fire_stone.cm");
  get_level_globals().load_model("model/stones/air_fire_water_stone.cm");
  get_level_globals().load_animation("animation/stones/sliver_1.canim");
  get_level_globals().load_animation("animation/stones/sliver_2.canim");
  get_level_globals().load_animation("animation/stones/sliver_3.canim");
  get_level_globals().load_animation("animation/stones/sliver_4.canim");
  get_level_globals().load_animation("animation/stones/sliver_5.canim");
} // plee::pre_cache()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do post creation actions.
 */
void ptb::plee::build()
{
  super::build();
 
  game_variables::set_lives_count(m_index, 3);

  set_model_actor( get_level_globals().get_model("model/plee/plee.cm") );

  if ( m_index == 1 )
    m_monster_type = monster::player_1_monster;
  else
    {
      m_monster_type = monster::player_2_monster;
      get_rendering_attributes().set_intensity(1, 1, 0.7);
    }
  
  m_air_float = false;
  save_position( get_center_of_mass() );
  start_action_model("idle");
  
  bear::engine::level_globals& glob = get_level_globals();

  m_halo_animation = new bear::visual::animation
    ( glob.get_animation("animation/plee/halo.canim") );

  m_halo_hand_animation = new bear::visual::animation
    ( glob.get_animation("animation/plee/halo_hand.canim") );

  get_level().add_interest_around(this);

  m_can_throw_power[air_attack] = true;
  m_can_throw_power[fire_attack] = true;
  m_can_throw_power[water_attack] = true;
} // plee::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool ptb::plee::is_valid() const
{
  return (m_index != 0) && super::is_valid();
} // player::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <unsigned integer>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::plee::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool result = true;

  if (name == "index")
    set_index(value);
  else
    result = super::set_u_integer_field(name, value);

  return result;
} // player::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the position of the plee.
 * \param p The center of mass to remember.
 */
void ptb::plee::save_position( const bear::universe::position_type& p )
{
  m_saved_position = p;
} // plee::save_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to Plee to start an action.
 * \param a The action to start.
 */
void ptb::plee::start_action( player_action::value_type a )
{
  switch( a )
    {
    case player_action::idle : break;
    case player_action::move_left : break;
    case player_action::move_right : break;
    case player_action::jump :
      m_states[m_current_state]->do_jump(); break;
    case player_action::get_camera : do_get_camera(); break;
    case player_action::slap :
      m_states[m_current_state]->do_slap(); break;
    case player_action::look_upward : do_start_look_upward(); break;
    case player_action::crouch :  do_start_crouch(); break;
    case player_action::drop : do_start_drop(); break;
    case player_action::chain : break;
    case player_action::unchain : break;
    case player_action::throw_stone : do_start_throw(); break;
    case player_action::die : apply_die(); break;
    case player_action::roar : start_action_model("roar"); break;
    case player_action::action_null: break;
    default:
      claw::logger << claw::log_warning << "Action ignored (start): " << a
                   << std::endl;;
    }
} // plee::start_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to the plee to do an action.
 * \param elapsed_time How long the action is done.
 * \param a The action to do.
 */
void ptb::plee::do_action
( bear::universe::time_type elapsed_time, player_action::value_type a )
{
  switch( a )
    {
    case player_action::idle : set_state(idle_state); break;
    case player_action::move_left :
      m_states[m_current_state]->do_move_left(); break;
    case player_action::move_right :
      m_states[m_current_state]->do_move_right(); break;
    case player_action::jump : break;
    case player_action::get_camera : break;
    case player_action::slap : break;
    case player_action::look_upward :
      m_states[m_current_state]->do_continue_look_upward(); break;
    case player_action::crouch :
      m_states[m_current_state]->do_continue_crouch(); break;
    case player_action::drop : break;
    case player_action::chain :
      m_states[m_current_state]->chain(); break;
    case player_action::unchain :
      m_states[m_current_state]->unchain(); break;
    case player_action::throw_stone : break;
    case player_action::die : break;
    case player_action::action_null: break;
    default:
      claw::logger << claw::log_warning << "Action ignored: " << a
                   << std::endl;
    }
} // plee::do_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to plee to stop an action.
 * \param a The action to stop.
 */
void ptb::plee::stop_action( player_action::value_type a )
{
  switch( a )
    {
    case player_action::idle : break;
    case player_action::move_left : break;
    case player_action::move_right : break;
    case player_action::jump :
      m_states[m_current_state]->do_stop_vertical_jump(); break;
    case player_action::get_camera : break;
    case player_action::slap : break;
    case player_action::look_upward : do_stop_look_upward(); break;
    case player_action::crouch :
      do_stop_crouch(); break;
    case player_action::throw_stone :
      m_states[m_current_state]->do_stop_throw(); break;
    case player_action::drop : 
      m_states[m_current_state]->do_stop_throw(); break;
    case player_action::chain : break;
    case player_action::unchain : break;
    case player_action::die : break;
    case player_action::action_null: break;
    default:
      claw::logger << claw::log_warning << "Action ignored (stop): " << a
                   << std::endl;
    }
} // plee::stop_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start a new action of model.
 */
void ptb::plee::start_action_model(const std::string& action)
{
  start_model_action(action);
} // plee::start_action_model()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a gauge.
 * \param t The type of the gauge
 */
const ptb::gauge* ptb::plee::get_gauge(gauge_type t) const
{
  if ( (unsigned int)t >= s_gauges_cardinality )
    {
      claw::logger << claw::log_verbose << "It is not a valid gauge_type."
                   << std::endl;
      return NULL;
    }
  else
    return m_gauges[t];
} // plee::get_gauge()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the ratio of the length of preparation of the jump.
 */
bear::universe::time_type ptb::plee::get_jump_time_ratio() const
{
  return m_jump_time_ratio;
} // plee::get_jump_time_ratio()

/*----------------------------------------------------------------------------*/
/**
 * \brief Updtae the ratio of the preparation of throw state.
 */
void ptb::plee::update_throw_time_ratio()
{
  m_throw_time_ratio = m_state_time / s_time_to_start_throw;
} // plee::update_throw_time_ratio()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of throw_down.
 * \param value The state of throw_down.
 */
void ptb::plee::set_throw_down(bool value)
{
  m_throw_down = value;
} // plee::set_throw_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to throw an azelnut
 */
void ptb::plee::throw_azelnut()
{
  m_current_object = azelnut_object;
  m_animation_to_throw = new bear::visual::animation
    ( get_level_globals().get_animation
      ("animation/owl/azelnut.canim") );
  m_states[m_current_state]->do_start_throw();
} // plee::throw_azelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the value of throw_up.
 * \param value The state of throw_up.
 */
void ptb::plee::set_throw_up(bool value)
{
  m_throw_up = value;
} // plee::set_throw_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if Plee is a marionette.
 */
bool ptb::plee::is_a_marionette() const
{
  return m_marionette;
} // plee::is_a_marionette()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn on/off the marionette state of Plee.
 * \param b The new value of the state.
 */
void ptb::plee::set_marionette( bool b )
{
  m_marionette = b;
} // plee::set_marionette()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn on/off the air_float status of Plee.
 * \param b The new value of the state.
 */
void ptb::plee::set_air_float( bool b )
{
  m_air_float = b;
  m_air_float_time = 0; 
} // plee::set_air_float()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the look up status.
 */
void ptb::plee::set_status_look_upward(bool status)
{
  m_status_look_upward = status;
} // plee::set_status_look_upward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the look up status.
 */
bool ptb::plee::get_status_look_upward() const
{
  return m_status_look_upward;
} // plee::get_status_look_upward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the crouch status.
 */
void ptb::plee::set_status_crouch(bool status)
{
  m_status_crouch = status;
} // plee::set_status_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the crouch status.
 */
bool ptb::plee::get_status_crouch() const
{
  return m_status_crouch;
} // plee::get_status_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee start to look up.
 */
void ptb::plee::do_start_look_upward()
{
  m_status_look_upward = true;
  m_look_upward_time = 0;
  m_states[m_current_state]->do_look_upward();
} // plee::do_start_look_upward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee start to crouch.
 */
void ptb::plee::do_start_crouch()
{
  m_status_crouch = true;
  m_crouch_time = 0;
  m_states[m_current_state]->do_crouch();
} // plee::do_start_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee start to crouch.
 */
void ptb::plee::do_start_throw()
{
  if ( ( game_variables::get_stones_count(m_index) > 0 ) 
       && m_air_stones.empty() )
    {
      m_current_object = stone_object;

      if ( game_variables::get_air_power(m_index) )
        if ( game_variables::get_fire_power(m_index) )
          if ( game_variables::get_water_power(m_index) )
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/air_fire_water_stone.canim") );
          else
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/air_fire_stone.canim") );
        else
          if ( game_variables::get_water_power(m_index) )
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/air_water_stone.canim") );
          else
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/air_stone.canim") );
      else
        if ( game_variables::get_fire_power(m_index) )
          if ( game_variables::get_water_power(m_index) )
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/water_fire_stone.canim") );
          else
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/fire_stone.canim") );
        else
          if ( game_variables::get_water_power(m_index) )
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/water_stone.canim") );
          else
            m_animation_to_throw = new bear::visual::animation
              ( get_level_globals().get_animation
                ("animation/stones/stone.canim") );

      m_states[m_current_state]->do_start_throw();
    }
  else
    apply_blast_stone();
} // plee::do_start_throw()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee start to drop.
 */
void ptb::plee::do_start_drop()
{
  if ( has_a_power() )
    m_states[m_current_state]->do_start_drop();
} // plee::do_start_drop()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set if Plee can cling.
 * \param status The new status.
 */
void ptb::plee::set_can_cling(bool status)
{
  m_can_cling = status;
} // plee::set_can_cling()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee receives energy.
 *
 * \param energy The energy received.
 */
void ptb::plee::receive_energy(double energy)
{
  m_energy += energy;

  if ( m_energy > game_variables::get_max_energy(m_index) )
    m_energy = game_variables::get_max_energy(m_index);
} // plee::receive_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the invincible status.
 * \param invincible The new invincible status.
 */
void ptb::plee::set_invincible(const bool invincible)
{
  super::set_invincible(invincible);
  
  if ( m_invincible )
    {
      m_invincible_time = 0;
      m_last_visual_time = 0;
      m_last_visuals.clear();

      bear::engine::transition_effect_message<invincibility_effect> msg;
      get_level_globals().send_message
        ( PTB_TRANSITION_EFFECT_DEFAULT_TARGET_NAME, msg );
    }
} // plee::set_invincible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee receives oxygen.
 *
 * \param oxygen The oxygen received.
 */
void ptb::plee::receive_oxygen(double oxygen)
{
  m_gauges[oxygen_gauge]->add_value(oxygen);
} // plee::receive_oxygen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a air_stone in the air stone set.
 *
 * \param s The stone to add.
 */
void ptb::plee::add_air_stone(ptb::stone* s)
{
  m_air_stones.insert(s);
} // plee::add_air_stone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove a air_stone in the air stone set.
 *
 * \param s The stone to remove.
 */
void ptb::plee::remove_air_stone( ptb::stone* s)
{
  std::set< ptb::stone* >::iterator it;
  it = m_air_stones.find(s);

  if ( it != m_air_stones.end() )
    m_air_stones.erase(it);
} // plee::remove_air_stone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if Plee can throw a given power.
 * \param p The status.
 * \param a The corresponding attack
 */
void ptb::plee::can_throw_power( bool b, monster::attack_type a)
{
  m_can_throw_power[a] = b;
} // plee::can_throw_power()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the coordinates of the point to target to (for camera or active
 *        region).
 */
bear::universe::position_type ptb::plee::hot_spot() const
{
  bear::universe::position_type p;
  
  bear::universe::size_type w,h;
  get_action("idle")->get_max_size(w,h);
  const bear::universe::size_box_type half_plee( w / 2, h / 2 );

  if ( get_rendering_attributes().is_mirrored() )
    {
      p = get_bottom_right();
      p.x -= half_plee.x;
    }
  else
    {
      p = get_bottom_left();
      p.x += half_plee.x;
    }

  p.y += half_plee.y;

  return p + m_hot_spot_position;
} // player::hot_spot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the index of this player.
 */
unsigned int ptb::plee::get_index() const
{
  return m_index;
} // player::get_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the index of this player.
 * \param index The index of the player.
 */
void ptb::plee::set_index( unsigned int index )
{
  m_index = index;
  set_name( player_name(index) );
  set_player_index(index);
} // player::set_index()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the player associated to an index.
 * \param player_index The index of the player.
 */
std::string ptb::plee::player_name( unsigned int player_index )
{
  std::ostringstream oss;
  oss << "player_" << player_index;

  return oss.str();
} // player::player_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the minimum acceptable value for the position of the hot spot.
 * \param x Minimum value on the X-axis.
 * \param y Minimum value on the Y-axis.
 */
void ptb::plee::set_spot_minimum
( bear::universe::coordinate_type x, bear::universe::coordinate_type y )
{
  m_hot_spot_minimum.set
    ( std::min(x, m_hot_spot_maximum.x), std::min(y, m_hot_spot_maximum.y) );
} // player::set_spot_minimum()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the maximum acceptable value for the position of the hot spot.
 * \param x Maximum value on the X-axis.
 * \param y Maximum value on the Y-axis.
 */
void ptb::plee::set_spot_maximum
( bear::universe::coordinate_type x, bear::universe::coordinate_type y )
{
  m_hot_spot_maximum.set
    ( std::max(x, m_hot_spot_minimum.x), std::max(y, m_hot_spot_minimum.y) );
} // player::set_spot_maximum()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the maximum movement allowed when balancing the hot spot.
 * \param x Maximum movement on the X-axis.
 * \param y Maximum movement on the Y-axis.
 */
void ptb::plee::set_spot_balance_move
( bear::universe::coordinate_type x, bear::universe::coordinate_type y )
{
  m_hot_spot_balance_move.set(std::abs(x), std::abs(y));
} // player::set_spot_balance_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the gap of the spot.
 * \param gap The vector of gap.
 */
void ptb::plee::add_spot_gap( const bear::universe::position_type& gap )
{
  m_hot_spot_position += gap;

  if ( m_hot_spot_position.x > m_hot_spot_maximum.x )
    m_hot_spot_position.x = m_hot_spot_maximum.x;
  else if ( m_hot_spot_position.x < m_hot_spot_minimum.x )
    m_hot_spot_position.x = m_hot_spot_minimum.x;

  if ( m_hot_spot_position.y > m_hot_spot_maximum.y )
    m_hot_spot_position.y = m_hot_spot_maximum.y;
  else if ( m_hot_spot_position.y < m_hot_spot_minimum.y )
    m_hot_spot_position.y = m_hot_spot_minimum.y;
} // player::add_spot_gap()

/*----------------------------------------------------------------------------*/
/**
 * \brief Balance the gap of the spot.
 * \param x Tell if we balance on the X-axis.
 * \param y Tell if we balance on the Y-axis.
 */
void ptb::plee::balance_spot(bool x, bool y)
{
  if ( x )
    {
      if ( m_hot_spot_position.x > m_hot_spot_balance_move.x )
        m_hot_spot_position.x -= m_hot_spot_balance_move.x;
      else if ( m_hot_spot_position.x < -m_hot_spot_balance_move.x)
        m_hot_spot_position.x += m_hot_spot_balance_move.x;
      else
        m_hot_spot_position.x = 0;
    }

  if ( y )
    {
      if ( m_hot_spot_position.y > m_hot_spot_balance_move.y )
        m_hot_spot_position.y -= m_hot_spot_balance_move.y;
      else if ( m_hot_spot_position.y < -m_hot_spot_balance_move.y)
        m_hot_spot_position.y += m_hot_spot_balance_move.y;
      else
        m_hot_spot_position.y = 0;
    }
} // player::balance_spot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Drops one power.
 */
void ptb::plee::apply_drop()
{
  std::vector<std::string> powers;
  
  if ( game_variables::get_air_power(m_index) && 
       m_can_throw_power[monster::air_attack] )
    powers.push_back("air");
  
  if ( game_variables::get_fire_power(m_index) && 
       m_can_throw_power[monster::air_attack] )
    powers.push_back("fire");
  
  if ( game_variables::get_water_power(m_index) && 
       m_can_throw_power[monster::air_attack] )
    powers.push_back("water");
  
  if ( ! powers.empty() ) 
    {
      unsigned int a = (unsigned int)(powers.size() * rand() / RAND_MAX);
      std::string drop_power;
      
      if ( a < powers.size() )
	drop_power = powers[a];
      else
	drop_power = powers[0];
      
      if ( drop_power == "air" )
	{
	  m_current_object = air_honeypot_object;
	  m_animation_to_throw = new bear::visual::animation
	    ( get_level_globals().get_animation
	      ("animation/powerup/small_air.canim") );
	  game_variables::set_air_power(m_index, false);
	  m_air_float = false;
	}
      else if ( drop_power == "fire" )
	{
	  m_current_object = fire_honeypot_object;
	  m_animation_to_throw = new bear::visual::animation
	    ( get_level_globals().get_animation
	      ("animation/powerup/small_fire.canim") );
	  game_variables::set_fire_power(m_index, false);
	}
      else if ( drop_power == "water" )
	{
	  m_current_object = water_honeypot_object;
	  m_animation_to_throw = new bear::visual::animation
	    ( get_level_globals().get_animation
	      ("animation/powerup/small_water.canim") );
	  game_variables::set_water_power(m_index, false);
	}

      update_powers();
      m_states[m_current_state]->do_start_throw();
    }
  else
    start_action_model("idle");
} // plee::apply_drop()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action move_right.
 */
void ptb::plee::apply_move_right()
{
  m_move_right = true;
  add_internal_force( bear::universe::force_type(m_right_move_force, 0) );
} // plee::apply_move_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action move_left.
 */
void ptb::plee::apply_move_left()
{
  m_move_left = true;
  add_internal_force( bear::universe::force_type(-m_right_move_force, 0) );
} // plee::apply_move_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply impulsion of a jump.
 */
void ptb::plee::apply_impulse_jump()
{
  if ( m_current_state == float_state )
    add_internal_force( bear::universe::force_type(0, 7500000) );
  else
    {
      add_external_force( bear::universe::force_type(0, 5750000) );
      add_internal_force( bear::universe::force_type(0, 5750000) );
    }
} // plee::apply_impulse_jump()


/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action jump.
 */
void ptb::plee::apply_jump()
{
  m_right_move_force = 100000;
  set_state(plee::jump_state);
  m_progress = &plee::progress_jump;
} // plee::apply_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action start_jump.
 */
void ptb::plee::apply_start_jump()
{
  set_state(plee::start_jump_state);
  m_progress = &plee::progress_start_jump;
  m_halo_animation->reset();
} // plee::apply_start_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action vertical_jump.
 *
 */
void ptb::plee::apply_vertical_jump()
{
  m_right_move_force = 50000;
  m_run_time = 0;

  if ( m_state_time >= s_time_to_jump )
    m_jump_time_ratio = 1;
  else
    m_jump_time_ratio = m_state_time / s_time_to_jump;

  set_state(plee::vertical_jump_state);
  m_progress = &plee::progress_vertical_jump;
} // plee::apply_vertical_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action fall.
 *
 */
void ptb::plee::apply_fall()
{
  m_right_move_force = 100000;
  set_state(plee::fall_state);
  m_progress = &plee::progress_fall;
} // plee::apply_fall()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action idle.
 *
 */
void ptb::plee::apply_idle()
{
  m_right_move_force = 200000;
  set_state(plee::idle_state);
  m_progress = &plee::progress_idle;
} // plee::apply_idle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action walk.
 *
 */
void ptb::plee::apply_walk()
{
  m_right_move_force = 200000;
  set_state(plee::walk_state);
  m_progress = &plee::progress_walk;
} // plee::apply_walk()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action run.
 *
 */
void ptb::plee::apply_run()
{
  m_right_move_force = 450000;
  set_state(plee::run_state);
  m_progress = &plee::progress_run;
} // plee::apply_run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action slap.
 *
 */
void ptb::plee::apply_slap()
{
  m_right_move_force = 200000;
  set_state(plee::slap_state);
  m_progress = &plee::progress_slap;
} // plee::apply_slap()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn the attack mode.
 *
 */
void ptb::plee::apply_attack()
{
  set_offensive_phase(true);
  set_defensive_power(monster::normal_attack,true);
} // plee::apply_attack()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action chain.
 *
 */
void ptb::plee::apply_chain()
{
  set_state(plee::captive_state);
  m_progress = &plee::progress_captive;
} // plee::apply_chain()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action unchain.
 *
 */
void ptb::plee::apply_unchain()
{
  set_state(plee::idle_state);
  m_progress = &plee::progress_idle;
} // plee::apply_unchain()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action throw.
 *
 */
void ptb::plee::apply_throw()
{
  if ( get_current_action_name() == "throw_and_fall" )
    m_right_move_force = 100000;
  else
    m_right_move_force = 200000;
  
  set_state(plee::throw_state);
  m_progress = &plee::progress_throw;
} // plee::apply_throw()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action maintain.
 *
 */
void ptb::plee::apply_maintain()
{
   if ( get_current_action_name() == "maintain_and_fall" )
    m_right_move_force = 100000;
  else
    m_right_move_force = 200000;

  m_throw_time_ratio = 0;
  set_state(plee::maintain_state);
  m_progress = &plee::progress_maintain;
} // plee::apply_maintain()

/*----------------------------------------------------------------------------*/
/**
 * \brief Throw an object.
 *
 */
void ptb::plee::apply_throw_object()
{
  if ( m_throw_time_ratio == 0 )
    m_throw_time_ratio = 1;

  if ( m_current_object == stone_object ) 
    create_stone( get_stone_force() );
  else if ( m_current_object == air_honeypot_object ) 
    create_small_honeypot( get_stone_force() );
  else if ( m_current_object == fire_honeypot_object ) 
    create_small_honeypot( get_stone_force() );
  else if ( m_current_object == water_honeypot_object ) 
    create_small_honeypot( get_stone_force() );
  else if (  m_current_object == azelnut_object )
    create_azelnut();

  m_current_object = none_object;
  if ( m_animation_to_throw != NULL )
    {
      delete m_animation_to_throw;
      m_animation_to_throw = NULL;
    }
} // plee::apply_throw_object()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action blast_stone.
 *
 */
void ptb::plee::apply_blast_stone()
{
  if ( ! m_air_stones.empty() )
    {
      std::set<ptb::stone*> save_stones(m_air_stones);
      std::set<ptb::stone*>::iterator it;
      for ( it = save_stones.begin(); it != save_stones.end(); ++it )
        {
          // the call can change the set m_air_stones
          (*it)->inform_new_stone();
        }
    }
} // plee::apply_blast_stone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action wait. *
 */
void ptb::plee::apply_wait()
{
  set_state(plee::wait_state);
  m_progress = &plee::progress_wait;
} // plee::apply_wait()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action crouch.
 *
 */
void ptb::plee::apply_crouch()
{
  m_right_move_force = 200000;
  set_state(plee::crouch_state);
  m_progress = &plee::progress_crouch;
} // plee::apply_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action look_upward.
 *
 */
void ptb::plee::apply_look_upward()
{
  m_right_move_force = 200000;
  set_state(plee::look_upward_state);
  m_progress = &plee::progress_look_upward;
} // plee::apply_look_upward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Die.
 */
void ptb::plee::apply_die()
{
  game_variables::set_lives_count
    ( m_index, game_variables::get_lives_count(m_index) - 1);

  if ( game_variables::get_lives_count( m_index ) == 0 )
    {
      bear::engine::transition_effect_message<game_over_effect> msg;
      get_level_globals().send_message
        ( PTB_TRANSITION_EFFECT_DEFAULT_TARGET_NAME, msg );
      start_action_model("game_over");
      apply_game_over();
    }
  else
    {
      bear::engine::transition_effect_message<bear::engine::fade_effect> msg;
      msg.get_effect().set_duration(0, 0.25, 0.75);
      msg.get_effect().set_color( 1, 1, 1 );
      get_level_globals().send_message
        ( PTB_TRANSITION_EFFECT_DEFAULT_TARGET_NAME, msg );
      set_state(dead_state);
      m_progress = &plee::progress_dead;
    }
} // plee::apply_die()

/*----------------------------------------------------------------------------*/
/**
 * \brief Roar.
 */
void ptb::plee::apply_roar()
{
  set_state(roar_state);
  m_progress = &plee::progress_roar;
} // plee::apply_roar()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the state "game over".
 */
void ptb::plee::apply_game_over()
{
  m_right_move_force = 0;
  set_state(game_over_state);
  m_progress = &plee::progress_game_over;
} // plee::apply_game_over()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to cling.
 */
void ptb::plee::apply_start_cling()
{
  set_state(start_cling_state);
  m_progress = &plee::progress_start_cling;
} // plee::apply_start_cling()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee cling.
 */
void ptb::plee::apply_cling()
{
  m_run_time = 0;
  set_state(cling_state);
  m_progress = &plee::progress_cling;
} // plee::apply_cling()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee is clung and jump.
 */
void ptb::plee::apply_clung_jump()
{
  if ( get_rendering_attributes().is_mirrored() )
    add_internal_force( bear::universe::force_type(5000000, 10000000) );
  else
    add_internal_force( bear::universe::force_type(-5000000, 10000000) );
} // plee::apply_clung_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to hang.
 */
void ptb::plee::apply_start_hang()
{
  set_state(start_hang_state);
  m_progress = &plee::progress_start_hang;
} // plee::apply_start_hang()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee hang.
 */
void ptb::plee::apply_hang()
{
  set_state(hang_state);
  m_progress = &plee::progress_hang;
} // plee::apply_hang()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee swim.
 */
void ptb::plee::apply_swimming()
{
  m_right_move_force = 150000;
  set_state(swimming_state);
  m_progress = &plee::progress_swimming;
} // plee::apply_swimming()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee sink.
 */
void ptb::plee::apply_sink()
{
  m_right_move_force = 150000;
  set_state(sink_state);
  m_progress = &plee::progress_sink;
} // plee::apply_swimming()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action swim_up.
 *
 */
void ptb::plee::apply_swim_up()
{
  add_internal_force( bear::universe::force_type(0, 300000) );
} // plee::apply_swim_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action swim_down.
 *
 */
void ptb::plee::apply_swim_down()
{
  add_internal_force( bear::universe::force_type(0,-50000) );
} // plee::apply_swim_down()


/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action swim_down.
 *
 */
void ptb::plee::apply_swim_jump()
{
  if ( get_rendering_attributes().is_mirrored() )
    add_internal_force( bear::universe::force_type(-4000000, 7000000));
  else
    add_internal_force( bear::universe::force_type(4000000, 7000000));
} // plee::apply_swim_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action float.
 *
 */
void ptb::plee::apply_float()
{
  m_right_move_force = 150000;
  set_state(float_state);
  m_progress = &plee::progress_float;
} // plee::apply_float()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action dive.
 *
 */
void ptb::plee::apply_dive()
{
  add_internal_force( bear::universe::force_type(0, -200000));
} // plee::apply_float()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the action injured.
 */
void ptb::plee::apply_injured()
{
  m_right_move_force = 0;
  m_nb_bottom_contact = 0;
  set_state(plee::injured_state);
  m_progress = &plee::progress_injured;
} // plee::apply_injured()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call a function.
 * \param name The function to call.
 */
void ptb::plee::execute_function( const std::string& name )
{
  if ( name == "start_roar" )
    apply_roar();
  if ( name == "start_game_over" )
    apply_game_over();
  if ( name == "start_captive" )
    apply_chain();
  else if ( name == "start_crouch" )
    apply_crouch();
  else if ( name == "start_fall" )
    apply_fall();
  else if ( name == "start_float" )
    apply_float();
  else if ( name == "start_idle" )
    apply_idle();
  else if ( name == "start_jump" )
    apply_jump();
  else if ( name == "start_look_upward" )
    apply_look_upward();
  else if ( name == "start_run" )
    apply_run();
  else if ( name == "start_sink" )
    apply_sink();
  else if ( name == "start_slap" )
    apply_slap();
  else if ( name == "start_attack" )
    apply_attack();
  else if ( name == "start_start_cling" )
    apply_start_cling();
  else if ( name == "start_start_hang" )
    apply_start_hang();
  else if ( name == "start_cling" )
    apply_cling();
  else if ( name == "start_hang" )
    apply_hang();
  else if ( name == "start_clung_jump" )
    apply_clung_jump();
  else if ( name == "start_throw" )
    apply_throw();
  else if ( name == "start_maintain" )
    apply_maintain();
  else if ( name == "start_start_jump" )
    apply_start_jump();
  else if ( name == "start_swimming" )
    apply_swimming();
  else if ( name == "start_vertical_jump" )
    apply_vertical_jump(); 
  else if ( name == "wait" )
    apply_wait();
  else if ( name == "start_walk" )
    apply_walk();
  else if ( name == "throw" )
    apply_throw_object();
  else if ( name == "start_injured" )
    apply_injured();
  else 
    super::execute_function(name);
} // plee::execute_function()

/*----------------------------------------------------------------------------*/
/**
 * \brief  Inform the item that he have no energy now.
 * \param attacker The attacker monster.
 */
void ptb::plee::inform_no_energy(const monster& attacker)
{
  apply_die();
} // plee::inform_no_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief The monster is injured.
 * \param attacker The monster attacking me.
 */
void ptb::plee::injure
(const monster& attacker, const bear::universe::collision_info& info)
{
  double orientation = -1;

  if ( ( info.get_collision_side() == 
         bear::universe::zone::middle_right_zone ) || 
       ( info.get_collision_side() == 
         bear::universe::zone::top_right_zone ) || 
       ( info.get_collision_side() == 
         bear::universe::zone::bottom_right_zone ) )
    orientation = 1;
    
  set_speed(bear::universe::speed_type(0,0));
  add_external_force
    ( bear::universe::force_type(orientation * 4000000, 5500000) );

  super::injure(attacker,info);

  m_states[m_current_state]->do_injured();
} // ptb::plee::injure()

/*----------------------------------------------------------------------------*/
/**
 * \brief The monster isn't injure any more.
 */
void ptb::plee::finish_injure()
{
  super::finish_injure();

  m_states[m_current_state]->do_finish_injured();
} // ptb::monster_item::finish_injure()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a given offensive coefficient.
 * param index The index of the attack.
 * param other The other monster.
 * param info Information on collision.
 */
unsigned int ptb::plee::get_offensive_coefficient
( unsigned int index,
  const monster& other,
  const bear::universe::collision_info& info ) const
{
  unsigned int result = super::get_offensive_coefficient(index,other,info);
  
  const base_item* item = dynamic_cast<const base_item*>(&other);
  
  if ( ( item != NULL ) && ( result != 0 ) )
    {
      if ( get_rendering_attributes().is_mirrored() )
	{
	  if ( item->get_left() >= get_left() )
	    result = 0;
	}
      else if ( item->get_right() <= get_right() )
	result = 0;
    }
  
  return result;
} // plee::get_offensive_coefficient()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void ptb::plee::to_string( std::string& str ) const
{
  std::ostringstream oss;

  super::to_string(str);

  oss << "oxygen_gauge: ";
  if ( m_gauges[oxygen_gauge]->is_activated() )
    oss << "active : ";
  else
    oss << "not active : ";
  oss << m_gauges[oxygen_gauge]->get_value() << "/"
      << m_gauges[oxygen_gauge]->get_max_value() << "\n";

  oss << "fire_gauge: ";
  if ( m_gauges[fire_gauge]->is_activated() )
    oss << "active : ";
  else
    oss << "not active : ";
  oss << m_gauges[fire_gauge]->get_value() << "/"
      << m_gauges[fire_gauge]->get_max_value() << "\n";

  oss << "ice_gauge: ";
  if ( m_gauges[ice_gauge]->is_activated() )
    oss << "active : ";
  else
    oss << "not active : ";
  oss << m_gauges[ice_gauge]->get_value() << "/"
      << m_gauges[ice_gauge]->get_max_value() << "\n";

  oss << "state: ";
  oss << m_states[m_current_state]->get_name();
  oss << "\n";
  oss << "action: ";
  oss << get_current_action_name();
  oss << "\n";
  oss << "can_cling: " << m_can_cling << "\n";
  oss << "status_look_upward: " << m_status_look_upward << "\n";
  oss << "status_crouch: " << m_status_crouch << "\n";
  oss << "powers: " << " : ";
  if ( game_variables::get_air_power(m_index) )
    oss << "air  ";
  if ( game_variables::get_fire_power(m_index) )
    oss << "fire  ";
  if ( game_variables::get_water_power(m_index) )
    oss << "water  ";
  oss << "\n";
  oss << "air_float: ";
  if ( m_air_float )
    oss << "y";
  else
    oss << "n"; 
  oss << " : " << m_air_float_time;
  oss << "\n";

  str += oss.str();
} // plee::to_string()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_walk( bear::universe::time_type elapsed_time )
{
  brake();

  if ( has_bottom_contact() )
    {
      if ( m_last_bottom_left == get_bottom_left() )
        start_action_model("idle");
      else
        {
          bear::universe::speed_type speed( get_speed() );
          // calculate the speed in the intern axis
          bear::universe::coordinate_type speed_x =
            speed.dot_product(get_x_axis());

          if( std::abs(speed_x) >= s_speed_to_run )
            {
              if ( m_run_time >= s_time_to_run ) 
                start_action_model("run");
            }
          else if ( speed_x == 0 )
            start_action_model("idle");
          else
            m_right_move_force = 
              50000 + std::min(m_run_time,s_time_to_run)*(300000-50000) 
              / s_time_to_run;
        }
    }
  else
    test_in_sky_or_swimm();
} // plee::progress_walk()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_idle( bear::universe::time_type elapsed_time )
{
  brake();

  if ( has_bottom_contact() )
    {
      m_run_time = 0;
  
      if ( m_state_time >= s_time_to_wait )
        {
          std::ostringstream s;
          s << "wait" << (rand() % 3 + 1);
          start_action_model(s.str());
        }
    }
  else
    test_in_sky_or_swimm();
} // plee::progress_idle()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_jump( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    {
      if ( is_only_in_environment(bear::universe::water_environment) )
        {
          if( get_speed().y <= 0 )
            start_action_model("sink");
          else
            start_action_model("swimming");
        }
      else if ( is_in_floating() )
        start_action_model("float");
      else if( get_speed().y <= 0 )
        start_action_model("fall");
    }
} // plee::progress_jump()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_fall( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    {
      if ( is_only_in_environment(bear::universe::water_environment) )
        if( get_speed().y <= 0 )
          start_action_model("sink");
        else
          start_action_model("swimming");
      else if ( is_in_floating() )
        start_action_model("float");
      if( get_speed().y > 0 )
        start_action_model("jump");
    }
} // plee::progress_fall()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_dead( bear::universe::time_type elapsed_time )
{
  regenerate();

  start_action_model("roar");
} // plee::progress_dead()


/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_game_over( bear::universe::time_type elapsed_time )
{
  set_status_look_upward(false);
  set_status_crouch(false);
} // plee::progress_game_over()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_roar( bear::universe::time_type elapsed_time )
{
  // do_nothing
} // plee::progress_roar()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_run( bear::universe::time_type elapsed_time )
{
  if ( has_bottom_contact() )
    {
      bear::universe::speed_type speed;
      speed = get_speed();

      // calculate the speed in the intern axis
      bear::universe::coordinate_type speed_x =
        speed.dot_product(get_x_axis());

      if( std::abs(speed_x) < s_speed_to_run )
        {
          if( speed_x == 0 )
            start_action_model("idle");
          else
            start_action_model("walk");
        }
    }
  else
    test_in_sky_or_swimm();
} // plee::progress_run()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_slap( bear::universe::time_type elapsed_time )
{
  brake();

  bear::universe::speed_type speed(get_speed());
  bear::universe::coordinate_type speed_x = speed.dot_product(get_x_axis());
  
  if ( get_current_action_name() == "slap" )
    {
      if ( std::abs(speed_x) != 0 )
        switch_to_model_action("slap_and_walk");
    }
  else if ( get_current_action_name() == "slap_and_walk") 
    {
      if( std::abs(speed_x) == 0 )
        switch_to_model_action("slap");
    }

  if ( has_right_contact() ) 
    apply_move_right();
  else if ( has_left_contact() ) 
    apply_move_left();
} // plee::progress_slap()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_start_jump( bear::universe::time_type elapsed_time )
{
  m_halo_animation->next(elapsed_time);

  brake();

  if ( ! has_bottom_contact() )
    test_in_sky_or_swimm();
} // plee::progress_start_jump()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_vertical_jump
( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    if( get_speed().y <= 0 )
      start_action_model("fall");
} // plee::progress_vertical_jump()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_crouch( bear::universe::time_type elapsed_time )
{
  brake();

  if ( ! has_bottom_contact() )
    test_in_sky_or_swimm();
} // plee::progress_crouch()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_look_upward( bear::universe::time_type elapsed_time )
{
  brake();

  if ( ! has_bottom_contact() )
    test_in_sky_or_swimm();
} // plee::progress_look_upward()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_captive( bear::universe::time_type elapsed_time )
{
  // do nothing
} // plee::progress_captive()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_throw( bear::universe::time_type elapsed_time )
{
  brake();

  bear::universe::speed_type speed(get_speed());
  bear::universe::coordinate_type speed_x = speed.dot_product(get_x_axis());
  
  if ( get_current_action_name() == "throw" )
    {
      if ( ! has_bottom_contact() )
        switch_to_model_action("throw_and_fall");
      else if( std::abs(speed_x) != 0 )
        switch_to_model_action("throw_and_walk");
    }
  else if ( get_current_action_name() == "throw_and_walk") 
    {
      if ( ! has_bottom_contact() )
        switch_to_model_action("throw_and_fall");
      else if( std::abs(speed_x) == 0 )
        switch_to_model_action("throw");
    }
  else if ( get_current_action_name() == "throw_and_fall") 
    {
      if ( has_bottom_contact() )
        {
          if( std::abs(speed_x) == 0 )
            switch_to_model_action("throw");
          else 
            switch_to_model_action("throw_and_walk");
        }
    }

  if ( ( m_current_object == air_honeypot_object ) && 
       ! m_can_throw_power[monster::air_attack] )
    {
      game_variables::set_air_power(m_index, true);      
      start_action_model("idle");
    }
  if ( ( m_current_object == fire_honeypot_object ) && 
       ! m_can_throw_power[monster::fire_attack] )
    {
      game_variables::set_fire_power(m_index, true); 
      start_action_model("idle");
    }
  if ( ( m_current_object == water_honeypot_object ) && 
       ! m_can_throw_power[monster::water_attack] )
    {
      game_variables::set_water_power(m_index, true); 
      start_action_model("idle");
    }
} // plee::progress_throw()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_maintain( bear::universe::time_type elapsed_time )
{
  brake();

  bear::universe::speed_type speed(get_speed());
  bear::universe::coordinate_type speed_x = speed.dot_product(get_x_axis());
  
  if ( get_current_action_name() == "maintain" )
    {
      if ( ! has_bottom_contact() )
        switch_to_model_action("maintain_and_fall");
      else if( std::abs(speed_x) != 0 )
        switch_to_model_action("maintain_and_walk");
    }
  else if ( get_current_action_name() == "maintain_and_walk") 
    {
      if ( ! has_bottom_contact() )
        switch_to_model_action("maintain_and_fall");
      else if( std::abs(speed_x) == 0 )
        switch_to_model_action("maintain");
    }
  else if ( get_current_action_name() == "maintain_and_fall") 
    {
      if ( has_bottom_contact() )
        {
          if( std::abs(speed_x) == 0 )
            switch_to_model_action("maintain");
          else 
            switch_to_model_action("maintain_and_walk");
        }
    }
} // plee::progress_maintain()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_wait( bear::universe::time_type elapsed_time )
{
  brake();

  if ( ! has_bottom_contact() )
    test_in_sky_or_swimm();
} // plee::progress_wait()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_start_cling
( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    {
      if ( ( !get_rendering_attributes().is_mirrored() 
            && has_right_contact() ) ||
           ( get_rendering_attributes().is_mirrored() && has_left_contact() ) )
        {
          // test if the wall can be clung
          if ( m_can_cling )
            start_action_model("cling");
          else
            start_action_model("idle");
        }
    }
} // plee::progress_start_cling()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_cling( bear::universe::time_type elapsed_time )
{
  if ( ! test_bottom_contact() )
    {
      if ( has_top_contact() || ( ! m_can_cling  ) )
        start_action_model("fall");
      else
        {
          bear::universe::speed_type speed = get_speed();
          speed.x = 0;
          if ( speed.y > 0 )
            speed.y = 0;
          else
            speed.y *= 0.9;
          
          set_speed(speed);

	  
	  bear::universe::force_type force
	    ( - get_mass() * get_owner().get_gravity() );
  
	  if ( (get_density() != 0) &&
           ( get_mass() != std::numeric_limits<double>::infinity() )  )
	    force += get_owner().get_gravity() * get_mass() *
	      get_owner().get_average_density(get_bounding_box()) / 
	      get_density();

          add_external_force(force);
        }
    }
} // plee::progress_cling()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_clung_jump
( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    if( get_speed().y <= 0 )
      start_action_model("fall");
} // plee::progress_clung_jump()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_start_hang
( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    {
      if ( ( !get_rendering_attributes().is_mirrored() && 
             has_right_contact() ) ||
           ( get_rendering_attributes().is_mirrored() && has_left_contact() ) )
        start_action_model("hang");
      else
        if ( m_state_time >= s_max_time_to_hang )
          start_action_model("idle");
    }
} // plee::progress_start_hang()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_hang( bear::universe::time_type elapsed_time )
{
  if ( ! test_bottom_contact() )
    {
      if ( has_top_contact() )
        start_action_model("fall");
      else
        add_external_force(- get_mass() * get_owner().get_gravity());
    }
} // plee::progress_hang()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_swimming( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    {
      if ( is_only_in_environment(bear::universe::water_environment) )
        {
          if( get_speed().y <= 0 )
            start_action_model("sink");
        }
      else if ( is_in_floating() )
        start_action_model("float");
      else if( get_speed().y <= 0 )
        start_action_model("fall");
      else
        start_action_model("jump");
    }
} // plee::progress_swimming()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_sink( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    {
      if ( is_only_in_environment(bear::universe::water_environment) )
        {
          if( get_speed().y > 0 )
            start_action_model("swimming");
        }
      else if ( is_in_floating() )
        start_action_model("float");
      else if( get_speed().y > 0 )
        start_action_model("jump");
      else
        start_action_model("fall");
    }
} // plee::progress_sink()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_float( bear::universe::time_type elapsed_time )
{
  if ( !test_bottom_contact() )
    {
      if ( is_only_in_environment(bear::universe::water_environment) )
        {
          if( get_speed().y > 0 )
            start_action_model("swimming");
          else
            start_action_model("sink");
        }
      else if ( ! is_in_floating() )
        {
          if( get_speed().y > 0 )
            start_action_model("jump");
          else
            start_action_model("fall");
        }
    }
} // plee::progress_float()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the state .
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_injured( bear::universe::time_type elapsed_time )
{
  brake();

  if ( has_bottom_contact() )
    {
      m_nb_bottom_contact++;

      if ( m_nb_bottom_contact > 2 ) 
        {
          if ( m_last_bottom_left == get_bottom_left() )
            start_action_model("idle");
          else
            {
              bear::universe::speed_type speed( get_speed() );
              // calculate the speed in the intern axis
              bear::universe::coordinate_type speed_x =
                speed.dot_product(get_x_axis());
              
              if( std::abs(speed_x) >= s_speed_to_run )
                start_action_model("run");
              else if ( speed_x == 0 )
                start_action_model("idle");
              else 
                start_action_model("walk");
            }
        }
    }
} // plee::progress_injured()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the state of Plee.
 */
void ptb::plee::set_state(plee_state_name state)
{
  if ( state != m_current_state )
    {
      m_states[m_current_state]->stop();
      m_current_state = state;
      m_states[m_current_state]->start();
      m_state_time = 0;
    }
} // plee::set_state()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if Plee is crushed.
 */
bool ptb::plee::is_crushed() const
{
  return ( ( has_bottom_contact() && has_top_contact() ) ||
           ( has_right_contact() && has_left_contact() ) );
     // &&   has_contact_after_collision() );
} // plee::is_crushed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Spawn the plee after his death.
 * \todo Game over if the plee has no more lifes.
 * \todo Update lifes and any other counters.
 */
void ptb::plee::regenerate()
{
  set_center_of_mass( m_saved_position );
  set_speed( bear::universe::speed_type(0, 0) );
  set_internal_force(bear::universe::force_type(0, 0));
  set_external_force(bear::universe::force_type(0, 0));
  m_energy = game_variables::get_max_energy(m_index);
  m_invincible_time = 0;
  set_invincible(false);
  m_last_visual_time = 0;
  m_last_visuals.clear();
  m_run_time = 0;
  m_can_cling = false;
  m_air_float = false;
  finish_injure();
  game_variables::set_air_power(m_index,false);
  game_variables::set_fire_power(m_index,false);
  game_variables::set_water_power(m_index,false);
  update_powers();
  m_gauges[oxygen_gauge]->set_value(s_max_oxygen);
  m_gauges[fire_gauge]->set_value(s_max_fire_gauge);
  m_gauges[ice_gauge]->set_value(s_max_ice_gauge);
  m_air_stones.clear();
  m_move_right = false;
  m_move_left = false;
  m_last_bottom_left = bear::universe::position_type(0,0);
  get_rendering_attributes().mirror(false);
  stop_to_speak();
  if ( m_animation_to_throw ) 
    {
      delete m_animation_to_throw;
      m_animation_to_throw = NULL;
    }
  remove_all_links();
} // plee::regenerate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if Plee has a power.
 */
bool ptb::plee::has_a_power() const
{
  return game_variables::get_air_power(m_index) ||
    game_variables::get_fire_power(m_index) ||
    game_variables::get_water_power(m_index);
} // plee::has_power()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if Plee is in the water and in the air.
 *
 */
bool ptb::plee::is_in_floating() const
{
  bool result = false;

  if ( has_owner() )
    {
      std::set<bear::universe::environment_type> environments;
      get_owner().get_environments(get_bounding_box(), environments);
      result =
        (environments.find(bear::universe::water_environment) !=
         environments.end()) &&
        ( environments.size() >= 1 );
    }

  return result;
} // plee::is_only_in_floatting()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the orientation.
 */
void ptb::plee::update_orientation()
{
  if ( get_speed().x < 0 )
    {
      if ( get_current_action_name()!= "injured" ) 
        get_rendering_attributes().mirror(true);
      else        
        get_rendering_attributes().mirror(false);
    }
  else if ( get_speed().x > 0 )
    {
      if ( get_current_action_name()!= "injured" )
        get_rendering_attributes().mirror(false);
      else
        get_rendering_attributes().mirror(true);
    }
} // plee::update_orientation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the offensive and defensive parameters according to
 * a power.
 * \param p The power status.
 * \param a The corresponding attack
 */
void ptb::plee::update_power( bool b, monster::attack_type a)
{
  if ( b )
    {
      m_defensive_powers[a] = true;
      m_offensive_coefficients[a] = 1;
    }
  else
    {
      m_defensive_powers[a] = false;
      m_offensive_coefficients[a] = 0;
    }
} // plee::update_power()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the offensive and defensive parameters according to
 * the powers.
 */
void ptb::plee::update_powers()
{
  update_power(game_variables::get_air_power(m_index),monster::air_attack);
  update_power(game_variables::get_fire_power(m_index),monster::fire_attack);
  update_power(game_variables::get_water_power(m_index),monster::water_attack);

  if ( has_a_power() && ( m_halo_hand_animation != NULL ) )
    {
      if ( game_variables::get_air_power(m_index) )
        if ( game_variables::get_fire_power(m_index) )
          if ( game_variables::get_water_power(m_index) )
            m_halo_hand_animation->set_intensity(0.5, 0.5, 0.5);
          else
            m_halo_hand_animation->set_intensity(1, 0, 0);
        else
          if ( game_variables::get_water_power(m_index) )
            m_halo_hand_animation->set_intensity(0.19, 0.78, 0.94);
          else
            m_halo_hand_animation->set_intensity(1, 1, 1);
      else
        if ( game_variables::get_fire_power(m_index) )
          if ( game_variables::get_water_power(m_index) )
            m_halo_hand_animation->set_intensity(1, 0.5, 0);
          else
            m_halo_hand_animation->set_intensity(0.66, 0.13, 0);
        else
          if ( game_variables::get_water_power(m_index) )
            m_halo_hand_animation->set_intensity(0, 0, 1);
    }
} // plee::update_powers()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the camera on the current plee.
 */
void ptb::plee::do_get_camera()
{
  shared_camera::placement_mode mode;

  if ( get_name() == player_name(1) )
    mode = shared_camera::lock_first_player;
  else
    mode = shared_camera::lock_second_player;

  shared_camera::set_placement_message msg(mode);
  bear::engine::level_globals& glob = get_level_globals();

  glob.send_message( "camera", msg );
} // plee::do_get_camera()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop to look up.
 */
void ptb::plee::do_stop_look_upward()
{
  m_status_look_upward = false;
  m_states[m_current_state]->do_stop_look_upward();
} // state_plee::do_stop_look_upward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop to crouch.
 */
void ptb::plee::do_stop_crouch()
{
  m_status_crouch = false;
  m_states[m_current_state]->do_stop_crouch();
} // state_plee::do_stop_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the gauges.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_gauges( bear::universe::time_type elapsed_time )
{
  progress_oxygen_gauge(elapsed_time);
  progress_fire_gauge(elapsed_time);
  progress_ice_gauge(elapsed_time);
} // plee::progress_gauges()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the oxygen_gauge.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_oxygen_gauge( bear::universe::time_type elapsed_time )
{
  m_gauges[oxygen_gauge]->set_activated
    (is_in_environment(bear::universe::water_environment));

  bear::engine::model_mark_placement m;
  
  if ( is_in_environment(bear::universe::water_environment) )
    if ( get_mark_placement("mouth",m) ) 
      {
        if ( get_owner().is_in_environment
             (m.get_position(), bear::universe::water_environment) )
          m_gauges[oxygen_gauge]->remove_value
            (s_oxygen_loss_speed*elapsed_time);
        else
          m_gauges[oxygen_gauge]->add_value
            (s_oxygen_inspiration_speed*elapsed_time);
      }
    else
      m_gauges[oxygen_gauge]->remove_value(s_oxygen_loss_speed*elapsed_time);
  else
    m_gauges[oxygen_gauge]->fill();
  
  if ( m_gauges[oxygen_gauge]->get_value() == 0 )
    remove_energy(*this,0.1);
} // plee::progress_oxygen_gauge()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the fire_gauge.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_fire_gauge( bear::universe::time_type elapsed_time )
{
  m_gauges[fire_gauge]->set_activated
    (is_in_environment(bear::universe::fire_environment));

  if ( is_only_in_environment(bear::universe::fire_environment) )
    m_gauges[fire_gauge]->remove_value(s_fire_loss_speed*elapsed_time);
  else
    m_gauges[fire_gauge]->add_value(s_fire_increase_speed*elapsed_time);

  if ( m_gauges[fire_gauge]->get_value() == 0 )
    remove_energy(*this,0.1);
} // plee::progress_fire_gauge()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the ice_gauge.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_ice_gauge( bear::universe::time_type elapsed_time )
{
  m_gauges[ice_gauge]->set_activated
    (is_in_environment(bear::universe::ice_environment));

  if ( is_only_in_environment(bear::universe::ice_environment) )
    m_gauges[ice_gauge]->remove_value(s_ice_loss_speed*elapsed_time);
  else
    m_gauges[ice_gauge]->add_value(s_ice_increase_speed*elapsed_time);

  if ( m_gauges[ice_gauge]->get_value() == 0 )
    remove_energy(*this,0.1);
} // plee::progress_ice_gauge()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the spot.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_spot( bear::universe::time_type elapsed_time )
{
  bool balance_x(true);
  bool balance_y(true);

  if ( m_status_look_upward )
    {
      m_look_upward_time += elapsed_time;

      if ( m_look_upward_time >= s_time_to_look_upward )
        {
          balance_y = false;
          add_spot_gap( bear::universe::position_type(0, 5) );
        }
    }

  if ( m_status_crouch )
    {
      m_crouch_time += elapsed_time;

      if ( m_crouch_time >= s_time_to_crouch )
        {
          balance_y = false;
          add_spot_gap( bear::universe::position_type(0, -5) );
        }
    }

  if ( get_speed().x > s_speed_to_run )
    {
      balance_x = false;
      add_spot_gap( bear::universe::position_type(3, 0) );
    }
  else if ( get_speed().x < -s_speed_to_run )
    {
      balance_x = false;
      add_spot_gap( bear::universe::position_type(-3, 0) );
    }

  balance_spot(balance_x, balance_y);
} // plee::progress_spot()


/*---------------------------------------------------------------------------*/
/**
 * \brief Teste if Plee walk and change state thereof.
 *         Return true if Plee walk.
 */
bool ptb::plee::test_walk()
{
  bool result = false;

  if ( has_bottom_contact() )
    {
      bear::universe::speed_type speed;
      speed = get_speed();
      // calculate the speed in the intern axis
      bear::universe::coordinate_type speed_x =
        speed.dot_product(get_x_axis());

      if( std::abs(speed_x) >= s_speed_to_run )
        {
          result = true;
          start_action_model("run");
        }
      else if ( ( speed_x != 0 ) &&
                ( get_bottom_left() != m_last_bottom_left ) )
        {
          result = true;
          start_action_model("walk");
        }
    }

  return result;
} // plee::test_walk()

/*---------------------------------------------------------------------------*/
/**
 * \brief Teste if Plee walk, run or is idle and change state thereof.
 *         Return true if Plee has a bottom contact.
 */
bool ptb::plee::test_bottom_contact()
{
  bool result = false;

  if ( has_bottom_contact() )
    {
      bear::universe::speed_type speed;
      speed = get_speed();

      // calculate the speed in the intern axis
      bear::universe::coordinate_type speed_x =
        speed.dot_product(get_x_axis());

      if( std::abs(speed_x) >= s_speed_to_run )
        start_action_model("run");
      else if( speed_x != 0 )
        start_action_model("walk");
      else
        start_action_model("idle");

      result = true;
    }

  return result;
} // plee::test_bottom_contact()

/*---------------------------------------------------------------------------*/
/**
 * \brief Teste if Plee is in the sky or swimm and change state thereof.
 *         Return true if Plee is in the sky.
 */
bool ptb::plee::test_in_sky_or_swimm()
{
  bool result = false;

  if ( !has_bottom_contact() )
    {
      result = true;

      if ( is_only_in_environment(bear::universe::water_environment) )
        if( get_speed().y <= 0 )
          start_action_model("sink");
        else
          start_action_model("swimming");
      else if ( is_in_floating() )
        start_action_model("float");
      else if( get_speed().y <= 0 )
        start_action_model("fall");
      else
        start_action_model("jump");
    }

  return result;
} // plee::test_in_sky_or_swimm()

/*---------------------------------------------------------------------------*/
/**
 * \brief Brake on x axis if Player has no direction.
 */
void ptb::plee::brake()
{
  if ( ( ! m_move_right ) && ( ! m_move_left ) )
    {
      bear::universe::speed_type speed = get_speed();
      speed.x *= 0.9;
      set_speed(speed);
    }
  m_move_right = false;
  m_move_left = false;
} // plee::brake()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compute the ejection force of stone.
 */
bear::universe::force_type ptb::plee::get_stone_force() const
{
  double force;
  bear::universe::force_type result;

  if ( m_throw_up && ! m_throw_down)
    {
      force = 4 + 4 * m_throw_time_ratio;
      result.x = 1;
      result.y = 5;
    }
  else if ( ! m_throw_up && m_throw_down)
    {
      force = 1 + 6 * m_throw_time_ratio;
      result.x = 4;
      result.y = -1;
    }
  else
    {
      force = 3 + 5 * m_throw_time_ratio;
      result.x = 3;
      result.y = 3;
    }

  result *= 3500;

  if ( get_rendering_attributes().is_mirrored() )
    result.x = -result.x;

  result.x = result.x * force + 100*get_speed().x;
  result.y = result.y * force + get_speed().y;

  return result;
} // plee::get_stone_force()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a stone.
 *
 * \param a The force applied to thes stone.
 */
void ptb::plee::create_stone( const bear::universe::force_type& a )
{
  if ( ! m_air_stones.empty() )
    {
      std::set<ptb::stone*> save_stones(m_air_stones);
      std::set<ptb::stone*>::iterator it;
      for ( it = save_stones.begin(); it != save_stones.end(); ++it )
        {
          // the call can change the set m_air_stones
          (*it)->inform_new_stone();
        }
    }
  else if ( game_variables::get_stones_count(m_index) > 0 )
    {
      game_variables::set_stones_count
	( m_index, game_variables::get_stones_count(m_index) - 1);

      stone* new_stone;
      if ( game_variables::get_air_power(m_index) )
        if ( game_variables::get_fire_power(m_index) )
          if ( game_variables::get_water_power(m_index) )
            new_stone= new air_fire_water_stone(get_id());
          else
            new_stone= new air_fire_stone(get_id());
        else
          if ( game_variables::get_water_power(m_index) )
            new_stone= new air_water_stone(get_id());
          else
            new_stone= new air_stone(get_id());
      else
        if ( game_variables::get_fire_power(m_index) )
          if ( game_variables::get_water_power(m_index) )
            new_stone= new water_fire_stone(get_id());
          else
            new_stone= new fire_stone(get_id());
        else
          if ( game_variables::get_water_power(m_index) )
            new_stone= new water_stone(get_id());
          else
            new_stone= new stone(get_id());

      
      bear::engine::model_mark_placement m;
      if ( get_mark_placement("hand",m) ) 
	new_stone->set_center_of_mass(m.get_position());
      else
	new_stone->set_center_of_mass(get_center_of_mass());

      new_stone->set_z_position(get_z_position() + 2);
      if ( m_index == 1 )
        new_stone->set_monster_type(monster::stone_1_monster);
      else
        new_stone->set_monster_type(monster::stone_2_monster);
      new_item( *new_stone );

      new_stone->add_external_force(a);
    }
} // ptb::plee::create_stone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a honeypot.
 *
 * \param a The force to give at the honeypot.
 */
void ptb::plee::create_small_honeypot( const bear::universe::force_type& a)
{
  small_honeypot* new_small_honeypot = new small_honeypot();

  bear::engine::model_mark_placement m;
  if ( get_mark_placement("hand",m) ) 
    {
      new_small_honeypot->set_z_position(m.get_depth_position());
      new_small_honeypot->set_center_of_mass(m.get_position());
    }
  else
    {
      new_small_honeypot->set_z_position(get_z_position() + 1);
      new_small_honeypot->set_center_of_mass(get_center_of_mass());
    }

  if ( m_current_object == air_honeypot_object )
    new_small_honeypot->set_type(base_bonus::air_power);
  else if ( m_current_object == fire_honeypot_object )
    new_small_honeypot->set_type(base_bonus::fire_power);
  else
    new_small_honeypot->set_type(base_bonus::water_power);
    
  new_item( *new_small_honeypot );

  new_small_honeypot->add_external_force(a);
} // ptb::plee::create_honeypot()


/*----------------------------------------------------------------------------*/
/**
 * \brief Create an azelnut.
 */
void ptb::plee::create_azelnut()
{
  azelnut* new_azelnut = new azelnut();

  bear::engine::model_mark_placement m;
  if ( get_mark_placement("hand",m) ) 
    {
      new_azelnut->set_z_position(m.get_depth_position());
      new_azelnut->set_center_of_mass(m.get_position());
    }
  else
    {
      new_azelnut->set_z_position(get_z_position() + 1);
      new_azelnut->set_center_of_mass(get_center_of_mass());
    }

  new_item( *new_azelnut );

  bear::universe::force_type force(600000,600000);
      
  if ( get_rendering_attributes().is_mirrored() )
    force.x = -force.x;

  force.x = force.x + 2000*get_speed().x;
  force.y = force.y + 20*get_speed().y;

  new_azelnut->add_external_force(force);
} // ptb::plee::create_azelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render halos.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void ptb::plee::render_halos
( std::list<bear::engine::scene_visual>& visuals ) const
{
  render_jump_halo(visuals);
  render_hand_halo(visuals);
} // plee::render_halos()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render jump halo.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void ptb::plee::render_jump_halo
( std::list<bear::engine::scene_visual>& visuals ) const
{
  if ( m_current_state == plee::start_jump_state )
    {
      bear::universe::size_type height
        ( s_max_halo_height * (m_state_time / s_time_to_jump) );

      bear::visual::sprite current_sprite(m_halo_animation->get_sprite());
      current_sprite.set_height((unsigned int)height);
      current_sprite.set_angle
        ( current_sprite.get_angle() + get_visual_angle() );

      visuals.push_front
        ( bear::engine::scene_visual
          ( get_bottom_middle().x - current_sprite.width() / 2,
            get_bottom() + current_sprite.height() - 2,
            current_sprite, get_z_position()+1 ) );
    }
} // plee::render_jump_halo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render halos.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void
ptb::plee::render_hand_halo
( std::list<bear::engine::scene_visual>& visuals ) const
{
  if ( has_a_power() && 
       ( ( m_current_state == plee::maintain_state ) || 
         ( m_current_state == plee::throw_state ) ) )
    {
      bear::visual::sprite current_sprite(m_halo_hand_animation->get_sprite());
      double angle(current_sprite.get_angle() + get_visual_angle() );
      bear::universe::position_type pos(get_center_of_mass());
      bear::engine::model_mark_placement m;
  
      if ( get_mark_placement("hand_over",m) ) 
        if ( m.is_visible() ) 
          {
            angle += m.get_angle(); 
            pos = m.get_position();
            
            current_sprite.set_angle(angle);      
            visuals.push_front
              ( bear::engine::scene_visual
                ( pos.x - current_sprite.width() / 2,
                  pos.y + current_sprite.height() / 2,
                  current_sprite, get_z_position()+1 ) );
          }
    }
} // plee::render_hand_halo()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the throw state.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void
ptb::plee::render_throw
( std::list<bear::engine::scene_visual>& visuals ) const
{
  if ( m_current_state == plee::throw_state )
    if ( m_animation_to_throw != NULL )
      {
        bear::engine::model_mark_placement m;
            
        if ( get_mark_placement("hand",m) ) 
          if ( m.is_visible() )
            {
              bear::visual::sprite current_sprite
                (m_animation_to_throw->get_sprite());
              
              double angle(current_sprite.get_angle() + 
                           get_visual_angle() + m.get_angle() );
              bear::universe::position_type pos = m.get_position();            
              current_sprite.set_angle(angle);  
              
              visuals.push_front
                ( bear::engine::scene_visual
                  ( pos.x - current_sprite.width() / 2,
                    pos.y + current_sprite.height() / 2,
                    current_sprite, m.get_depth_position() ) );
            }
      }
} // plee::render_throw()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item without invincible visuals.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void
ptb::plee::get_visuals_without_invincibility
( std::list<bear::engine::scene_visual>& visuals ) const
{
  super::get_visual(visuals);

  render_halos(visuals);
  render_throw(visuals);
} // plee::get_visuals_without_invincibility()

/*---------------------------------------------------------------------------*/
/**
 * \brief Progress the invincibility statut.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_invincibility
( bear::universe::time_type elapsed_time )
{
  m_last_visual_time += elapsed_time;

  if ( is_invincible() )
    {
      m_invincible_time += elapsed_time;
      if ( m_invincible_time >= invincibility_effect::get_total_duration() )
        set_invincible(false);
    }

  if ( m_last_visual_time > 0.03 ) 
    {
      m_last_visual_time = 0;

      if ( is_invincible() )
        {
          if ( m_last_visuals.size() > 6 ) 
            m_last_visuals.pop_front();
          
          std::list<bear::engine::scene_visual> visuals;
          get_visuals_without_invincibility(visuals);
          
          std::list<bear::engine::scene_visual>::iterator it;
          for ( it = visuals.begin(); it != visuals.end(); ++it )
            it->scene_element.get_rendering_attributes().set_intensity
              (1,0.5,0.5);
          
          m_last_visuals.push_back(visuals);
        }
      else if ( ! m_last_visuals.empty() ) 
        m_last_visuals.pop_front();
      
      std::list< std::list<bear::engine::scene_visual> >::iterator it;
      
      for ( it = m_last_visuals.begin(); it != m_last_visuals.end(); ++it )
        {
          std::list<bear::engine::scene_visual>::iterator it2;
          
          for (it2 = it->begin(); it2 != it->end(); ++it2 )
            it2->scene_element.get_rendering_attributes().set_opacity
              ( 0.8 * 
                it2->scene_element.get_rendering_attributes().get_opacity());
        }  
    }
} // plee::progress_invincibilty()

/*---------------------------------------------------------------------------*/
/**
 * \brief Progress the air_float movement.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_air_float
( bear::universe::time_type elapsed_time )
{
  m_air_float_time += elapsed_time;
  
  if ( m_air_float_time >= s_max_time_air_float )
    m_air_float = false;
  else
    {
      if ( get_speed().y < 0 )
	{
	  double r=(double)rand()/(double)RAND_MAX; 
	  
	  bear::universe::force_type force
	    ( -(0.5+r/2)*get_mass() * get_owner().get_gravity() );
	  
	  if ( (get_density() != 0) &&
	       ( get_mass() != std::numeric_limits<double>::infinity() )  )
	    force += get_owner().get_gravity() * get_mass() *
	      get_owner().get_average_density(get_bounding_box()) / 
	      get_density();
	  
	  add_external_force(force);
	}
    }
} // progress_air_float()
