/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */



#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>

#include "codegen.h"



/* Globals */

dtRelocRec_t reloc[16];
unsigned relocN;



void
dtInitialize(void)
{
  /* Initialize the DT environment */

  /* Initialize the fragment cache */
  dtInitFragCache();
}



static Bit8u *
dtPass(Bit8u * tcodePtr, Bit8u * isPtr, int sz)
{
  memcpy(tcodePtr, isPtr, sz);
  return (tcodePtr + sz);
}



Bit32u
dtTranslateSequence(unsigned metaIndex, Bit32u gOff, Bit32u gla)
{
  /*
   *  Dynamically translate the guest code sequence at the given
   *  offset and linear address
   */

  int i;
  Bit32u pOff;
  Bit8u tcode[128], *tcodePtr, *iPtr, *isPtr;
  unsigned b0, b1, modrm, sib;
  Bit32u targetOffset32, gOffInstructionStart;
  unsigned tcodeLen;
  Bit8s displ8;

  pOff = gla & 0x00000fff;

/* +++ page checks here */

  /* +++ Braindead instruction decoder for now */
  iPtr = (Bit8u *) gla;
  tcodePtr = tcode;

  /* clear the relocation records */
  relocN = 0;

  /* main decoder loop */
loop:
  isPtr = iPtr;
  b0 = *iPtr++;
  gOffInstructionStart = gOff;

/* +++ All instructions in stream could register addresses */

  switch (b0) {
    case 0x0f: /* 2-byte primary opcode */
      b1 = *iPtr++;
      switch (b1) {
        case 0x84: /* JZ  Jd */
          gOff += 6;
          targetOffset32 = gOff + *(Bit32s *) iPtr;
          iPtr += 4;

          /* Emit code */
          tcodePtr = dtGenJzJd(tcodePtr, targetOffset32);
          goto loop;

        case 0x85: /* JNZ Jd */
          gOff += 6;
          targetOffset32 = gOff + *(Bit32s *) iPtr;
          iPtr += 4;

          /* Emit code */
          tcodePtr = dtGenJnzJd(tcodePtr, targetOffset32);
          goto loop;

        default:
          printf("dtTS: opcode 0x0f 0x%02x unhandled\n", b1);
          exit(1);
        }

      break;

    case 0x01: /* ADD EvGv */
      b1 = *iPtr++;
      if ((b1 & 0xc0) == 0xc0) {
        /* ADD Reg,Reg */
        gOff += 2;

        /* Emit code */
        tcodePtr = dtPass(tcodePtr, isPtr, 2);
        goto loop;
        }
      else {
        printf("dtTS: ADD not Reg,Reg\n");
        exit(1);
        }

    case 0x49: /* decl %ecx */
    case 0x51: /* pushl %ecx */
    case 0x59: /* popl %ecx */
    case 0x60: /* pusha */
    case 0x61: /* popa */
    case 0x90: /* NOP */
      *tcodePtr++ = b0; /* pass through */
      gOff++;
      goto loop;

    case 0xe2: /* LOOP Jb */
      gOff += 2;
      displ8 = *(Bit8s *) iPtr++;

      tcodePtr = dtGenLoopJb(tcodePtr, displ8);
      goto loop;

    case 0xb9: /* mov ECX,Id */
      gOff += 5;
      iPtr += 4;
      tcodePtr = dtPass(tcodePtr, isPtr, 5);
      goto loop;

    case 0xc3: /* RET */
      gOff++;

      /* Emit code */
      tcodePtr = dtGenRet(tcodePtr);
      goto finish;

    case 0xe9: /* JMP_Jd */
      gOff += 5;
      targetOffset32 = gOff + *(Bit32s *) iPtr;
      iPtr += 4;

      tcodePtr = dtGenJmpJd(tcodePtr, targetOffset32);
      goto finish;

    case 0xff: /* Group5 */
      modrm = *iPtr++;
      switch (modrm) {
        case 0x0d: /* DEC_Ed */
          gOff += 6;
          iPtr += 4;

          /* Emit pass through */
          tcodePtr = dtPass(tcodePtr, isPtr, 6);
          goto loop;

        case 0x24:
          sib = *iPtr++;
          if (sib == 0x8d) {    /* jmp *table(,%ecx,4) */
            gOff += 7;
            targetOffset32 = *(Bit32s *) iPtr;
            iPtr += 4;

            /* Emit code */
            tcodePtr = dtGenJmpEcx4(tcodePtr, targetOffset32);
            goto finish;
            }
          else {
            printf("dtTS: G5 sib=0x%x unhandled\n", sib);
            exit(1);
            }

        default:
          printf("dtTS: G5 modrm 0x%02x unhandled\n", modrm);
          exit(1);
        }

      break;

    default:
      printf("dtTS: opcode 0x%02x unhandled\n", b0);
      exit(1);
    }

  printf("dtTS: finished prematurely\n");
  exit(1);

finish:
  /* Find length of translated sequence */
  tcodeLen = (tcodePtr - tcode);

  /* Add the fragment to the fragment cache */
  tcodePtr = dtCreateFragment(tcode, tcodeLen, pOff, metaIndex);

  /*
   *  Relocate branches.  Because branches use displacements relative
   *  to the EIP offset, they depend on the location where the tcode
   *  is actually placed.
   */

  for (i = 0; i < relocN; i++) {
    /*
     *  Translate the address of the displacement to patch,
     *  from the original tcode buffer to the real tcode home
     */

    Bit32u loc = (Bit32u) tcodePtr +
      ((Bit32u) reloc[i].tcodePtr - (Bit32u) tcode);

    /* Now patch the displacement */
    *(Bit32u *) loc = reloc[i].tcodeOff - (loc + 4);
    }

  return (Bit32u) tcodePtr;
}
