

//////////////////////////////////////////////////////////////////
//                                                              //
//           PLINK (c) 2005-2008 Shaun Purcell                  //
//                                                              //
// This file is distributed under the GNU General Public        //
// License, Version 2.  Please see the file COPYING for more    //
// details                                                      //
//                                                              //
//////////////////////////////////////////////////////////////////

#include <iostream>
#include <fstream>
#include <iomanip>
#include <cmath>

#include "plink.h"
#include "helper.h"
#include "options.h"
#include "perm.h"
#include "stats.h"


void Plink::permTestRareDistribution(Perm & perm)
{

  printLOG("Testing for regions of rarer variants ( MAF < "
	   +dbl2str(par::rarer_maf_threshold)
	   +", within "
	   +int2str(int(par::rarer_dist_threshold/1000))
	   +" kb)\n");

  if ( par::SNP_major )
    SNP2Ind();
  


  ///////////////////////
  // Set up permutations

  perm.setTests(nl_all);
  perm.setPermClusters(*this);
  perm.originalOrder();



  ///////////////////////
  // Original

  vector_t original = testRareDistribution(perm, true);

  if (par::mperm_rank)
    perm.setOriginalRanking(original);



  //////////////////////
  // Begin permutations

  bool finished = false;
  while(!finished)
    {
      vector<double> pr(nl_all);
      perm.permuteInCluster();
      pr = testRareDistribution(perm,false);
      finished = perm.update(pr,original);
    } 
  
  if (!par::silent)
    cout << "\n\n";



  /////////////////////////////////////////////////////////////////////
  // Write results to file

  ofstream ASC;
  string f;

  if (par::adaptive_perm) f = par::output_file_name + ".rarer.perm";
  else f = par::output_file_name + ".rarer.mperm";

  ASC.open(f.c_str(),ios::out);
  ASC.precision(4);
  printLOG("Writing permutation association results to [ " + f + " ] \n");

  ASC << setw(4) << "CHR" << " "
      << setw(par::pp_maxsnp)<< "SNP" << " "
      << setw(12)<< "STAT" << " "
      << setw(12) << "EMP1" << " ";
  if (par::adaptive_perm)
    ASC << setw(12)<< "NP" << " ";
  else if ( par::mperm_rank )
    ASC << setw(12)<< "EMP3" << " "
        << setw(12)<< "RANK" << " ";
  else
    ASC << setw(12)<< "EMP2" << " ";
  ASC << "\n";
  
  
  for (int l=0; l<nl_all; l++)
    {
      
      // Skip?, if filtering p-values
      if ( par::pfilter && perm.pvalue(l) > par::pfvalue )
        continue;
      
      ASC << setw(4)  << locus[l]->chr << " "
          << setw(par::pp_maxsnp) << locus[l]->name << " ";
      
      ASC << setw(12) << original[l]  << " "
          << setw(12) << perm.pvalue(l) << " ";

      if (par::adaptive_perm)
        ASC << setw(12) << perm.reps_done(l) << " ";
      else if ( par::mperm_rank )
        ASC << setw(12) << perm.max_pvalue(l) << " "
            << setw(12) << perm.rank(l) << " ";
      else
        ASC << setw(12) << perm.max_pvalue(l) << " ";
      
      ASC << "\n";
  
    }
  
  ASC.close();

}



vector_t Plink::testRareDistribution(Perm & perm , bool display)
{
  
  /////////////////////////////////////////////////////////////////////
  // Write results to file

  ofstream OUT;
  
  if ( display )
    {
      string f = par::output_file_name + ".rarer";
      OUT.open(f.c_str(),ios::out);
      printLOG("Writing results to [ " + f + " ]\n");

      OUT << setw(4) << "CHR" << " "
	  << setw(par::pp_maxsnp) << "SNP" << " "
	  << setw(8) << "AFF" << " "
	  << setw(8) << "UNAFF" << " "
	  << setw(5) << "NSNP" << " " 
	  << setw(8) << "AFF2" << " "
	  << setw(8) << "UNAFF2" << " "
	  << setw(8) << "OR" << "\n";

    }

  vector<double> rcount_case(nl_all,0);
  vector<double> rcount_control(nl_all,0);
  
  iIndividual person = sample.begin();
  
  int tot_case = 0;
  int tot_control = 0;

  while ( person != sample.end() )
    {

      Individual * pperson = (*person)->pperson;
      
      if ( pperson->missing ) 
	continue;

      if ( pperson->aff ) 
	tot_case++;
      else
	tot_control++;
      
      //////////////////////////////////////////
      // Consider each SNP for this person
      
      for (int l=0; l<nl_all; l++)
	{
	  
	  //////////////////////////////////////////
	  // X-chromosome, haploid?
	  
	  if ( par::chr_sex[locus[l]->chr] ||
	       par::chr_haploid[locus[l]->chr] )
	    continue;
	  
	  //////////////////////////////////////////////
	  // Only consider alleles below a certain MAF
	  
	  if ( locus[l]->freq > par::rarer_maf_threshold )
	    continue;
	    

	  //////////////////////////////////////////
	  // Get and parse genotypes

	  bool one = (*person)->one[l];
	  bool two = (*person)->two[l];
	  
	  // Skip if missing

	  if ( one && !two )
	    continue;

	  if ( ! one ) 
	    {
	      if ( pperson->aff ) 
		rcount_case[l]++;
	      else
		rcount_control[l]++;
	    }

	  if ( ! two )
	  {
	    if ( pperson->aff ) 
	      rcount_case[l]++;
	    else
	      rcount_control[l]++;
	  }
	  
	} // Next SNP


      person++;


    } // Next Individual




  /////////////////////
  // Smooth over loci

  // Use helper function designed for segmental
  // data

  par::seg_test_window_bp = par::rarer_dist_threshold;

  vector<double> r_case(nl_all);
  vector<double> r_control(nl_all);

  vector<double> nr_case(nl_all);
  vector<double> nr_control(nl_all);

  vector<int> invlist;
  

  // Obtain smoothed scores for case counts

  smoother( *this, 
	    rcount_case, 
	    tot_case, 
	    r_case,
	    nr_case,
	    invlist );

  // Obtain smoothed scores for control counts

  smoother( *this, 
	    rcount_control, 
	    tot_control, 
	    r_control,
	    nr_control,
	    invlist );



  // Calculate statistics on smoothed data
  
  vector_t results(nl_all);
  vector_t oratio;
	    	    
  for (int l=0; l<nl_all; l++)  
    {
      
      double x2 = r_case[l];
      double x1 = r_control[l];

      double c2 = nr_case[l];
      double c1 = nr_control[l];

      
      // pick l-indexes here

      double ratio = ( x2*c1 ) / ( x1*c2 ) ;
      
      // Test statistic (1-sided) average chi-sq
      if ( ratio > 1 )
	results[l] = chi2x2(x2,x1,c2,c1) / (double)invlist[l];
      else
	results[l] = 0;

      if ( display )
	oratio.push_back( ratio );	  
      
    }



  if ( display )
    {
      
      for (int l=0; l<nl_all; l++)
	OUT << setw(4) << locus[l]->chr << " "
	    << setw(par::pp_maxsnp) << locus[l]->name << " "
	    << setw(8) << rcount_case[l] << " "
	    << setw(8) << rcount_control[l] << " "
	    << setw(4) << invlist[l] << " "
	    << setw(8) << r_case[l] << " "
	    << setw(8) << r_control[l] << " "
	    << setw(8) << oratio[l] << "\n";
      
      OUT.close();
    }
  
  
 
  return results;

}

