/*  Plzip - A parallel compressor compatible with lzip
    Copyright (C) 2009 Laszlo Ersek.
    Copyright (C) 2009, 2010 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

class Pretty_print
  {
  const char * const stdin_name;
  unsigned int longest_name;
  std::string name_;
  mutable bool first_post;

public:
  Pretty_print( const std::vector< std::string > & filenames )
    : stdin_name( "(stdin)" ), longest_name( 0 ), first_post( false )
    {
    const unsigned int stdin_name_len = std::strlen( stdin_name );
    for( unsigned int i = 0; i < filenames.size(); ++i )
      {
      const std::string & s = filenames[i];
      const unsigned int len = ( ( s == "-" ) ? stdin_name_len : s.size() );
      if( len > longest_name ) longest_name = len;
      }
    if( longest_name == 0 ) longest_name = stdin_name_len;
    }

  void set_name( const std::string & filename )
    {
    if( filename.size() && filename != "-" ) name_ = filename;
    else name_ = stdin_name;
    first_post = true;
    }

  void reset() const throw() { if( name_.size() ) first_post = true; }
  const char * name() const throw() { return name_.c_str(); }
  void operator()( const char * const msg = 0 ) const throw();
  };


/*--------------------- Defined in compress.cc ---------------------*/

void xinit( pthread_cond_t * cond, pthread_mutex_t * mutex );
void xdestroy( pthread_cond_t * cond, pthread_mutex_t * mutex );
void xlock( pthread_mutex_t * mutex );
void xunlock( pthread_mutex_t * mutex );
void xwait( pthread_cond_t * cond, pthread_mutex_t * mutex );
void xsignal( pthread_cond_t * cond );
void xbroadcast( pthread_cond_t * cond );


class Slot_tally
  {
public:
  unsigned long check_counter;
  unsigned long wait_counter;
private:
  const int num_slots;			// total slots
  int num_free;				// remaining free slots
  pthread_mutex_t mutex;
  pthread_cond_t slot_av;		// free slot available

public:
  Slot_tally( const int slots )
    : check_counter( 0 ), wait_counter( 0 ),
      num_slots( slots ), num_free( slots )
    { xinit( &slot_av, &mutex ); }

  ~Slot_tally() { xdestroy( &slot_av, &mutex ); }

  bool all_free() { return ( num_free == num_slots ); }

  void get_slot()			// wait for a free slot
    {
    xlock( &mutex );
    ++check_counter;
    while( num_free <= 0 )
      { ++wait_counter; xwait( &slot_av, &mutex ); ++check_counter; }
    --num_free;
    xunlock( &mutex );
    }

  void leave_slot()			// return a slot to the tally
    {
    xlock( &mutex );
    if( ++num_free == 1 ) xsignal( &slot_av );
    xunlock( &mutex );
    }
  };


int compress( const int data_size, const int dictionary_size,
              const int match_len_limit, const int num_workers,
              const int num_slots, const int infd, const int outfd,
              const Pretty_print & pp, const int debug_level );


/*-------------------- Defined in decompress.cc --------------------*/

int decompress( const int num_workers, const int num_slots,
                const int infd, const int outfd, const Pretty_print & pp,
                const int debug_level, const bool testing );


/*----------------------- Defined in main.cc -----------------------*/

extern int verbosity;

void fatal();		// terminate the program

void show_error( const char * const msg, const int errcode = 0, const bool help = false ) throw();
void internal_error( const char * const msg );
int readblock( const int fd, uint8_t * const buf, const int size ) throw();
int writeblock( const int fd, const uint8_t * const buf, const int size ) throw();
