
#include "moon.h"
#include "moondata.h"
#include "tws.h"
#include "phase.h"
#include <stdlib.h>
#include <math.h>

static const unsigned char  leftmask[8] = { 0xff, 0x7f, 0x3f, 0x1f, 0x0f, 0x07, 0x03, 0x01 };
static const unsigned char rightmask[8] = { 0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe };

static const unsigned char shades[16][8] = 
{
   { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff },
   { 0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff },
   { 0x7f,0xff,0xdf,0xff,0xff,0xff,0xff,0xff },
   { 0x7f,0xff,0xdf,0xff,0xfe,0xff,0xff,0xff },
   { 0x7f,0xff,0xdf,0xff,0xfe,0xff,0xff,0xf7 },
   { 0x7f,0xfd,0xdf,0xff,0xfe,0xff,0xff,0xf7 },
   { 0x7f,0xfd,0xdf,0xfb,0xfe,0xff,0xff,0xf7 },
   { 0x7f,0xfd,0xdf,0xfb,0xfe,0xff,0x7f,0xf7 },
   { 0x7f,0xfd,0xdf,0xfb,0xfe,0xfb,0x7f,0xf7 },
   { 0x7f,0xfd,0xdf,0xfb,0xbe,0xfb,0x7f,0xf7 },
   { 0x7f,0xfd,0xdf,0xfb,0xbe,0xfb,0x5f,0xf7 },
   { 0x7f,0xfd,0xdf,0xfb,0xbe,0xfb,0x5f,0xf5 },
   { 0x7f,0xfd,0x5f,0xfb,0xbe,0xfb,0x5f,0xf5 },
   { 0x7f,0xf5,0x5f,0xfb,0xbe,0xfb,0x5f,0xf5 },
   { 0x7f,0xf5,0x5f,0xfb,0xae,0xfb,0x5f,0xf5 },
   { 0x5f,0xf5,0x5f,0xfb,0xae,0xfb,0x5f,0xf5 }
};


#define PI 3.14159265358979323846  /* assume not near black hole or in
				      Tennessee */

moon_t *mooncreate()
{
   moon_t *moondata;
   
   moondata = (moon_t*)malloc(sizeof( moon_t ));
   moondata->bitmap = (png_bytep)moon_bits;
   moondata->width  = moon_width;
   moondata->height = moon_height;
   moondata->xbytes = (moon_width + 7) / 8;

   return moondata;
}


void moondestroy( moon_t *moon )
{
   free( moon );
   moon = NULL;
}


void mooncopy(image_t *image, moon_t *moondata, int cx, int cy, int blackflag)
{
   double jd, angphase, cphase, aom, cdist, cangdia, csund, csuang;
   struct tws* t;
   int r = moondata->width/2;
   int i;
   register int x, y;
   int xleft, xright;
   int txleft, txright;
   double fxleft, fxright;
   double fy;
   int bytexleft, bitxleft, bytexright, bitxright;
   int bytetxleft, bittxleft, bytetxright, bittxright;
   int moonoff, imageoff;
   double cap, ratio;
   int shadeindex;
   unsigned char shade;
   int imageoffset;
   
   t = dtwstime();
   
   imageoffset = (cy - moondata->height/2) * image->width + (cx - moondata->width/2);
   
   jd = jtime( t );
   
   angphase = phase( jd, &cphase, &aom, &cdist, &cangdia, &csund, &csuang );
   cap = cos( angphase );
   
   /* Hack to figure approximate earthlighting. */
   if ( cphase < 0.1 ) cphase = 0.1;
   if ( cphase > 0.9 ) cphase = 0.9;
   ratio = (1.0 - cphase) / cphase;	/* ratio varies from 9.0 to 0.111 */
   shadeindex = (int) ( ratio / 9.0 * 15.9999 );
   
   for ( i = 0; i < 2 * r; ++i )
   {
      y = moondata->height/2 - r + i;
      if ( y < 0 || y >= moondata->height )
	 continue;
      fy = i - r;
      fxright = r * sqrt( 1.0 - ( fy * fy ) / ( r * r ) );
      fxleft = - fxright;
      txleft = fxleft + moondata->width/2 + 0.5;
      txright = fxright + moondata->width/2 + 0.5;
      
      if ( angphase >= 0.0 && angphase < M_PI )
	 fxright *= cap;
      else
	 fxleft *= cap;
      
      xleft = fxleft + moondata->width/2 + 0.5;
      xright = fxright + moondata->width/2 + 0.5;
      
      bytexleft = xleft / 8;
      bitxleft = xleft % 8;
      
      bytexright = (xright-1) / 8;
      bitxright = (xright-1) % 8;
      
      bytetxleft = txleft / 8;
      bittxleft = txleft % 8;
      
      bytetxright = (txright-1) / 8;
      bittxright = (txright-1) % 8;
      
      moonoff  = y * ( ( moondata->width + 7 ) / 8 );
      imageoff = (y * ( image->width    ) + imageoffset + 7) / 8;
      
      if ( blackflag )
	 shade = 0xff;
      else
	 shade = shades[shadeindex][y % 8];

      for( x = bytetxleft; x <= bytetxright; x++ )
      {
	 if( bytexleft == x )
	 {
	    image->bitmap[x + imageoff] = moondata->bitmap[x + moonoff] & ~(leftmask[bitxleft] & shade);
	 }
	 else if( bytexright == x )
	 {
	    image->bitmap[x + imageoff] = moondata->bitmap[x + moonoff] & ~(rightmask[bitxright] & shade);
	 }
	 else if( (bytexleft < x) && (x < bytexright) )
	 {
	    image->bitmap[x + imageoff] = moondata->bitmap[x + moonoff] & ~shade;
	 }
	 else
	 {
	    image->bitmap[x + imageoff] = moondata->bitmap[x + moonoff];
	 }
      }
   }
}
