#define __USE_W32_SOCKETS
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#include <sys/wcebase.h>
#include <sys/wceerror.h>
#include <sys/wcetrace.h>
#include <sys/wcenetwork.h>
#include <sys/io.h>

#define CHECKFD(afd) \
  if (afd < 0 || afd >= MAXFDS || _fdtab[afd].fd == -1) { \
    errno = EBADF; \
    return -1; \
  } \
  if( _fdtab[afd].type != IO_FILE_TYPE_SOCKET) { \
    errno = EBADF; \
    return -1; \
  }

static BOOL M$NET_initialized = FALSE;
static WSADATA M$NET_wsadata;
  
BOOL
_m$net_init()
{
  WORD version;

  if (!M$NET_initialized) {
    /* Windows CE 3.0 has only WINSOCK Version 1.1 */
    version = MAKEWORD(1, 1);
    if (WSAStartup(version, &M$NET_wsadata) == 0) {
      WCETRACE(WCE_NETWORK, "_m$net_init: winsock 1.1 initialized");
      M$NET_initialized = TRUE;
      return(TRUE);
    } else {
      WCETRACE(WCE_NETWORK, "_m$net_init: ERROR - can't initialize winsock 1.1");
      return(FALSE);
    }
  }

  return(TRUE);
}
  
int
WSAGetLastError()
{
  return GetLastError();
}

int
accept(int afd, struct sockaddr *addr, socklen_t *addrlen)
{
  SOCKET s;
  SOCKET sr;
  int werr;
  int fd;

  WCETRACE(WCE_IO, "accept: CALLED\n");
  CHECKFD(afd);

  s = (SOCKET)_fdtab[afd].hnd;

  if ((sr = M$_accept(s, addr, addrlen)) == INVALID_SOCKET) {
    werr = GetLastError();
    errno = _winerr2errno(werr);
    return(-1);
  }

  fd = _getnewfd();

  _fdtab[fd].fd = fd;
  _fdtab[fd].type = IO_FILE_TYPE_SOCKET;
  _fdtab[fd].hnd = (HANDLE) sr;

  /*!! is this correct? */
  _fdtab[fd].flags = _fdtab[afd].flags;

  WCETRACE(WCE_IO, "accept: fd = %d, handle = %d", fd, sr);
  return(fd);
}

int
bind(int afd, const struct sockaddr *addr, socklen_t addrlen)
{
  SOCKET s;
  int    werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  if (M$_bind(s, addr, addrlen) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  }

  return(0);
}

int
connect(int afd, const struct sockaddr *addr, socklen_t addrlen)
{
  SOCKET s;
  int    werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  if (M$_connect(s, addr, addrlen) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  }   

  return(0);
}

struct hostent *
gethostbyaddr(const void *addr, socklen_t len, int type)
{
  struct hostent *hp;
  int werr;

  WCETRACE(WCE_IO, "gethostbyaddr CALLED w/\"%s\"", addr);
  if ((hp = M$_gethostbyaddr(addr, len, type)) == NULL) {
    werr = WSAGetLastError();
    errno = _winerr2errno(werr);
    WCETRACE(WCE_IO, "gethostbyaddr ERROR errno %d", errno);
    return(NULL);
  }

  WCETRACE(WCE_IO, "gethostbyaddr returns %p", hp);
  return(hp);
}

struct hostent *
gethostbyname(const char *name)
{
  struct hostent *hp;
  int werr;

  if ((hp = M$_gethostbyname(name)) == NULL) {
    werr = WSAGetLastError();
    errno = _winerr2errno(werr);
    return(NULL);
  }

  return(hp);
}

int
gethostname(char *name, size_t buflen)
{
  int werr;

  if (M$_gethostname(name, buflen) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno = _winerr2errno(werr);
    return(-1);
  }

  return(0);
}

int
getpeername(int afd, struct sockaddr *addr, socklen_t *addrlen)
{
  SOCKET s;
  int werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  if (M$_getpeername(s, addr, addrlen) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  }   

  return(0);
}

int
getsockname(int afd, struct sockaddr *addr, socklen_t *addrlen)
{
  SOCKET s;
  int    werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  if (M$_getsockname(s, addr, addrlen) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  } 

  return(0);
}

int
getsockopt(int afd, int level, int optname, void *optval, socklen_t *optlen)
{
  SOCKET s;
  int r, werr;


  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  WCETRACE(WCE_NETWORK, "getsockopt: fd %d level %d optname %d optval %p optlen %d", 
           afd, level, optname, optval, optlen);
  if ((r = M$_getsockopt(s, level, optname, optval, optlen)) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    WCETRACE(WCE_NETWORK, "getsockopt ERROR r %d errno %d werr %d", r, errno, werr);
    return(-1);
  } 
 
  WCETRACE(WCE_NETWORK, "getsockopt SUCCEEDS, r %d", r);
  return(r);
}

int
listen(int afd, int backlog)
{
  SOCKET s;
  int r, werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  if ((r = M$_listen(s, backlog)) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  }

  return(r);
}

ssize_t
recv(int afd, void *buf, size_t len, int flags)
{
  SOCKET s;
  int r, werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  if ((r = M$_recv(s, buf, len, flags)) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  }

  return(r);
}

ssize_t
recvfrom(int afd, void *buf, size_t len, int flags, struct sockaddr *from, socklen_t *fromlen)
{
  SOCKET s;
  int r, slen, werr;
  int frombufsize = *fromlen;
  struct sockaddr_in remaddr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  WCETRACE(WCE_NETWORK, "recvfrom: fd %d s %d", afd, s);
  slen = sizeof(remaddr);
  memset(&remaddr, 0, sizeof(remaddr));
  if ((r = M$_recvfrom(s, buf, len, 0, (struct sockaddr *)&remaddr, &slen)) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    WCETRACE(WCE_NETWORK, "recvfrom: ERROR errno %d werr %d", errno, werr);
    return(-1);
  }

  /* Winsock's recvfrom() only returns a valid 'from' when the socket
   * is connectionless.  Perl expects a valid 'from' for all types
   * of sockets, so go the extra mile.
   */

 *fromlen = slen;
  memcpy(from, &remaddr, slen);
  if (frombufsize == *fromlen)
    getpeername(s, from, fromlen);

  WCETRACE(WCE_NETWORK, "recvfrom: OK and DONE");
  return(r);
}

ssize_t
send(int afd, const void *buf, size_t len, int flags)
{
  SOCKET s;
  int r, werr;

  CHECKFD(afd);
  s = (SOCKET) _fdtab[afd].hnd;

  if ((r = M$_send(s, (char *)buf, len, flags)) == SOCKET_ERROR) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  }

  return(r);
}

ssize_t
sendto(int afd, const void *buf, size_t len, int flags, const struct sockaddr *to, socklen_t tolen)
{
  SOCKET s;
  int r, werr;

  CHECKFD(afd);
  s = (SOCKET) _fdtab[afd].hnd;

  if ((r = M$_sendto(s, buf, len, flags, to, tolen)) == SOCKET_ERROR) {
     werr = WSAGetLastError();
     errno =  _winerr2errno(werr);
     return(-1);
  }

  return(r);
}

int
setsockopt(int afd, int level, int optname, const void *optval, socklen_t optlen)
{
  SOCKET s;
  int r, werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  WCETRACE(WCE_NETWORK, "setsockopt: fd %d level %d optname %d optval %p optlen %d", 
           afd, level, optname, optval, optlen);
  if ((r = M$_setsockopt(s, level, optname, (char *)optval, optlen)) == SOCKET_ERROR) {
      werr = WSAGetLastError();
      if (werr == WSAENOPROTOOPT) {
       	errno = EINVAL;
      } else {
        errno = _winerr2errno(werr);
      }
      WCETRACE(WCE_NETWORK, "setsockopt ERROR r %d errno %d werr %d", r, errno, werr);
      return(-1);
    }

  WCETRACE(WCE_NETWORK, "setsockopt SUCCEEDS, r %d", r);
  return(r);
}

int
shutdown(int afd, int how)
{
  SOCKET s;
  int r, werr;

  CHECKFD(afd);
  s = (SOCKET)_fdtab[afd].hnd;

  if ((r = M$_shutdown(s, how)) < 0) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    return(-1);
  }

  return(r);
}

int
socket(int af, int type, int protocol)
{
  SOCKET s;
  int fd, werr;

  WCETRACE(WCE_IO, "socket: CALLED");

  if ((s = M$_socket(af, type, protocol)) == INVALID_SOCKET) {
    werr = WSAGetLastError();
    errno =  _winerr2errno(werr);
    WCETRACE(WCE_IO, "socket: ERROR errno %d", errno);
    return(-1);
  }

  fd = _getnewfd();

  _fdtab[fd].fd = fd;
  _fdtab[fd].type = IO_FILE_TYPE_SOCKET;
  _fdtab[fd].hnd = (HANDLE) s;
  _fdtab[fd].flags = 0;

  WCETRACE(WCE_IO, "socket: fd = %d, handle = %d", fd, s);

  return(fd);
}

int
ioctl(int fd, unsigned long request, void *arg)
{
  int werr;
  SOCKET s;

  WCETRACE(WCE_IO, "ioctl(%d, 0x%x)", fd, request);

  CHECKFD(fd);

  /* On WINCE, only IO_FILE_TYPE_SOCKET is handled */
  if (_fdtab[fd].type == IO_FILE_TYPE_SOCKET) {
    s = (SOCKET) _fdtab[fd].hnd;
    WCETRACE(WCE_IO, "doing ioctlsocket w/0x%x (fd %d hnd %x)", request, fd, s);
    if (M$_ioctlsocket(s, request, arg) == SOCKET_ERROR) {
      werr = WSAGetLastError();
      errno =  _winerr2errno(werr);
      return(-1);
    }
  }

  return(0);
}


static int
fd_to_socket(fd_set* set)
{
	int i;
	if (set == NULL)
		return 0;
	for (i = 0; i < set->fd_count; i++) {
		int fd = set->fd_array[i];
		CHECKFD(fd);

		/* On WINCE, only IO_FILE_TYPE_SOCKET is handled */
		if (_fdtab[fd].type == IO_FILE_TYPE_SOCKET) {
			SOCKET s = (SOCKET) _fdtab[fd].hnd;
			WCETRACE(WCE_IO, "select: fd = %d, handle = %d", fd, s);
			set->fd_array[i] = s;
		} else
			WCETRACE(WCE_IO, "select: fd = %d is not a socket", fd);
	}
	return 0;
}


static int
socket_to_fd(fd_set* set)
{
	if (set == NULL)
		return 0;
	// xxx todo translate socket descriptors back to file descriptors
	set->fd_count = 0;
	return 0;
}


int
select(int n, fd_set *readfds, fd_set *writefds, fd_set *exceptfds,
		struct timeval *timeout)
{ 
	int status;

	WCETRACE(WCE_IO, "select(%d, %p, %p, %p, %p)",
			n, readfds, writefds, exceptfds, timeout);
	if (fd_to_socket(readfds) < 0 ||
			fd_to_socket(writefds) < 0 ||
			fd_to_socket(exceptfds) < 0) {
		WCETRACE(WCE_IO, "select: bad fd");
		return -1;
	}

	status = M$_select(n, readfds, writefds, exceptfds, timeout);
	socket_to_fd( readfds);
	socket_to_fd( writefds);
	socket_to_fd( exceptfds);

	if (status == SOCKET_ERROR) {
		int werr = WSAGetLastError();
		errno = _winerr2errno(werr);
		return -1;
	} else
		return status;
}
