//
// StrategyCollection.h
//
// $Id: //poco/1.2/Foundation/include/Poco/StrategyCollection.h#2 $
//
// Library: Foundation
// Package: Cache
// Module:  StrategyCollection
//
// Definition of the StrategyCollection class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef  Foundation_StrategyCollection_INCLUDED
#define  Foundation_StrategyCollection_INCLUDED


#include "Poco/KeyValueArgs.h"
#include "Poco/ValidArgs.h"
#include "Poco/AbstractStrategy.h"
#include "Poco/SharedPtr.h"
#include <vector>


namespace Poco {


template <class TKey, class TValue> 
class StrategyCollection: public AbstractStrategy<TKey, TValue>
	/// An StrategyCollection is a decorator masking n collections as a single one
{
public:
	typedef std::vector<SharedPtr<AbstractStrategy<TKey, TValue> > > Strategies;
	typedef typename Strategies::iterator       Iterator;
	typedef typename Strategies::const_iterator ConstIterator;

public:
	StrategyCollection()
	{
	}

	~StrategyCollection()
	{
	}

	void pushBack(AbstractStrategy<TKey, TValue>* pStrat)
		/// Adds an AbstractStrategy to the collection. Class takes ownership of pointer
	{
		_strategies.push_back(SharedPtr<AbstractStrategy<TKey, TValue> >(pStrat));
	}

	void popBack()
		/// Removes the last added AbstractStrategy from the collection.
	{
		_strategies.pop_back();
	}

	void onAdd(const void* pSender, const KeyValueArgs <TKey, TValue>& key)
		/// Adds the key to the strategy.
		/// If for the key already an entry exists, it will be overwritten.
	{
		Iterator it = _strategies.begin();
		Iterator endIt = _strategies.end();
		for (; it != endIt; ++it)
		{
			(*it)->onAdd(pSender, key);
		}
	}

	void onRemove(const void* pSender, const TKey& key)
		/// Removes an entry from the strategy. If the entry is not found
		/// the remove is ignored.
	{
		Iterator it = _strategies.begin();
		Iterator endIt = _strategies.end();
		for (; it != endIt; ++it)
		{
			(*it)->onRemove(pSender, key);
		}
	}

	void onGet(const void* pSender, const TKey& key)
	{
		Iterator it = _strategies.begin();
		Iterator endIt = _strategies.end();
		for (; it != endIt; ++it)
		{
			(*it)->onGet(pSender, key);
		}
	}

	void onClear(const void* pSender, const EventArgs& args)
	{
		Iterator it = _strategies.begin();
		Iterator endIt = _strategies.end();
		for (; it != endIt; ++it)
		{
			(*it)->onClear(pSender, args);
		}
	}

	void onIsValid(const void* pSender, ValidArgs<TKey>& key)
	{
		Iterator it = _strategies.begin();
		Iterator endIt = _strategies.end();
		for (; it != endIt && key.isValid(); ++it)
		{
			(*it)->onIsValid(pSender, key);
		}
	}

	void onReplace(const void* pSender, std::set<TKey>& elemsToRemove)
	{
		Iterator it = _strategies.begin();
		Iterator endIt = _strategies.end();
		for (; it != endIt; ++it)
		{
			(*it)->onReplace(pSender, elemsToRemove);
		}
	}

protected:
	Strategies _strategies;
};


} // namespace Poco


#endif
