//
// ZipLocalFileHeader.h
//
// $Id: //poco/Main/template/class.h#8 $
//
// Library: Zip
// Package: ChangeThis
// Module:  ZipLocalFileHeader
//
// Definition of the ZipLocalFileHeader class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Zip_ZipLocalFileHeader_INCLUDED
#define Zip_ZipLocalFileHeader_INCLUDED


#include "Poco/Zip/Zip.h"
#include "Poco/Zip/ZipConstants.h"
#include "Poco/DateTime.h"
#include <iostream>

namespace Poco {
namespace Zip {


class Zip_API ZipLocalFileHeader
	/// TODO: Describe class ZipLocalFileHeader...
{
public:
	enum CompressionMethod
	{
		CM_STORE   = 0,
		CM_SHRUNK  = 1,
		CM_FACTOR1 = 2,
		CM_FACTOR2 = 3,
		CM_FACTOR3 = 4,
		CM_FACTOR4 = 5,
		CM_IMPLODE = 6,
		CM_TOKENIZE= 7,
		CM_DEFLATE = 8,
		CM_ENHANCEDDEFLATE = 9,
		CM_DATECOMPRIMPLODING = 10,
		CM_UNUSED = 11
	};


	enum HostSystem
	{
		HS_FAT = 0,  // + PKZIPW 2.50 VFAT, NTFS
		HS_AMIGA = 1,
		HS_VMS = 2,
		HS_UNIX = 3,
		HS_VM_CMS = 4,
		HS_ATARI = 5,
		HS_HPFS = 6,
		HS_MACINTOSH = 7,
		HS_ZSYSTEM = 8,
		HS_CP_M = 9,
		HS_TOPS20 = 10, // used by pkzip2.5 to indicate ntfs
		HS_NTFS = 11,
		HS_SMS_QDOS = 12,
		HS_ACORN = 13,
		HS_VFAT = 14,
		HS_MVS = 15,
		HS_BEOS = 16,
		HS_TANDEM = 17,
		HS_UNUSED = 18,
	};

	ZipLocalFileHeader(std::istream& inp);
		/// Creates the ZipLocalFileHeader.

	virtual ~ZipLocalFileHeader();
		/// Destroys the ZipLocalFileHeader.

	HostSystem getHostSystem() const;

	int getMajorVersionNumber() const;

	int getMinorVersionNumber() const;

	std::size_t getHeaderSize() const;
		/// Returns the size of the header including filename + extra field size

	std::size_t getStartPos() const;
		/// Returns the position of the first byte of the header in the file stream

	std::size_t getEndPos() const;
		/// Points past the last byte of the file entry (ie. either the first byte of the next header, or the directory)

	CompressionMethod getCompressionMethod() const;

	bool isEncrypted() const;

	const Poco::DateTime& lastModifiedAt() const;

	Poco::UInt32 getCRC() const;

	Poco::UInt32 getCompressedSize() const;

	Poco::UInt32 getUncompressedSize() const;

	const std::string& getFileName() const;

private:
	void parse(std::istream& inp);
	void parseDateTime();

	Poco::UInt16 getFileNameLength() const;
	
	Poco::UInt16 getExtraFieldLength() const;

	Poco::UInt16 get16BitValue(Poco::UInt32 pos) const;
	Poco::UInt32 get32BitValue(Poco::UInt32 pos) const;

	bool searchCRCAndSizesAfterData() const;

private:
	enum 
	{
		HEADER_SIZE = 4,
		HEADER_POS = 0,
		VERSION_SIZE = 2,
		VERSION_POS = HEADER_POS+HEADER_SIZE,
		GENERAL_PURPOSE_SIZE = 2,
		GENERAL_PURPOSE_POS = VERSION_POS + VERSION_SIZE,
		COMPR_METHOD_SIZE = 2,
		COMPR_METHOD_POS = GENERAL_PURPOSE_POS + GENERAL_PURPOSE_SIZE,
		LASTMODEFILETIME_SIZE = 2,
		LASTMODEFILETIME_POS = COMPR_METHOD_POS + COMPR_METHOD_SIZE,
		LASTMODEFILEDATE_SIZE = 2,
		LASTMODEFILEDATE_POS = LASTMODEFILETIME_POS + LASTMODEFILETIME_SIZE,
		CRC32_SIZE = 4,
		CRC32_POS = LASTMODEFILEDATE_POS + LASTMODEFILEDATE_SIZE,
		COMPRESSEDSIZE_SIZE = 4,
		COMPRESSEDSIZE_POS = CRC32_POS + CRC32_SIZE,
		UNCOMPRESSEDSIZE_SIZE = 4,
		UNCOMPRESSEDSIZE_POS = COMPRESSEDSIZE_POS + COMPRESSEDSIZE_SIZE,
		FILELENGTH_SIZE = 2,
		FILELENGTH_POS = UNCOMPRESSEDSIZE_POS + UNCOMPRESSEDSIZE_SIZE,
		EXTRAFIELD_LENGTH = 2,
		EXTRAFIELD_POS = FILELENGTH_POS + FILELENGTH_SIZE,
		FULLHEADER_SIZE = 30
	};

	static const char HEADER[HEADER_SIZE];

	char _rawHeader[FULLHEADER_SIZE];
	std::size_t _startPos;
	std::size_t _endPos;
	std::string _fileName;
	Poco::DateTime _lastModifiedAt;
};


inline ZipLocalFileHeader::HostSystem ZipLocalFileHeader::getHostSystem() const
{
	return static_cast<ZipLocalFileHeader::HostSystem>(_rawHeader[VERSION_POS + 1]);
}


inline int ZipLocalFileHeader::getMajorVersionNumber() const
{
	return (_rawHeader[VERSION_POS]/10);
}


inline int ZipLocalFileHeader::getMinorVersionNumber() const
{
	return (_rawHeader[VERSION_POS]%10);
}


inline Poco::UInt16 ZipLocalFileHeader::getFileNameLength() const
{
	return get16BitValue(FILELENGTH_POS);
}


inline Poco::UInt16 ZipLocalFileHeader::getExtraFieldLength() const
{
	return get16BitValue(EXTRAFIELD_POS);
}


inline Poco::UInt16 ZipLocalFileHeader::get16BitValue(Poco::UInt32 pos) const
{
	return static_cast<Poco::UInt16>((unsigned char)_rawHeader[pos])+ (static_cast<Poco::UInt16>((unsigned char)_rawHeader[pos+1]) << 8);
}


inline Poco::UInt32 ZipLocalFileHeader::get32BitValue(Poco::UInt32 pos) const
{
	return static_cast<Poco::UInt32>((unsigned char)_rawHeader[pos])+ (static_cast<Poco::UInt32>((unsigned char)_rawHeader[pos+1]) << 8)+
		(static_cast<Poco::UInt32>((unsigned char)_rawHeader[pos+2]) << 16) + (static_cast<Poco::UInt32>((unsigned char)_rawHeader[pos+3]) << 24);
}


inline std::size_t ZipLocalFileHeader::getHeaderSize() const
{
	return FULLHEADER_SIZE+getExtraFieldLength()+getFileNameLength();
}


inline std::size_t ZipLocalFileHeader::getStartPos() const
{
	return _startPos;
}


inline std::size_t ZipLocalFileHeader::getEndPos() const
{
	return _endPos;
}


inline ZipLocalFileHeader::CompressionMethod ZipLocalFileHeader::getCompressionMethod() const
{
	return static_cast<CompressionMethod>(get16BitValue(COMPR_METHOD_POS));
}


inline bool ZipLocalFileHeader::isEncrypted() const
{
	// bit 0 indicates encryption
	return ((get16BitValue(GENERAL_PURPOSE_POS) & 0x0001) != 0);
}


inline const Poco::DateTime& ZipLocalFileHeader::lastModifiedAt() const
{
	return _lastModifiedAt;
}


inline Poco::UInt32 ZipLocalFileHeader::getCRC() const
{
	return get32BitValue(CRC32_POS);
}


inline Poco::UInt32 ZipLocalFileHeader::getCompressedSize() const
{
	return get32BitValue(COMPRESSEDSIZE_POS);
}


inline Poco::UInt32 ZipLocalFileHeader::getUncompressedSize() const
{
	return get32BitValue(UNCOMPRESSEDSIZE_POS);
}


inline const std::string& ZipLocalFileHeader::getFileName() const
{
	return _fileName;
}


} } // namespace Poco::Zip


#endif // Zip_ZipLocalFileHeader_INCLUDED
