/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric PINSON <cpinson@freesheep.org>
 *  Bruno 'Beosil' Heidelberger
 *
 */

#include "undercal3dStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif


//****************************************************************************//
// Includes                                                                   //
//****************************************************************************//

#include "cal3d/error.h"
#include "cal3d/mixer.h"
#include "cal3d/coremodel.h"
#include "cal3d/corebone.h"
#include "cal3d/coreanimation.h"
#include "cal3d/coretrack.h"
#include "cal3d/corekeyframe.h"
#include "cal3d/model.h"
#include "cal3d/skeleton.h"
#include "cal3d/bone.h"

#ifndef WIN32
#	include "cal3d/blender.h"
#	include "cal3d/animationalt.h"
#endif // WIN32


 /*****************************************************************************/
/** Constructs the blender instance.
  *
  * This function is the default constructor of the blender instance.
  *****************************************************************************/

CalBlender::CalBlender()
{
}

 /*****************************************************************************/
/** Destructs the blender instance.
  *
  * This function is the destructor of the blender instance.
  *****************************************************************************/

CalBlender::~CalBlender()
{
  destroy();
}

/**
 *  Add animation in a channel
 *
 */
bool CalBlender::addAnimation(Channel _channel,CalAnimationAlt* _animation)
{
  if (!_animation)
  {
    CalError::setLastError(CalError::INVALID_HANDLE, __FILE__, __LINE__);
    return false;
  }

  if (_channel==FOREGROUND)
    m_animationForeground.push_back(_animation);

  if (_channel==BACKGROUND)
    m_animationBackground.push_back(_animation);

  return true;
}

/**
 *  Add animation in a channel
 *
 */
void CalBlender::removeAnimation(CalAnimationAlt* _animation)
{
  if (!_animation)
  {
    CalError::setLastError(CalError::INVALID_HANDLE, __FILE__, __LINE__);
    return;
  }

  m_animationForeground.remove(_animation);
  m_animationBackground.remove(_animation);
}

/*****************************************************************************/
/** Creates the blender instance.
 *
 * This function creates the blender instance.
 *
 * @param pModel A pointer to the model that should be managed with this blender
 *               instance.
 *
 * @return One of the following values:
 *         \li \b true if successful
 *         \li \b false if an error happend
 *****************************************************************************/

bool CalBlender::create(CalModel *pModel)
{
  if(pModel == 0)
  {
    CalError::setLastError(CalError::INVALID_HANDLE, __FILE__, __LINE__);
    return false;
  }

  m_pModel = pModel;

  CalCoreModel *pCoreModel;
  pCoreModel = m_pModel->getCoreModel();
  if(pCoreModel == 0)
  {
    CalError::setLastError(CalError::INVALID_HANDLE, __FILE__, __LINE__);
    return false;
  }

  // set the animation time/duration values to default
  m_animationTime = 0.0f;
  m_animationDuration = 0.0f;

  return true;
}

/**
 *  destroy, clear list
 *
 */
void CalBlender::destroy()
{
  m_animationForeground.clear();
  m_animationBackground.clear();
  m_pModel = 0;
}

void CalBlender::updateChannel(Animations &animations, CalSkeleton *pSkeleton)
{
  // get the bone vector of the skeleton
  std::vector<CalBone *>& vectorBone = pSkeleton->getVectorBone();
  
  // loop through the container
  Animations::iterator it;
  for(it = animations.begin(); it != animations.end(); it++) {
    CalAnimationAlt *pAnimation = (*it);
    // get the core animation instance
    CalCoreAnimation *pCoreAnimation = pAnimation->getCoreAnimation();

    // get the list of core tracks of above core animation
    std::list<CalCoreTrack *>& listCoreTrack = pCoreAnimation->getListCoreTrack();

    // loop through all core tracks of the core animation
    std::list<CalCoreTrack *>::iterator itCoreTrack;

#ifdef USE_CACHE_KEYFRAME
    int indexOfCoreTrack=0;
    CalBone** ppBone=&vectorBone.front();
    for(itCoreTrack = listCoreTrack.begin(); itCoreTrack != listCoreTrack.end(); itCoreTrack++) {
      
      CalCoreTrack* trackCandidate=(*itCoreTrack);

      // get the appropriate bone of the track
      CalBone *pBone;
      pBone = ppBone[trackCandidate->getCoreBoneId()];

      // get the current translation and rotation
      CalVector translation;
      CalQuaternion rotation;
      pAnimation->getState(trackCandidate,indexOfCoreTrack,pAnimation->getTime(), translation, rotation);

      // blend the bone state with the new state
      pBone->blendState(pAnimation->getWeight(), translation, rotation);
      indexOfCoreTrack++;
    }
#else
    // previous version
    for(itCoreTrack = listCoreTrack.begin(); itCoreTrack != listCoreTrack.end(); itCoreTrack++) {

      // get the appropriate bone of the track
      CalBone *pBone;
      pBone = vectorBone[(*itCoreTrack)->getCoreBoneId()];

      // get the current translation and rotation
      CalVector translation;
      CalQuaternion rotation;
      (*itCoreTrack)->getState(pAnimation->getTime(), translation, rotation);

      // blend the bone state with the new state
      pBone->blendState(pAnimation->getWeight(), translation, rotation);
    }
#endif
  }
}

/*****************************************************************************/
/** Updates the skeleton.
 *
 * This function updates the skeleton to match the current animation state.
 *****************************************************************************/

void CalBlender::update()
{
  // get the skeleton we need to update
  CalSkeleton *pSkeleton;
  pSkeleton = m_pModel->getSkeleton();
  assert(pSkeleton);

  // clear the skeleton state
  pSkeleton->clearState();

  // update for foreground
  updateChannel(m_animationForeground, pSkeleton);

  // lock the skeleton state
  pSkeleton->lockState();

  // update for foreground
  updateChannel(m_animationBackground, pSkeleton);

  // lock the skeleton state
  pSkeleton->lockState();

  // let the skeleton calculate its final state
  pSkeleton->calculateState();
}
