/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric PINSON <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *
 */

#include "undercal3dStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#include "cal3d/coremodel.h"
#include "cal3d/model.h"

#ifndef WIN32
#	include "cal3d/scheduler.h"
#endif // WIN32

#include <cassert>
#include <algorithm>

#ifdef HACK_SOUND_IN_CAL3D
#include <maf/audio.h>
#endif

/**
 *  Base classe to manage command in pipe
 */
class Entry {

protected:
  double m_launchTime;
  int m_animationId;

public:
  Entry(float launchTime, int animationId) : m_launchTime(launchTime), m_animationId(animationId) {}
  virtual ~Entry() {}

  void setAnimationId(int animationId) { m_animationId = animationId; }
  int getAnimationId() const { return m_animationId; }

  void setLaunchTime(double launchTime) { m_launchTime = launchTime; }
  double getLaunchTime() { return m_launchTime; }

  virtual void launch(CalScheduler* _scheduler) = 0;
};


/**
 *  Entry for command run
 *  used when you want to start an animation
 */
class EntryStart : public Entry
{
public:
  EntryStart(float launchTime, int animationId) : Entry(launchTime, animationId) {}
  virtual void launch(CalScheduler* scheduler);
};

/**
 *  Entry for command stop
 *  used when you want to stop an animation
 */
class EntryStop : public Entry
{
public:
  EntryStop(float launchTime, int animationId) : Entry(launchTime, animationId) {}
  virtual void launch(CalScheduler* scheduler);
};

void EntryStart::launch(CalScheduler* pScheduler)
{
  CalAnimationAlt *pAnimation = pScheduler->getAnimation(m_animationId);
  if (!pAnimation)
    return;

  pScheduler->m_activeAnimations.push_back(pAnimation);
  pScheduler->getBlender()->addAnimation(pAnimation->getChannel(), pAnimation);

#ifdef HACK_SOUND_IN_CAL3D
//   std::cout << "Launch id " << m_animationId << std::endl;
  if (pScheduler->mAnimation2Sounds) {
    MAFAudioController* sound=0;
    std::map<int,osg::ref_ptr<MAFAudioController> >::iterator it=pScheduler->mAnimation2Sounds->find(pAnimation->getCoreAnimationId());
    if (it!=pScheduler->mAnimation2Sounds->end())
      sound=it->second.get();
    if (sound) {
      std::cout << "Play Sound " << sound->GetModel()->GetName() << std::endl;
      sound->Play();
    }
  }
#endif
}


void EntryStop::launch(CalScheduler* pScheduler)
{
  CalAnimationAlt *pAnimation = pScheduler->getAnimation(m_animationId);
  if (!pAnimation)
    return;

  pAnimation->setState(CalAnimationAlt::WANT_TO_STOP);
}

/**
 *  Class to generate unique id for instance animation.
 *  Generated id for a animation start always at 65536
 *  It permits to identify a unique animation ran.
 *  @todo improve it (currently it's too simple to be robust)
 */
const int AnimationIdStart=1<<16;

class IdGenerator
{
  int counter;
  
 public:
  IdGenerator();
  int getUniqId();
  void releaseId(int _id);
};

IdGenerator::IdGenerator()
{
  counter=AnimationIdStart;
}

int IdGenerator::getUniqId()
{
  assert(counter<1<<30); // quick and dirty method
  return counter++;
}

void IdGenerator::releaseId(int _id)
{
  ; // later we would like to control id
}

static IdGenerator idGenerator;

// check if _id is an instance id or a core id
bool CalScheduler::isAnimationId(int anyId)
{
  return anyId >= AnimationIdStart;
}

const int CalScheduler::ALL = -1;
const float CalScheduler::ONCE = 0.0f;
const float CalScheduler::FOREVER = -1.0f;
const float CalScheduler::NOW = 0.f;

CalScheduler::CalScheduler() : m_time(0), m_pModel(0), m_pBlender(0)
{
#ifdef HACK_SOUND_IN_CAL3D
  mAnimation2Sounds=0;
#endif
}

CalScheduler::~CalScheduler()
{
  destroy();
  if(m_pBlender)
    delete m_pBlender;
}

void CalScheduler::destroy()
{
  for(Entries::iterator it = m_entries.begin(); it != m_entries.end(); it++)
    delete (*it);

  if(m_pBlender)
    m_pBlender->destroy();

  for(Id2Animation::iterator it = m_id2animation.begin(); it != m_id2animation.end(); it++)
    delete it->second;

  m_entries.clear();
  m_activeAnimations.clear();
  m_id2animation.clear();
}

bool CalScheduler::create(CalModel *pModel)
{
  if(m_pBlender == 0)
    m_pBlender = new CalBlender;

  destroy();
  
  m_time=0;

  m_pBlender->create(pModel);

  m_pModel = pModel;

  return true;
}

void CalScheduler::updateAnimation(float deltaInSeconds)
{
//  printf("%.9f\n",m_time);
  m_time += deltaInSeconds;
//  printf("%.9f\n",m_time);
  bool possible_side_effect;
  int maxloop = 5;

  do {
    maxloop--;
    possible_side_effect = false;

    // launch entries for which time has come and delete them
    for(Entries::iterator it = m_entries.begin(); it != m_entries.end();)
    {
      if((*it)->getLaunchTime() < m_time + 1e-4)
      {
	(*it)->launch(this);
	delete (*it);
	it = m_entries.erase(it);
	possible_side_effect = true;
      }
      else
      {
	it++;
      }
    }

    for (ActiveAnimations::iterator it = m_activeAnimations.begin(); it != m_activeAnimations.end(); )
    {
      CalAnimationAlt* animation = *it;
      animation->update(deltaInSeconds);
      if(animation->getState() == CalAnimationAlt::STOPPED)
      {
	m_pBlender->removeAnimation(animation);
	m_id2animation.erase(animation->getAnimationId());
	it = m_activeAnimations.erase(it);
	if(animation->getStopCallback())
	{
	  animation->getStopCallback()->process(getModel(), animation);
	  possible_side_effect = true;
	}
	delete animation;
      }
      else
      {
	it++;
      }
    }

    deltaInSeconds = 0.f;
  } while(possible_side_effect && maxloop > 0);

  if(maxloop <= 0) {
    //    abort();
  }
}

void CalScheduler::updateSkeleton()
{
  m_pBlender->update();
}


/**
 *  Create an animation from \b coreAnimationId and start running it
 *  after \b delay seconds. The animation will last \b length
 *  seconds. Animations run in the CalBlender::FOREGROUND \b channel
 *  have precendence over the animations run in the
 *  CalBlender::BACKGROUND \b channel. An animation will be granted
 *  a weight relative to the total number of animations running in
 *  the same channel (i.e. two animations will have a weight of 0.5
 *  (1/2), three animations a weight of 0.3 (1/3) and so on). The \b
 *  weight argument controls the actual weight that will be used. If
 *  two animations are running and one of them was launched with a
 *  \b weight argument equal to 0.25 (1/4), it will only use a
 *  quarter of what it could (i.e a 1/4 of 1/2). The \b weight
 *  argument may be further altered while the animation is running
 *  if a \b pWeightFunction function is provided.
 *
 *  The \b run arguments allow to specify the most commonly used
 *  parameters. Additional tuning of the animation to run may be
 *  done using the returned CalAnimationAlt instance, such as
 *  setting a callback function to be called when the animation
 *  stops (see CalAnimationAlt::setStopCallback) or a function that
 *  accelerates or slow down the animation (see
 *  CalAnimationAlt::setTimeFunction).
 *
 *  The return pointer MUST NOT be permanently stored by the
 *  caller. Instead, if a reference to the running animation is
 *  needed for later reference, the caller MUST rely on the
 *  CalAnimationAlt::getAnimationId method that will return an uniq
 *  identifier.
 *
 *  Example 1: run animation 3 once in the foreground.
 *  \verbatim
 *  scheduler->run(CalScheduler::FOREGROUND, 3, CalScheduler::ONCE);
 *  \endverbatim
 *
 *  Example 2: cycle animation 4 in the background.
 *  \verbatim
 *  scheduler->run(CalScheduler::BACKGROUND, 4, CalScheduler::FOREVER);
 *  \endverbatim
 * 
 *  Example 3: fade in animation 4 (it takes 0.5f (1/2) seconds to reach 
 *  the nominal weight 1.f) and cycle in the background.
 *  \verbatim
 *  scheduler->run(CalScheduler::BACKGROUND, 4, CalScheduler::FOREVER, 
 *                 1.f, new CalScheduler::FadeIn(.5f));
 *  \endverbatim
 *
 *  Example 4: fade in animation 4 during 1/2 second and cycle in the background.
 *  \verbatim
 *  scheduler->run(CalScheduler::BACKGROUND, 4, CalScheduler::FOREVER, 
 *                 1.f, new CalScheduler::FadeIn(.5f));
 *  \endverbatim
 * 
 *  Example 5: cycle animation 4 in the background, starting in 5 seconds.
 *  \verbatim
 *  scheduler->run(CalScheduler::BACKGROUND, 4, CalScheduler::FOREVER, 
 *                 1.f, NULL, 5.f);
 *  \endverbatim
 * 
 *  Example 6: run animation 5 twice in the foreground, with a fade in 
 *  of 1 second and a fade out of 2 second.
 *  \verbatim
 *  scheduler->run(CalScheduler::FOREGROUND, 5, 2,
 *                 1.f, CalScheduler::FadeInOut(1.f, 2.f));
 *  \endverbatim
 *
 *  Example 7: cycle animation 6 in the background and register
 *  a callback that is to be called when it will stop. 
 *
 *  \verbatim
 *  class MyCallback : public CalScheduler::StopCallback { ... }
 *  MyCallback* cb = new MyCallback;
 *
 *  scheduler->run(CalScheduler::BACKGROUND, 6, CalScheduler::FOREVER)
 *    ->setStopCallback(cb);
 *
 *  \endverbatim
 * 
 *  Example 8: cycle animation 20 in the background, get the 
 *  id that uniquely identifies the animation, and schedule it to be
 *  stopped after 5 minutes, with a fade out of 1 second.
 *  \verbatim
 *  int id = scheduler->run(CalScheduler::BACKGROUND, 20, CalScheduler::FOREVER)
 *             ->getAnimationId();
 *  scheduler->stop(id, CalScheduler::FadeOut(1.f), 5 * 60.f);
 *  \endverbatim
 *
 *  @param channel Either CalScheduler::FOREGROUND or CalScheduler::BACKGROUND
 * 
 *  @param coreAnimationId The id of a CalCoreAnimation as returned by 
 *  CalCoreModel::loadCoreAnimation.
 *
 *  @param length The animation will last exactly length seconds.
 *  The constants CalScheduler::ONCE runs the animation once,
 *  CalScheduler::FOREVER runs the animation forever.
 *  
 *  @param weight The weight of the animation in the range [0.f,1.f].
 *  If N animations run in a given \b channel, the actual weight of
 *  the animation will be (1/N) * \b weight.
 *
 *  @param pWeightFunction An instance of a subclass of
 *  CalScheduler::WeightFunction that modifies the \b weight of the
 *  animation according to the current state of the
 *  animation. Pre-defined functions include CalScheduler::FadeIn,
 *  CalScheduler::FadeOut and CalScheduler::FadeInOut. The
 *  pWeightFunction pointer will be deleted when the animation stops
 *  or when another WeightFunction is set using the CalAnimationAlt
 *  accessors. The caller MUST NOT delete the pWeightFunction
 *  pointer. If a NULL pointer is specified it means that the \b
 *  weight is not altered in any way.
 *
 *  @param delay The number of seconds to wait before running the
 *  animation.
 *  
 *  @return A pointer to an internal CalAnimationAlt instance or a
 *  null pointer if an error occured. The caller MUST NOT delete the
 *  returned pointer.
 */
CalAnimationAlt* CalScheduler::run(Channel channel,
				   int coreAnimationId,
				   float length,
				   float weight,
				   WeightFunction *pWeightFunction,
				   float delay)
{
  int animationId = createAnimation(coreAnimationId);
  if(animationId == -1)
  {
    return 0;
  }

  CalAnimationAlt* pAnimation = getAnimation(animationId);
  if(!pAnimation)
    return 0;

  pAnimation->setChannel((CalBlender::Channel)channel);
  pAnimation->setWeightFunction(pWeightFunction);
  pAnimation->setCoreAnimationId(coreAnimationId);
  pAnimation->setAnimationId(animationId);
  pAnimation->setLength(length);
  pAnimation->setReferenceWeight(weight);

  m_entries.push_back(new EntryStart((float)m_time + delay, animationId));
  if (pWeightFunction)
    delete pWeightFunction;
  return pAnimation;
}

/**
 *  Ask the \b anyId animation(s) to stop running. If a \b
 *  pWeightFunction is specified it may control the fade out of the
 *  animation (if it is an instance * of CalScheduler::FadeOut). If
 *  the \b delay argument is specified, the animation will be asked
 *  to stop running after \b delay seconds.
 *
 *  Note that the \b stop function does not stop and delete the
 *  animation at the moment of the call. It politely asks the
 *  animation to stop at its earliest convenience. In the most
 *  simple case, such as an animation run with
 *  run(CalBlender::BACKGROUND, 1, CalBlender::FOREVER), it will
 *  stop immediately. However, if a fade out was specified using
 *  CalScheduler::FadeOut(2.f), the animation will fade out during 2
 *  seconds before stopping. Stated in more general terms, a
 *  CalScheduler::WeightFunction or CalScheduler::TimeFunction
 *  registered in the animation may decide to delay the termination
 *  of the animation. If no such functions are registered in the
 *  animation, it will terminate without delay.
 * 
 *  The \b stop arguments allow to specify the most commonly used
 *  parameters. Additional tuning of the animation to stop may be
 *  done using the returned CalAnimationAlt instance, such as
 *  setting a callback function to be called when the animation
 *  actually stops (see CalAnimationAlt::setStopCallback) or a
 *  function that accelerates or slow down the animation (see
 *  CalAnimationAlt::setTimeFunction).
 *
 *  The return pointer MUST NOT be permanently stored by the
 *  caller. Instead, if a reference to the stopped animation is
 *  needed for later reference, the caller MUST rely on the
 *  CalAnimationAlt::getAnimationId method that will return an uniq
 *  identifier.
 *
 *  @param anyId A uniq number designating either a CalAnimationAlt
 *  or a CalCoreAnimation. The ids returned by
 *  CalAnimationAlt::getAnimationId or CalScheduler::createAnimation
 *  designate CalAnimationAlt instances. The ids returned by
 *  CalCoreModel::loadCoreAnimation designate CalCoreAnimation
 *  instances. If a CalCoreAnimation id is provided, all
 *  CalAnimationAlt instance created on the basis of this
 *  CalCoreAnimation are stopped. If a CalAnimationAlt id is
 *  provided, the corresponding running animation is stopped.
 *  If \b anyId is -1, all running animations are stopped.
 *
 *  @param pWeightFunction An instance of a subclass of
 *  CalScheduler::WeightFunction that modifies the \b weight of the
 *  animation according to the current state of the animation. The
 *  pre-defined function CalScheduler::FadeOut is likely to be the
 *  most useful. The pWeightFunction pointer will be deleted when
 *  the animation stops or when another WeightFunction is set using
 *  the CalAnimationAlt accessors. The caller MUST NOT delete the
 *  pWeightFunction pointer. If a NULL pointer is specified, this
 *  will remove any previously registered function, reverting to the
 *  default behaviour (i.e. the reference weight is not altered).
 *
 *  @param delay The number of seconds to wait before stopping the
 *  animation.
 *
 *  @return A pointer to an internal CalAnimationAlt instance or a
 *  null pointer if an error occured. If more than one
 *  CalAnimationAlt instance was stopped, the last one is
 *  returned. The caller MUST NOT delete the returned pointer.
 */
CalAnimationAlt* CalScheduler::stop(int anyId,
				    WeightFunction *pWeightFunction,
				    float delay)
{
  typedef std::vector<int> AnimationIds;
  AnimationIds animationIds;
  getAnimationIdsFromAnyId(anyId, animationIds);
  if(animationIds.empty())
    return 0;

  CalAnimationAlt* animation = 0;
  for(AnimationIds::iterator it = animationIds.begin(); it != animationIds.end(); it++) 
  {
    animation = getAnimation(*it);
    if(!animation)
      return 0;
    if(pWeightFunction)
      animation->setWeightFunction(pWeightFunction);
    Entry* entry = new EntryStop((float)m_time + delay, (*it));
    m_entries.push_back(entry);
  }

  if (pWeightFunction)
    delete pWeightFunction;

  return animation;
}

/**
 *  Return a list of \b animationIds (i.e. ids of CalAnimationAlt
 *  instances) matching \b anyId. If \b anyId designate a
 *  CalAnimationAlt instance, it is returned unmodified as a 
 *  result. If \b anyId designate a CalCoreAnimation instance,
 *  the ids of all CalAnimationAlt instances created on the basis
 *  of this CalCoreAnimation instance are returned.
 *
 *  It is most useful for a function that needs to translate an id
 *  (not knowing if it designates a CalCoreAnimation or a
 *  CalAnimationAlt instance) into one or more CalAnimationAlt
 *  instances ids.
 *
 *  @param anyId A uniq number designating either a CalAnimationAlt
 *  or a CalCoreAnimation. The ids returned by
 *  CalAnimationAlt::getAnimationId or CalScheduler::createAnimation
 *  designate CalAnimationAlt instances. The ids returned by
 *  CalCoreModel::loadCoreAnimation designate CalCoreAnimation
 *  instances. 
 *
 *  @param animationIds A reference to a vector of CalAnimationAlt
 *  ids.  The vector is emptied at the beginning of the function. It
 *  is filled with the CalAnimationAlt ids matching the argument \b
 *  anyId, as described above. When the function returns, the
 *  \b animationIds vector is empty if no CalAnimationAlt instance id 
 *  matched \b anyId.
 *
 *  @return A list of CalAnimationAlt instance ids in the \b animationIds
 *  argument.
 *
 */
void CalScheduler::getAnimationIdsFromAnyId(int anyId, std::vector<int>& animationIds)
{
  animationIds.clear();
  if(isAnimationId(anyId))
  {
    if(m_id2animation.find(anyId) != m_id2animation.end())
      animationIds.push_back(anyId);
  }
  else
  {
    for(Id2Animation::iterator it = m_id2animation.begin(); it != m_id2animation.end(); it++)
      // anyId < 0 means get all ids
      if(anyId < 0 || (*it).second->getCoreAnimationId() == anyId)
        animationIds.push_back((*it).first);
  }
}

/*
 * Return animation from is instance id
 */
CalAnimationAlt* CalScheduler::getAnimation(int anyId)
{
  std::vector<int> animationIds;
  getAnimationIdsFromAnyId(anyId, animationIds);

  if(animationIds.empty())
    return 0;

  int animationId = animationIds.front();

  Id2Animation::iterator it = m_id2animation.find(animationId);
  if(it == m_id2animation.end())
    return 0;

  return (*it).second;
}

int CalScheduler::createAnimation(int coreAnimationId)
{
  int animationId = idGenerator.getUniqId();
  CalCoreAnimation *coreAnimation = getModel()->getCoreModel()->getCoreAnimation(coreAnimationId);
  assert(coreAnimation);

  // create a new animation instance
  CalAnimationAlt* pAnimation = new CalAnimationAlt(coreAnimation);

  // check if the generated id is ok
  Id2Animation::iterator it = m_id2animation.find(animationId);
  if(it != m_id2animation.end())
    assert(0 && "duplicate animationId");
  m_id2animation[animationId] = pAnimation;
  pAnimation->setAnimationId(animationId);

  return animationId;
}

bool CalScheduler::isAnimationActive(int anyId)
{
  CalAnimationAlt *pAnimation = getAnimation(anyId);
  if (pAnimation == 0)
    return false;

  ActiveAnimations::iterator begin = m_activeAnimations.begin();
  ActiveAnimations::iterator end =  m_activeAnimations.end();
  ActiveAnimations::iterator found = std::find(begin, end, pAnimation);
  return (found != end);
}

#if 0

#include <iostream>
void CalScheduler::dump()
{
  for(Id2Animation::iterator it = m_id2animation.begin(); it != m_id2animation.end(); it++)
    std::cout << "animation " << (*it).second->getCoreAnimationId() << " is in m_id2animation\n";
}

#endif
