/* *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Henry Prcheur <henry@precheur.org>
 *  Cedric Pinson <cpinson@freesheep.org>
 *  Igor Kravtchenko <igor@obraz.net>
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#include <cstdio>
# define snprintf _snprintf
#endif


#include <maf/profile.h>

#include <osg/MatrixTransform>

#include <maf/maferror.h>
#include <maf/packets.h>
#include <maf/scene.h>
#include <maf/cursor.h>
#include <maf/utils.h>
#include <maf/animate2d.h>
#include <maf/MultipleAnimationPathCallback.h>
#include <maf/window.h>
#include <ugame/animated.h>
#include <ugame/debug.h>
#include <string>

#ifndef WIN32
#	include "PokerApplication.h"
#	include "PokerPlayer.h"
#	include "PokerCard.h"
#	include "Poker.h"
#	include "PokerChipsStack.h"
#	include "PokerSeat.h"
#	include "PokerMoveChips.h"
#	include "PokerCamera.h"
#	include "PokerCardProjector.h"
#       include "PokerCursor.h"
#endif // WIN32

#include <osgDB/Registry>
#include <osgDB/ReadFile>
#include <osgDB/WriteFile>
#include <PokerBoard.h>
#include <PokerBody.h>

struct PokerTableController : UGAMEArtefactController
{
};


PokerModel::PokerModel(PokerApplication* game) : mSeats(0), mMe(0), mGameSerial(0), mShowStackToolTip(false), mGame(game), mGameId(0) {
  std::string seats_string = game->HeaderGet("sequence", "/sequence/seat/@count");
  if(seats_string == "") g_error("PokerModel::PokerModel: failed to read /sequence/seat/@count");
  mSeats = atoi(seats_string.c_str());

  for(int seat = 0; seat < mSeats; seat++) {
    mSeat2Serial.push_back(0);
    mSeat2Player.push_back(0);
  }


  //
  // The board
  //
  mBoard=new PokerBoardController(game); // problem here for win32 in debug
  game->AddController(mBoard.get());
  
  {
    std::string dealer_name = mGame->HeaderGet("sequence", "/sequence/dealer/@url");
    MAFOSGData* data = dynamic_cast<MAFOSGData*>(mGame->mDatas->GetVision(dealer_name));
    if(data == 0)
      g_error("PokerModel: cannot do without dealer button");
    mDealerButton = data->GetGroup();
  }

  mSeatManager = new PokerSeatManager();
  mSeatManager->Init(game);
  game->AddController(mSeatManager.get());

  mPotCenter= new PokerPotController(game); // problem here for win32 in debug
  game->AddController(mPotCenter.get());

  mFrontDoor = new PokerDoorController();
  mFrontDoor->Init(game, "/sequence/doors/door01");
  game->AddController(mFrontDoor.get());

  mBackDoor = new PokerDoorController();
  mBackDoor->Init(game, "/sequence/doors/door02");
  game->AddController(mBackDoor.get());

  mActiveBet2Pot=new PokerTrackActiveMoveChips(mSerial2Player);
  mActivePot2Player=new PokerTrackActiveMoveChips(mSerial2Player);
  mEventRunAnimationProcess=false;

  mDealCard=0;
  mDealTime=0;
  mLastPlayerDealedCard=0;
  mDealPlayerCards.clear();

  std::string paramMinTimeToDealCard=game->HeaderGet("sequence", "/sequence/seat/@minTimeToDealCard");
  if (paramMinTimeToDealCard.empty())
    g_error("PokerModel::PokerModel /sequence/seat/@minTimeToDealCard not found");
  mParamTimeToDealCard=atof(paramMinTimeToDealCard.c_str());


#if 0 // not cat for the demo
  if (mGame->GetScene()->GetModel()->GetAudio()) {
    std::string sound = mGame->HeaderGet("sequence", "/sequence/sound/endgame/@sound");
    std::string soundanchor = mGame->HeaderGet("sequence", "/sequence/sound/endgame/@anchor");
    if (soundanchor.empty())
      g_error("PokerModel::PokerModel /sequence/sound/endgame/@anchor not found in configuration file");

    MAFAnchor* anchor=mGame->mSetData->GetAnchor(soundanchor.c_str());
    if (!anchor)
      g_error("PokerModel::PokerModel anchor %s not found",soundanchor.c_str());
    
    if (sound.empty())
      g_error("PokerModel::PokerModel /sequence/sound/endgame/@sound not found in configuration file");
    MAFAudioData* data = mGame->mDatas->GetAudio(sound);
    if (!data)
      g_error("PokerModel::PokerModel sound %s not found in data",sound.c_str());

    MAFAudioModel* audio_model = new MAFAudioModel;
    audio_model->SetName(sound);
    audio_model->SetData(data);
    audio_model->SetAmbient(true);
    mSoundEndGame = new MAFAudioController;
    mSoundEndGame->SetModel(audio_model);
    mSoundEndGame->Init();
    mSoundEndGame->AttachTo(anchor->asGroup());
  }
#endif

}

PokerModel::~PokerModel() {
  mActiveBet2Pot=0;
  mActivePot2Player=0;

  mGame->RemoveController(mPotCenter.get());
  mPotCenter=0;

  mGame->RemoveController(mSeatManager.get());
  mSeatManager = 0;

  RecursiveClearUserData(mFrontDoor->GetModel()->mDoorCollNode.get());
  mGame->RemoveController(mFrontDoor.get());
  mFrontDoor = 0;

  RecursiveClearUserData(mBackDoor->GetModel()->mDoorCollNode.get());
  mGame->RemoveController(mBackDoor.get());
  mBackDoor = 0;

  //  g_debug("PokerModel::~PokerModel");

}



void PokerModel::SortStack(std::vector<PokerMoveChipsCommand>& stackToSort,bool toPlayer)
{
  for (std::vector<PokerMoveChipsCommand>::iterator i=stackToSort.begin();i!=stackToSort.end();)
    if ((*i).IsEmpty() || mSerial2Player.find((*i).mSerial) == mSerial2Player.end() )
      i=stackToSort.erase(i);
    else
      i++;
}


bool PokerModel::IsAnyChipsToMoveToPot()
{
  int e=(int)mBet2PotCommand.size();
  for (int i=0;i<e;i++)
    if (!mBet2PotCommand[i].IsEmpty())
      return true;
  return false;
}

bool PokerModel::IsAnyChipsToMoveToPlayer()
{
  int e=(int)mPot2PlayerCommand.size();
  for (int i=0;i<e;i++)
    if (!mPot2PlayerCommand[i].IsEmpty())
      return true;
  return false;
}

float PokerModel::RunChipsAnimationBet2Pot()
{
  std::vector<int> betNull(0);
  float res=0;
  int e=(int)mBet2PotCommand.size();
  for (int i=0;i<e;i++) {
    if (mSerial2Player.find(mBet2PotCommand[i].mSerial)!=mSerial2Player.end()) {
      PokerPlayer* p=mSerial2Player[mBet2PotCommand[i].mSerial].get();
      PokerMoveChipsBet2PotController* a=p->GetFreeAnimationBet2Pot();
      if (a) {
	mPotCenter->BuildAnimationBetToPot(a,mBet2PotCommand[i].mIndex);
	a->mChips->SetChips(mBet2PotCommand[i].mChips);
	a->StartAnimation();
 	a->SetTargetChipsStack(mPotCenter->GetPotChipsStack(mBet2PotCommand[i].mIndex));
	mActiveBet2Pot->AddEntry(PokerTrackActiveMoveChips::EntryElement(mBet2PotCommand[i].mSerial,a));
	p->SetBet(betNull);
      }
    }
  }
  return res;
}
float PokerModel::RunChipsAnimationPot2Player()
{
  float res=0;
  int e=(int)mPot2PlayerCommand.size();
  for (int i=0;i<e;i++) {
    if (mSerial2Player.find(mPot2PlayerCommand[i].mSerial)!=mSerial2Player.end()) {
      PokerPlayer* p=mSerial2Player[mPot2PlayerCommand[i].mSerial].get();
      PokerMoveChipsPot2PlayerController* a=p->GetFreeAnimationPot2Player();
      if (a) {
	mPotCenter->BuildAnimationPotToPlayer(a,mPot2PlayerCommand[i].mIndex);
	// sub this tokens to pot but it does not work because the stack is not empty at the end so reset instead
	//mPotCenter->GetPotChipsStack(mPot2PlayerCommand[i].mIndex)->SubTokens(mPot2PlayerCommand[i].mChips);
	a->mChips->SetChips(mPot2PlayerCommand[i].mChips);
	a->SetTargetChipsStack(p->mBetStack.get());
	a->StartAnimation();
	mActivePot2Player->AddEntry(PokerTrackActiveMoveChips::EntryElement(mPot2PlayerCommand[i].mSerial,a));
      }
    }
  }
  mPotCenter->ResetPots();
  return res;
}


/*
 * End chips animation related
 */ 










PokerController::PokerController(PokerApplication* game)
{
  SetModel(new PokerModel(game));
  SetView(new PokerView());
  Init();
  
  
#ifdef TEST_PERFORMANCE
  mPerfCharacteres=1;  
#endif
}

PokerController::~PokerController() {

}

void PokerController::SetPot(const std::vector<int>& amount,int index) {
  static_cast<PokerModel*>(GetModel())->mPotCenter->SetPotValue(amount,index);
}

void PokerController::SetCards(const std::vector<int>& cards) 
{
  GetModel()->mBoard->SetCards(cards);
}


void PokerController::FoldCards(void) 
{
  GetModel()->mBoard->FoldCards();
  GetModel()->mBoard->StopToDisplayShowDown();
}

void PokerController::PythonAccept(MAFPacket* packet)
{
  PROFILE_SAMPLE("PokerController::PythonAccept");

  PokerApplication* game = GetModel()->mGame;

  if(packet->IsType("POKER_SEATS")) {
    std::vector<guint>& seat2serial = GetModel()->mSeat2Serial;
    std::vector<osg::ref_ptr<PokerPlayer> >& seat2player = GetModel()->mSeat2Player;
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    std::vector<int> seats;
    packet->GetMember("seats", seats);

    int index = 0;

    unsigned int seatsCount = GetModel()->mSeatManager->GetSeatsCount();
    if(seatsCount > seats.size())
    g_error("PokerController::PythonAccept: seatsCount = %d > seats = %d", seatsCount, seats.size());
    for(unsigned int i = 0; i < seatsCount; i++) {
    int serial = seats[i];
    if(serial == 0) {
      //player leave
      if (seat2serial[index] != 0)
      {
        GetModel()->mSeatManager->PlayerLeave(index);
      }
      seat2serial[index] = 0;
      seat2player[index] = 0;
    } else {

      //player arrive
      if (seat2serial[index] == 0) {
	GetModel()->mSeatManager->PlayerSeated(index);
      }
      if(serial2player.find(serial) == serial2player.end())
        g_error("PokerController::Update: SEATS: serial %d not matching any player", serial);

      PokerPlayer* player = serial2player[serial].get();
      seat2serial[index] = serial;
      seat2player[index] = player;
      player->SetSeatId(index);
    }      
    index++;
    }

  } else if(packet->IsType("POKER_TABLE") ||
	    packet->IsType("POKER_TABLE_DESTROY")) { 
    g_debug("PokerController::Update: POKER_TABLE(_DESTROY)");

    if(packet->IsType("POKER_TABLE")) {
      packet->GetMember("id", GetModel()->mGameSerial);
      unsigned int seatsCount = 0;
      packet->GetMember("seats", seatsCount);
      g_assert((seatsCount > 0) && (seatsCount <= 10));
      std::vector<int> seatsAll;
      packet->GetMember("seats_all", seatsAll);
      g_assert(seatsAll.size() == seatsCount);
      GetModel()->mSeatManager->SetSeats(seatsAll);

    } else {
      GetModel()->mSeatManager->DisableAllSeats();
    }

    GetModel()->mPotCenter->ResetPots();

    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    std::vector<osg::ref_ptr<PokerPlayer> >& seat2player = GetModel()->mSeat2Player;
    std::vector<guint>& seat2serial = GetModel()->mSeat2Serial;
    for(std::map<guint,osg::ref_ptr<PokerPlayer> >::iterator i = serial2player.begin(); i != serial2player.end(); i++) {
      PokerPlayer* player = i->second.get();
      // Leave the seat
      seat2serial[player->GetSeatId()] = 0;
      seat2player[player->GetSeatId()] = 0;
      player->SetSeatId(POKER_PLAYER_NO_SEAT);

      // remove player controler else the player is not deleted
      game->RemoveController(player);
      
    }
    serial2player.clear();

    std::vector<int> cards;
    GetModel()->mBoard->SetCards(cards);
      
  } else if(packet->IsType("POKER_START")) {

    packet->GetMember("game_id", GetModel()->mGameId);
    //
    // Fold all cards
    //
    GetModel()->mBoard->SetShadeOfCardsToDefault();
    //GetModel()->mBoard->FoldCards();
    GetModel()->mBoard->StopToDisplayShowDown();
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    for(std::map<guint,osg::ref_ptr<PokerPlayer> >::iterator i = serial2player.begin(); i != serial2player.end(); i++) {
      PokerPlayer* player = i->second.get();
      guint serial = i->first;
      player->NoCards();
    if(serial == GetModel()->mMe)
      ; // player->EndFolding(); now in python
    }

    GetModel()->mBoard->SetHandValue("");

  } else if(packet->IsType("POKER_PLAYER_NO_CARDS")) { 
    guint serial = 0;
    packet->GetMember("serial", serial);
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(serial) == serial2player.end()) 
      g_error("PokerController::Update: PLAYER_NO_CARDS: serial %d not in player list", serial);
    PokerPlayer* player = serial2player[serial].get();
    player->NoCards();

  } else if(packet->IsType("POKER_IN_GAME")) { 
    g_debug("PokerController::Update: POKER_IN_GAME");

    //
    // Build a map of the the InGame status for each player based
    // on the list of InGame players found in the packet.
    //
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    std::map<guint,bool> serial2inGame;
    for(std::map<guint,osg::ref_ptr<PokerPlayer> >::iterator i = serial2player.begin();
	i != serial2player.end();
	i++) {
      serial2inGame[i->first] = false;
    }

    std::vector<int> serials;
    packet->GetMember("players", serials);

    for(std::vector<int>::iterator i = serials.begin(); i != serials.end(); i++) {
      int serial = *i;
      if(serial2player.find(serial) == serial2player.end())
	g_error("PokerController::Update: IN_GAME: serial %d not matching any player", serial);
      serial2inGame[serial] = true;
    }

    for(std::map<guint,osg::ref_ptr<PokerPlayer> >::iterator i = serial2player.begin();
	i != serial2player.end();
	i++) {
      PokerPlayer* player = (i->second).get();
      guint serial = i->first;
      player->SetInGame(serial2inGame[serial]);
      if(serial2inGame[serial] == false && player->GetInGame() == true)
	player->SetInGame(false);
    }

  } else if(packet->IsType("POKER_SIT")) {
    guint serial = 0;
    packet->GetMember("serial", serial);
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(serial) == serial2player.end()) 
      g_error("PokerController::Update: SIT: serial %d not in player list", serial);
    PokerPlayer* player = serial2player[serial].get();
    player->SetSit(true);
      
  } else if(packet->IsType("POKER_SIT_OUT")) { 
    guint serial = 0;
    packet->GetMember("serial", serial);
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(serial) == serial2player.end()) 
      g_error("PokerController::Update: SIT_OUT: serial %d not in player list", serial);
    PokerPlayer* player = serial2player[serial].get();
    player->SetSit(false);
      
  } else if(packet->IsType("POKER_CALL") || packet->IsType("POKER_RAISE")) { 
    guint serial = 0;
    packet->GetMember("serial", serial);
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(serial) == serial2player.end()) 
      g_error("PokerController::Update: BET: serial %d not in player list", serial);
    PokerPlayer* player = serial2player[serial].get();
    player->Bet();
	player->SetLastActionString("Bet");

  } else if(packet->IsType("POKER_CLIENT_ACTION")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(GetModel()->mMe) != serial2player.end()) {
      PokerPlayer* player = serial2player[GetModel()->mMe].get();
      std::string name;
      packet->GetMember("action", name);
      bool valid;
      packet->GetMember("display", valid);
      (void)player;

      //player->DisplayInteractor(name, valid);
    } else {
      g_debug("PokerController::Update: CLIENT_ACTION: cannot find self (%d) in player list", GetModel()->mMe);
    }
  } else if(packet->IsType("POKER_CHECK")) { 
    guint serial = 0;
    packet->GetMember("serial", serial);
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(serial) == serial2player.end()) 
      g_error("PokerController::Update: CHECK: serial %d not in player list", serial);
    PokerPlayer* player = serial2player[serial].get();
    player->Check();
	player->SetLastActionString("Check");
  } else if(packet->IsType("POKER_FOLD")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    unsigned	serial;
    packet->GetMember("serial", serial);
    if(serial2player.find(serial) != serial2player.end()) {
//       serial2player[serial]->FoldPocketCards();
//      serial2player[serial]->FoldHoleCards();
      // serial2player[serial]->Folding(); now in python
      serial2player[serial]->GetShowdown()->Reset();
	  serial2player[serial]->SetLastActionString("Fold");
    } else {
      g_critical("PokerController::Update: FOLD: serial %d not matching any player (ignored)", serial);
    }

  } else if(packet->IsType("POKER_PLAYER_CHIPS")) { 
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    std::vector<int> amounts;
    packet->GetMember("bet", amounts);

	int money = 0;
	for(unsigned int i = 0; i < amounts.size(); i += 2) {
	  money = amounts[i] * amounts[i + 1];
	}

    guint serial = 0;
    packet->GetMember("serial", serial);
      
    if(serial2player.find(serial) == serial2player.end())
      g_error("PokerController::Update: PLAYER_CHIPS: serial %d not matching any player", serial);

    PokerPlayer* player = serial2player[serial].get();
    player->SetBet(amounts);
	player->SetLastActionString("Bet");

    packet->GetMember("money", amounts);
    player->SetMoney(amounts);
	player->SetLastBet(money);


  } else if(packet->IsType("POKER_POT_CHIPS")) { 
//     std::vector<int> amounts;
//     int index=0;
//     packet->GetMember("bet", amounts);
//     SetPot(amounts,index); now we use the BET2POT packet


  } else if(packet->IsType("POKER_PLAYER_CARDS") ||
	    packet->IsType("POKER_BEST_CARDS")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    guint serial = 0;
    packet->GetMember("serial", serial);
    std::vector<int> cards;
    packet->GetMember("cards", cards);

    if(serial2player.find(serial) == serial2player.end())
      g_error("PokerController::Update: {PLAYER|BEST}_CARDS: serial %d not matching any player", serial);

    PokerPlayer* player = serial2player[serial].get();
    if(packet->IsType("POKER_PLAYER_CARDS")) {
      std::string visibles;
      packet->GetMember("visibles", visibles);

      // first set pockets cards if any
      const int	visible_mask = ~(64 - 1);
      const int	value_mask = (64 - 1);

      std::vector<int>	tmp;
      //      int nbMaskedCards=0;
	
      for (std::vector<int>::iterator i = cards.begin(); i != cards.end(); i++)
	if (*i & visible_mask) {
	  tmp.push_back(*i & value_mask);
	}
      player->SetPocketCards(tmp);

      // here get card to display it after animation of card
      // display them now if now animation is aplying
      if (!(GetModel()->mDealCard && !GetModel()->mDealPlayerCards.empty())) {
// 	g_debug("Tmp size %d",tmp.size());
	for (int c=0;c<(int)tmp.size();c++)
	  player->ShowCard(c);
      }

      tmp.clear();
      for (std::vector<int>::iterator i = cards.begin(); i != cards.end(); i++)
	if (!(*i & visible_mask))
	  tmp.push_back(*i & value_mask);
      if(visibles == "hole")
	player->SetHoleCards(tmp);
      else {
	std::vector<int> dummy;
	player->GetShowdown()->Set("", tmp, dummy, "");
      }

    } else if(packet->IsType("POKER_BEST_CARDS")) {
      std::string side;
      std::string hand;
      std::vector<int> board;
      packet->GetMember("board", board);
      packet->GetMember("hand", hand);
      packet->GetMember("side", side);
      player->GetShowdown()->Set(side, cards, board, hand);

      GetModel()->mBoard->ShadeUsedCardForBestHand(cards);
      GetModel()->mBoard->SetHandValue(hand);
      GetModel()->mBoard->StartToDisplayShowDown();
      
    } else {
      g_error("PokerController::Update: {PLAYER|BEST}_CARDS: unknown packet type");
    }

  } else if(packet->IsType("POKER_BOARD_CARDS")) {
    std::vector<int> cards;
    packet->GetMember("cards", cards);
    SetCards(cards);

  } else if(packet->IsType("POKER_BET_LIMIT")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(GetModel()->mMe) != serial2player.end()) {
      PokerPlayer* player = serial2player[GetModel()->mMe].get();
      int min_bet;
      packet->GetMember("min", min_bet);
      int max_bet;
      packet->GetMember("max", max_bet);
      int step;
      packet->GetMember("step", step);
      int call;
      packet->GetMember("call", call);
      player->BetLimits(min_bet, max_bet, step, call);
    }
      
  } else if(packet->IsType("POKER_POSITION")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    unsigned	serial_in_position;
    packet->GetMember("serial", serial_in_position);
    PokerPlayer* player_in_position = 0;

    if(serial_in_position != 0) {
      if(serial2player.find(serial_in_position) == serial2player.end())
	g_error("PokerController::Update: POSITION: serial_in_position %d not matching any player", serial_in_position);
      player_in_position = serial2player[serial_in_position].get();
    }

    std::vector<guint>& seat2serial = GetModel()->mSeat2Serial;
    for(std::vector<guint>::iterator i = seat2serial.begin();
	i != seat2serial.end();
	i++) {
      unsigned int serial = *i;
      if(serial != 0) {
	if(serial2player.find(serial) == serial2player.end())
	  g_error("PokerController::Update: POSITION: found serial %d not matching any player", serial);
	PokerPlayer* player = serial2player[serial].get();
	if(serial != serial_in_position) {
	  //
	  // Other players look at the player in position
	  // and stop any animation they may have if they were previously
	  // in position.
	  //
	  if(player_in_position  && player->GetInGame())
	    player->LookAt(player_in_position);
	  //
	  // We are about to lose position: no interactor selected
	  //
	  if(serial_in_position != GetModel()->mMe && player->HasPosition())
	    ; //player->ToggleInteractor("none");
	  player->LostPosition();
	} else {
	  player->InPosition();
	  //
	  // We are in position: interactors that were selected
	  // while we were not in position must return to their
	  // unselected state.
	  //
	  if(serial_in_position == GetModel()->mMe)
	    ;//player->ToggleInteractor("none");
	}
      }
    }

  } else if(packet->IsType("POKER_DEALER")) {
    std::vector<guint>& seat2serial = GetModel()->mSeat2Serial;
    unsigned int dealer_seat;
    packet->GetMember("serial", dealer_seat);

    assert(dealer_seat>=0 && dealer_seat<seat2serial.size());
    if(GetModel()->mDealerButton->getParents().size() > 0)
      GetModel()->mDealerButton->getParent(0)->removeChild(GetModel()->mDealerButton.get());
    char tmp[32];
    snprintf(tmp, 32, "transform_button%02d", dealer_seat + 1);
    osg::Group* anchor = game->mSetData->GetAnchor(tmp);
    if(anchor == 0)
      g_critical("POKER_DEALER: could not find anchor %s", tmp);
    else
      anchor->addChild(GetModel()->mDealerButton.get());

  } else if(packet->IsType("SERIAL")) {
    guint serial = 0;
    packet->GetMember("serial", serial);
    g_debug("my serial is %d", serial);
    GetModel()->mMe = serial;
      
  } else if(packet->IsType("POKER_CHAT_HISTORY")) {
    std::string yesno;
    packet->GetMember("show", yesno);
    MAFName2Animate& name2animate = game->GetInterface()->GetName2Animate();
    MAFApplication2DAnimate* animate = name2animate["ChatHistory"];
    g_assert(animate);
    for(MAFApplication2DAnimate::iterator i = animate->begin(); i != animate->end(); i++) {
      MAFApplication2DSlide* slide = dynamic_cast<MAFApplication2DSlide*>(i->get());
      if(slide)
	slide->SetVisible(yesno == "yes");
    }

  } else if(packet->IsType("POKER_PLAYER_ARRIVE")) { // new player
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
      
    guint serial = 0;
    packet->GetMember("serial", serial);
    g_debug("packet player arrives with serial %d", serial);

    std::string name;
    packet->GetMember("name", name);

    std::string outfitName;
    packet->GetMember("outfit", outfitName);

    std::string display_body = game->HeaderGet("sequence", "/sequence/player/@display");

    PokerPlayer* player = new PokerPlayer(game, GetModel()->mMe == serial, outfitName);

    if (serial == GetModel()->mMe)
      GetModel()->mSeatManager->MainPlayerArrive(GetModel()->mSeat2Serial);


    player->SetName(name);

    if(serial2player.find(serial) != serial2player.end())
      g_error("PokerController::Update: PLAYER_ARRIVE: serial %d is matching an existing player", serial);

    serial2player[serial] = player;
	player->SetLastActionString("Joined");

	
	game->AddController(player);
  } else if(packet->IsType("POKER_WIN")) { 
#if 0 // now in python
    std::vector<int> winner_serials;
    packet->GetMember("serials", winner_serials);
    bool i_win = false;
    std::vector<guint>& seat2serial = GetModel()->mSeat2Serial;
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;

    g_debug("PokerController::Update: WIN got %d winners", winner_serials.size());

    for(std::vector<guint>::iterator i = seat2serial.begin();
	i != seat2serial.end();
	i++) {
      guint serial = *i;

      if(serial != 0) {
	if(serial2player.find(serial) == serial2player.end())
	  g_error("PokerController::Update: WIN: found serial %d not matching any player", serial);
	PokerPlayer* player = serial2player[serial].get();
	player->SetLastActionString("Won");

	//
	// Animate winner and losers
	//
	bool winner = false;
	for(std::vector<int>::iterator j = winner_serials.begin(); j != winner_serials.end(); j++) {
	  g_debug("PokerController::Update: WIN %d == %d", *j, serial);
	  if((guint)*j == serial) {
	    winner = true;
	    break;
	  }
	}
	    
	if(winner) {
	  player->Win();
	} else {
	  player->Loose();
	}

	if(serial == GetModel()->mMe && winner)
	  i_win = true;
      }
    }

#endif
#if 0 // not cat for the demo
	if (GetModel()->mSoundEndGame.get())
      GetModel()->mSoundEndGame->Play();
#endif

  } else if (packet->IsType("POKER_CHAT")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    guint serial;
    packet->GetMember("serial", serial);
    typedef std::map<guint,osg::ref_ptr<PokerPlayer> >::iterator It;
    It playerIt = serial2player.find(serial);
    if (playerIt != serial2player.end())
      {
	PokerPlayer* player = playerIt->second.get();
	std::string message;
	packet->GetMember("message", message);
	player->SetTextMessage(message);
      }

  } else if(packet->IsType("POKER_PLAYER_LEAVE")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    guint serial;
    packet->GetMember("serial", serial);
    if(serial2player.find(serial) == serial2player.end()) {
      g_critical("PokerController::Update POKER_PLAYER_LEAVE, unexpected serial %d (ignored)\n", serial);
    } else {

      std::vector<osg::ref_ptr<PokerPlayer> >& seat2player = GetModel()->mSeat2Player;
      std::vector<guint>& seat2serial = GetModel()->mSeat2Serial;
      PokerPlayer* player = serial2player[serial].get();
      int index = player->GetSeatId();
      GetModel()->mSeatManager->PlayerLeave(index);

      {

	game->RemoveController(player);
	// Leave the seat
	seat2serial[player->GetSeatId()] = 0;
	seat2player[player->GetSeatId()] = 0;
	player->SetSeatId(POKER_PLAYER_NO_SEAT);
	if(player->referenceCount() != 1)
	  g_critical("POKER_LAYER_LEAVE: refcount: %d, should be 1", player->referenceCount());
	// Leave game
	serial2player.erase(serial);
      }

      {
	//seatManager
	if (serial == GetModel()->mMe)
		GetModel()->mSeatManager->MainPlayerLeave(GetModel()->mSeat2Serial);
      }
    }
  } else if(packet->IsType("QUIT")) {
    game->Quit();
  } else if (packet->IsType("POKER_TIMEOUT_WARNING")) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    guint serial;
    guint timeout;
    packet->GetMember("serial", serial);
    packet->GetMember("timeout", timeout);

    if(serial2player.find(serial) == serial2player.end()) {
      g_critical("PokerController::Update POKER_TIMEOUT_WARNING, unexpected serial %d (ignored)\n", serial);
    }
    PokerPlayer* player = serial2player[serial].get();

    /*
     *  If i am on another table and I received a POKER_TIMEOUT_WARNING player=0 and it crashes
     *
     */
    if (!player)
      g_critical("PokerController::PythonAccept POKER_TIMEOUT_WARNING received but no player with id %d on current game",serial);
    else
      player->TimeoutAdvise(timeout*1.0);

    g_debug("timeout warning serial %d time %d",serial,timeout);

  } else if(packet->IsType("POKER_CHIPS_BET2POT")) { 
    guint serial = 0;
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    std::vector<int> amounts;
    int pot=0;
    packet->GetMember("serial", serial);
    packet->GetMember("chips", amounts);
    packet->GetMember("pot", pot);
      
    if(serial2player.find(serial) == serial2player.end())
      g_error("PokerController::Update: POKER_MOVE_CHIPS: serial %d not matching any player", serial);

    PokerPlayer* player = serial2player[serial].get();
    if (!player)
      g_error("Player with serial %d does not exist",serial);
    else {
      GetModel()->mBet2PotCommand.push_back(PokerModel::PokerMoveChipsCommand(serial,amounts,pot));
    }

  } else if(packet->IsType("POKER_CHIPS_POT2PLAYER")) { 
    guint serial = 0;
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    std::vector<int> amounts;
    int pot;
    packet->GetMember("serial", serial);
    packet->GetMember("chips", amounts);
    packet->GetMember("pot", pot);
      
    if(serial2player.find(serial) == serial2player.end())
      g_error("PokerController::Update: POKER_POT2PLAYER: serial %d not matching any player", serial);

    PokerPlayer* player = serial2player[serial].get();
    if (!player)
      g_error("Player with serial %d does not exist",serial);
    else {
      GetModel()->mPot2PlayerCommand.push_back(PokerModel::PokerMoveChipsCommand(serial,amounts,pot));
    }

  } else if(packet->IsType("POKER_CHIPS_PLAYER2BET")) { 
//     guint serial = 0;
//     std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
//     std::vector<int> amounts;
//     packet->GetMember("serial", serial);
//     packet->GetMember("chips", amounts);
  } else if (packet->IsType("POKER_END_ROUND")) {

    GetModel()->SortStack(GetModel()->mBet2PotCommand,false);
    GetModel()->SortStack(GetModel()->mPot2PlayerCommand,true);

  } else if (packet->IsType("POKER_CHIPS_POT_RESET")) {
//     GetModel()->mResetPot=true;
//     GetModel()->mPotCenter->ResetPots();

  } else if (packet->IsType("POKER_DEAL_CARDS")) {

//     if (!GetModel()->mDealPlayerCards.empty())
//       assert(0);

    int nbCards=0;
    std::vector<int> serials;
    packet->GetMember("numberOfCards", nbCards);
    packet->GetMember("serials", serials);
    GetModel()->mNbCardToDealPerPlayer=nbCards;
    GetModel()->mDealCard=1;
    GetModel()->mDealTime=-1;
    GetModel()->mCurrentPlayerToSendCard=0;
    GetModel()->mDealPlayerCards=serials;
    

  } else if (packet->IsType("POKER_DISPLAY_NODE")) {
    std::string state;
    std::string node;
    packet->GetMember("state", state);
    packet->GetMember("node", node);
    

    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(GetModel()->mMe) != serial2player.end()) {
      PokerPlayer* player = serial2player[GetModel()->mMe].get();
      std::string name;
      packet->GetMember("name", name);
      // consider doing an std::map
      if (name == "check")
	player->mCheckInteractor->Accept(packet);
      else if (name == "fold")
	player->mFoldInteractor->Accept(packet);
    }

  } else if (packet->IsType("POKER_DISPLAY_CARD")) {
    guint game_id;
    packet->GetMember("game_id", game_id);

    if(game_id == GetModel()->mGameId) {
      std::vector<int> indexes;
      int state=0;
      guint serial;
      packet->GetMember("index", indexes);
      packet->GetMember("serial", serial);
      packet->GetMember("display", state);
    
      std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
      if(serial2player.find(serial) != serial2player.end()) {
	PokerPlayer* player = serial2player[serial].get();
	int end=(int)indexes.size();
	for (int i=0;i<end;i++)
	  if (state)
	    player->ShowCard(indexes[i]);
	  else
	    player->HideCard(indexes[i]);

      }
    }
  } else if (packet->IsType("POKER_ANIMATION_PLAYER_NOISE")) {
    guint serial;
    std::string action;
    packet->GetMember("serial", serial);
    packet->GetMember("action", action);
    
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(serial) != serial2player.end()) {
      PokerPlayer* player = serial2player[serial].get();
      g_debug("noise action %s",action.c_str());
      if (action=="start") {
// 	assert(0);
	player->GetBody()->PlayFacialNoise();
      } else if (action=="stop") {
// 	assert(0);
	player->GetBody()->StopFacialNoise();
      }
    }
  } else if (packet->IsType("POKER_ANIMATION_PLAYER_FOLD")) {
    guint serial;
    std::string animation;
    packet->GetMember("serial", serial);
    packet->GetMember("animation", animation);
    
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(serial) != serial2player.end()) {
      PokerPlayer* player = serial2player[serial].get();
      g_debug("fold with animation %s",animation.c_str());
      player->GetBody()->GetModel()->PlayFold(animation);
    }
  } else {
    g_critical("PokerController::Update unknown packet type (%d)", (int)packet->GetType());
  }
}

static void	request_quit_application(PokerApplication* game)
{
  game->PythonCall("quit");
}

bool PokerController::Update(MAFApplication* application) {
  PokerApplication* game = dynamic_cast<PokerApplication*>(application);

  // check if ESC has been pressed even when keyboard is locked by
  // another controller
  SDL_Event* event = game->GetLastEventIgnoreLocking();
  if (event &&
      event->type == SDL_KEYDOWN &&
      event->key.keysym.sym == SDLK_ESCAPE)
    request_quit_application(game);

  event = game->GetLastEvent(this);
  if (event) { // && game->GetFocus() == 0) {
    if (event->type == SDL_KEYDOWN) {
      switch(event->key.keysym.sym) {
      case SDLK_F11:
	{
	  game->PythonCall("reload");
	}
	break;
      case SDLK_F1:
	{
	  game->PythonCall("showKeyBinding");
	}
	break;
      case SDLK_F2:
	{
	  game->PythonCall("showUserInfo");
	}
	break;
      case SDLK_F3:
	{
	  game->PythonCall("showOutfits");
	}
	break;
      case SDLK_F4:
	{

	  GetModel()->mDealCard=1;
	  GetModel()->mDealTime=-1;
	  GetModel()->mLastPlayerDealedCard=(*GetModel()->mSerial2Player.begin()).first;
	}
	break;
      case SDLK_F6:
	{
	  game->PythonCall("logout");
	}
	break;
      case SDLK_F5: // dump osg tree in file
	{
	  osgDB::writeObjectFile(*(game->mSetData->GetGroup()),"dump.osg");
	}
	break;
      case SDLK_F7:
	{
	  PokerCameraController* camera = dynamic_cast<PokerCameraController*>(game->GetScene()->GetModel()->mCamera.get());
	  PokerCameraModel::Mode mode = camera->GetModel()->GetMode();
	  if (mode == PokerCameraModel::CAMERA_GAME_MODE)
	    camera->GetModel()->SetMode(PokerCameraModel::CAMERA_LEAVE_MODE);
	  game->PythonCall("queryHands");
	}
	break;
      case SDLK_F8:
	{
	  game->PythonCall("queryAllHands");
	}
	break;
      case SDLK_F10:
	{
	  std::vector<int> cards;
	  cards.push_back(0);
	  cards.push_back(1);
	  cards.push_back(2);
	  cards.push_back(3);
	  cards.push_back(4);
	  GetModel()->mBoard->SetCards(cards);
	  std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
	  if(serial2player.find(GetModel()->mMe) != serial2player.end()) {
	    PokerPlayer* player = serial2player[GetModel()->mMe].get();
	    std::vector<int> cards;
	    std::vector<int> board;
	    cards.push_back(10);
	    cards.push_back(11);
	    cards.push_back(12);
	    cards.push_back(13);
	    player->SetPocketCards(cards);
	    cards.push_back(14);
	    player->GetShowdown()->Set("hi", cards, board, "Hi Hand");
	    player->SetHoleCards(cards);
	    cards.clear();
	    cards.push_back(20);
	    cards.push_back(21);
	    cards.push_back(22);
	    cards.push_back(23);
	    cards.push_back(24);
	    player->GetShowdown()->Set("low", cards, board, "Low Hand");
	  }
	}
	break;
      case SDLK_F12:
	{
	  game->PythonCall("replayStep");
	}
	break;
      case SDLK_TAB:
	{
	  // toggle tooltips
	}
	break;
      default:
// 	game->GetInterface()->HandleDefault(event); // FIXME input
	break;
      }
    }
    else if (event->type == SDL_KEYUP)
      ;
//       game->GetInterface()->HandleDefault(event); // FIXME input
    else if (event->type == SDL_QUIT)
      {
	// see escape
	request_quit_application(game); // FIXME input
      }
  } else if(game->GetLastEvent(this) == NULL) {
    std::map<guint,osg::ref_ptr<PokerPlayer> >& serial2player = GetModel()->mSerial2Player;
    if(serial2player.find(GetModel()->mMe) != serial2player.end()) {
      PokerPlayer* player = serial2player[GetModel()->mMe].get();

      if (player->GetSelectedValue() != 0) {

	g_debug("PokerController::Update: bet");
	osg::ref_ptr<MAFPacket> packet = game->GetPacketsModule()->Create("PacketPokerRaise");
	packet->SetMember("serial", GetModel()->mMe);
	std::vector<int>	_chips(42, 0); // FIXME !!!! wrong
					       // vector size
	_chips[0] = player->GetSelectedValue();
	  
	packet->SetMember("amount", _chips);
	//	packet->SetMember("serial", GetModel()->mMe);
	//	packet->SetMember("game_id", GetModel()->mGameSerial);
	//	game->PythonCall("scheduleAction", packet.get());
	game->PythonCall("sendBetPacket", packet.get());

	player->ResetSelectedValue();

	// no more raise interactor ?
	//player->ToggleInteractor("raise");
      }
      
      bool checkInteractorSelected = player->mCheckInteractor->GetSelected();
      bool foldInteractorSelected = player->mFoldInteractor->GetSelected();
      
      if (checkInteractorSelected) {
	osg::ref_ptr<MAFPacket> packet = game->GetPacketsModule()->Create("Packet");
	packet->SetMember("name", std::string("check"));
	packet->SetMember("event", std::string("selected"));
	game->PythonCall("interactorSelected", packet.get());
	player->mCheckInteractor->SetSelected(false);
      }

      if (foldInteractorSelected) {
	osg::ref_ptr<MAFPacket> packet = game->GetPacketsModule()->Create("Packet");
	packet->SetMember("name", std::string("fold"));
	packet->SetMember("event", std::string("selected"));
	game->PythonCall("interactorSelected", packet.get());
	player->mFoldInteractor->SetSelected(false);
      }

    }

    float dt=GetDeltaFrame()/1000.0;

    // Deal card for each player in game
    if (GetModel()->mDealCard && !GetModel()->mDealPlayerCards.empty()) {

      float mParamAnimationToDeal1Card=0;
      PokerModel::Serial2Player& s2p=GetModel()->mSerial2Player;
      PokerPlayer* player=0;
      if (!s2p.empty()) {
	player=s2p.begin()->second.get();
	osg::MultipleAnimationPathCallback* cbTmp=static_cast<osg::MultipleAnimationPathCallback*>(player->GetAnimationCard(0)->getUpdateCallback());
	mParamAnimationToDeal1Card=cbTmp->getAnimationDuration();
      }

      if (GetModel()->mDealCard<=GetModel()->mNbCardToDealPerPlayer) {
	if (GetModel()->mDealTime<0) {

	  // play animation of a player
	  PokerModel::Serial2Player::iterator it=s2p.find(GetModel()->mDealPlayerCards[GetModel()->mCurrentPlayerToSendCard]);
	  player=0;
	  if (it!=s2p.end()) {
	    player=(*it).second.get();
	    player->AnimateCard(GetModel()->mDealCard-1);
	  
	    GetModel()->mDealTime=GetModel()->mParamTimeToDealCard;

	  // we have sent a card to all player, restart to the first
	    GetModel()->mCurrentPlayerToSendCard++;
	    if (GetModel()->mCurrentPlayerToSendCard==(int)GetModel()->mDealPlayerCards.size()) {
	      GetModel()->mCurrentPlayerToSendCard=0;
	      GetModel()->mDealCard++;
	    }
	  } else {
	    GetModel()->mDealPlayerCards.erase(GetModel()->mDealPlayerCards.begin()+GetModel()->mCurrentPlayerToSendCard);
	    if (GetModel()->mCurrentPlayerToSendCard==(int)GetModel()->mDealPlayerCards.size()) {
	      GetModel()->mCurrentPlayerToSendCard=0;
	      GetModel()->mDealCard++;
	    }
	  }
	}
      }


      // check if cards are at the target position
      int cardAtTheTarget=0;
      for (int i=0;i<(int)GetModel()->mDealPlayerCards.size();i++) {
	PokerModel::Serial2Player::iterator it=s2p.find(GetModel()->mDealPlayerCards[i]);
	player=0;
	if (it!=s2p.end())
	  player=(*it).second.get();
	if (player)
	  for (int j=0;j<GetModel()->mNbCardToDealPerPlayer;j++) {
	    osg::MultipleAnimationPathCallback* cb=0;
	    cb=static_cast<osg::MultipleAnimationPathCallback*>(player->GetAnimationCard(j)->getUpdateCallback());
	    
	    if (cb->getAnimationTime()>cb->getAnimationDuration()-1e-4) {
	      player->HideAnimateCard(j);
// 	      g_debug("display card %d",player->GetNbCardsDisplayed()+1);
	      player->ShowCard(player->GetNbCardsDisplayed());
	    } else if (cb->getAnimationTime()<1e-4)
	      cardAtTheTarget++;
	  }
	else
	  cardAtTheTarget++; // we assume that card for this player is ok
      }

      if (GetModel()->mDealCard>GetModel()->mNbCardToDealPerPlayer && cardAtTheTarget>=(int)GetModel()->mDealPlayerCards.size()*GetModel()->mNbCardToDealPerPlayer) {
 	GetModel()->mDealCard=0;
	GetModel()->mDealPlayerCards.clear();
      }
      GetModel()->mDealTime-=dt;
    }




    //Door interaction
    PokerDoorController &frontDoor = *GetModel()->mFrontDoor.get();
    PokerDoorController &backDoor = *GetModel()->mBackDoor.get();
    if (frontDoor.GetSelected() || backDoor.GetSelected())
      {	
	PokerCameraController* camera = dynamic_cast<PokerCameraController*>(game->GetScene()->GetModel()->mCamera.get());
	PokerCameraModel::Mode mode = camera->GetModel()->GetMode();
	if (mode == PokerCameraModel::CAMERA_FREE_MODE)
	  {
	    // wait to return in freemode
	    // before handle door selection
	    game->PythonCall("wantToLeave");
	    frontDoor.SetSelected(false);
	    backDoor.SetSelected(false);
	  }
      }
    

    // chips animation
    bool isBet2PotChips=GetModel()->IsAnyChipsToMoveToPot();
    bool isPot2PlayerChips=GetModel()->IsAnyChipsToMoveToPlayer();
    bool isfreezed=GetModel()->mPotCenter->IsFreezed();
    bool isStoped=GetModel()->mPotCenter->IsStopped();


    if (GetModel()->mEventRunAnimationProcess && !GetModel()->mActivePot2Player->HasAnimation() && !GetModel()->mActiveBet2Pot->HasAnimation()
	&& isStoped && isfreezed) {
      GetModel()->mPotCenter->UnFreezeCenter();
      GetModel()->mEventRunAnimationProcess=false;
    }


    if (isBet2PotChips) {
      if (!GetModel()->mPotCenter->IsFreezed()) {
	GetModel()->mPotCenter->FreezeCenter();
      } else if (GetModel()->mPotCenter->IsStopped()) {
	GetModel()->RunChipsAnimationBet2Pot();
	GetModel()->mBet2PotCommand.clear();
	GetModel()->mEventRunAnimationProcess=true;
      }
    }


    if (!GetModel()->mActiveBet2Pot->HasAnimation() && !GetModel()->IsAnyChipsToMoveToPot()) { // don't play winner chips if there are Bet2Pot Animation active or if packets Bet2Pot need to be animated
      if (isPot2PlayerChips) {
	if (!GetModel()->mPotCenter->IsFreezed()) {
	  GetModel()->mPotCenter->FreezeCenter();
	} else if (GetModel()->mPotCenter->IsStopped()) {
	  GetModel()->mEventRunAnimationProcess=true;
	  GetModel()->RunChipsAnimationPot2Player();
	  GetModel()->mPot2PlayerCommand.clear();
	}
      }
    }


    // update track of animation
    GetModel()->mActivePot2Player->RemoveFinishedEntry();
    GetModel()->mActiveBet2Pot->RemoveFinishedEntry();
  }


//   for (PokerModel::Serial2Player::iterator i=GetModel()->mSerial2Player.begin();i!=GetModel()->mSerial2Player.end();i++) {
//     (*i).second->GetCount();
//   }
      

  return true;
}


bool PokerModel::PokerTrackActiveMoveChips::HasAnimation() 
{
  return mActives.size()!=0;
}

void PokerModel::PokerTrackActiveMoveChips::RemoveFinishedEntry() 
{
  for (std::vector<EntryElement>::iterator i=mActives.begin();i!=mActives.end();) {
    if (mSerial2Player.find((*i).mSerial)==mSerial2Player.end()) {// test first if the player is already here
//       g_debug("Serial not found remove animation");
      i=mActives.erase(i);
    } else if ((*i).mAnimation->mFinished) {
//       g_debug("Finished %d -> %d",i,mActives[i].mAnimation->mFinished);
      i=mActives.erase(i);
    } else
      i++;
  }
}



#if 0



class AnimateFoldCards 
{

  int mShake;

  osg::ref_ptr<osg::MatrixTransform> mTransformShake;
  osg::ref_ptr<osg::MatrixTransform> mTransformPosition;
  osg::Matrix mTarget;
  osg::ref_ptr<osg::MultipleAnimationPathCallback> mAnimationCallback;

  MAFBezierInterpolator<float> mInterpolator;
  double mTimer;
  double mMoveDuration;

  void LoadKeys(std::vector<osg::Vec2> &keys, MAFXmlData *data, const std::string &name);

  template<typename T> void LoadSpline(T &interpolator, MAFXmlData *data, const std::string &name) {
    std::vector<osg::Vec2> keys;
    LoadKeys(keys, data, name);
    interpolator.InitKeys(keys);
  }

public:

  AnimateFoldCards() {}


  void Init() {
    const std::string &spline=game->HeaderGet("sequence", "/sequence/cardsfold/@spline");
    if (spline.empty())
      g_error("AnimateFoldCards::AnimateFoldCards /sequence/cardsfold/@spline not found");
    LoadSpline(mInterpolator, data, spline.c_str());
    mInterpolator.Init(0,1);

    const std::string &duration=game->HeaderGet("sequence", "/sequence/cardsfold/@duration");
    if (duration.empty())
      g_error("AnimateFoldCards::AnimateFoldCards /sequence/cardsfold/@duration not found");
    mMoveDuration=atof(duration);

    const std::string &node=game->HeaderGet("sequence", "/sequence/cardsfold/@target");
    if (node.target())
      g_error("AnimateFoldCards::AnimateFoldCards /sequence/cardsfold/@target not found");
    
    if (!game->mSetData->GetAnchor(node))
      g_error("AnimateFoldCards::AnimateFoldCards anchor %s not found",node.c_str());
    mTarget = MAFComputeLocalToWorld(game->mSetData->GetAnchor(node));
  }



  void SetCardPosition(const osg::MatrixTransform& transform) { mTransform=transform;}

  void StartAnimation()
  {
    mTransformPosition->setMatrix(osg::Matrix::identity());
    mShake=1;
    mMoveCards=0;
    mAnimationCallback->start();
    g_debug("tremblements");
  }

  void UpdateCards() {

    double dt=0;

    // cards end shakes so we start move cards
    if (mShake && mAnimationCallback->isEnd()) {
      mShake=0;
      mMoveCards=1;
    }
    
    // shakes is finished so move card to table center
    if (mMoveCards) {
      mTimer+=dt;
      float v=mTimer/mMoveDuration;
      float result;
      mInterpolator.Get(v,result);
      osg::Vec3 translation=(end-start)*result;
      mTransformPosition->getMatrix().setTranslation(translation);

      if (mTimer>=mMoveDuration) {
	mMoveCards=0;;
      }
    }

  }

};


#endif
