/**
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Henry Prcheur <henry@precheur.org>
 *  Cedric Pinson <cpinson@freesheep.org>
 *  Igor Kravtchenko <igor@obraz.net>
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#include <cstdio>
# define snprintf _snprintf
#endif


#include <maf/profile.h>

#include <osg/Geode>
#include <osg/MatrixTransform>

#include <maf/maferror.h>
#include <maf/scene.h>
#include <maf/utils.h>
#include <maf/osghelper.h>
#include <string>

#ifndef WIN32
#	include "PokerApplication.h"
#	include "PokerCard.h"
#	include "Poker.h"
#endif // WIN32

#include <PokerBoard.h>
#include <osg/AutoTransform>
#include <osg/BlendFunc>
#include <osg/ShapeDrawable>
#include <osg/Quat>
#include <osg/Material>
#include <osg/CullFace>

#define MAX_V 2.832f

MAFESCNData			*g_setData;
MAFESCNData			*g_cardData;

osg::Node			*g_lightRay[5];
osg::TexMat			*g_lightRayProjMat[5][2];
osg::Material		*g_lightRayMaterial[5];

osg::Group			*g_lightCone;
osg::TexMat			*g_lightConeTexMat[2];

osg::Group			*g_cachePot;
osg::TexMat			*g_cachePotTexMat[2];
osg::Material		*g_cachePotMaterial;

osg::Group			*g_cpath[5];

std::vector<int>	g_ray0_index_base;
std::vector<int>	g_ray0_index_topleft;
std::vector<int>	g_ray0_index_topright;
std::vector<int>	g_ray0_index_bottomright;
std::vector<int>	g_ray0_index_bottomleft;

std::vector<int>	g_ray1_index_base;
std::vector<int>	g_ray1_index_topleft;
std::vector<int>	g_ray1_index_topright;
std::vector<int>	g_ray1_index_bottomright;
std::vector<int>	g_ray1_index_bottomleft;
std::vector<int>	g_ray1_index_topleft_medium;
std::vector<int>	g_ray1_index_topright_medium;
std::vector<int>	g_ray1_index_bottomright_medium;
std::vector<int>	g_ray1_index_bottomleft_medium;

std::vector<int>	g_lightcone_index_vertex_top;
std::vector<int>	g_lightcone_index_vertex_bottom;

osg::Vec3f			*g_lightcone_vertex;
osg::Vec3f			*g_lightcone_vertex_top_bak;
osg::Vec3f			*g_lightcone_vertex_bottom_bak;

PokerBoardController *g_projectorCard = NULL;

PokerBoardController::PokerBoardController(PokerApplication* game) : mGame(game) 
{

	g_projectorCard = this;

  //
  // The board
  //
  char tmp[128];
  std::string card_url = game->HeaderGet("sequence", "/sequence/board/@url");
  std::string anchor_card = game->HeaderGet("sequence", "/sequence/board/@anchor");
  std::string board_count = game->HeaderGet("sequence", "/sequence/board/@count");
  std::vector<std::string> anchors;

  /*

  O main point for board
  |-------------------------------------------------------------------------------------O billboard
  O mCardPosition                                                                       | 
  |                                                                                     O mBBCardPosition
  O mTransformTranslattion contain offset translation when cards increase scale factor  |
  |                                                                                     O mBBtranslate for scale
  O mTransformCard contain orientation data and scale                                   |
  |                                                                                     O mBBtransform card offset y
  o card                                                                                |
                                                                                        O mBBYrotation of card
                                                                                        |
	                                                                                o card

  |
  O mTransformTransition
  |
  O mTransformTransitionTranslation
  |
  O mTransformTransitionCard because i need to change the x pos on zdir to camera in transition state
  |
  o card

  */                      




  mCards.resize(atoi(board_count.c_str()));
  for(unsigned int i = 0; i < mCards.size(); i++) {
    snprintf(tmp, sizeof(tmp), anchor_card.c_str(), i + 1);
    anchors.push_back(tmp);
    PokerCardController* card = new PokerCardController(game, card_url);
    card->Fold();
    std::string anchorname=/*std::string("auto")+*/ tmp;
    assert(game->mSetData->GetAnchor(anchorname));

    // make transition tree, and main tree
    osg::MatrixTransform* anim=new osg::MatrixTransform;
    osg::MatrixTransform* translateForScale=new osg::MatrixTransform;
    translateForScale->setMatrix(osg::Matrix::identity());
    osg::MatrixTransform* cardTransform=0;
    //    osg::Node* mainTransform=0;
    osg::AutoTransform* node=dynamic_cast<osg::AutoTransform*>(game->mSetData->GetAnchor(anchorname)->asTransform());

    //main tree
    mCardPosition[i]=node->getParent(0)->asGroup()->asTransform()->asMatrixTransform();
    cardTransform=new osg::MatrixTransform;
    mCardPosition[i]->addChild(translateForScale);
    translateForScale->addChild(cardTransform);
    mCardPosition[i]->getParent(0)->addChild(anim);
    mCardPosition[i]->removeChild(node);
    mTransformTranslations[i]=translateForScale;

    // transition tree
    mTransformTransitions.push_back(anim);
    osg::MatrixTransform* tscale=new osg::MatrixTransform;
    anim->addChild(tscale);
    mTransformTransitionTranslations.push_back(tscale);
    osg::MatrixTransform* tcard2=new osg::MatrixTransform;
    tscale->addChild(tcard2);
    mTransformTransitionCards.push_back(tcard2);

      
    mTransformCards.push_back(cardTransform);
    card->Anchor(cardTransform);
    mMatrixOriginal.push_back(mCardPosition[i]->asTransform()->asMatrixTransform()->getMatrix());
    mCards[i] = card;

    std::string nametmp=anchorname; //std::string("auto") + tmp;
    g_debug("anchor %s",anchorname.c_str());

  }


  std::string anchor_center = game->HeaderGet("sequence", "/sequence/board/@board_center");
  if (anchor_center.empty())
    g_error("PokerModel::PokerModel /sequence/board/@board_center not found");
  mBoardCenter=game->mSetData->GetAnchor(anchor_center)->asTransform()->asMatrixTransform();
//    mBoardCenter=dynamic_cast<osg::AutoTransform*>(game->mSetData->GetAnchor(anchor_center)->asTransform());
  if (!mBoardCenter.get())
    g_error("PokerModel::PokerModel anchor %s not found",anchor_center.c_str());

  mInitialMatrix=mBoardCenter->getMatrix();
  //  mInitialMatrix=osg::Matrix::rotate(0,osg::Vec3(0,1,0));

  mSamples.SetDelay(0.5);
  mSamples.SetSampleRate(1.0/50.0);

  mStartToDisplayHandValue=false;
  mOffsetCard=0;
  mTransitionTime=0;



  
  {
  std::string paramTextYOffset=game->HeaderGet("sequence", "/sequence/board/@text_yoffset");
  if (paramTextYOffset.empty())
    g_error("PokerModel::PokerModel /sequence/board/@text_yoffset not found");
  mParamYoffsetForText=atof(paramTextYOffset.c_str());
  }

  mChangeSpace=false;
  mAutoHandValueSpace=new osg::AutoTransform;
  mAutoHandValueSpace->setAutoRotateMode(osg::AutoTransform::ROTATE_TO_SCREEN);
  mCardPosition[0]->getParent(0)->addChild(mAutoHandValueSpace.get());
  for (int i=0;i<5;i++) {
    mHandValueSpace[i]=new osg::MatrixTransform;
    mHandValueTranslations[i]=new osg::MatrixTransform;
    mHandValueScaleSpace[i]=new osg::MatrixTransform;
    mHandValueTransform[i]=new osg::MatrixTransform;
    mHandValueSpace[i]->addChild(mHandValueTranslations[i].get());
    mHandValueTranslations[i]->addChild(mHandValueScaleSpace[i].get());
    mHandValueScaleSpace[i]->addChild(mHandValueTransform[i].get());
    mAutoHandValueSpace->addChild(mHandValueSpace[i].get());
  }
  mTextAnchor=new osg::MatrixTransform;
  mTextAnchor->setMatrix(osg::Matrix::translate(0,mParamYoffsetForText,0));
  mAutoHandValueSpace->addChild(mTextAnchor.get());

  PokerLabelController* label = new PokerLabelController();
  label->Init();
  label->GetView()->SetText("salut les aminches");
  std::string datadir=game->HeaderGet("settings", "/settings/data/@path");
  if (datadir.empty())
    g_assert(0 && "Stop to play remount data please");
  label->GetView()->mText->setFont(MAFLoadFont(datadir + "/FreeSansBold.ttf", game->GetOptions()));
  label->GetView()->SetCharacterSize(10);
  label->GetView()->SetText("yopa");
  label->GetView()->SetBackgroundColor(osg::Vec4(0,0,0,0));
//   label->GetView()->SetPosition(osg::Vec3(0,0,0));
  label->GetView()->SetCenter(osg::Vec3(0,0,0));
  label->Anchor(mTextAnchor.get());
  label->GetView()->mBackgroundGeode->setNodeMask(~(MAF_VISIBLE_MASK|MAF_COLLISION_MASK)); // remove background box ...
  osg::BlendFunc *bf = new osg::BlendFunc();
  bf->setFunction(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  label->GetView()->mTextGeode->getOrCreateStateSet()->setTextureAttributeAndModes(0, bf, osg::StateAttribute::ON);
  label->GetView()->mTextGeode->getOrCreateStateSet()->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);

//  label->GetView()->mTextGeode->getOrCreateStateSet()->setMode(GL_BLEND, osg::StateAttribute::ON);
//   osg::Material *material = dynamic_cast<osg::Material*>(sstate->getAttribute(osg::StateAttribute::MATERIAL));
//   if (material) 
//     {	    
//       material->setAlpha(osg::Material::FRONT,1);
//       sstate->setMode(GL_BLEND, osg::StateAttribute::OFF|osg::StateAttribute::OVERRIDE);
//       sstate->setRenderingHint(osg::StateSet::OPAQUE_BIN);
//     }


  mLabel=label;

  //  mTargetPositionInHandSpace
  // setup card position for showdown
  {
    float sizey=40;
    float sizex=75;

    std::string paramSizeX=game->HeaderGet("sequence", "/sequence/board/@showdown_sizex");
    if (paramSizeX.empty())
      g_error("PokerModel::PokerModel /sequence/board/@showdown_sizex not found");
    sizex=atof(paramSizeX.c_str());

    std::string paramSizeY=game->HeaderGet("sequence", "/sequence/board/@showdown_sizey");
    if (paramSizeY.empty())
      g_error("PokerModel::PokerModel /sequence/board/@showdown_sizey not found");
    sizex=atof(paramSizeY.c_str());


    float part=sizex/2.0;
//     float posy=sizey*0.5;
    float posy=sizey;
    for (int i=0;i<3;i++) {
//       float posx=-sizex*0.5+i*part;
//       mTargetPos[i]=osg::Vec3(posx,posy,0);
//       mHandValueSpace[i]->setMatrix(osg::Matrix::translate(mTargetPos[i]));
      mTargetPositionInHandSpace[i]=osg::Vec3(0,posy,0);
    }

    float part2=sizex/4;
//     float posy2=-sizey*0.5;
    float posy2=0;
    part=2*part2;
    for (int i=0;i<2;i++) {
//       float posx=-sizex*0.5+part2+i*part;
//       mTargetPos[3+i]=osg::Vec3(posx,posy,0);
//       mHandValueSpace[3+i]->setMatrix(osg::Matrix::translate(mTargetPos[3+i]));
      mTargetPositionInHandSpace[i+3]=osg::Vec3(0,posy2,0);
    }
    
    label->GetView()->SetCenter(osg::Vec3(0,(posy-posy2)*0.5,0));
  }

  
  StopToDisplayShowDown();
  
  mTransitionTime=0;
  {
  std::string paramDuration=game->HeaderGet("sequence", "/sequence/board/@transition_duration");
  if (paramDuration.empty())
    g_error("PokerModel::PokerModel /sequence/board/@transition_duration not found");
  mParamAnimationDuration=atof(paramDuration.c_str());
  }

  mChangeSpaceFinal=false;


  {
  std::string paramDuration=game->HeaderGet("sequence", "/sequence/board/@translation_duration");
  if (paramDuration.empty())
    g_error("PokerModel::PokerModel /sequence/board/@translation_duration not found");
  mParamTransitionTranslationDuration=atof(paramDuration.c_str());
  }

  {
  std::string paramDuration=game->HeaderGet("sequence", "/sequence/board/@fadein_duration");
  if (paramDuration.empty())
    g_error("PokerModel::PokerModel /sequence/board/@fadein_duration not found");
  mParamFadeInDuration=atof(paramDuration.c_str());
  }


  {
    std::map<std::string,std::string> colors = game->HeaderGetProperties("sequence", "/sequence/bestHand/boardShadeColor");
    mParamShadeColorForCardUsed.x() = atof(colors["red"].c_str()) / 255;
    mParamShadeColorForCardUsed.y() = atof(colors["green"].c_str()) / 255;
    mParamShadeColorForCardUsed.z() = atof(colors["blue"].c_str()) / 255;
    mParamShadeColorForCardUsed.w() = 1.f;
    g_debug("PokerBoardController::PokerBoardController: boardShadeColor red %s, green %s, blue %s", colors["red"].c_str(), colors["green"].c_str(), colors["blue"].c_str());
  }


  g_setData = (MAFESCNData*) mGame->mDatas->GetVision("set.escn");
  g_cardData = (MAFESCNData*) mGame->mDatas->GetVision("flop01.escn");

  {
	  MAFESCNData *data = (MAFESCNData*) mGame->mDatas->GetVision("lightrayB.escn");
	  if (!data)
	    g_error("PokerBoardController::PokerBoardController lightrayB.escn not found");
	    
	  //	  osg::Group *lightRayB = data->GetGroup();
	  osg::Geode *geode = (osg::Geode*) OSGHelper_getNodeByName(*data->GetGroup(), "set/mesh_lightrayB.emsh");
	  if (!geode)
	    g_error("PokerBoardController::PokerBoardController set/mesh_lightrayB.emsh not found");
	  osg::Geometry *geom = (osg::Geometry*) geode->getDrawable(0);
	  osg::StateSet *ss = geom->getOrCreateStateSet();
	  ss->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

	  for (int i = 0; i < 3; i++) {
		  osg::Geode *geode = new osg::Geode();
		  g_lightRay[i] = geode;

		  LightRayGeometry *gg = new LightRayGeometry(*geom, i);
		  geode->addDrawable( gg );

		  osg::StateSet *ss = new osg::StateSet( *geom->getOrCreateStateSet() );
		  gg->setStateSet(ss);

		  osg::TexMat *texMat = new osg::TexMat();
		  g_lightRayProjMat[i][0] = texMat;
		  ss->setTextureAttributeAndModes(0, texMat);
		  texMat = new osg::TexMat();
		  g_lightRayProjMat[i][1] = texMat;
		  ss->setTextureAttributeAndModes(1, texMat);

		  osg::TexEnv *env;
		  env = new osg::TexEnv;
		  env->setMode(osg::TexEnv::MODULATE);
		  ss->setTextureAttributeAndModes(0, env);
		  env = new osg::TexEnv;
		  env->setMode(osg::TexEnv::MODULATE);
		  ss->setTextureAttributeAndModes(1, env);

		  osg::Material *newmat = new osg::Material();
		  newmat->setDiffuse(osg::Material::FRONT_AND_BACK, osg::Vec4(1, 1, 1, 1));
		  newmat->setAmbient(osg::Material::FRONT_AND_BACK, osg::Vec4(0, 0, 0, 1));
		  newmat->setEmission(osg::Material::FRONT_AND_BACK, osg::Vec4(0, 0, 0, 1));
		  newmat->setSpecular(osg::Material::FRONT_AND_BACK, osg::Vec4(0, 0, 0, 1));
		  ss->setTextureAttributeAndModes(0, newmat, osg::StateAttribute::ON);
		  g_lightRayMaterial[i] = newmat;
	  }

	  osg::Vec3Array *arr = (osg::Vec3Array*) geom->getVertexArray();

	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(-10.147f, 31.387f, -9.986f), g_ray1_index_topleft );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3( 10.147f, 31.387f, -9.986f), g_ray1_index_topright );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3( 10.147f, 31.387f,  9.986f), g_ray1_index_bottomright );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(-10.147f, 31.387f,  9.986f), g_ray1_index_bottomleft );

	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(- 9.042f, 27.970f, -8.899f), g_ray1_index_topleft_medium );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(  9.042f, 27.970f, -8.899f), g_ray1_index_topright_medium );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(  9.042f, 27.970f,  8.899f), g_ray1_index_bottomright_medium );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(- 9.042f, 27.970f,  8.899f), g_ray1_index_bottomleft_medium );

	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(0, 0, 0), g_ray1_index_base );
  }

  {
	  MAFESCNData *data = (MAFESCNData*) mGame->mDatas->GetVision("lightray.escn");
	  if (!data)
	    g_error("PokerBoardController::PokerBoardController lightray.escn not found");
	  //	  osg::Group *lightRay = data->GetGroup();
	  osg::Geode *geode = (osg::Geode*) OSGHelper_getNodeByName(*data->GetGroup(), "set/mesh_lightray.emsh");
	  if (!geode)
	    g_error("PokerBoardController::PokerBoardController set/mesh_lightray.emsh not found");
	  osg::Geometry *geom = (osg::Geometry*) geode->getDrawable(0);
	  osg::StateSet *ss = geom->getOrCreateStateSet();
	  ss->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

	  for (int i = 3; i < 5; i++) {
		  osg::Geode *geode = new osg::Geode();
		  g_lightRay[i] = geode;

		  LightRayGeometry *gg = new LightRayGeometry(*geom, i);
		  geode->addDrawable( gg );

		  osg::StateSet *ss = new osg::StateSet( *geom->getOrCreateStateSet() );
		  gg->setStateSet(ss);

		  osg::TexMat *texMat = new osg::TexMat();
		  g_lightRayProjMat[i][0] = texMat;
		  ss->setTextureAttributeAndModes(0, texMat);
		  texMat = new osg::TexMat();
		  g_lightRayProjMat[i][1] = texMat;
		  ss->setTextureAttributeAndModes(1, texMat);

		  osg::TexEnv *env;
		  env = new osg::TexEnv;
		  env->setMode(osg::TexEnv::MODULATE);
		  ss->setTextureAttributeAndModes(0, env);
		  env = new osg::TexEnv;
		  env->setMode(osg::TexEnv::MODULATE);
		  ss->setTextureAttributeAndModes(1, env);

		  osg::Material *newmat = new osg::Material();
		  ss->setTextureAttributeAndModes(0, newmat, osg::StateAttribute::ON);
		  g_lightRayMaterial[i] = newmat;
	  }

	  osg::Vec3Array *arr = (osg::Vec3Array*) geom->getVertexArray();
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(-10.147f, 31.387f, -9.986f), g_ray0_index_topleft );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3( 10.147f, 31.387f, -9.986f), g_ray0_index_topright );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3( 10.147f, 31.387f,  9.986f), g_ray0_index_bottomright );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3(-10.147f, 31.387f,  9.986f), g_ray0_index_bottomleft );
	  OSGHelper_getPointsEqualTo(*arr, osg::Vec3( 0, 0, 0), g_ray0_index_base );
  }

  {
	  MAFESCNData *data = (MAFESCNData*) mGame->mDatas->GetVision("lightcone.escn");
	  if (!data)
	    g_error("PokerBoardController::PokerBoardController lightcone.escn not found");

	  osg::Geode *geode = (osg::Geode*) OSGHelper_getNodeByName(*data->GetGroup(), "set/mesh_lightcone.emsh");
	  if (!geode)
	    g_error("PokerBoardController::PokerBoardController set/mesh_lightcone.emsh not found");
	  osg::Geometry *geom = (osg::Geometry*) geode->getDrawable(0);

	  osg::Vec3Array *arr = (osg::Vec3Array*) geom->getVertexArray();
	  osg::Vec3f *pnts = (osg::Vec3f*) arr->getDataPointer();
	  g_lightcone_vertex = pnts;

	  OSGHelper_getPointsWithYOf(*arr, 25.0f, g_lightcone_index_vertex_top, 1.0f );
	  OSGHelper_getPointsWithYOf(*arr, 0.0f, g_lightcone_index_vertex_bottom, 1.0f );

	  int nb = g_lightcone_index_vertex_top.size();
	  g_lightcone_vertex_top_bak = new osg::Vec3f[nb];
	  for (int i = 0; i < nb; i++) {
		  int index = g_lightcone_index_vertex_top[i];
		  g_lightcone_vertex_top_bak[i] = pnts[index];
	  }

	  nb = g_lightcone_index_vertex_bottom.size();
	  g_lightcone_vertex_bottom_bak = new osg::Vec3f[nb];
	  for (int i = 0; i < nb; i++) {
		  int index = g_lightcone_index_vertex_bottom[i];
		  g_lightcone_vertex_bottom_bak[i] = pnts[index];
	  }

	  g_lightCone = data->GetGroup();
	  g_lightCone->setNodeMask(0);

	  osg::StateSet *ss = g_lightCone->getOrCreateStateSet();
	  g_lightConeTexMat[0] = new osg::TexMat();
	  g_lightConeTexMat[1] = new osg::TexMat();
	  ss->setTextureAttributeAndModes(0, g_lightConeTexMat[0]);
	  ss->setTextureAttributeAndModes(1, g_lightConeTexMat[1]);
	  ss->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
  }

  {
	  MAFESCNData *data = (MAFESCNData*) mGame->mDatas->GetVision("cachepot.escn");
	  if (!data)
	    g_error("PokerBoardController::PokerBoardController cachepot.escn not found");
	  g_cachePot = data->GetGroup();
	  osg::Geode *geode = (osg::Geode*) g_cachePot->getChild(0);

	  osg::StateSet *ss = geode->getDrawable(0)->getOrCreateStateSet();
	  osg::TexMat *texMat = new osg::TexMat();
	  ss->setTextureAttributeAndModes(0, texMat);
	  g_cachePotTexMat[0] = texMat;
	  texMat = new osg::TexMat();
	  ss->setTextureAttributeAndModes(1, texMat);
	  g_cachePotTexMat[1] = texMat;

	  osg::TexEnv *env;
	  env = new osg::TexEnv;
	  env->setMode(osg::TexEnv::MODULATE);
	  ss->setTextureAttributeAndModes(0, env);
	  env = new osg::TexEnv;
	  env->setMode(osg::TexEnv::MODULATE);
	  ss->setTextureAttributeAndModes(1, env);

	  ss->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

	  osg::Material *newmat = new osg::Material();
	  ss->setTextureAttributeAndModes(0, newmat, osg::StateAttribute::ON);
	  g_cachePotMaterial = newmat;
  }

  {
	  for (int i = 0; i < 5; i++) {
		  char str[200];
		  sprintf(str, "cpath%.2d.escn", i+1);
		  MAFESCNData *data = (MAFESCNData*) mGame->mDatas->GetVision(str);
		  if (!data)
		    g_error("PokerBoardController::PokerBoardController %s not found",str);
		    
		  osg::Group *cpath = data->GetGroup();
		  g_cpath[i] = cpath;
	  }
  }


  int i;

  {
	  osg::Group *lr = g_cardData->GetGroup();
	  osg::Geode *geode = (osg::Geode*) OSGHelper_getNodeByName(*lr, "set/mesh_flop01.emsh");
	  if (!geode)
	    g_error("PokerBoardController::PokerBoardController set/mesh_flop01.emsh not found");
	  osg::BoundingBox bb = geode->getBoundingBox();
	  float w  = bb.xMax() - bb.xMin();
	  float h  = bb.yMax() - bb.yMin();

	  // top left
	  m_cardCorners[0].x() = -w * 0.5f;
	  m_cardCorners[0].y() = h * 0.5f;
	  m_cardCorners[0].z() = 0;

	  // top right
	  m_cardCorners[1].x() = w * 0.5f;
	  m_cardCorners[1].y() = h * 0.5f;
	  m_cardCorners[1].z() = 0;

	  // bottom right
	  m_cardCorners[2].x() = w * 0.5f;
	  m_cardCorners[2].y() = -h * 0.5f;
	  m_cardCorners[2].z() = 0;

	  // bottom left
	  m_cardCorners[3].x() = -w * 0.5f;
	  m_cardCorners[3].y() = -h * 0.5f;
	  m_cardCorners[3].z() = 0;
  }

  for (i = 0; i < 5; i++) {
	  osg::TexMat *texmat = new osg::TexMat();
	  char str[200];
	  sprintf(str, "set/mesh_cpath%.2d.emsh", i+1);
	  osg::Geode *geode = (osg::Geode*) OSGHelper_getNodeByName(*g_cpath[i], str);
	  if (!geode)
	    g_error("PokerBoardController::PokerBoardController %s not found",str);
	    
	  osg::Geometry *geom0 = (osg::Geometry*) geode->getDrawable(0);
	  osg::Geometry *geom1 = (osg::Geometry*) geode->getDrawable(1);

	  geode->setNodeMask(0);

	  osg::AlphaFunc *alphaFunc = new osg::AlphaFunc();
	  alphaFunc->setReferenceValue(0);
	  alphaFunc->setFunction(osg::AlphaFunc::NOTEQUAL);

	  osg::StateSet *ss0 = geom0->getStateSet();
	  osg::StateSet *ss1 = geom1->getStateSet();

	  ss0->setAttributeAndModes(alphaFunc);
	  ss1->setAttributeAndModes(alphaFunc);

	  osg::Texture *tx = (osg::Texture*) ss1->getTextureAttribute(0, osg::StateAttribute::TEXTURE);
	  tx->setBorderColor( osg::Vec4f(0, 0, 0, 0) );
	  tx->setWrap(osg::Texture::WRAP_S, osg::Texture::CLAMP_TO_BORDER);
	  tx->setWrap(osg::Texture::WRAP_T, osg::Texture::CLAMP_TO_BORDER);

	  geom0->getOrCreateStateSet()->setTextureAttributeAndModes(0, texmat);
	  geom1->getOrCreateStateSet()->setTextureAttributeAndModes(0, texmat);

	  m_slots[i].cpath_texmat = texmat;
	  m_slots[i].geode = geode;
  }

  osg::MatrixTransform *node_transform = (osg::MatrixTransform*) OSGHelper_getNodeByName(*g_setData->GetGroup(), "transform_ProjCenter");
  if (!node_transform)
	g_critical("Leaks \"transform_ProjCenter\" matrix!");
  g_projectorCard->m_rayBaseInv = osg::Matrix::inverse( node_transform->getMatrix() );
  g_projectorCard->m_rayBase = node_transform->getMatrix().getTrans();
  node_transform->addChild( g_lightRay[0] );
  node_transform->addChild( g_lightRay[1] );
  node_transform->addChild( g_lightRay[2] );
  node_transform->addChild( g_lightRay[3] );
  node_transform->addChild( g_lightRay[4] );
  node_transform->addChild( g_lightCone );

  osg::MatrixTransform *node_cache_pot = (osg::MatrixTransform*) OSGHelper_getNodeByName(*g_setData->GetGroup(), "transform_cachepot");
  if (!node_cache_pot)
	  g_critical("Leaks \"transform_cachepot\" matrix!");
  node_cache_pot->addChild( g_cachePot );

  osg::MatrixTransform *cpath01_transform = (osg::MatrixTransform*) OSGHelper_getNodeByName(*g_setData->GetGroup(), "transform_cpath01");
  osg::MatrixTransform *cpath02_transform = (osg::MatrixTransform*) OSGHelper_getNodeByName(*g_setData->GetGroup(), "transform_cpath02");
  osg::MatrixTransform *cpath03_transform = (osg::MatrixTransform*) OSGHelper_getNodeByName(*g_setData->GetGroup(), "transform_cpath03");
  osg::MatrixTransform *cpath04_transform = (osg::MatrixTransform*) OSGHelper_getNodeByName(*g_setData->GetGroup(), "transform_cpath04");
  osg::MatrixTransform *cpath05_transform = (osg::MatrixTransform*) OSGHelper_getNodeByName(*g_setData->GetGroup(), "transform_cpath05");
  cpath01_transform->addChild( g_cpath[0] );
  cpath02_transform->addChild( g_cpath[1] );
  cpath03_transform->addChild( g_cpath[2] );
  cpath04_transform->addChild( g_cpath[3] );
  cpath05_transform->addChild( g_cpath[4] );

  m_previousCardsSize = 0;
  for (i = 0; i < 5; i++)
	m_previousCardsValue[i] = -999; // safety impossible to have first value

  m_bAllCardsGo = false;
  m_counterAllCardsGo = 0;
  m_coneMatrixCounter = 0;
  m_coneFactor = 0;
  m_coneStatus = 0;
  m_cachePotMatrixcounter = 0;
}



PokerBoardController::~PokerBoardController() {}


void PokerBoardController::SetCards(const std::vector<int>& cards)
{
	if (m_previousCardsSize == 0 && cards.size() > 0)
		makeConeArrive();


	int nbVisible=0;
	PokerCardControllerVector& ocards=mCards;
	for(unsigned int i = 0; i < ocards.size(); i++) {
		if(i < cards.size()) {
			//ocards[i]->Receive();

			if (m_previousCardsValue[i] != cards[i]) {
				makeCardArrive(i, cards[i]);
				m_previousCardsValue[i] = cards[i];
			}

			if(cards[i] != 255) {
				ocards[i]->SetValue(cards[i]);
				ocards[i]->Visible(true);
				nbVisible++;
			}
			else {
				ocards[i]->Visible(false);
			}
		}
		else {
//			if (m_previousCardsValue[i] != -123) {
//				m_previousCardsValue[i] = -123;
//				makeCardGo(i);
//			}
//			ocards[i]->Fold();
		}
	}
	if (!nbVisible) {
		StopToDisplayShowDown();

		makeConeGo();
		makeAllCardsGo();
	}

	m_previousCardsSize = cards.size();

	Update(mGame);

	//StartToDisplayShowDown();
}

void PokerBoardController::FoldCards(void)
{
  for(unsigned int i = 0; i < mCards.size(); i++)
    mCards[i]->Fold();
  StopToDisplayShowDown();
}



static void ComputeSignAndAngle(const osg::Vec3& base,const osg::Vec3& vec2,float& sign, float& angle)
{
  sign=-((osg::Vec3(0,1,0)^base)*vec2>0?1:-1); // get the current signe to go to target
  float dirdotcdir=vec2*base;
  if (dirdotcdir>=1.0)
    dirdotcdir=0.9999999;
  if (dirdotcdir<=-1.0)
    dirdotcdir=-0.9999999;
  angle=sign*acos(dirdotcdir);
}


bool PokerBoardController::StartToDisplayShowDown()
{
	makeAllCardsNikel();

  mStartToDisplayHandValue=true;
  mOffsetCard=0;
  mTransitionTime=0;
  mChangeSpaceFinal=false;
  mTextAnchor->setNodeMask(MAF_VISIBLE_MASK);
  SetTextColor(osg::Vec4(1,1,1,0));
  mActivateFadein=false;
  return true;
}

bool PokerBoardController::StopToDisplayShowDown()
{
  mStartToDisplayHandValue=false;
  mOffsetCard=0;
  mTransitionTime=0;
  ActionFromHandvalueToMain();
  mTextAnchor->setNodeMask(0);  
  mActivateFadein=false;
  return true;
}

static void Get3VectorFromMatrix(const osg::Matrix& m,osg::Vec3& xdir,osg::Vec3& ydir,osg::Vec3& zdir)
{
  xdir=osg::Vec3(m(0,0),m(0,1),m(0,2));
  ydir=osg::Vec3(m(1,0),m(1,1),m(1,2));
  zdir=osg::Vec3(m(2,0),m(2,1),m(2,2));
  
  xdir.normalize();
  ydir.normalize();
  zdir.normalize();
}


bool PokerBoardController::Update(MAFApplication* application)
{
  if(mGame->HasEvent())
    return true;


  int i;

  float dt=GetDeltaFrame()*1.0/1000;

  float g_deltaFrameTime = dt;

  PokerCameraModel* camera=(dynamic_cast<PokerCameraController*>(mGame->GetScene()->GetModel()->mCamera.get()))->GetModel();

//   const float mParamTimeToDisplayShowdown = 1;
//  const float mParamFadeInSpeed = 2.0;
//  const float mParamMoveCardSpeed = 35;

  const float mParamScaleRatio = 0.003;
  const float mParamScaleMin = 1;
  const float mParamScaleMax = 1.5;
  const float mParamScaleTranslate = .8;
  const float mParamScaleTranslateMin = mParamScaleMin;
  const float mParamScaleTranslateMax = mParamScaleMax;

  const float yTranslationAzimuthFactor = -20;

  //  float transitionTranslation=0; // used when card are billboarded at showdown
  float fadeInValue=0; //used to fadein hand value

  float yTranslationAzimuth = 0;


  float d=(camera->GetPosition()-mInitialMatrix.getTrans()).length();
  float scale=d*mParamScaleRatio;
//   g_debug("scale %f dist %f",scale,d);
  if (scale<mParamScaleMin)
    scale=mParamScaleMin;
  if (scale>mParamScaleMax)
    scale=mParamScaleMax;


  // manage the scale
  int end=(int)mTransformCards.size();
  for (int i=0;i<end;i++) {
    float scalet=scale;
    if (scalet<mParamScaleTranslateMin)
      scalet=mParamScaleTranslateMin;
    else if (scalet>mParamScaleTranslateMax)
      scalet=mParamScaleTranslateMax;
    float yoff;
    yoff=scalet*mMatrixOriginal[i].getTrans()[2]-mMatrixOriginal[i].getTrans()[2];
    scalet=scalet*mMatrixOriginal[i].getTrans()[0]-mMatrixOriginal[i].getTrans()[0];
    scalet*=mParamScaleTranslate;
    yoff*=-mParamScaleTranslate;
    if (i>=3)
      yoff=0; //fabs((1-scale)*mParamScaleTranslateMax);
    else
      mTextAnchor->setMatrix(osg::Matrix::scale(scale,scale,scale)*osg::Matrix::translate(0,mParamYoffsetForText,0));
	float mf = m_slots[i].matrix_factor;
    mTransformTranslations[i]->setMatrix(osg::Matrix::translate(scalet*mf,yoff*mf,0));
    mHandValueTranslations[i]->setMatrix(osg::Matrix::translate(scalet,yoff,0));
  }
  


  if (mActivateFadein) {
    mFadeInTime+=dt;
    fadeInValue=mFadeInTime/mParamFadeInDuration;
    if (fadeInValue>1)
      fadeInValue=1;
  }


  if (mChangeSpaceFinal && mTransitionTranslationTime>mParamTransitionTranslationDuration) {
    mChangeSpaceFinal=false;
    mActivateFadein=true;
    mFadeInTime=0;
  }


  /*
   *  manage translation in y at the end part of sequence
   */
  if (mChangeSpaceFinal) {
    mTransitionTranslationTime+=dt;
    float f=mTransitionTranslationTime/mParamTransitionTranslationDuration;
    if (f>1)
      f=1;
    for (int i=0;i<5;i++) {
      osg::Vec3 res=mSourcePositionInHandSpace[i];
      float yoffset=mTargetPositionInHandSpace[i][1]-mSourcePositionInHandSpace[i][1];
      res[1]+=yoffset*f;
      osg::Matrix mtemp=mHandValueSpace[i]->getMatrix();
      mtemp.setTrans(res);
      mHandValueSpace[i]->setMatrix(mtemp);
    }
  }



  /*
   *  manage transition from animation to align card in one plane and translation in y
   */
  if (mStartToDisplayHandValue && mTransitionTime>mParamAnimationDuration) {
    if (!mChangeSpaceFinal) {
      SetupCardInNewSpace(camera);
      ActionFromTransitionToHandvalue();
    }
    mStartToDisplayHandValue=false;
    mTransitionTranslationTime=0;
    for (int i=0;i<5;i++) {
      osg::Vec3 trans=mHandValueSpace[i]->getMatrix().getTrans();
      mSourcePositionInHandSpace[i]=trans;
    }
  }


  // transition state
  if (mStartToDisplayHandValue) {
    if (!mChangeSpace) {
      ActionFromMainToTransition();
    }
    mTransitionTime+=dt;
  }

  
  // feature only available in standard mode
  if (1 || !mStartToDisplayHandValue && !mChangeSpaceFinal) {
    osg::Vec3 cam=camera->GetTarget()-camera->GetPosition();
    cam.normalize();
    yTranslationAzimuth=1.0-fabs(cam*osg::Vec3(0,1,0));
    yTranslationAzimuth*=yTranslationAzimuthFactor;

    //osg::Matrix tryazimuth=osg::Matrix::translate(0,yTranslationAzimuth*m_slots[i].matrix_factor,0);
	for (int i=3;i<5;i++) {
		osg::Matrix tryazimuth=osg::Matrix::translate(0,yTranslationAzimuth*m_slots[i].matrix_factor,0);
		mTransformTranslations[i]->setMatrix(mTransformTranslations[i]->getMatrix()*tryazimuth);
	}

  }


  osg::Matrix tmat;
  osg::Vec3 camDirection=camera->GetPosition()-mInitialMatrix.getTrans(); camDirection[1]=0; // lock in y orientation
  camDirection.normalize();

  mSamples.Update(dt);
  mSamples.SetSample(camDirection);
  osg::Vec3 dir=camDirection;
  if (/*!mChangeSpaceFinal &&*/ !mChangeSpace) { // use direction modified only when free mode
    mSamples.GetSample(dir);
    dir.normalize();
  }

  // give me a smooth curve scaled by an angle
  float sign,diffAngle;
  ComputeSignAndAngle(camDirection,dir,sign,diffAngle);
  if (diffAngle<-osg::PI)
    diffAngle=-osg::PI;
  else if (diffAngle>osg::PI)
    diffAngle=osg::PI;
  float value=-sinf(diffAngle);
//   g_debug("diffAngle %f sign %f value %f",diffAngle,sign,value);
//   int end=(int)mTransformCard.size();

  {
    osg::Vec3 dd=camera->GetTarget()-camera->GetPosition();
    dd.normalize();
    osg::Vec3 dd2=dd;
    dd2[1]=0;
    dd2.normalize();
    float angle=acos(dd*dd2); //azimuth angle
    float trsign=1.0;

    for (int i=0;i<end;i++) {
      if (i>=3)
	trsign=0;

      osg::Matrix m2;
	  float rscale = _lerp(1, scale, m_slots[i].matrix_factor);
	  m2=osg::Matrix::scale(rscale,rscale,rscale)*osg::Matrix::rotate(value*m_slots[i].matrix_factor,osg::Vec3(0,1,0));
      osg::Matrix m3=osg::Matrix::rotate(-angle,osg::Vec3(1,0,0));
      m2=m3*m2;

      if (mChangeSpace) {
	osg::Matrix m2bis=m2;
	float f=mTransitionTime/mParamAnimationDuration;
	if (f>1)
	  f=1;
	osg::Vec3 ori=mMatrixOriginal[i].getTrans();
	osg::Vec3 space[3];
	Get3VectorFromMatrix(mTransformCards[i]->getMatrix(),space[0],space[1],space[2]);
// 	space[2]=osg::Vec3(0,0,1);
	float start=space[2]*ori;

	osg::Vec3 fromPlane=space[2]*f*start;
	
	osg::Vec3 tr=ori-fromPlane;
	m2bis.setTrans(tr);
	mTransformTransitionCards[i]->setMatrix(m2bis);
      }
      mTransformCards[i]->setMatrix(m2);
    }
  }

//   SetTextPosition(osg::Vec3(0,0,0));
  if (mActivateFadein) {
    //    float v=1-fadeInValue;
    //    SetTextColor(osg::Vec4(1,1,1,fadeInValue));
    SetTextColor(osg::Vec4(1,1,1,fadeInValue));
  }

  for (int k=0;k<5;k++) {
//     mHandValueScaleSpace[k]->setMatrix(osg::Matrix::scale(1,1,1));
    mHandValueScaleSpace[k]->setMatrix(osg::Matrix::scale(scale,scale,scale)*osg::Matrix::identity());
  }

//   mAutoHandValueSpace->setScale(scale);    
//   mHandValue->setColor(osg::Vec4(1,1,1,mFadeInValue));
//   mBox->setColor(osg::Vec4(0,0,0,mFadeInValue));


  
  // update is ok put result in matrix
  osg::Matrix mat;
  osg::Vec3 side=osg::Vec3(0,1,0)^dir;
  mat.makeIdentity();
  mat(0,0)=side[0];
  mat(0,1)=side[1];
  mat(0,2)=side[2];
  mat(0,1)=0;
  mat(1,1)=1;
  mat(2,1)=0;
  mat(2,0)=dir[0];
  mat(2,1)=dir[1];
  mat(2,2)=dir[2];
  mat.setTrans(mInitialMatrix.getTrans());
  mBoardCenter->setMatrix(mat);





  //return true;

  float speed = 1.0f;

  if (m_bAllCardsGo) {
	  for (i = 0; i < 5; i++) {
		  Slot &slot = m_slots[i];
		  //			slot.matrix_factor -= g_deltaFrameTime * speed * 2;
		  //			if (slot.matrix_factor < 0) {
		  slot.matrix_factor = 0;
		  //			}

		  osg::MatrixTransform *mt = mCardPosition[i].get();
		  PokerCardController *card = mCards[i].get();
		  osg::Vec3f pp = mt->getMatrix().getTrans();

		  osg::MatrixTransform *grp = (osg::MatrixTransform*) card->GetModel()->GetAnchor();
		  osg::Matrix mat = grp->getMatrix();

		  if (m_counterAllCardsGo < 1) {
			  pp *= m_counterAllCardsGo;
			  mat *= osg::Matrix::scale( _lerp(1, 0.05, m_counterAllCardsGo), _lerp(1, 3, m_counterAllCardsGo), 1);
		  }
		  else {
			  mat *= osg::Matrix::scale( 0.05, _lerp(3, 0, m_counterAllCardsGo-1), 1);
		  }

		  mat.setTrans(-pp);
		  grp->setMatrix(mat);
	  }

	  m_counterAllCardsGo += g_deltaFrameTime * 2; // * 0.4f;
	  if (m_counterAllCardsGo > 2) {
		  for (i = 0; i < 5; i++) {
			  Slot &slot = m_slots[i];
			  slot.state = -1;
			  //slot.geode->setNodeMask(0);
			  mCards[i]->Fold();
		  }
		  //m_counterAllCardsGo = 0;
		  m_bAllCardsGo = false;
	  }
  }


  for (i = 0; i < 3; i++) {
	  Slot &slot = m_slots[i];
	  if (slot.state == -1)
		  continue;

	  if (slot.state == 0) {

		  slot.rayState = 1;

		  slot.matrix_factor += g_deltaFrameTime * speed * 2;
		  if (slot.matrix_factor > 1) {
			  slot.matrix_factor = 1;
		  }

		  osg::MatrixTransform *mt = mCardPosition[i].get();
		  PokerCardController *card = mCards[i].get();
		  osg::Vec3f pp = mt->getMatrix().getTrans();

		  osg::MatrixTransform *grp = (osg::MatrixTransform*) card->GetModel()->GetAnchor();
		  osg::Matrix mat = grp->getMatrix();

		  pp.y() += 10;

		  if (slot.scale < 1) {
			  mat *= osg::Matrix::scale( 0.05, _lerp(0, 3, slot.scale), 1);
		  }
		  else {
			  if (slot.scale > 2) {
				  slot.scale = 2;
				  slot.state = 1;
			  }
			  pp *= (2 - slot.scale);
			  mat *= osg::Matrix::scale( _lerp(0.05, 1, slot.scale - 1), _lerp(3, 1, slot.scale - 1), 1);
		  }

		  mat.setTrans(-pp);
		  grp->setMatrix(mat);

		  slot.scale += g_deltaFrameTime * 2; // * 0.4f;
	  }

  }

  for (i = 3; i < 5; i++) {

	  Slot &slot = m_slots[i];
	  if (slot.state == -1)
		  continue;

	  //slot.cpath_currentPos = 0.05f;
	  osg::Matrix mat = osg::Matrix::translate(0, -slot.cpath_currentPos, 0);
	  slot.cpath_texmat->setMatrix(mat);

	  if (slot.state == 0) {
		  slot.cpath_currentPos += g_deltaFrameTime * speed * 5.5;
		  slot.rayState = 1;
		  if (slot.cpath_currentPos > MAX_V - 1) {
			  slot.cpath_currentPos = MAX_V - 1;
			  slot.state = 1;
			  mCards[i]->Receive();
			  slot.geode->setNodeMask(0);
		  }
	  }
	  else if (slot.state == 1) {
		  slot.matrix_factor += g_deltaFrameTime * speed * 2;
		  if (slot.matrix_factor > 1.0f) {
			  slot.matrix_factor = 1;
			  slot.state = 2;
		  }
	  }
	  else if (slot.state == 10) {
		  slot.matrix_factor -= g_deltaFrameTime * speed * 2;
		  if (slot.matrix_factor < 0) {
			  slot.state = 11;
			  mCards[i]->Fold();
			  slot.geode->setNodeMask(0xffffffff);
		  }
	  }
	  else if (slot.state == 11) {
		  slot.rayState = 0;
		  slot.cpath_currentPos -= g_deltaFrameTime * speed * 4;
		  if (slot.cpath_currentPos < -1) {
			  slot.cpath_currentPos = -1;
			  slot.state = -1;
			  slot.geode->setNodeMask(0);
		  }
	  }
  }

  for (i = 0; i < 5; i++) {
	  Slot &slot = m_slots[i];

	  slot.textureAngle += g_deltaFrameTime;

	  osg::Matrix texm;
	  texm = osg::Matrix::translate(-0.5f, -0.5f, 0);
	  texm = texm * osg::Matrix::rotate(slot.textureAngle*0.33f, osg::Vec3(0, 0, 1) );
	  texm = texm * osg::Matrix::translate(0.5f, 0.5f, 0);
	  g_lightRayProjMat[i][0]->setMatrix(texm);

	  texm = osg::Matrix::translate(-0.5f, -0.5f, 0);
	  texm = texm * osg::Matrix::rotate(-slot.textureAngle*0.25f, osg::Vec3(0, 0, 1) );
	  texm = texm * osg::Matrix::translate(0.5f, 0.5f, 0);
	  g_lightRayProjMat[i][1]->setMatrix(texm);

	  float alpha = slot.alphaRay;

	  osg::Material *mat = g_lightRayMaterial[i];
	  mat->setDiffuse( osg::Material::FRONT_AND_BACK, osg::Vec4f(1, 1, 1, alpha) );

	  if (slot.rayState == 0) {
		  slot.alphaRay -= g_deltaFrameTime * speed * 2;
		  if (slot.alphaRay < 0)
			  slot.alphaRay = 0;
	  }
	  else if (slot.rayState == 1) {
		  slot.alphaRay += g_deltaFrameTime * speed;
		  if (slot.alphaRay > 1)
			  slot.alphaRay = 1;
	  }
  }


  {
	osg::Matrix texm;
	texm = osg::Matrix::translate(-0.5f, -0.5f, 0);
	texm = texm * osg::Matrix::rotate(m_coneMatrixCounter * 0.33f, osg::Vec3(0, 0, 1) );
	texm = texm * osg::Matrix::translate(0.5f, 0.5f, 0);
	g_lightConeTexMat[0]->setMatrix(texm);

	texm = osg::Matrix::translate(-0.5f, -0.5f, 0);
	texm = texm * osg::Matrix::rotate(-m_coneMatrixCounter * 0.25f, osg::Vec3(0, 0, 1) );
	texm = texm * osg::Matrix::translate(0.5f, 0.5f, 0);
	g_lightConeTexMat[1]->setMatrix(texm);

	m_coneMatrixCounter += g_deltaFrameTime;
  }

  if (m_coneStatus == 1) {
	  int nb;
	  nb = g_lightcone_index_vertex_top.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_top_bak[i];
		  int index = g_lightcone_index_vertex_top[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;
		  pt.y() = _lerp(0, orgPt.y()*6, m_coneFactor);
		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;
		  g_lightcone_vertex[index] = pt;
	  }

	  nb = g_lightcone_index_vertex_bottom.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_bottom_bak[i];
		  int index = g_lightcone_index_vertex_bottom[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;
		  pt.y() = _lerp(0, orgPt.y()*6, m_coneFactor);
		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;
		  g_lightcone_vertex[index] = pt;
	  }

	  m_coneFactor += g_deltaFrameTime * 2.0f; // * 0.4f;
	  m_cachePotAlpha += g_deltaFrameTime * 4.0f;
	  if (m_coneFactor > 1) {
		  m_coneFactor = 1;
		  m_coneStatus = 2;
	  }
  }
  else if (m_coneStatus == 2) {
	  int nb;
	  nb = g_lightcone_index_vertex_top.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_top_bak[i];
		  int index = g_lightcone_index_vertex_top[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;
		  pt.x() = _lerp(pt.x(), orgPt.x(), m_coneFactor - 1);

		  pt.y() = _lerp(orgPt.y()*6, orgPt.y(), m_coneFactor-1);

		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;
		  pt.z() = _lerp(pt.z(), orgPt.z(), m_coneFactor - 1);

		  g_lightcone_vertex[index] = pt;
	  }

	  nb = g_lightcone_index_vertex_bottom.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_bottom_bak[i];
		  int index = g_lightcone_index_vertex_bottom[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;
		  pt.x() = _lerp(pt.x(), orgPt.x(), m_coneFactor - 1);

		  pt.y() = _lerp(orgPt.y()*6, orgPt.y(), m_coneFactor-1);

		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;
		  pt.z() = _lerp(pt.z(), orgPt.z(), m_coneFactor - 1);

		  g_lightcone_vertex[index] = pt;
	  }

	  m_coneFactor += g_deltaFrameTime * 2.0f; // * 0.4f;
	  if (m_coneFactor > 2) {
		  m_coneFactor = 2;
		  m_coneStatus = 3;
	  }
  }
  else if (m_coneStatus == 10) {
	  int nb;
	  nb = g_lightcone_index_vertex_top.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_top_bak[i];
		  int index = g_lightcone_index_vertex_top[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;
		  pt.x() = _lerp(orgPt.x(), pt.x(), m_coneFactor);

		  pt.y() = orgPt.y() * (m_coneFactor*5+1);

		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;
		  pt.z() = _lerp(orgPt.z(), pt.z(), m_coneFactor);

		  g_lightcone_vertex[index] = pt;
	  }

	  nb = g_lightcone_index_vertex_bottom.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_bottom_bak[i];
		  int index = g_lightcone_index_vertex_bottom[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;
		  pt.x() = _lerp(pt.x(), orgPt.x(), m_coneFactor);

		  pt.y() = orgPt.y() * (m_coneFactor*5+1);

		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;
		  pt.z() = _lerp(pt.z(), orgPt.z(), m_coneFactor);

		  g_lightcone_vertex[index] = pt;
	  }

	  m_coneFactor += g_deltaFrameTime * 2.0f; // * 0.4f;
	  if (m_coneFactor > 1) {
		  m_coneFactor = 1;
		  m_coneStatus = 11;
	  }
  }
  else if (m_coneStatus == 11) {

	  m_coneFactor += g_deltaFrameTime * 2.0f; // * 0.4f;
	  m_cachePotAlpha -= g_deltaFrameTime * 4.0f;
	  if (m_coneFactor > 2) {
		  m_coneFactor = 2;
		  m_coneStatus = 0;
	  }

	  int nb;
	  nb = g_lightcone_index_vertex_top.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_top_bak[i];
		  int index = g_lightcone_index_vertex_top[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;

		  pt.y() = _lerp(orgPt.y()*6, 0, m_coneFactor-1);

		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;

		  g_lightcone_vertex[index] = pt;
	  }

	  nb = g_lightcone_index_vertex_bottom.size();
	  for (i = 0; i < nb; i++) {
		  const osg::Vec3f &orgPt = g_lightcone_vertex_bottom_bak[i];
		  int index = g_lightcone_index_vertex_bottom[i];
		  osg::Vec3f pt;
		  if (orgPt.x() < 0) pt.x() = -1.0f;
		  else pt.x() = 1.0f;

		  pt.y() = _lerp(orgPt.y()*6, 0, m_coneFactor-1);

		  if (orgPt.z() < 0) pt.z() = -1.0f;
		  else pt.z() = 1.0f;

		  g_lightcone_vertex[index] = pt;
	  }

  }

  {
	  osg::Matrix texm;

	  texm = osg::Matrix::translate(-0.5f, -0.5f, 0);
	  texm = texm * osg::Matrix::rotate(m_cachePotMatrixcounter, osg::Vec3(0, 0, 1) );
	  texm = texm * osg::Matrix::translate(0.5f, 0.5f, 0);
	  g_cachePotTexMat[0]->setMatrix(texm);

	  texm = osg::Matrix::translate(-0.5f, -0.5f, 0);
	  texm = texm * osg::Matrix::rotate(-m_cachePotMatrixcounter, osg::Vec3(0, 0, 1) );
	  texm = texm * osg::Matrix::translate(0.5f, 0.5f, 0);
	  g_cachePotTexMat[1]->setMatrix(texm);

	  m_cachePotMatrixcounter += g_deltaFrameTime;
  }

  if (m_cachePotAlpha > 1)
	  m_cachePotAlpha = 1;
  else if (m_cachePotAlpha < 0)
	  m_cachePotAlpha = 0;

  g_cachePotMaterial->setDiffuse(osg::Material::FRONT_AND_BACK, osg::Vec4f(1, 1, 1, m_cachePotAlpha) );

  return true;
}



void PokerBoardController::ActionFromMainToTransition()
{
  int end=(int)mTransformCards.size();
  for (int i=0;i<end;i++) {
    mTransformTransitionTranslations[i]->setMatrix(mTransformTranslations[i]->getMatrix());
    mTransformTransitionCards[i]->setMatrix(mTransformCards[i]->getMatrix());
    mCards[i]->Anchor(mTransformTransitionCards[i].get());
  }

  mChangeSpace=true;
  g_debug("card attached");
}

void PokerBoardController::ActionFromHandvalueToMain()
{
  int end=(int)mTransformCards.size();
  for (int i=0;i<end;i++) {
    mCards[i]->Anchor(mTransformCards[i].get());
  }
  g_debug("card detach");
  mChangeSpace=false;
}


void PokerBoardController::ActionFromTransitionToHandvalue()
{
  int end=(int)mCards.size();
  for (int i=0;i<end;i++) {
    mCards[i]->Anchor(mHandValueTransform[i].get());
  }
  mChangeSpaceFinal=true;
}

void PokerBoardController::SetupCardInNewSpace(PokerCameraModel* camera)
{
  osg::Vec3 space[3];

//   osg::Vec3 z=camera->GetTarget()-camera->GetPosition();
//   osg::Vec3 y=camera->GetUp();
//   z.normalize();
  
  
  Get3VectorFromMatrix(mTransformTransitions[0]->getMatrix(),space[0],space[1],space[2]);
  Get3VectorFromMatrix(mTransformTransitionCards[0]->getMatrix(),space[0],space[1],space[2]);
  for (int i=0;i<5;i++) {
//     osg::Vec3 pos=mTransformTransitions[i]->getMatrix().getTrans();
    osg::Vec3 pos=mTransformTransitionCards[i]->getMatrix().getTrans();
    osg::Vec3 res=osg::Vec3(pos*space[0],pos*space[1],0);
    mHandValueSpace[i]->setMatrix(osg::Matrix::translate(res));

    osg::Vec3 pos2=mTransformTransitionTranslations[i]->getMatrix().getTrans();
    osg::Vec3 res2=osg::Vec3(pos2*space[0],pos2*space[1],0);
    mHandValueTranslations[i]->setMatrix(osg::Matrix::translate(res2));
  }
}




class AlterMaterialColor2 : public osg::NodeVisitor {
public:
  AlterMaterialColor2(const osg::Vec4& color) : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN), mColor(color) {
    setNodeMaskOverride(0xffffffff);
  }

  virtual void apply(osg::Geode& geode) {
    int num_drawables = geode.getNumDrawables();
    g_debug("Num drawable %d",num_drawables);
    if (!num_drawables)
      return;
//     g_assert(num_drawables == 1);
    for (int i=0;i<num_drawables;i++) {
      osg::Drawable* drawable = geode.getDrawable(i);
      osg::StateSet* state = drawable->getStateSet();
//     g_assert(state != 0);
      if(state) {
	osg::Material* material = dynamic_cast<osg::Material*>(state->getAttribute(osg::StateAttribute::MATERIAL));
	if(!material) material = new osg::Material;
	material->setColorMode(osg::Material::DIFFUSE);
	material->setDiffuse(osg::Material::FRONT_AND_BACK, mColor);
	state->setAttributeAndModes(material, osg::StateAttribute::ON);
      }
    }
  }

private:
  osg::Vec4 mColor;
};

void PokerBoardController::ShadeUsedCardForBestHand(const std::vector<int>& cards)
{
  g_debug("check for shade cards %d",cards.size());
  std::cout << "cards candidate ";
  for (unsigned int i=0;i<cards.size();i++)
    std::cout << cards[i] << " ";
  std::cout << "cards candidate ";
  std::cout << std::endl;
  std::cout << "cards board " << mCards.size() << " ";
  for (unsigned int i=0;i<mCards.size();i++)
    std::cout << mCards[i]->GetValue() << " ";
  std::cout << std::endl;
  for(unsigned j = 0; j < mCards.size(); j++) {
    bool found=false;
    for(unsigned i = 0; i < cards.size(); i++)
      if (cards[i] == mCards[j]->GetValue()) {
	found=true;
	break;
      }

    if (!found) {
      g_debug("card %d is shaded",j);
      osg::Node* node=mCards[j]->GetModel()->GetArtefact();
      AlterMaterialColor2 alter(mParamShadeColorForCardUsed);
      node->accept(alter);
    }
  }
}


void PokerBoardController::SetShadeOfCardsToDefault()
{
  for(int j = 0; j < (int)mCards.size(); j++) {
    osg::Node* node=mCards[j]->GetModel()->GetArtefact();
    AlterMaterialColor2 alter(osg::Vec4(1,1,1,1));
    node->accept(alter);
  }
}


void PokerBoardController::makeCardArrive(int _slot, int _iCard)
{
	Slot &slot = m_slots[_slot];
	slot.state = 0;
	slot.matrix_factor = 0;
	slot.cpath_currentPos = -1;
	slot.alphaRay = 0;
	slot.scale = 0;

	mCards[_slot]->SetValue(_iCard);

	if (_slot < 3) {
		mCards[_slot]->Receive();
	}
	else {
		mCards[_slot]->Fold();
		slot.geode->setNodeMask(0xffffffff);
	}

	osg::Drawable *drawable = slot.geode->getDrawable(0);
	osg::Texture2D *texture = (osg::Texture2D*) drawable->getOrCreateStateSet()->getTextureAttribute(0, osg::StateAttribute::TEXTURE);

	osg::Image *image = mGame->mDeck->GetImage(_iCard);
	texture->setImage( image );
	texture->setBorderColor( osg::Vec4f(0, 0, 0, 0) );
	texture->setWrap(osg::Texture::WRAP_S, osg::Texture::CLAMP_TO_BORDER);
	texture->setWrap(osg::Texture::WRAP_T, osg::Texture::CLAMP_TO_BORDER);
}

void PokerBoardController::makeCardGo(int _slot)
{
	Slot &slot = m_slots[_slot];

//	if (slot.state != 2)
//		return;

	slot.state = 10;
	slot.cpath_currentPos = MAX_V - 1;
	slot.matrix_factor = 1;
}

void PokerBoardController::makeAllCardsGo()
{
	//return;
	m_bAllCardsGo = true;
	m_counterAllCardsGo = 0;
	for (int i = 0; i < 5; i++) {
		Slot &slot = m_slots[i];
		slot.rayState = 0;
		slot.matrix_factor = 1;
		m_previousCardsValue[i] = -999; // safety impossible to have first value
	}
}

void PokerBoardController::makeConeArrive()
{
//	if (m_coneStatus != 0)
//		return;

	m_coneStatus = 1;
	m_coneFactor = 0;
	m_cachePotAlpha = 0;
	g_lightCone->setNodeMask(0xffffffff);
	//	m_bAllCardsGo = false;
}

void PokerBoardController::makeConeGo()
{
//	if (m_coneStatus != 3)
//		return;

	m_coneStatus = 10;
	m_coneFactor = 0;

	//makeAllCardsGo();
	//	for (int i = 0; i < 5; i++)
	//		g_projectorCard->makeCardGo(i);
}

void PokerBoardController::makeAllCardsNikel()
{
	for (int i = 0; i < 5; i++) {
		Slot &slot = m_slots[i];
		mCards[i]->Receive();
		slot.geode->setNodeMask(0);
		slot.state = -1;
		slot.scale = 1;
	}
}

void LightRayGeometry::drawImplementation(osg::State &_state) const
{
	int iCard = m_rayIndex;

	//	PokerBoardController::Slot &slot = g_projectorCard->m_slots[iCard];

	osg::Geometry *geom = (osg::Geometry*) this;
	osg::Array *arr = geom->getVertexArray();
	osg::Vec3f *data = (osg::Vec3f*) arr->getDataPointer();

	osg::MatrixTransform *mt = g_projectorCard->mTransformCards[iCard].get();
	osg::Matrix cm = MAFComputeLocalToWorld(mt);

	osg::Vec3 raybase = g_projectorCard->m_rayBase;
	osg::Matrix inv = g_projectorCard->m_rayBaseInv;

	osg::Vec3 vec_pt0 = osg::Vec3(-8, 23, 0) * cm - raybase;
	osg::Vec3 vec_pt1 = osg::Vec3(8, 23, 0) * cm - raybase;
	osg::Vec3 vec_pt2 = osg::Vec3(8, 0, 0) * cm - raybase;
	osg::Vec3 vec_pt3 = osg::Vec3(-8, 0, 0) * cm - raybase;

	if (iCard < 3) {
		osg::Vec3f m_pt0 = (vec_pt0 * 1.0f + raybase) * inv;
		osg::Vec3f m_pt1 = (vec_pt1 * 1.0f + raybase) * inv;
		osg::Vec3f m_pt2 = (vec_pt2 * 1.0f + raybase) * inv;
		osg::Vec3f m_pt3 = (vec_pt3 * 1.0f + raybase) * inv;

		osg::Vec3f u_pt0 = (vec_pt0 * 1.2f + raybase) * inv;
		osg::Vec3f u_pt1 = (vec_pt1 * 1.2f + raybase) * inv;
		osg::Vec3f u_pt2 = (vec_pt2 * 1.2f + raybase) * inv;
		osg::Vec3f u_pt3 = (vec_pt3 * 1.2f + raybase) * inv;

		// up top left -> u_pt3
		{
			std::vector<int> &vec = g_ray1_index_topleft;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = u_pt3;
			}
		}
		// medium top left -> m_pt3
		{
			std::vector<int> &vec = g_ray1_index_topleft_medium;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = m_pt3;
			}
		}

		// up top right -> u_pt2
		{
			std::vector<int> &vec = g_ray1_index_topright;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = u_pt2;
			}
		}
		// medium top right -> m_pt2
		{
			std::vector<int> &vec = g_ray1_index_topright_medium;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = m_pt2;
			}
		}

		// up bottom right -> u_pt1
		{
			std::vector<int> &vec = g_ray1_index_bottomright;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = u_pt1;
			}
		}
		// medium bottom right -> m_pt1
		{
			std::vector<int> &vec = g_ray1_index_bottomright_medium;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = m_pt1;
			}
		}


		// up bottom left -> u_pt0
		{
			std::vector<int> &vec = g_ray1_index_bottomleft;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = u_pt0;
			}
		}
		// medium bottom left -> m_pt0
		{
			std::vector<int> &vec = g_ray1_index_bottomleft_medium;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = m_pt0;
			}
		}
	}
	else {

		osg::Vec3f pt0 = (vec_pt0 * 1.2f + raybase) * inv;
		osg::Vec3f pt1 = (vec_pt1 * 1.2f + raybase) * inv;
		osg::Vec3f pt2 = (vec_pt2 * 1.2f + raybase) * inv;
		osg::Vec3f pt3 = (vec_pt3 * 1.2f + raybase) * inv;

		// top left -> pt0
		{
			std::vector<int> &vec = g_ray0_index_topleft;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = pt0;
			}
		}

		// top right -> pt1
		{
			std::vector<int> &vec = g_ray0_index_topright;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = pt1;
			}
		}

		// bottom right -> pt2
		{
			std::vector<int> &vec = g_ray0_index_bottomright;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = pt2;
			}
		}

		// bottom left -> pt3
		{
			std::vector<int> &vec = g_ray0_index_bottomleft;
			int nb = vec.size();
			for (int i = 0; i < nb; i++) {
				int ii = vec[i];
				osg::Vec3f *p = &data[ii];
				*p = pt3;
			}
		}
	}

	Geometry::drawImplementation(_state);
}
