/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Johan Euphrosine <johan@mekensleep.com>
 *  Loic Dachary <loic@gnu.org>
 *  Henry Precheur <henry@precheur.org>
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#include <algorithm>
#include <iostream>
#include <math.h>

#include <osg/Material>
#include <osg/MatrixTransform>
#include <osg/AutoTransform>
// #include <osg/PositionAttitudeTransform>

#include <ugame/text.h>
#include <ugame/debug.h>

#ifndef WIN32
#	include <PokerError.h>
#	include <PokerApplication.h>
#	include <PokerBubble.h>
#endif // WIN32

PokerBubbleModel::PokerBubbleModel(PokerApplication* game)
{
  std::string	data_path = game->HeaderGet("settings", "/settings/data/@path");
  osgText::Font*	font =
    MAFLoadFont(data_path + "/FreeSansBold.ttf", game->GetOptions());
  osg::Image*	img = MAFLoadImage(data_path + "/seat/bubble.tga", game->GetOptions());

  mDisplayedText = "...";
  mText = new UGAMEFramedText(mDisplayedText, img, font);
  mPAT = new osg::PositionAttitudeTransform;
}

PokerBubbleModel::~PokerBubbleModel()
{
  mText = 0;
}

// PokerBubble osg tree
// 
//  + Artefact PAT
//  |
//  + AutoTransform (autoscale to screen)
//  |
//  + PAT to scale bubble
//  |
//  + UGAMEFramedText
void PokerBubbleModel::Init()
{
  UGAMEArtefactModel::Init();
  GetNode()->setName("PokerBubble");

  mState = PokerBubbleModel::NONE;

  osg::AutoTransform*	at = new osg::AutoTransform;
  at->setAutoScaleToScreen(true);
  mPAT->addChild(mText.get());
  at->addChild(mPAT.get());
  GetPAT()->addChild(at);

  mMinScale = osg::Vec3(0.1f, 0.1f, 0.1f);
  mCurrentScale = mMinScale;

  const float growTimeout = 200.f;
  const float shrinkTimeout = 200.f;
  const float showTimeout = 3000.f;
  
  mGrowTimeout = growTimeout;
  mShrinkTimeout = shrinkTimeout;
  mShowTimeout = showTimeout;

  // SetArtefact(at);

  // avoid text showing
  mPAT->setNodeMask(0);
  mState = NONE;
}

void PokerBubbleModel::SetTextMessage(const std::string &msg)
{
  mDisplayedText = msg;
  mText->setStringUTF8(msg);

  const osg::Vec3 &from = mMinScale;
  const osg::Vec3 &to = osg::Vec3(4.f, 4.f, 4.f);

  SetInterpolator(from, to, mGrowTimeout);
  mPAT->setScale(from);
  mPAT->setNodeMask(MAF_VISIBLE_MASK);
  mState = GROW;
}

void PokerBubbleModel::AddTimeAndScale(float delta)
{
  mTimer.AddTime(delta);
  mTimer.Get(mCurrentScale, mInterpolator);
  mPAT->setScale(mCurrentScale);
}

void PokerBubbleModel::SetInterpolator(const osg::Vec3 &from, const osg::Vec3 &to, float timeout)
{
  mTimer.Init(timeout);
  mInterpolator.Init(from, to);
  // mText->setNodeMask(0);
}

void  PokerBubbleModel::Update(float delta)
{
  switch(mState)
    {
    case GROW:
      {
	AddTimeAndScale(delta);
	if (mTimer.Finished())
	  {
	    mText->setNodeMask(MAF_VISIBLE_MASK);
	    // calculate display time based on the lenght of the text
	    size_t	str_size = mDisplayedText.size();
 	    float	time = MIN(10.0f,
				   MAX(2.0f,
				       static_cast<float>(str_size) / 5.0f));
	    time = str_size ? time : 0;

	    mTimeout = 1000.0f * time;
	    mState = SHOW;
	  }
      }
      break;
    case SHOW:
      {
	mTimeout -= delta;
	if (mTimeout <= 0)
	  {
	    mState = SHRINK;
	    SetInterpolator(mCurrentScale, mMinScale, mShrinkTimeout);
	  }
      }
      break;
    case SHRINK:
      {
	AddTimeAndScale(delta);
	if (mTimer.Finished())
	  {
	    mPAT->setNodeMask(0);
	    mState = NONE;
	  }
      }
      break;
    case NONE:
      {
      }
      break;
    }
}

PokerBubbleController::PokerBubbleController(PokerApplication* game) 
{
  SetModel(new PokerBubbleModel(game));
  Init();
}

PokerBubbleController::~PokerBubbleController()
{
  //this->~UGAMEArtefact();
}

void PokerBubbleController::SetTextMessage(const std::string &msg)
{
  GetModel()->SetTextMessage(msg);
}

bool PokerBubbleController::Update(MAFApplication* application)
{
  GetModel()->Update(GetDelta());
  return true;
}
