/*
 *
 * Copyright (C) 21004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Henry Prcheur <henry@precheur.org>
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/utils.h>
#include <maf/maferror.h>

#include <osgDB/ReadFile>
#include <osg/Geode>
#include <osg/Drawable>
#include <osg/MatrixTransform>
#include <osg/BoundingBox>
#include <osg/Material>

#include <poker_defs.h>

#ifndef WIN32
#	include <PokerError.h>
#	include <PokerApplication.h>
#	include <PokerCard.h>
#endif // WIN32

struct vec3cmp {
  bool operator()(const osg::Vec3 v1, const osg::Vec3 v2) const {
    return v1 < v2;
  }
};

struct distancecmp {
  osg::Vec3 eye;
  
  bool operator()(const osg::Node* n1, const osg::Node* n2) const {
    osg::Vec3 n1_center = n1->getBound().center();
    osg::Vec3 n1_distance = n1_center - eye;
    osg::Vec3 n2_center = n2->getBound().center();
    osg::Vec3 n2_distance = n2_center - eye;
      
    return n1_distance.length() < n2_distance.length();
  }
};

class Square {
public:
  Square() {}
  Square(float left, float right, float bottom, float top) : mLeft(left), mRight(right), mBottom(bottom), mTop(top) {}

  bool intersect(const Square& other) {
    return
      ((other.mLeft < mRight && other.mRight > mRight) ||
       (other.mRight > mLeft && other.mLeft < mLeft) ) &&
      ((other.mBottom < mTop && other.mTop > mTop) ||
       (other.mTop > mBottom && other.mBottom < mBottom));
  }

  float mLeft;
  float mRight;
  float mBottom;
  float mTop;
};

typedef std::set<osg::Node*,distancecmp> Overlapping;
typedef std::map<osg::Vec3,osg::Node*> Vec2Node;
typedef std::map<osg::Node*,Square> Node2Square;

PokerCardsCallback::PokerCardsCallback() : mScale(1.f) {
  mEyePoint.set(0.f, 0.f, 0.f);
}

osg::PositionAttitudeTransform* PokerCardsCallback::FindTransform(osg::Node* node)
{
  //
  // Retreive the PositionAttitudeTransform of the first PokerCardModel
  // below node.
  //
  while(node) {
    osg::Group* group = node->asGroup();
    g_assert(group != 0);
    for(unsigned int i = 0; i < group->getNumDescriptions(); i++) {
      if(group->getDescription(i) == "PokerCard") {
	osg::Transform* transform = group->asTransform();
	g_assert(transform != 0);
	osg::PositionAttitudeTransform* pat = transform->asPositionAttitudeTransform();
	g_assert(pat != 0);
	return pat;
      }
    }
    g_assert(group->getNumChildren() == 1);
    node = group->getChild(0);
  }

    
  return 0;
}

void PokerCardsCallback::AutoScale(osg::Group* group, osg::CullStack* cs, float& width, float& height, float& width_on_screen, float& height_on_screen) {
  const osg::RefMatrix& mvpw = cs->getMVPW();
  //
  // Get width and height of the card in local space coordinates
  //
  if(group->getNumChildren() > 0) {
    osg::PositionAttitudeTransform* pat = FindTransform(group->getChild(0));
    pat->setScale(osg::Vec3(GetScale(), GetScale(), GetScale()));
    if(mNode2Controller.find(pat) == mNode2Controller.end())
      g_error("PokerCardsCallback::AutoScale: no controller for node");
    PokerCardController* card = mNode2Controller[pat];
    MAFOSGData* data = card->GetModel()->GetData();
    osg::BoundingBox bb = data->GetBound();
    height = bb.yMax() - bb.yMin();
    width = bb.xMax() - bb.xMin();
    height*=pat->getScale()[0];
    width*=pat->getScale()[0];
  }

  //
  // Move all cards so that they float slightly (10%) above their 
  // anchor position when billboarded
  //
  for (unsigned int i = 0; i < group->getNumChildren(); i++) {
    osg::PositionAttitudeTransform* pat = FindTransform(group->getChild(i));
    pat->setScale(osg::Vec3(GetScale(), GetScale(), GetScale()));
    pat->setPosition(osg::Vec3(0.f, height * mPercentUp,0));
  }

  //
  // Get width and height of the card on the screen
  //
  if(group->getNumChildren() > 0) {
    osg::Node* child = group->getChild(0);
    osg::Vec3 center = child->getBound().center();
    osg::Vec3 center_on_screen = center * mvpw;
    osg::Vec3 top = center + cs->getUpLocal() * height / 2.f;
    osg::Vec3 top_on_screen = top * mvpw;
    osg::Vec3 half = top_on_screen - center_on_screen;
    height_on_screen = half.length() * 2;
    width_on_screen = height_on_screen * (width / height);
  }

  //
  // Fix the scale to reach the desired height, in the
  // [GetScaleMin,GetScaleMax] boundaries.
  //
  if(!osg::equivalent(height_on_screen, mDesiredHeight, 3.f)) {
    float delta_scale = mDesiredHeight / height_on_screen;
    float scale = GetScale() * delta_scale;
    scale = scale > GetScaleMax() ? GetScaleMax() : scale;
    scale = scale < GetScaleMin() ? GetScaleMin() : scale;
    if(scale != GetScale()) {
      SetScale(scale);
      AutoScale(group, cs, width, height, width_on_screen, height_on_screen);
    }
  } else {
    //
    // Pretend that width and height are smaller (or bigger) than
    // the actual size. The calculated width and height are
    // only an approximation of the actual width and height (larger
    // than the actual width and height), reducing it to 90% or 80%
    // prevents cards to behave as if they interesect although they
    // do not on the screen.
    //
    height_on_screen *= mPercentIntersect;
    width_on_screen *= mPercentIntersect;

  }
}

void PokerCardsCallback::SetControllers(PokerCardControllerVector& controllers) {
  for(PokerCardControllerVector::iterator i = controllers.begin();
      i != controllers.end();
      i++) {
    mNode2Controller[(*i)->GetModel()->GetNode()] = i->get();
  }
}

void PokerCardsCallback::operator()(osg::Node* node, osg::NodeVisitor* nv) {
  if(mEyePoint != nv->getEyePoint() &&
     nv->getVisitorType() == osg::NodeVisitor::CULL_VISITOR) {
    mEyePoint = nv->getEyePoint();
    osg::CullStack* cs = dynamic_cast<osg::CullStack*>(nv);
    const osg::RefMatrix& mvpw = cs->getMVPW();
    float height_on_screen = 0.f;
    float height = 0.f;
    float width_on_screen = 0.f;
    float width = 0.f;
    osg::Group* group = node->asGroup();


    AutoScale(group, cs, width, height, width_on_screen, height_on_screen);

    //
    // Build tables for later (position => card and card => 
    // square on screen)
    //
    Vec2Node pos2card;
    Node2Square card2square;
    for (unsigned int ii = 0; ii < group->getNumChildren(); ii++) {
      osg::Node* child = group->getChild(ii);
      osg::Vec3 center = child->getBound().center();

      osg::Vec3 where = center * mvpw;
      pos2card[where] = child;
      card2square[child] = Square(where.x() - width_on_screen / 2.f,
				  where.x() + width_on_screen / 2.f,
				  where.y() - height_on_screen / 2.f,
				  where.y() + height_on_screen / 2.f);
    }

    //
    // Find cards that overlap almost completely and store
    // them in sets sorted according to distance from eyepoint
    //
    Vec2Node::iterator i = pos2card.begin();
    osg::Vec3 previous_pos = i->first;
    osg::Node* previous_node = i->second;
    std::list<Overlapping> overlappings;
    distancecmp sort_distance;
    sort_distance.eye = mEyePoint;
    Overlapping overlapping(sort_distance);
    for(i++; i != pos2card.end(); i++) {
      osg::Vec3 current_pos = i->first;
      osg::Node* current_node = i->second;
      if(card2square[previous_node].intersect(card2square[current_node])) {
	overlapping.insert(previous_node);
	overlapping.insert(current_node);
      } else {
	overlappings.push_back(overlapping);
	overlapping.clear();
      }
      previous_pos = current_pos;
      previous_node = current_node;
    }
    if(!overlapping.empty())
      overlappings.push_back(overlapping);

    //
    // Raise cards that are obscured by a card in front of them
    //
    for(std::list<Overlapping>::iterator j = overlappings.begin();
	j != overlappings.end();
	j++) {
      std::map<osg::Node*,int> node2elevation;
      Overlapping& overlapping = *j;
      for(Overlapping::iterator k = overlapping.begin();
	  k != overlapping.end();
	  k++) {
	//
	// Raise the card, using the recorded elevation if
	// found.
	//
	osg::Node* node = *k;
	float delta_z = 0.f;
	int elevation = 0;
	if(node2elevation.find(node) != node2elevation.end()) {
	  elevation = node2elevation[node];
	  delta_z = height * mElevateRatio * elevation;
	}
	osg::PositionAttitudeTransform* pat = FindTransform(node);
	pat->setPosition(osg::Vec3(0.f, height * mPercentUp + delta_z, 0.f));
	//
	// Increase elevation for all cards that intersect with
	// the current one. 
	//
	for(Overlapping::iterator l = k;
	    l != overlapping.end();
	    l++) {
	  osg::Node* other = *l;
	  if(card2square[node].intersect(card2square[other]))
	    node2elevation[other] = elevation + 1;
	}
      }
    }
  }

  traverse(node, nv);
}

PokerDeck::PokerDeck(PokerApplication* game) {
  std::list<std::string> urls = game->HeaderGetList("sequence", "/sequence/deck/@url");
  std::list<std::string> names = game->HeaderGetList("sequence", "/sequence/deck/@name");
  std::string base = game->HeaderGet("settings", "/settings/data/@path");
  //  g_debug("PokerDeck::PokerDeck: Loading deck (%d)", urls.size());
  int index = 0;
  std::list<std::string>::iterator i;
  std::list<std::string>::iterator j;
  for(i = urls.begin(), j = names.begin(); i != urls.end(); index++, i++, j++) {
    // g_debug(" %s", j->c_str());
    std::string path = base + "/" + *i;
	osg::ref_ptr<osg::Image> image = osgDB::readImageFile(path.c_str(), game->GetOptions());
    g_assert(image != 0);
    int card = 0;
    if(Deck_stringToCard((char*)j->c_str(), &card) == 0) {
      g_critical("PokerDeck::PokerDeck: card name %s is unknown, ignored", j->c_str());
    } else {
      mImages[card] = image;
    }
  }
}

osg::Image* PokerDeck::GetImage(int index) {
  if(mImages.find(index) != mImages.end())
    return mImages[index].get();
  else
    throw new PokerError(UNDERWARE_POKER_ERROR_CARD, "PokerDeck::GetImage: unknown card index %d", index);
}

class CardSetupVisitor : public osg::NodeVisitor {
public:
  CardSetupVisitor() : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN) {
    setNodeMaskOverride(0xffffffff);
  }

  virtual void apply(osg::Geode& geode) {
    unsigned int num_drawables = geode.getNumDrawables();
    for(unsigned int i = 0; i < num_drawables; i++) {
      osg::Drawable* drawable = geode.getDrawable(i);
      osg::StateSet* state = drawable->getStateSet();
      g_assert(state != 0);
      if(state) {
	//
	// Self illum
	//
	state->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
	osg::Material* material = dynamic_cast<osg::Material*>(state->getAttribute(osg::StateAttribute::MATERIAL));
	if(!material) material = new osg::Material;
// 	material->setEmission(osg::Material::FRONT, osg::Vec4(1,1,1,1));
// 	material->setColorMode(osg::Material::EMISSION);
	material->setDiffuse(osg::Material::FRONT, osg::Vec4(1,1,1,1));
	state->setAttributeAndModes(material, osg::StateAttribute::ON);

	//
	// Linear
	//
	osg::Texture2D* texture = dynamic_cast<osg::Texture2D*>(state->getTextureAttribute(0, osg::StateAttribute::TEXTURE));
	if(texture) {
	  texture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
	  texture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
	}
      }
    }
  }
};

PokerCardModel::PokerCardModel(PokerApplication* game, const std::string& url) : mGame(game), mValue(0), mVisible(false) {
  MAFData* data = game->mDatas->GetVision(url);
  SetData(dynamic_cast<MAFVisionData*>(data->Clone()));
  CardSetupVisitor visitor;
  GetData()->GetGroup()->accept(visitor);
}

PokerCardModel::~PokerCardModel() {
  //  g_debug("PokerCardModel::~PokerCardModel");
  if(GetData())
    delete GetData();
}

void PokerCardModel::Init(void) {
  UGAMEArtefactModel::Init();
  SetArtefact(GetData()->GetGroup());
}

float PokerCardModel::GetHeight(void) {
  osg::BoundingBox bb = GetData()->GetBound();
  return bb.yMax() - bb.yMin();
}

float PokerCardModel::GetWidth(void) {
  osg::BoundingBox bb = GetData()->GetBound();
  return bb.xMax() - bb.xMin();
}

int PokerCardModel::GetValue(void) { return mValue; }

void PokerCardModel::SetValue(int value) { mValue = value; }

PokerCardController::PokerCardController(PokerApplication* game, const std::string& url) : mGame(game) {
  SetModel(new PokerCardModel(game, url));
  Init();
  GetModel()->GetNode()->addDescription("PokerCard");
}

void PokerCardController::SetScale(float scale) {
  GetModel()->GetPAT()->setScale(osg::Vec3(scale, scale, scale));
}

int PokerCardController::GetValue() {
  return GetModel()->GetValue();
}


void PokerCardController::SetValue(int value) {
  GetModel()->SetValue(value);

  osg::Image* image = mGame->mDeck->GetImage(value);
  MAFOSGData* data = GetModel()->GetData();
  osg::Geode* geode = GetGeode(data->GetGroup());
  unsigned int num_drawables = geode->getNumDrawables();
  for(unsigned int i = 0; i < num_drawables; i++) {
    osg::Drawable* drawable = geode->getDrawable(i);
    osg::StateSet* state = drawable->getStateSet();
    if(state) {
      osg::StateAttribute* attr = state->getTextureAttribute(0, osg::StateAttribute::TEXTURE);
      if(attr) {
	osg::Texture2D* current_texture = dynamic_cast<osg::Texture2D*>(state->getTextureAttribute(0, osg::StateAttribute::TEXTURE));
	osg::Image* current_image = current_texture->getImage();
	if(current_image) {
	  std::string tn=current_image->getFileName();
	  if (tn.find("cardcovr") == std::string::npos) {
	    current_texture->setImage(image);
	  }
	}
      }
    }
  }
}

void PokerCardController::Visible(bool visible) {
  osg::Quat flip;
  flip.makeRotate((visible ? 0 : osg::PI), osg::Vec3(1.f, .0f, .0f));
  GetModel()->GetPAT()->setAttitude(flip);

  GetModel()->SetVisible(visible);
}

void PokerCardController::Receive() {
  Displayed(true);
}

void PokerCardController::Fold() {
  Displayed(false);
}
