/*
*
* Copyright (C) 2004 Mekensleep
*
*	Mekensleep
*	24 rue vieille du temple
*	75004 Paris
*       licensing@mekensleep.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*
* Authors:
*  Igor Kravtchenko <igor@obraz.net>
*
*/

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#include <cstdio>
# define snprintf _snprintf
#endif


#include <maf/profile.h>

#include <osg/Geode>
#include <osg/MatrixTransform>

#include <maf/maferror.h>
#include <maf/scene.h>
#include <maf/utils.h>
#include <string>

#ifndef WIN32
#	include "PokerApplication.h"
#	include "PokerCard.h"
#	include "Poker.h"
#endif // WIN32

#include <PokerCardProjector.h>
#include <osg/AutoTransform>
#include <osg/ShapeDrawable>
#include <osg/Quat>
#include <osg/Material>

PokerCardProjector::PokerCardProjector(PokerApplication* game) : mGame(game) 
{

	//
	// The board
	//
	char tmp[128];
	std::string card_url = game->HeaderGet("sequence", "/sequence/board/@url");
	std::string anchor_card = game->HeaderGet("sequence", "/sequence/board/@anchor");
	std::string board_count = game->HeaderGet("sequence", "/sequence/board/@count");
	std::vector<std::string> anchors;

	/*

	O main point for board
	|-------------------------------------------------------------------------------------O billboard
	O mCardPosition                                                                       | 
	|                                                                                     O mBBCardPosition
	O mTransformTranslattion contain offset translation when cards increase scale factor  |
	|                                                                                     O mBBtranslate for scale
	O mTransformCard contain orientation data and scale                                   |
	|                                                                                     O mBBtransform card offset y
	o card                                                                                |
	O mBBYrotation of card
	|
	o card

	|
	O mTransformTransition
	|
	O mTransformTransitionTranslation
	|
	O mTransformTransitionCard because i need to change the x pos on zdir to camera in transition state
	|
	o card

	*/                      




	mCards.resize(atoi(board_count.c_str()));
	for(unsigned int i = 0; i < mCards.size(); i++) {
		snprintf(tmp, sizeof(tmp), anchor_card.c_str(), i + 1);
		anchors.push_back(tmp);
		PokerCardController* card = new PokerCardController(game, card_url);
		card->Fold();
		std::string anchorname=/*std::string("auto")+*/ tmp;
		assert(game->mSetData->GetAnchor(anchorname));

		// make transition tree, and main tree
		osg::MatrixTransform* anim=new osg::MatrixTransform;
		osg::MatrixTransform* translateForScale=new osg::MatrixTransform;
		translateForScale->setMatrix(osg::Matrix::identity());
		osg::MatrixTransform* cardTransform=0;
		//    osg::Node* mainTransform=0;
		osg::AutoTransform* node=dynamic_cast<osg::AutoTransform*>(game->mSetData->GetAnchor(anchorname)->asTransform());

		//main tree
		mCardPosition[i]=node->getParent(0)->asGroup()->asTransform()->asMatrixTransform();
		cardTransform=new osg::MatrixTransform;
		mCardPosition[i]->addChild(translateForScale);
		translateForScale->addChild(cardTransform);
		mCardPosition[i]->getParent(0)->addChild(anim);
		mCardPosition[i]->removeChild(node);
		mTransformTranslations[i]=translateForScale;

		// transition tree
		mTransformTransitions.push_back(anim);
		osg::MatrixTransform* tscale=new osg::MatrixTransform;
		anim->addChild(tscale);
		mTransformTransitionTranslations.push_back(tscale);
		osg::MatrixTransform* tcard2=new osg::MatrixTransform;
		tscale->addChild(tcard2);
		mTransformTransitionCards.push_back(tcard2);


		mTransformCards.push_back(cardTransform);
		card->Anchor(cardTransform);
		mMatrixOriginal.push_back(mCardPosition[i]->asTransform()->asMatrixTransform()->getMatrix());
		mCards[i] = card;

		std::string nametmp=anchorname; //std::string("auto") + tmp;
		g_debug("anchor %s",anchorname.c_str());

	}


	std::string anchor_center = game->HeaderGet("sequence", "/sequence/board/@board_center");
	if (anchor_center.empty())
		g_error("PokerModel::PokerModel /sequence/board/@board_center not found");
	mBoardCenter=game->mSetData->GetAnchor(anchor_center)->asTransform()->asMatrixTransform();
	//    mBoardCenter=dynamic_cast<osg::AutoTransform*>(game->mSetData->GetAnchor(anchor_center)->asTransform());
	if (!mBoardCenter.get())
		g_error("PokerModel::PokerModel anchor %s not found",anchor_center.c_str());

	mInitialMatrix=mBoardCenter->getMatrix();
	//  mInitialMatrix=osg::Matrix::rotate(0,osg::Vec3(0,1,0));

	mSamples.SetDelay(0.5);
	mSamples.SetSampleRate(1.0/50.0);

	mStartToDisplayHandValue=false;
	mOffsetCard=0;
	mTransitionTime=0;




	{
		std::string paramTextYOffset=game->HeaderGet("sequence", "/sequence/board/@text_yoffset");
		if (paramTextYOffset.empty())
			g_error("PokerModel::PokerModel /sequence/board/@text_yoffset not found");
		mParamYoffsetForText=atof(paramTextYOffset.c_str());
	}

	mChangeSpace=false;
	mAutoHandValueSpace=new osg::AutoTransform;
	mAutoHandValueSpace->setAutoRotateMode(osg::AutoTransform::ROTATE_TO_SCREEN);
	mCardPosition[0]->getParent(0)->addChild(mAutoHandValueSpace.get());
	for (int i=0;i<5;i++) {
		mHandValueSpace[i]=new osg::MatrixTransform;
		mHandValueTranslations[i]=new osg::MatrixTransform;
		mHandValueScaleSpace[i]=new osg::MatrixTransform;
		mHandValueTransform[i]=new osg::MatrixTransform;
		mHandValueSpace[i]->addChild(mHandValueTranslations[i].get());
		mHandValueTranslations[i]->addChild(mHandValueScaleSpace[i].get());
		mHandValueScaleSpace[i]->addChild(mHandValueTransform[i].get());
		mAutoHandValueSpace->addChild(mHandValueSpace[i].get());
	}
	mTextAnchor=new osg::MatrixTransform;
	mTextAnchor->setMatrix(osg::Matrix::translate(0,mParamYoffsetForText,0));
	mAutoHandValueSpace->addChild(mTextAnchor.get());
	PokerLabelController* label = new PokerLabelController();
	label->Init();
	label->GetView()->SetText("salut les aminches");
	label->GetView()->SetCharacterSize(10);
	label->GetView()->SetText("yopa");
	label->GetView()->SetBackgroundColor(osg::Vec4(0,0,0,1));
	//   label->GetView()->SetPosition(osg::Vec3(0,0,0));
	label->GetView()->SetCenter(osg::Vec3(0,0,0));
	label->Anchor(mTextAnchor.get());
	mLabel=label;


	//  mTargetPositionInHandSpace
	// setup card position for showdown
	{
		float sizey=40;
		float sizex=75;

		std::string paramSizeX=game->HeaderGet("sequence", "/sequence/board/@showdown_sizex");
		if (paramSizeX.empty())
			g_error("PokerModel::PokerModel /sequence/board/@showdown_sizex not found");
		sizex=atof(paramSizeX.c_str());

		std::string paramSizeY=game->HeaderGet("sequence", "/sequence/board/@showdown_sizey");
		if (paramSizeY.empty())
			g_error("PokerModel::PokerModel /sequence/board/@showdown_sizey not found");
		sizex=atof(paramSizeY.c_str());


		float part=sizex/2.0;
		//     float posy=sizey*0.5;
		float posy=sizey;
		for (int i=0;i<3;i++) {
			//       float posx=-sizex*0.5+i*part;
			//       mTargetPos[i]=osg::Vec3(posx,posy,0);
			//       mHandValueSpace[i]->setMatrix(osg::Matrix::translate(mTargetPos[i]));
			mTargetPositionInHandSpace[i]=osg::Vec3(0,posy,0);
		}

		float part2=sizex/4;
		//     float posy2=-sizey*0.5;
		float posy2=0;
		part=2*part2;
		for (int i=0;i<2;i++) {
			//       float posx=-sizex*0.5+part2+i*part;
			//       mTargetPos[3+i]=osg::Vec3(posx,posy,0);
			//       mHandValueSpace[3+i]->setMatrix(osg::Matrix::translate(mTargetPos[3+i]));
			mTargetPositionInHandSpace[i+3]=osg::Vec3(0,posy2,0);
		}

		label->GetView()->SetCenter(osg::Vec3(0,(posy-posy2)*0.5,0));
	}


	StopToDisplayShowDown();

	mTransitionTime=0;
	{
		std::string paramDuration=game->HeaderGet("sequence", "/sequence/board/@transition_duration");
		if (paramDuration.empty())
			g_error("PokerModel::PokerModel /sequence/board/@transition_duration not found");
		mParamAnimationDuration=atof(paramDuration.c_str());
	}

	mChangeSpaceFinal=false;


	{
		std::string paramDuration=game->HeaderGet("sequence", "/sequence/board/@translation_duration");
		if (paramDuration.empty())
			g_error("PokerModel::PokerModel /sequence/board/@translation_duration not found");
		mParamTransitionTranslationDuration=atof(paramDuration.c_str());
	}

	{
		std::string paramDuration=game->HeaderGet("sequence", "/sequence/board/@fadein_duration");
		if (paramDuration.empty())
			g_error("PokerModel::PokerModel /sequence/board/@fadein_duration not found");
		mParamFadeInDuration=atof(paramDuration.c_str());
	}


	{
		std::map<std::string,std::string> colors = game->HeaderGetProperties("sequence", "/sequence/bestHand/boardShadeColor");
		mParamShadeColorForCardUsed.x() = atof(colors["red"].c_str()) / 255;
		mParamShadeColorForCardUsed.y() = atof(colors["green"].c_str()) / 255;
		mParamShadeColorForCardUsed.z() = atof(colors["blue"].c_str()) / 255;
		mParamShadeColorForCardUsed.w() = 1.f;
		g_debug("PokerCardProjector::PokerCardProjector: boardShadeColor red %s, green %s, blue %s", colors["red"].c_str(), colors["green"].c_str(), colors["blue"].c_str());
	}
}



PokerCardProjector::~PokerCardProjector() {}


void PokerCardProjector::SetCards(const std::vector<int>& cards)
{
	int nbVisible=0;
	PokerCardControllerVector& ocards=mCards;
	for(unsigned int i = 0; i < ocards.size(); i++) {
		if(i < cards.size()) {
			ocards[i]->Receive();
			if(cards[i] != 255) {
				ocards[i]->SetValue(cards[i]);
				ocards[i]->Visible(true);
				nbVisible++;
			} else {
				ocards[i]->Visible(false);
			}
		} else {
			ocards[i]->Fold();
		}
	}
	if (!nbVisible)
		StopToDisplayShowDown();
}

void PokerCardProjector::FoldCards(void)
{
	for(unsigned int i = 0; i < mCards.size(); i++)
		mCards[i]->Fold();
	StopToDisplayShowDown();
}



static void ComputeSignAndAngle(const osg::Vec3& base,const osg::Vec3& vec2,float& sign, float& angle)
{
	sign=-((osg::Vec3(0,1,0)^base)*vec2>0?1:-1); // get the current signe to go to target
	float dirdotcdir=vec2*base;
	if (dirdotcdir>=1.0)
		dirdotcdir=0.9999999;
	if (dirdotcdir<=-1.0)
		dirdotcdir=-0.9999999;
	angle=sign*acos(dirdotcdir);
}


bool PokerCardProjector::StartToDisplayShowDown()
{
	mStartToDisplayHandValue=true;
	mOffsetCard=0;
	mTransitionTime=0;
	mChangeSpaceFinal=false;
	mTextAnchor->setNodeMask(MAF_VISIBLE_MASK);  
	SetTextColor(osg::Vec4(0,0,0,1));
	mActivateFadein=false;
	return true;
}

bool PokerCardProjector::StopToDisplayShowDown()
{
	mStartToDisplayHandValue=false;
	mOffsetCard=0;
	mTransitionTime=0;
	ActionFromHandvalueToMain();
	mTextAnchor->setNodeMask(0);  
	mActivateFadein=false;
	return true;
}

static void Get3VectorFromMatrix(const osg::Matrix& m,osg::Vec3& xdir,osg::Vec3& ydir,osg::Vec3& zdir)
{
	xdir=osg::Vec3(m(0,0),m(0,1),m(0,2));
	ydir=osg::Vec3(m(1,0),m(1,1),m(1,2));
	zdir=osg::Vec3(m(2,0),m(2,1),m(2,2));

	xdir.normalize();
	ydir.normalize();
	zdir.normalize();
}


bool PokerCardProjector::Update(MAFApplication* application)
{
	if(mGame->HasEvent())
		return true;

	float dt=GetDeltaFrame()*1.0/1000;
	PokerCameraModel* camera=(dynamic_cast<PokerCameraController*>(mGame->GetScene()->GetModel()->mCamera.get()))->GetModel();

	//   const float mParamTimeToDisplayShowdown = 1;
	//  const float mParamFadeInSpeed = 2.0;
	//  const float mParamMoveCardSpeed = 35;

	const float mParamScaleRatio = 0.003;
	const float mParamScaleMin = 1;
	const float mParamScaleMax = 1.5;
	const float mParamScaleTranslate = .8;
	const float mParamScaleTranslateMin = mParamScaleMin;
	const float mParamScaleTranslateMax = mParamScaleMax;

	const float yTranslationAzimuthFactor = -20;

	//  float transitionTranslation=0; // used when card are billboarded at showdown
	float fadeInValue=0; //used to fadein hand value

	float yTranslationAzimuth = 0;


	float d=(camera->GetPosition()-mInitialMatrix.getTrans()).length();
	float scale=d*mParamScaleRatio;
	//   g_debug("scale %f dist %f",scale,d);
	if (scale<mParamScaleMin)
		scale=mParamScaleMin;
	if (scale>mParamScaleMax)
		scale=mParamScaleMax;


	// manage the scale
	int end=(int)mTransformCards.size();
	for (int i=0;i<end;i++) {
		float scalet=scale;
		if (scalet<mParamScaleTranslateMin)
			scalet=mParamScaleTranslateMin;
		else if (scalet>mParamScaleTranslateMax)
			scalet=mParamScaleTranslateMax;
		float yoff;
		yoff=scalet*mMatrixOriginal[i].getTrans()[2]-mMatrixOriginal[i].getTrans()[2];
		scalet=scalet*mMatrixOriginal[i].getTrans()[0]-mMatrixOriginal[i].getTrans()[0];
		scalet*=mParamScaleTranslate;
		yoff*=-mParamScaleTranslate;
		if (i>=3)
			yoff=0; //fabs((1-scale)*mParamScaleTranslateMax);
		else
			mTextAnchor->setMatrix(osg::Matrix::scale(scale,scale,scale)*osg::Matrix::translate(0,mParamYoffsetForText,0));
		mTransformTranslations[i]->setMatrix(osg::Matrix::translate(scalet,yoff,0));
		mHandValueTranslations[i]->setMatrix(osg::Matrix::translate(scalet,yoff,0));
	}



	if (mActivateFadein) {
		mFadeInTime+=dt;
		fadeInValue=mFadeInTime/mParamFadeInDuration;
		if (fadeInValue>1)
			fadeInValue=1;
	}


	if (mChangeSpaceFinal && mTransitionTranslationTime>mParamTransitionTranslationDuration) {
		mChangeSpaceFinal=false;
		mActivateFadein=true;
		mFadeInTime=0;
	}


	/*
	*  manage translation in y at the end part of sequence
	*/
	if (mChangeSpaceFinal) {
		mTransitionTranslationTime+=dt;
		float f=mTransitionTranslationTime/mParamTransitionTranslationDuration;
		if (f>1)
			f=1;
		for (int i=0;i<5;i++) {
			osg::Vec3 res=mSourcePositionInHandSpace[i];
			float yoffset=mTargetPositionInHandSpace[i][1]-mSourcePositionInHandSpace[i][1];
			res[1]+=yoffset*f;
			osg::Matrix mtemp=mHandValueSpace[i]->getMatrix();
			mtemp.setTrans(res);
			mHandValueSpace[i]->setMatrix(mtemp);
		}
	}



	/*
	*  manage transition from animation to align card in one plane and translation in y
	*/
	if (mStartToDisplayHandValue && mTransitionTime>mParamAnimationDuration) {
		if (!mChangeSpaceFinal) {
			SetupCardInNewSpace(camera);
			ActionFromTransitionToHandvalue();
		}
		mStartToDisplayHandValue=false;
		mTransitionTranslationTime=0;
		for (int i=0;i<5;i++) {
			osg::Vec3 trans=mHandValueSpace[i]->getMatrix().getTrans();
			mSourcePositionInHandSpace[i]=trans;
		}
	}


	// transition state
	if (mStartToDisplayHandValue) {
		if (!mChangeSpace) {
			ActionFromMainToTransition();
		}
		mTransitionTime+=dt;
	}


	// feature only available in standard mode
	if (1 || !mStartToDisplayHandValue && !mChangeSpaceFinal) {
		osg::Vec3 cam=camera->GetTarget()-camera->GetPosition();
		cam.normalize();
		yTranslationAzimuth=1.0-fabs(cam*osg::Vec3(0,1,0));
		yTranslationAzimuth*=yTranslationAzimuthFactor;

		osg::Matrix tryazimuth=osg::Matrix::translate(0,yTranslationAzimuth,0);
		for (int i=3;i<5;i++)
			mTransformTranslations[i]->setMatrix(mTransformTranslations[i]->getMatrix()*tryazimuth);

	}


	osg::Matrix tmat;
	osg::Vec3 camDirection=camera->GetPosition()-mInitialMatrix.getTrans(); camDirection[1]=0; // lock in y orientation
	camDirection.normalize();

	mSamples.Update(dt);
	mSamples.SetSample(camDirection);
	osg::Vec3 dir=camDirection;
	if (/*!mChangeSpaceFinal &&*/ !mChangeSpace) { // use direction modified only when free mode
		mSamples.GetSample(dir);
		dir.normalize();
	}

	// give me a smooth curve scaled by an angle
	float sign,diffAngle;
	ComputeSignAndAngle(camDirection,dir,sign,diffAngle);
	if (diffAngle<-osg::PI)
		diffAngle=-osg::PI;
	else if (diffAngle>osg::PI)
		diffAngle=osg::PI;
	float value=-sinf(diffAngle);
	//   g_debug("diffAngle %f sign %f value %f",diffAngle,sign,value);
	//   int end=(int)mTransformCard.size();

	{
		osg::Vec3 dd=camera->GetTarget()-camera->GetPosition();
		dd.normalize();
		osg::Vec3 dd2=dd;
		dd2[1]=0;
		dd2.normalize();
		float angle=acos(dd*dd2); //azimuth angle
		float trsign=1.0;

		for (int i=0;i<end;i++) {
			if (i>=3)
				trsign=0;

			osg::Matrix m2;
			m2=osg::Matrix::scale(scale,scale,scale)*osg::Matrix::rotate(value,osg::Vec3(0,1,0));
			osg::Matrix m3=osg::Matrix::rotate(-angle,osg::Vec3(1,0,0));
			m2=m3*m2;

			if (mChangeSpace) {
				osg::Matrix m2bis=m2;
				float f=mTransitionTime/mParamAnimationDuration;
				if (f>1)
					f=1;
				osg::Vec3 ori=mMatrixOriginal[i].getTrans();
				osg::Vec3 space[3];
				Get3VectorFromMatrix(mTransformCards[i]->getMatrix(),space[0],space[1],space[2]);
				// 	space[2]=osg::Vec3(0,0,1);
				float start=space[2]*ori;

				osg::Vec3 fromPlane=space[2]*f*start;

				osg::Vec3 tr=ori-fromPlane;
				m2bis.setTrans(tr);
				mTransformTransitionCards[i]->setMatrix(m2bis);
			}
			mTransformCards[i]->setMatrix(m2);
		}
	}

	//   SetTextPosition(osg::Vec3(0,0,0));
	if (mActivateFadein) {
		//    float v=1-fadeInValue;
		SetTextColor(osg::Vec4(1,1,1,fadeInValue));
	}

	for (int k=0;k<5;k++) {
		//     mHandValueScaleSpace[k]->setMatrix(osg::Matrix::scale(1,1,1));
		mHandValueScaleSpace[k]->setMatrix(osg::Matrix::scale(scale,scale,scale)*osg::Matrix::identity());
	}

	//   mAutoHandValueSpace->setScale(scale);    
	//   mHandValue->setColor(osg::Vec4(1,1,1,mFadeInValue));
	//   mBox->setColor(osg::Vec4(0,0,0,mFadeInValue));



	// update is ok put result in matrix
	osg::Matrix mat;
	osg::Vec3 side=osg::Vec3(0,1,0)^dir;
	mat.makeIdentity();
	mat(0,0)=side[0];
	mat(0,1)=side[1];
	mat(0,2)=side[2];
	mat(0,1)=0;
	mat(1,1)=1;
	mat(2,1)=0;
	mat(2,0)=dir[0];
	mat(2,1)=dir[1];
	mat(2,2)=dir[2];
	mat.setTrans(mInitialMatrix.getTrans());
	mBoardCenter->setMatrix(mat);

	//	osg::MatrixTransform *rr = mTransformCards[0].get();
	//	const osg::Matrix &ma = rr->getMatrix();



	return true;
}



void PokerCardProjector::ActionFromMainToTransition()
{
	int end=(int)mTransformCards.size();
	for (int i=0;i<end;i++) {
		mTransformTransitionTranslations[i]->setMatrix(mTransformTranslations[i]->getMatrix());
		mTransformTransitionCards[i]->setMatrix(mTransformCards[i]->getMatrix());
		mCards[i]->Anchor(mTransformTransitionCards[i].get());
	}

	mChangeSpace=true;
	g_debug("card attached");
}

void PokerCardProjector::ActionFromHandvalueToMain()
{
	int end=(int)mTransformCards.size();
	for (int i=0;i<end;i++) {
		mCards[i]->Anchor(mTransformCards[i].get());
	}
	g_debug("card detach");
	mChangeSpace=false;
}


void PokerCardProjector::ActionFromTransitionToHandvalue()
{
	int end=(int)mCards.size();
	for (int i=0;i<end;i++) {
		mCards[i]->Anchor(mHandValueTransform[i].get());
	}
	mChangeSpaceFinal=true;
}

void PokerCardProjector::SetupCardInNewSpace(PokerCameraModel* camera)
{
	osg::Vec3 space[3];

	//   osg::Vec3 z=camera->GetTarget()-camera->GetPosition();
	//   osg::Vec3 y=camera->GetUp();
	//   z.normalize();


	Get3VectorFromMatrix(mTransformTransitions[0]->getMatrix(),space[0],space[1],space[2]);
	Get3VectorFromMatrix(mTransformTransitionCards[0]->getMatrix(),space[0],space[1],space[2]);
	for (int i=0;i<5;i++) {
		//     osg::Vec3 pos=mTransformTransitions[i]->getMatrix().getTrans();
		osg::Vec3 pos=mTransformTransitionCards[i]->getMatrix().getTrans();
		osg::Vec3 res=osg::Vec3(pos*space[0],pos*space[1],0);
		mHandValueSpace[i]->setMatrix(osg::Matrix::translate(res));

		osg::Vec3 pos2=mTransformTransitionTranslations[i]->getMatrix().getTrans();
		osg::Vec3 res2=osg::Vec3(pos2*space[0],pos2*space[1],0);
		mHandValueTranslations[i]->setMatrix(osg::Matrix::translate(res2));
	}
}




class AlterMaterialColor2 : public osg::NodeVisitor {
public:
	AlterMaterialColor2(const osg::Vec4& color) : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN), mColor(color) {
		setNodeMaskOverride(0xffffffff);
	}

	virtual void apply(osg::Geode& geode) {
		int num_drawables = geode.getNumDrawables();
		g_debug("Num drawable %d",num_drawables);
		if (!num_drawables)
			return;
		//     g_assert(num_drawables == 1);
		for (int i=0;i<num_drawables;i++) {
			osg::Drawable* drawable = geode.getDrawable(i);
			osg::StateSet* state = drawable->getStateSet();
			//     g_assert(state != 0);
			if(state) {
				osg::Material* material = dynamic_cast<osg::Material*>(state->getAttribute(osg::StateAttribute::MATERIAL));
				if(!material) material = new osg::Material;
				material->setColorMode(osg::Material::DIFFUSE);
				material->setDiffuse(osg::Material::FRONT_AND_BACK, mColor);
				state->setAttributeAndModes(material, osg::StateAttribute::ON);
			}
		}
	}

private:
	osg::Vec4 mColor;
};

void PokerCardProjector::ShadeUsedCardForBestHand(const std::vector<int>& cards)
{
	g_debug("check for shade cards %d",cards.size());
	std::cout << "cards candidate ";
	for (unsigned int i=0;i<cards.size();i++)
		std::cout << cards[i] << " ";
	std::cout << "cards candidate ";
	std::cout << std::endl;
	std::cout << "cards board " << mCards.size() << " ";
	for (unsigned int i=0;i<mCards.size();i++)
		std::cout << mCards[i]->GetValue() << " ";
	std::cout << std::endl;
	for(int j = 0; j < (int)mCards.size(); j++) {
		bool found=false;
		for(int i = 0; i < (int)cards.size(); i++)
			if (cards[i] == mCards[j]->GetValue()) {
				found=true;
				break;
			}

			if (!found) {
				g_debug("card %d is shaded",j);
				osg::Node* node=mCards[j]->GetModel()->GetArtefact();
				AlterMaterialColor2 alter(mParamShadeColorForCardUsed);
				node->accept(alter);
			}
	}
}


void PokerCardProjector::SetShadeOfCardsToDefault()
{
	for(int j = 0; j < (int)mCards.size(); j++) {
		osg::Node* node=mCards[j]->GetModel()->GetArtefact();
		AlterMaterialColor2 alter(osg::Vec4(1,1,1,1));
		node->accept(alter);
	}
}



