/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 * Loic Dachary <loic@gnu.org>
 * Cedric Pinson <cpinson@freesheep.org>
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#include <cstdio>
# define snprintf _snprintf
#endif

#include <glib.h>
#include <string>
#include <algorithm>
#include <list>
#include <vector>
#include <cstdlib>

#include <maf/maferror.h>
#include <maf/data.h>
#include <maf/scene.h>
#include <maf/window.h>
#include <maf/utils.h>

#include <ugame/debug.h>
#include <ugame/BetSlider>

#include <osg/Vec3>
#include <osg/Vec4>
#include <osg/MatrixTransform>
#include <osg/Geometry>
#include <osg/AutoTransform>
#include <osg/PolygonOffset>

#include <osgchips/Stacks>

#include <ugame/text.h>

#include <libintl.h>
#define _(String) gettext (String)

#ifndef WIN32
#	include "PokerChipsStack.h"
#	include "PokerCursor.h"
#	include "PokerApplication.h"
#endif // WIN32

static const char* fake = _("FAKE");

static const osg::Vec3	anchors[] = 
  {
    osg::Vec3(0,0,0),
    osg::Vec3(-0.89,0,-0.52),
    osg::Vec3(0.9,0,-0.51),
    osg::Vec3(-0.9,0,0.52),
    osg::Vec3(0.88,0,0.52),
    osg::Vec3(-1.78,0,0),
    osg::Vec3(1.79,0,0.01),
    osg::Vec3(0,0,1.05),
    osg::Vec3(-1.79,0,1.03),
    osg::Vec3(1.78,0,1.06),
    osg::Vec3(-0.9,0,1.56),
    osg::Vec3(0.88,0,1.57),
    osg::Vec3(0,0,-1.05),
  };
static const size_t	anchors_count = sizeof (anchors) / sizeof (osg::Vec3);

void PokerChipsStackModel::InitTooltip(PokerApplication* game)
{
	mTooltip=new UGAMEBasicText("Tooltip",MAFLoadFont("data/FreeSansBold.ttf", game->GetOptions()));
	mTooltip->getText()->setColor(osg::Vec4(1,1,1,0.8));
	mTooltip->getText()->setCharacterSize(10);
	mTooltip->getText()->setCharacterSizeMode(osgText::Text::OBJECT_COORDS);
	mTooltip->getText()->setPosition(osg::Vec3(0,0,0));

	mTooltip->getText()->setLayout(osgText::Text::LEFT_TO_RIGHT);
	mTooltip->getText()->setAlignment(osgText::Text::CENTER_CENTER);
	mTooltip->getText()->setAxisAlignment(osgText::Text::XY_PLANE);
	mTooltip->getText()->setText("");
	osg::AutoTransform* autot=new osg::AutoTransform;
	autot->setAutoRotateMode(osg::AutoTransform::ROTATE_TO_SCREEN);
	autot->addChild(mTooltip.get());
	autot->setPosition(osg::Vec3(0,35,0));
	GetPAT()->addChild(autot);
}
void PokerChipsStackModel::ShowTooltip(bool state)
{
  if (state) {
	unsigned int result=0;
	for (std::map<unsigned int,unsigned int>::iterator it=mChips.begin();it!=mChips.end();it++)
	  result+=it->first * it->second;
	char str[16];
	snprintf(str,15,"%d\n",result);
	mTooltip->getText()->setText(str);
    mTooltip->setNodeMask(MAF_VISIBLE_MASK);
  } else
	mTooltip->setNodeMask(~(MAF_VISIBLE_MASK|MAF_COLLISION_MASK));
}

PokerChipsStackModel::PokerChipsStackModel(PokerApplication* game) :
  mSelected(false)
{
  Init();
  mStacks = new osgchips::Stacks;
  const osg::BoundingBox& box = osgchips::ChipBank::instance()->getBoundingBox();
  float diameter = box.xMax() - box.xMin();
  for(size_t i = 0; i < anchors_count; i++) {
    osgchips::Stack* stack = new osgchips::Stack;
    stack->setCount(0);
    stack->setPosition(anchors[i] * diameter);
    mStacks->addStack(stack);
  }
  mStacks->getOrCreateStateSet()->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

  osg::Group* group = new osg::Group;
  group->setName("PokerChipsStack");
  group->addChild(mStacks.get());

  SetArtefact(group);
  InitTooltip(game);
}

PokerChipsStackModel::~PokerChipsStackModel() {
}

PokerChipsStackController::PokerChipsStackController(PokerApplication* game) {
  SetModel(new PokerChipsStackModel(game));
}

PokerChipsStackController::~PokerChipsStackController() {
}

void PokerChipsStackController::SetChips(const std::vector<int>& chips) {
  if(chips.size() % 2 != 0) {
    g_critical("PokerChipsStackController::SetChips: chips list size is not a multiple of two");
    return;
  }

  std::map<unsigned int,unsigned int> chipsMap;

  for(unsigned int i = 0; i < chips.size(); i += 2) {
    unsigned int value = chips[i];
    unsigned int count = chips[i + 1];
    chipsMap[value] = count;
  }

  GetModel()->mChips = chipsMap;

  SyncChips();
}

void PokerChipsStackController::AddChips(const std::map<unsigned int,unsigned int>& chips) {
  std::map<unsigned int,unsigned int>& chipsMap = GetModel()->mChips;

  for(std::map<unsigned int,unsigned int>::const_iterator i = chips.begin();
      i != chips.end();
      i++) {
    unsigned int value = i->first;
    unsigned int count = i->second;
    if(chipsMap.find(value) != chipsMap.end())
      chipsMap[value] += count;
    else
      chipsMap[value] = count;
  }

  SyncChips();
}

void PokerChipsStackController::SubChips(const std::map<unsigned int,unsigned int>& chips) {
  std::map<unsigned int,unsigned int>& chipsMap = GetModel()->mChips;
  std::vector<int> toRemove;
  for(std::map<unsigned int,unsigned int>::const_iterator i = chips.begin();
      i != chips.end();
      i++) {
    unsigned int value = i->first;
    unsigned int count = i->second;

    if(chipsMap.find(value) != chipsMap.end()) {
      if(chipsMap[value] > count) {
	chipsMap[value] -= count;
      } else {
	toRemove.push_back(value);
      }
    } else {
      g_critical("PokerChipsStackController::SubChips: cannot subtract %d from value %d", count, value);
    }
  }

  for(std::vector<int>::iterator i = toRemove.begin(); i != toRemove.end(); i++) {
    chipsMap.erase(*i);
  }
  
  SyncChips();
}

void PokerChipsStackController::SyncChips() {
  PokerChipsStackModel* model = GetModel();

  if(model->mChips.size() > model->mStacks->getNumStacks()) {
    g_critical("PokerChipsStackController::SetChips: not enough 3D stacks (%d) to represent %d stacks", model->mStacks->getNumStacks(), model->mChips.size());
  }

  unsigned int j = 0;
  for(std::map<unsigned int,unsigned int>::iterator i = model->mChips.begin();
      i != model->mChips.end();
      i++) {
    unsigned int value = i->first;
    unsigned int count = i->second;
    char tmp[16];
    snprintf(tmp, 16, "%d", value);
    osgchips::Stack* stack = model->mStacks->getStack(j);
    g_assert(stack != 0);
    osgchips::ChipBank::Chip* chip = osgchips::ChipBank::instance()->getChip(tmp);
    if(!chip) {
      g_critical("PokerChipsStackController::SetChips: chip value %d not defined in chip bank", value);
      continue;
    }
    stack->setChip(chip);
    stack->setCount(count);
    j++;
  }

  for(; j < model->mStacks->getNumStacks(); j++) {
    osgchips::Stack* stack = model->mStacks->getStack(j);
    stack->setChip(0);
  }
}

void PokerChipsStackController::CreateSlider(PokerApplication* game) {
  betslider::BetSlider* slider = new betslider::BetSlider;
  slider->unserialize(game->GetHeaders()["sequence"], game->GetOptions());
  PokerChipsStackModel* model = GetModel();
  model->mSlider = slider;
  model->mSliderPosition = new osg::PositionAttitudeTransform;
  model->mSliderPosition->setPosition(osg::Vec3(game->GetWindow(true)->GetWidth() / 2.f, 200.f, 0.f));
  model->mSliderPosition->addChild(slider);
}

bool PokerChipsStackController::Update(MAFApplication* application)
{
  PokerApplication* game = static_cast<PokerApplication*>(application);
  SDL_Event* event = game->GetLastEvent(this);

  PokerChipsStackModel* model = GetModel();
  if(event && model->mSlider.valid()) {
    if (game->GetFocus() == this || model->mSelected || ( event->type == SDL_MOUSEBUTTONDOWN && event->button.button == SDL_BUTTON_RIGHT )) {
      switch (event->type) {
      case SDL_MOUSEBUTTONDOWN:
	{
	  model->mSelected = true;
	  model->mX = event->button.x;
	  model->mY = event->button.y;
	  LOCK_MOUSE(game, this);
	  game->mCursor->ShowCursor(false);
	  game->GetScene()->GetModel()->mHUDGroup->removeChild(model->mSliderPosition.get());
	  game->GetScene()->GetModel()->mHUDGroup->addChild(model->mSliderPosition.get());
	}
	break;
      case SDL_MOUSEMOTION:
	if(model->mSelected) {
	  float pixel_delta = (float)(model->mY - event->motion.y);
	  if(pixel_delta) {
	    float delta = pixel_delta / application->GetWindow(true)->GetHeight();
	    model->mSlider->moveCursor(delta);
	    game->mCursor->WarpMouse(model->mX, model->mY);
	  }
	}
	break;
      case SDL_MOUSEBUTTONUP:
	if(model->mSelected) {
	  model->mValue = model->mSlider->getCurrentValue();
	  model->mSelected = false;
	  UNLOCK_MOUSE(game, this);
	  game->mCursor->ShowCursor(true);
	  game->GetScene()->GetModel()->mHUDGroup->removeChild(model->mSliderPosition.get());
	}
      }
    }
  }


  // focus ->Showtooltip
  if (game->GetFocus() == this)
    GetModel()->ShowTooltip(true);
  else
	GetModel()->ShowTooltip(false);



  return true;
}

void PokerChipsStackController::SetBetLimits(int min, int max, int step, int call) {
  g_assert(GetModel()->mSlider.valid());
  GetModel()->mSlider->setLimits(call, min, max, max + 1, max + 1, step);
}
