/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Johan Euphrosine <johan@mekensleep.com>
 *  Cedric Pinson <cpinson@freesheep.org>
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H


#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#include <cstdio>
# define snprintf _snprintf
#endif

#include <algorithm>

#include <osg/MatrixTransform>
#include <osg/Matrix>
#include <osg/AutoTransform>
#include <osg/Node>
#include <osg/Geode>
#include <osg/Geometry>
#include <osg/Array>
#include <osg/AnimationPath>
#include <osg/BlendFunc>
#include <maf/MultipleAnimationPathCallback.h>
//#include <osg/AnimationPathCallback>


#include <maf/maferror.h>
#include <maf/osghelper.h>
#include <maf/scene.h>
#include <maf/split.h>
#include <maf/scene.h>
#include <maf/shadow.h>
#include <maf/utils.h>
#include <maf/window.h>

#include <ugame/animated.h>
#include <ugame/debug.h>

#include "PokerMoveChips.h"
#include "PokerSound.h"

#ifndef WIN32
#	include "PokerPlayerCamera.h"
#	include "PokerApplication.h"
#	include "PokerChipsStack.h"
#	include "PokerCard.h"
#	include "PokerChipsStack.h"
#	include "PokerPlayer.h"
#	include "PokerBubble.h"
#	include "PokerBody.h"
#       include "PokerCursor.h"
#endif // WIN32

#include <osgAL/SoundManager>

// this flag should be remove later
#define NB_SOUND_PER_PLAYER 10
#define MAX_ANIMATION_CARD_TO_PLAY 30


void PokerDummyCaller::Init(MAFOSGData *seatData)
{
  UGAMEArtefactController::Init();
  osg::Shape *shape = new osg::Box(osg::Vec3(0.0f, 0.0f, 0.0f), 5.0f, 5.0f, 5.0f);
  osg::ShapeDrawable *drawable = new osg::ShapeDrawable(shape);
  osg::Geode *geode = new osg::Geode();
  geode->addDrawable(drawable);
  GetModel()->SetArtefact(geode);
  SetSelectable(true);

  const std::string &anchorName = "transform_btok";
  assert(anchorName.length() != 0);
  MAFAnchor *anchor = static_cast<MAFAnchor *>(seatData->GetNode(anchorName));
  assert(anchor != 0);
  Anchor(anchor);
}
bool PokerDummyCaller::Update(MAFApplication *application)
{
  UGAMEArtefactController::Update(application);
  if (GetSelected())
    {
      SendPacket(application);
      SetSelected(false);
    }
  return true;
}

void PokerDummyCaller::SendPacket(MAFApplication *application)
{
  PokerApplication *game = static_cast<PokerApplication *>(application);
  osg::ref_ptr<MAFPacket> packet = game->GetPacketsModule()->Create("PacketPokerClientAction");
  packet->SetMember("serial", game->GetPoker()->GetModel()->mMe);
  packet->SetMember("game_id", game->GetPoker()->GetModel()->mGameSerial);
  packet->SetMember("action", std::string("call"));
  game->PythonCall("scheduleAction", packet.get());
}




void PokerDummySlider::Init(MAFOSGData *seatData)
{
  UGAMEArtefactController::Init();
  osg::Shape *shape = new osg::Box(osg::Vec3(0.0f, 0.0f, 0.0f), 5.0f, 5.0f, 5.0f);
  osg::ShapeDrawable *drawable = new osg::ShapeDrawable(shape);
  osg::Geode *geode = new osg::Geode();
  drawable->setColor(osg::Vec4(1.0f, 0.0f, 0.0f, 1.0f));
  geode->addDrawable(drawable);
  GetModel()->SetArtefact(geode);
  SetSelectable(true);

  const std::string &anchorName = "transform_btok";
  assert(anchorName.length() != 0);
  MAFAnchor *anchor = static_cast<MAFAnchor *>(seatData->GetNode(anchorName));
  assert(anchor != 0);
  Anchor(anchor);
  GetModel()->GetPAT()->setPosition(osg::Vec3(10.f, 0.0f, 0.0f));
}
bool PokerDummySlider::Update(MAFApplication *application)
{
  UGAMEArtefactController::Update(application);
  if (GetSelected())
    {
      SendPacket(application);
      SetSelected(false);
    }
  return true;
}
void PokerDummySlider::SendPacket(MAFApplication *application)
{
  PokerApplication *game = static_cast<PokerApplication *>(application);
  osg::ref_ptr<MAFPacket> packet = game->GetPacketsModule()->Create("Packet");
  packet->SetMember("name", std::string("raise"));
  packet->SetMember("event", std::string("selected"));
  game->PythonCall("interactorSelected", packet.get());
}


PokerPlayer::PokerPlayer(PokerApplication* game, bool me, const std::string& outfitName) : mGame(game), mSeatId(POKER_PLAYER_NO_SEAT), mInGame(false), mSit(false), mCamera(0), mMe(me), mStandUpAnimationId(-1), mMoveTimeout(0.0), mSitInTimeout(0.0), mClickOnChair(false) {

  //  g_debug("PokerPlayer::PokerPlayer: 0x%08x", (int)this);

  //
  // Create objects
  //

  mShadowFlag=game->mShadowFlag;

  //
  // Interaction objects
  //
  {
    const std::list<std::string> &interactors = game->HeaderGetList("sequence", "/sequence/interactor/@name");
    for(std::list<std::string>::const_iterator i = interactors.begin();
	i != interactors.end();
	i++) {
      const std::string &name = *i;
      const std::string &url = game->HeaderGet("sequence", "/sequence/" + name + "/@url");
      MAFOSGData* data = dynamic_cast<MAFOSGData*>(game->mDatas->GetVision(url));
      
      mInteractors[name] = new UGAMEArtefactController();
      mInteractors[name]->Init();
      mInteractors[name]->Displayed(false);
      mInteractors[name]->GetModel()->SetData(data);
      mInteractors[name]->GetModel()->SetArtefact(data->GetGroup());
    }
  }

  //
  // Seat
  //
  {
    std::string seat_name = game->HeaderGet("sequence", "/sequence/seat/@url");
    mSeatData = dynamic_cast<MAFOSGData*>(game->mDatas->GetVision(seat_name)->Clone());
    UGAMEArtefactModel* vision_model = new UGAMEArtefactModel;
    vision_model->SetData(mSeatData);
    mSeat = new UGAMEArtefactController;
    mSeat->SetModel(vision_model);
    mSeat->Init();
    vision_model->SetArtefact(mSeatData->GetGroup());
  }

  //
  // Cards
  //
  {
    std::string hand_url = game->HeaderGet("sequence", "/sequence/hand/@url");
    std::string hand_count = game->HeaderGet("sequence", "/sequence/hand/@count");
    int count = atoi(hand_count.c_str());
    for(int i = 0; i < count; i++) {
      mHole.push_back(new PokerCardController(game, hand_url));
    }
    mShowdown = new PokerShowdownController(game, mSeatData);
    game->AddController(mShowdown.get());
  }


  std::string outfit_type;
  {
    //
    const std::string &player = game->HeaderGet("sequence", "/sequence/player/@url");

    std::string outfit_name;
    const std::string &defaultOutfit = game->HeaderGet("sequence", "/sequence/outfit/@default");
    if(defaultOutfit == "")
      g_critical("PokerPlayer::PokerPlayer: /sequence/outfit/@default does not define a default outfit");

    if(outfitName == "default") {
      GetTypeAndNameFromOutfit(defaultOutfit,outfit_name,outfit_type);
    } else
      GetTypeAndNameFromOutfit(outfitName,outfit_name,outfit_type);
      
    std::string dataPlayerName=player + "." + outfit_type + ".cal3d";
    MAFVisionData* data = game->mDatas->GetVision(dataPlayerName);
    if (!data)
      g_error("PokerPlayer::PokerPlayer data not found for %s",dataPlayerName.c_str());


    mBody = new PokerBodyController(game,data,mSeatData);

    if(outfitName == "default")
      mBody->GetModel()->SetDefaultOutfit(outfit_name);
    mBody->GetModel()->SetOutfit(outfit_name);




    mBody->SetMe(mMe);

    const std::string &useVertexProgram = game->HeaderGet("settings", "/settings/vprogram");
    mBody->SetUseVertexProgram(useVertexProgram == "on");

    const std::string &dataPath = game->HeaderGet("settings", "/settings/data/@path");
//     mBody->SetDataPath(dataPath); // now we want to have the data directory in the player directory not in another dir
    mBody->SetDataPath(dataPath+"/"+dataPlayerName);

    mBody->SetDeck(game->mDeck);
    

    if(game->HeaderGet("sequence", "/sequence/player/@animated") == "no")
      mBody->GetModel()->DisableAnimations();

    mBody->Init();

//     mBody->PlayFacialNoise();

    game->AddController(mBody.get());

  }

  //
  // Money & bet stack of the player
  //
  {
    mMoneyStack = new PokerChipsStackController(game);
    const std::string& name = game->HeaderGet("sequence", "/sequence/money/@anchor");
    MAFAnchor* anchor = mSeatData->GetAnchor(name);
    if(anchor == 0)
      g_error("PokerPlayer::PokerPlayer: anchor %s not found", name.c_str());
    else
      mMoneyStack->Anchor(anchor);
    game->AddController(mMoneyStack.get());
      mMoneyStack->SetSelectable(true);
	if(mMe) {
      mMoneyStack->CreateSlider(game);
    }
  }

  {
    mBetStack = new PokerChipsStackController(game);
    const std::string& name = game->HeaderGet("sequence", "/sequence/bet/@anchor");
    MAFAnchor* anchor = mSeatData->GetAnchor(name);
    if(anchor == 0)
      g_error("PokerPlayer::PokerPlayer: anchor %s not found", name.c_str());
    else
      mBetStack->Anchor(anchor);
    game->AddController(mBetStack.get());
  }

  mInPosition = false;

  char tmp[128];

  //
  // Move interaction objects
  //
  {
    std::list<std::string> interactors = game->HeaderGetList("sequence", "/sequence/interactor/@name");
    for(std::list<std::string>::iterator i = interactors.begin();
	i != interactors.end();
	i++) {
      std::string name = *i;

      std::string anchor_name = mGame->HeaderGet("sequence", "/sequence/" + name + "/@anchor");
      MAFAnchor* anchor = mSeatData->GetAnchor(anchor_name);
      g_assert(anchor != 0);
      mInteractors[name]->Anchor(anchor);
    }
  }

  //
  // Move Cards
  //
  {
    std::string anchor_card = mGame->HeaderGet("sequence", "/sequence/hand/@anchor");
    std::string hand_count = mGame->HeaderGet("sequence", "/sequence/hand/@count");

    mHole.resize(atoi(hand_count.c_str()));
    for(unsigned int i = 0; i < mHole.size(); i++) {
      snprintf(tmp, sizeof(tmp), anchor_card.c_str(), ('A' + i));
      mHole[i]->Anchor(mSeatData->GetAnchor(tmp));
    }
  }

  //
  // Move body
  //
  {
    std::string anchor_player = mGame->HeaderGet("sequence", "/sequence/player/@anchor");
    mBody->Anchor(mSeatData->GetAnchor(anchor_player));
    if (mMe)
      mBody->SetSelectable(true);
  }


  // Bubble 
  std::string displayBubble = mGame->HeaderGet("sequence", "/sequence/bubble/@enable");
  mDisplayBubble = (displayBubble == "yes");
  if (mDisplayBubble)
    {
      const std::string &anchor = mGame->HeaderGet("sequence", "/sequence/bubble/@anchor");
      mBubble = new PokerBubbleController(mGame);
      mBubble->Anchor(mSeatData->GetAnchor(anchor));
      game->AddController(mBubble.get());
    }
  
  // init time waiting
  mTimePlaying=0;
  mTimeWaiting=0;
  mTimeNoise=0;
  mInternalTime=0;

  // Camera parameters
  if(mMe) {
    PokerCameraController* camera = dynamic_cast<PokerCameraController*>(mGame->GetScene()->GetModel()->mCamera.get());
    std::map<std::string, std::string> params = game->HeaderGetProperties("sequence", "/sequence/cameras");
    mCamera = new PokerPlayerCamera(camera, params);
  }

  {
    // counter setting
    mCounterText = new osgText::Text;
    mCounterText->setFont(MAFLoadFont(mGame->HeaderGet("settings", "/settings/data/@path") + "/FreeSansBold.ttf", game->GetOptions()));
    mCounterText->setColor(osg::Vec4(1,1,1,1));
    mCounterText->setCharacterSize(10);
    //  text->setCharacterSize(64);
    mCounterText->setCharacterSizeMode(osgText::Text::OBJECT_COORDS);
//     int w=mGame->GetScene()->GetModel()->mScene->getViewport()->width();
//     int h=mGame->GetScene()->GetModel()->mScene->getViewport()->height();
//     mCounterText->setPosition(osg::Vec3(w/2,h/2,0));
    mCounterText->setLayout(osgText::Text::LEFT_TO_RIGHT);
    mCounterText->setAlignment(osgText::Text::CENTER_CENTER);
    mCounterText->setAxisAlignment(osgText::Text::SCREEN);
    mCounterText->setText("");

    mCounterGeode= new osg::Geode;
    mCounterGeode->setName("counter");
    mCounterGeode->addDrawable(mCounterText.get());
    //  mCounterGeode->setNodeMask(0);

    // set up the drawstate.
    osg::StateSet* dstate = mCounterGeode->getOrCreateStateSet();
    dstate->setMode(GL_CULL_FACE, osg::StateAttribute::OFF);
    dstate->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

    std::string anchor_player = mGame->HeaderGet("sequence", "/sequence/bubble/@anchor");
    osg::MatrixTransform* tr=new osg::MatrixTransform;
    tr->setMatrix(osg::Matrix::translate(0,15,0));
    tr->addChild(mCounterGeode.get());
    mSeatData->GetAnchor(anchor_player)->addChild(tr);
    mCounterGeode->setNodeMask(0);
  }


  // SitIn/Out Timeout
  if (mMe) {
    int sitInId = GetBody()->GetCoreAnimationId("seatDown");
    mSitInTimeout = GetBody()->GetDuration(sitInId);
  }

  /// from player to bet
  {
    std::string playerChipsAnchor = mGame->HeaderGet("sequence", "/sequence/player/@betzone");
    osg::Node* playerChipsAnchorNode=mSeatData->GetAnchor(playerChipsAnchor);
    assert(playerChipsAnchorNode);

    std::string playerBetZone = mGame->HeaderGet("sequence", "/sequence/player/@betzone");
    osg::Node* playerBetZoneNode=mSeatData->GetAnchor(playerBetZone);
    assert(playerBetZoneNode);


    /// player to bet
//     {
//       osg::Vec3 src=playerChipsAnchorNode->asTransform()->asMatrixTransform()->getMatrix().getTrans();
//       osg::Vec3 dst=playerBetZoneNode->asTransform()->asMatrixTransform()->getMatrix().getTrans();
//       mMoveChipsFromPlayerToBet=new PokerMoveChipsController(mGame,src,dst,0.5);
//       mSeatData->GetGroup()->addChild(mMoveChipsFromPlayerToBet->mTransform.get());
//       mGame->AddController(mMoveChipsFromPlayerToBet.get());
//     }

    /// from bet to pot reserve 10
    for (int i=0;i<10;i++) {
      PokerMoveChipsBet2PotController* a=new PokerMoveChipsBet2PotController(mGame,playerBetZoneNode);
      mMoveChipsFromBetToPot.push_back(a);
      game->mSetData->GetGroup()->addChild(a->mTransform.get());
      mGame->AddController(a);
    }

    /// from pot to player
    for (int i=0;i<10;i++) {
      PokerMoveChipsPot2PlayerController* a=new PokerMoveChipsPot2PlayerController(mGame,playerChipsAnchorNode);
      mMoveChipsFromPotToPlayer.push_back(a);
      game->mSetData->GetGroup()->addChild(a->mTransform.get());
      mGame->AddController(a);
    }
  }

  // new interactors
  {
    if (mMe)
      {	
	mFoldInteractor = new PokerInteractor();
	mFoldInteractor->Init(game, mSeatData, "/sequence/dummyinteractors/fold/");
	mCheckInteractor = new PokerInteractor();
	mCheckInteractor->Init(game, mSeatData, "/sequence/dummyinteractors/check/");
      }
  }


  mBeginPos = osg::Vec3(0.0f, 0.0f, 0.0f);
  mCurrentPos = osg::Vec3(0.0f, 0.0f, 0.0f);
  mDeltaPos = osg::Vec3(0.0f, 0.0f, 0.0f);
  mDeltaSet = false;
  mDiffPosLengthLimit = atof(game->HeaderGet("sequence", "/sequence/seat/sitout/@lengthToChooseDir").c_str());
  mDeltaIncrement = atof(game->HeaderGet("sequence", "/sequence/seat/sitout/@step").c_str());

  mCounterCurrent=1e6;


  // card animation
  {
    const std::string& animate = game->HeaderGet("sequence", "/sequence/seat/@animation1card").c_str();
    if (animate.empty())
      g_error("PokerPlayer::PokerPlayer /sequence/seat/@animation1card not found");
    MAFAnchor* aa=mSeatData->GetAnchor(animate.c_str());
    if (!aa)
      g_error("PokerPlayer::PokerPlayer /sequence/seat/@animation1card not found");
    osg::MatrixTransform* atr=aa->asGroup()->asTransform()->asMatrixTransform();
    if (!atr->getUpdateCallback())
      g_error("PokerPlayer::PokerPlayer no animation found in node %s",animate.c_str());
    osg::MultipleAnimationPathCallback* cb=dynamic_cast<osg::MultipleAnimationPathCallback*>(atr->getUpdateCallback());
    if (!cb)
      g_error("PokerPlayer::PokerPlayer update call back not found");

      
    const std::string& paramMultiplier = game->HeaderGet("sequence", "/sequence/seat/@timeScale").c_str();
    if (paramMultiplier.empty())
      g_error("PokerPlayer::PokerPlayer /sequence/seat/@timeScale not found");
    float mParamMultiplier=atof(paramMultiplier.c_str());

    atr->setNodeMask(0);
    for (int i=0;i<MAX_ANIMATION_CARD_TO_PLAY;i++) {
      osg::MatrixTransform* natr=(osg::MatrixTransform*)atr->clone(osg::CopyOp::DEEP_COPY_ALL);
      mAnimate1Card.push_back(natr);
      natr->setNodeMask(~(MAF_VISIBLE_MASK | MAF_COLLISION_MASK));
      osg::MultipleAnimationPathCallback* ncb=new osg::MultipleAnimationPathCallback(*cb);
      ncb->setTimeMultiplier(mParamMultiplier);
      natr->setUpdateCallback(ncb);
      osg::AnimationPath* pt=ncb->getAnimationPath();
      assert(pt);
      pt->setLoopMode(osg::AnimationPath::NO_LOOPING);
      atr->getParent(0)->addChild(natr);
      mAnimate1CardAnchor = atr->getParent(0);  
    }
    mCurrent1Card=0;
  }

  FoldHoleCards();

  mSounds.clear();
  if (mGame->GetScene()->GetModel()->GetAudio()) {
//     const std::string& soundsPath=game->HeaderGet("settings", "/settings/data/@sounds");
    const std::string& soundConfigFile=game->HeaderGet("sequence", "/sequence/player/@soundconfigfile");
	if (!soundConfigFile.empty()) {

		std::string cfile=outfit_type+"-"+soundConfigFile;
		MAFXmlData *data = game->mDatas->GetXml(cfile);
		if (data) {
	  
			std::vector<SoundInit> params;
			LoadSoundSettings(params,data);

			for (unsigned int i=0;i<params.size();i++) {
			const std::string& n=params[i].mName;
			mSounds[n]=GetSound(params[i]);
			//params[i].mSoundName,params[i].mAnchor,params[i].mReferenceDistance,params[i].mReferenceDistance);
			}
		} else
			g_critical("PokerPlayer::PokerPlayer config file %s not found",cfile.c_str());

	} else
		g_critical("PokerPlayer::PokerPlayer /sequence/player/@soundconfigfile not found in client.xml");

	mBody->GetModel()->BuildAnimationSoundMap(mSounds);
  }




  // init mousestate with the PokerCursor
  // A refactory is needed about mouse/cursor system !!!
  mMouseState.Init(game->mCursor);
  
  mShadowMT = NULL;
}



MAFAudioController* PokerPlayer::GetSound(const SoundInit& sound)//const std::string& name,const std::string& anchor,float refDistance)
{
  MAFAudioData* data = mGame->mDatas->GetAudio(sound.mSoundName);
  if (!data) {
    g_debug("PokerPlayer::GetSound Unable to load soundtrack %s",sound.mSoundName.c_str());
    return 0;
  }

  //  float rolloff=1;

  MAFAudioModel* audio_model = new MAFAudioModel;
  audio_model->SetName(sound.mSoundName);
  audio_model->SetData(data);
  MAFAudioController* audio_controller = new MAFAudioController;
  audio_controller->SetModel(audio_model);
  audio_controller->Init();
  audio_model->SetAmbient(false);
  if (!sound.mAnchor.empty() && mSeatData->GetAnchor(sound.mAnchor))
    audio_controller->AttachTo(mSeatData->GetAnchor(sound.mAnchor));
  else {
    g_debug("PokerPlayer::GetSound anchor %s not found to attach the sound check its existance in seat.escn",sound.mAnchor.c_str());
    // be careful is not really safe to do taht here
    std::string anchor_player = mGame->HeaderGet("sequence", "/sequence/player/@anchor");
    audio_controller->AttachTo(mSeatData->GetAnchor(anchor_player));
  }
  audio_model->SetReferenceDistance(sound.mReferenceDistance);
  audio_model->SetRolloff(sound.mRolloff);

  return audio_controller;
}



void PokerPlayer::GetTypeAndNameFromOutfit(const std::string& outfit,std::string& name,std::string& type)
{
  name.clear();
  type.clear();
  if (outfit=="none") {
    name="default";
    type="male";
    return;
  }

  if (outfit=="default") {
    name=outfit;
    type="male";
    return;
  } 


  //extract type of outfit
  std::string separator=" - ";
  std::string::size_type pos=outfit.find(separator);
  if (pos != std::string::npos) {
    type=outfit.substr(0,pos);
    name=outfit.substr(pos+separator.size());
  } else {
    // to work with current server
    type="male";
    name=outfit;
  }
}



// only to track the ref count
void PokerPlayer::GetCount()
{
  std::string anchor_player = mGame->HeaderGet("sequence", "/sequence/player/@anchor");
  osg::Node* node=mSeatData->GetAnchor(anchor_player);
  g_debug("PokerPlayer %s count %d",anchor_player.c_str(),node->referenceCount());
}


PokerPlayer::~PokerPlayer() {

  g_debug("PokerPlayer::~PokerPlayer");

  mSounds.clear();
  RecursiveClearUserData(mSeat->GetModel()->GetNode());

  int e=(int)mMoveChipsFromBetToPot.size();
  for (int i=0;i<e;i++) {
    mGame->mSetData->GetGroup()->removeChild(mMoveChipsFromBetToPot[i]->mTransform.get());
    mGame->RemoveController(mMoveChipsFromBetToPot[i].get());
  }
  mMoveChipsFromBetToPot.clear();
  
  e=(int)mMoveChipsFromPotToPlayer.size();
  for (int i=0;i<e;i++) {
    mGame->mSetData->GetGroup()->removeChild(mMoveChipsFromPotToPlayer[i]->mTransform.get());
    mGame->RemoveController(mMoveChipsFromPotToPlayer[i].get());
  }
  mMoveChipsFromPotToPlayer.clear();


  if (mMe)
    {
#if 0
      mGame->RemoveController(mSlider.get());
      RecursiveClearUserData(mSlider->GetModel()->GetNode());
      mSlider = 0;
      mGame->RemoveController(mCaller.get());
      RecursiveClearUserData(mCaller->GetModel()->GetNode());
      mCaller = 0;
#endif      
      mFoldInteractor->Finit(mGame);
      mCheckInteractor->Finit(mGame);
      mFoldInteractor = 0;
      mCheckInteractor = 0;
    }
  
  if (mDisplayBubble)
    mGame->RemoveController(mBubble.get());
  mGame->RemoveController(mShowdown.get());
  mGame->RemoveController(mMoneyStack.get());
  mGame->RemoveController(mBetStack.get());

  mGame->RemoveController(mBody.get());
  RecursiveClearUserData(mBody->GetModel()->GetNode());

  if(mCamera) delete mCamera;

  mSeat = 0;
  mMoneyStack = 0;
  mBetStack = 0;
  mHole.clear();
  mInteractors.clear();
  mBubble = 0;
  mShowdown = 0;
  //
  // Do *NOT* delete this assert. Having more than one reference count
  // at this point is the best way to produce huge memory leaks. If you
  // remove the assert, the program will run just fine but may eat
  // tons of memory when switching tables, for instance.
  // Check pokeranimation.py and c_animated.cpp as they can be the one
  // holding an additional reference.
  //
  //  if(mBody->referenceCount() != 1)
  //    g_critical("mBody->referenceCount() != 1");
  g_assert(mBody->referenceCount() == 1);
  mBody = 0;
  mCounterGeode = 0;
  mCounterText = 0;
//   mMoveChipsFromPlayerToBet = 0;

//  mSoundBet = 0;
//  mSoundStartLookCard = 0;
//  mTestSounds.clear();

  // Animate1Card Clean
  {
    for (unsigned int i = 0; i < mAnimate1Card.size(); i++)
      {
	mAnimate1CardAnchor->removeChild(mAnimate1Card[i].get());
	mAnimate1Card[i] = 0;
      }
    mAnimate1CardAnchor = 0;
    mAnimate1Card.clear();
  }

  osg::NodeVisitor* leakNodes = RecursiveLeakCollect(mSeatData->GetGroup());

  delete mSeatData;

  RecursiveLeakCheck(leakNodes);
}

void PokerPlayer::SetInGame(bool inGame)
{
  if(inGame != mInGame) {
    mInGame = inGame;
  }
}

void PokerPlayer::NoCards(void) {  
  mShowdown->Reset();
  FoldHoleCards();
  FoldPocketCards();
}

void PokerPlayer::FoldHoleCards(void) {
  PokerCardControllerVector& cards = mHole;
  for(unsigned int i = 0; i < cards.size(); i++)
    cards[i]->Fold();

  mBody->GetModel()->mNumberOfCardsReady=0;
  mBody->SetNbCards(0);
  mBody->DettachCardsDrawableToPlayer();

  // if only two player the position is not lost when folding, so
  DisableWarningTimer();
}

void PokerPlayer::SetHoleCards(const std::vector<int>& cards) {
  PokerCardControllerVector& card_controllers = mHole;

  for(unsigned int i = 0; i < card_controllers.size(); i++)
    {
      if (i < cards.size())
	{
	  card_controllers[i]->Receive();
	  if(cards[i] != 255)
	    {
	      card_controllers[i]->SetValue(cards[i]);
	      card_controllers[i]->Visible(true);
	    }
	  else
	    {
	      card_controllers[i]->Visible(false);
	    }
	}
      else
	{
	  card_controllers[i]->Visible(false);
	  card_controllers[i]->Fold();
	}
    }
}

void PokerPlayer::FoldPocketCards(void) {
  FoldHoleCards();
  return ;
//   if (mDisplayBubble) // why ???
    {
      mBody->SetNbCards(0);
      mBody->DettachCardsDrawableToPlayer();
    }
}

void PokerPlayer::SetPocketCards(const std::vector<int>& cards) 
{
  mBody->SetNbCards(cards.size());
  mBody->UpdateCardsOfPlayer(cards);
}

void PokerPlayer::SetMoney(const std::vector<int>& amount) {
  mMoneyStack->SetChips(amount);
}

void PokerPlayer::SetBet(const std::vector<int>& amount) {
  mBetStack->SetChips(amount);
}

void PokerPlayer::BetLimits(int min_bet, int max_bet, int step, int call) {
  mMoneyStack->SetBetLimits(min_bet, max_bet, step, call);
}

unsigned PokerPlayer::GetSelectedValue()
{
  return mMoneyStack->GetSelectedValue();
}

void PokerPlayer::ResetSelectedValue()
{
  mMoneyStack->ResetSelectedValue();
}

void PokerPlayer::SetName(const std::string& name) {
  mName = name;
  if(name == "")
    mBody->Displayed(false);
  else
    mBody->Displayed(true);
}

void PokerPlayer::SetSeatId(int seat_id) {
  if(seat_id != mSeatId) {
    if(seat_id == POKER_PLAYER_NO_SEAT) {
      if (mShadowFlag && mShadowMT) {
	mGame->mShadowedGroup->removeChild( mShadowMT );
	mShadowMT = NULL;
      }
      mSeat->Anchor(0);
      mBody->StopAll();
    } else {
      char tmp[128];
      std::string anchor_seat = mGame->HeaderGet("sequence", "/sequence/seat/@anchor");
      snprintf(tmp, sizeof(tmp), anchor_seat.c_str(), seat_id + 1);
      g_debug("PokerPlayer::SetSeatId: sitting at %s", tmp);
      MAFAnchor* anchor = mGame->mSetData->GetAnchor(tmp);
      mSeat->Anchor(anchor);
      mSeat->GetModel()->GetArtefact()->setName(mMe ? "self" : "player");

      {
	osg::NodePath path;
	osg::Node* node = mSeat->GetModel()->GetArtefact();
	while(node) {
	  path.push_back(node);
	  if(node->getNumParents() >= 1) {
	    if(node->getNumParents() > 1)
	      g_critical("PokerPlayer::SetSeatId: seat has more than one parent");
	    node = node->getParent(0);
	  } else {
	    node = 0;
	  }
	}
	std::reverse(path.begin(), path.end());

	/*
	  toworld get the orientation of seat
	  and toworld2 get a position to compute direction between players
	 */
	osg::Matrix toworld = MAFComputeLocalToWorld(mSeat->GetModel()->GetArtefact());
	std::string anchor_seatpos = mGame->HeaderGet("sequence", "/sequence/player/@anchorlookat");
	if (anchor_seatpos.empty())
	  g_error("anchor %s not found in /sequence/player/@anchorlookat",anchor_seatpos.c_str());

	osg::Node* nodelookat=mSeatData->GetAnchor(anchor_seatpos.c_str());
	osg::Matrix toworld2=MAFComputeLocalToWorld(nodelookat);
	mDirectionOfPlayerInWorldspace = toworld2.getTrans();
	mDirectionOfPlayerInWorldspace.normalize();
	mPositionOfPlayerInWorldspace = toworld2.getTrans();

	if (mShadowFlag) {
	  osg::Group *shadowedGroup = mGame->mShadowedGroup;

	  osg::Group *shadower = (osg::Group*) mBody->GetModel()->GetNode();
	  //	  osg::Group *parent = shadower->getParent(0);

	  osg::Geode *geode = (osg::Geode*) shadower->getChild(0);
	  osg::Matrix mat = MAFComputeLocalToWorld(shadower);
	  mShadowMT = new osg::MatrixTransform();
	  mShadowMT->setMatrix(mat);
	  shadowedGroup->addChild( mShadowMT );
	  geode->getOrCreateStateSet()->setRenderBinDetails(3, "RenderBin");
	  mShadowMT->addChild( geode );
	  shadower->setNodeMask( MAF_COLLISION_MASK );
	}

	if(mCamera) {
	  mCamera->SetPlayerDirection(mDirectionOfPlayerInWorldspace);
	  mCamera->SetPlayerPosition(mPositionOfPlayerInWorldspace);

	  std::string cameras_string = mGame->HeaderGet("sequence", "/sequence/cameras/@names");
	  std::vector<std::string> cameras = std::split(cameras_string, " ", 0);
	  for(std::vector<std::string>::iterator i = cameras.begin();
	      i != cameras.end();
	      i++) {
	    MAFCameraController* camera = mSeatData->GetCamera(*i);
	    g_assert(camera != 0);
	    MAFCameraModel model = *camera->GetModel();
	    model.SetPosition(model.GetPosition() * toworld);
	    model.SetTarget(model.GetTarget() * toworld);
	    mCamera->SetCameraModel(*i, model);
	  }
	}
      }

      if (GetSit())
	; // mBody->PlayWaiting(); now in python
      else
	mBody->PlayStanding();
    }
  }
  mSeatId = seat_id;
}

void PokerPlayer::SetSit(bool sit) {
  if(mSit != sit) {
    mSit = sit;
  }
}

void PokerPlayer::InPosition()
{
  if (!mInPosition) {
    mInPosition=true;
    //    mBody->PlayStartPlaying(); now in python
    mInteractors["position"]->Displayed(true);
    MarkLastAction();
  }
}


void PokerPlayer::DisableWarningTimer()
{
  // init counter for the timeout warning
  mCounterCurrent=1e6;
  mCounterGeode->setNodeMask(0);
}

void PokerPlayer::LostPosition()
{
  if(mInPosition) {
    mInPosition = false;
    mInteractors["position"]->Displayed(false);
    MarkLastAction();

    DisableWarningTimer();
    g_debug("lost position");
  }
}


void PokerPlayer::LookAt(PokerPlayer* player)
{
  mBody->PlayLookAt(GetDirectionOfPlayer(), GetPositionOfPlayer(), player->GetPositionOfPlayer());
}

void PokerPlayer::MarkLastAction()
{
  mTimeWaiting=0;
  mTimePlaying=0;
}

void PokerPlayer::SetTextMessage(const std::string &message)
{
  if (mDisplayBubble)
    mBubble->SetTextMessage(message);
}



void PokerPlayer::Win()
{
  if (mInGame)
    mBody->PlayWin();
  MarkLastAction();
}

double PokerPlayer::PlayStartLookCards()
{
  return mBody->PlayStartLookCards();
}

void PokerPlayer::Loose()
{
  if (mInGame)
    mBody->PlayLoose();
  MarkLastAction();
}

void PokerPlayer::Bet()
{

  //  mBody->PlayBet(); now in python
  MarkLastAction();
  g_debug("Betting !!!");
//   if (mSoundBet.get()) {
//     mSoundBet->Play();
// //     mSoundBet->GetModel()->DumpState();
//   }
}

void PokerPlayer::Check()
{
  // mBody->PlayCheck(); now in python
  MarkLastAction();
}


bool PokerPlayer::Update(MAFApplication* application)
{

  //PokerApplication* game = dynamic_cast<PokerApplication*>(application); // unused

  SDL_Event* event = mGame->GetLastEvent(this);
  PokerCameraModel* camera=(dynamic_cast<PokerCameraController*>(mGame->GetScene()->GetModel()->mCamera.get()))->GetModel();

  if(!mGame->HasEvent()) {

    const double MaxTimeToWait=10;
    const double MaxTimeToPlay=7;

    double deltaS=GetDeltaFrame()/1000.0;
    mInternalTime+=deltaS;

    if (mInPosition)
      mTimePlaying+=deltaS;
    else
      mTimeWaiting+=deltaS;
    mTimeNoise += deltaS;


    // the player is too slow when playing
    if (! (mMe && camera->GetMode()!=PokerCameraModel::CAMERA_FREE_MODE))
      if(mInPosition && mTimePlaying>MaxTimeToPlay) {
	mTimePlaying=0;
      }

    if(mTimeNoise > 2) {
      if(g_random_double() <= 0.5)
	{
	  if (mSit)
	    mBody->PlayBlink();
	}
      mTimeNoise = 0;
    }

    // the player wait to play too long
    if (! (mMe && camera->GetMode()!=PokerCameraModel::CAMERA_FREE_MODE))
      if(mSit && !mInPosition && mTimeWaiting>MaxTimeToWait) {
	mTimeWaiting=0;
      }

    // display counter from 5 to 0
    if (mInPosition) {
      float delta=GetDeltaFrame()/1000.0;
      float prevCounter=mCounterCurrent;
      mCounterCurrent-=delta;
      std::stringstream str;
      if (mCounterCurrent<10000) { // limit to fix currently we want to display it when a timout warning is received
	mCounterGeode->setNodeMask(MAF_VISIBLE_MASK);

	int number=(int)mCounterCurrent;
	if (number<0)
	  number=0;
      
	// check if we have changed of number
	if (number!=(int)prevCounter)
	  mCounterFadeOut=1.0;
      
	mCounterFadeOut-=delta;
	if (mCounterFadeOut<0) {
	  mCounterFadeOut=0;
	}
	str << number+1;
	mCounterText->setText(str.str());
	mCounterText->setColor(osg::Vec4(mCounterFadeOut,mCounterFadeOut,mCounterFadeOut,mCounterFadeOut));
	// 	g_debug("counter int %d %s fade %f counter %f delta %f",number,txt,mCounterFadeOut,mCounterCurrent,delta);

// 	g_debug("counter fade %f counter %f delta %f",mCounterFadeOut,mCounterCurrent,delta);
      }

      if (mCounterCurrent<0)
	mCounterGeode->setNodeMask(!MAF_VISIBLE_MASK);
	
    }
  }

  
  if (mMe)
    {

      bool chairFocused = (GetBody()->GetFocus() == "__chair");
      if (chairFocused) {
	if (mGame->mCursor->GetCursorType()!=PokerCursor::SITOUT)
	  mGame->mCursor->SetSitoutCursor();
      } else if (mGame->mCursor->GetCursorType()==PokerCursor::SITOUT)
	mGame->mCursor->RestoreCursor();

      mMouseState.Update(event);
      if (mMouseState.mStateChanged)
 	mClickOnChair = mMouseState.mLeftButtonClicked && chairFocused;

      if(mSit) {
	if(mStandUpAnimationId >= 0)
	  UpdateSittingOut();
	else
	  UpdateSitIn();
      } else {
	UpdateSitOut(); 
      }
    }

  if(mCamera && mSeatId != POKER_PLAYER_NO_SEAT) {
    bool interactorSelected = false; // todo: new interactorSelected test
    mCamera->SetInteractorSelected(interactorSelected);

    bool playerHasCards = GetBody()->GetNbCards() != 0;
    bool chairNotFocused = !(GetBody()->GetFocus() == "__chair");
    bool bodyFocusable = playerHasCards && chairNotFocused;
    bool bodyFocused = (mGame->GetFocus() == GetBody()) && bodyFocusable;

    switch(mCamera->Update(event, GetDelta(), bodyFocused)) {
    case PokerPlayerCamera::START_LOOK_CARDS:
      mGame->SendPacket("PacketPokerLookCards");
      mCamera->SetLookingCardAnimationTimeout(PlayStartLookCards());
      //      if (mSoundStartLookCard.get())
      //	mSoundStartLookCard->Play();
      break;
    case PokerPlayerCamera::END_LOOK_CARDS:
      mCamera->SetLookingCardAnimationTimeout(PlayEndLookCards());
      break;
    case PokerPlayerCamera::NONE:
      break;
    }
  }

  if(mGame->GetFocus() == GetBody()  && event && event->type == SDL_KEYDOWN) {
    switch(event->key.keysym.sym) {
    case SDLK_a:
      {
	osg::MultipleAnimationPathCallback* apc=dynamic_cast<osg::MultipleAnimationPathCallback*>(mAnimate1Card[0]->getUpdateCallback());
	mAnimate1Card[0]->setNodeMask(0xffffffff);
 	apc->setPause(false);
 	apc->reset();
	
	// mBody->PlayStartPlaying(); now in python
      }
      break;
    default:
      break;
    }
  }

#ifdef TEST_SOUND
  if (event && event->type == SDL_KEYDOWN)
  switch(event->key.keysym.sym) {
  case SDLK_SPACE:
    {
      int nb=mTestSounds.size();
      for (int i=0;i<nb;i++) {
	mTestSounds[i]->Play();
      }
    }
    break;
  default:
    break;
  }
#endif

#if 0
  if (event && event->type == SDL_KEYDOWN)
  switch(event->key.keysym.sym) {
  case SDLK_SPACE:
    {
      AnimateChipsFromBetToPot();      
      assert(0);
    }
    break;
  }
#endif
  return true;
}


void PokerPlayer::TimeoutAdvise(float timeToPlay)
{
  mCounterCurrent=timeToPlay;
}






double PokerPlayer::PlayEndLookCards()
{
  return mBody->PlayEndLookCards();
}

void PokerPlayer::MouseState::WarpMouse(bool state)
{
  mWarp = state; 
  mCursor->ShowCursor(!state);
}

void PokerPlayer::MouseState::Update(SDL_Event *event)
{
  if (mStateChanged)
    mStateChanged = false;
  
  mRX = 0;
  mRY = 0;
  if (event)
    switch(event->type)
      {
      case SDL_MOUSEBUTTONDOWN:
	if  (event->button.button == SDL_BUTTON_LEFT)	
	  LeftClick(true);
	break;
      case SDL_MOUSEBUTTONUP:
	if  (event->button.button == SDL_BUTTON_LEFT)
	  LeftClick(false);
	break;
      case SDL_MOUSEMOTION:
	{
	  mRX = event->motion.xrel;
	  mRY = event->motion.yrel;
	  
	  if (mWarp)
	    {
	      if (event->motion.x != mX || event->motion.y != mY)	      
		mCursor->WarpMouse(mX,mY);
// 		  SDL_WarpMouse(mX, mY);
	      else
		{
		  mRX = 0;
		  mRY = 0;
		}
	    }
	  else
	    {
	      mX = event->motion.x;
	      mY = event->motion.y;
	    }
	}
	break;
      default:
	break;
      }
}


struct PokerPlayerUtil
{
  static void ScreenPosNormalize(osg::Vec3 &out, const osg::Vec3 &in, PokerApplication *application)
  {
    float x = in.x();
    float y = in.y();
    float width = application->GetWindow(true)->GetWidth();
    float height = application->GetWindow(true)->GetHeight();
    float widthOver2 = width / 2.0f;
    float heightOver2 = height / 2.0f;
    out = osg::Vec3((x - widthOver2)/widthOver2, -((y - heightOver2)/heightOver2), 0.0f);
  }
  static void ScreenDirNormalize(osg::Vec3 &out, const osg::Vec3 &in, PokerApplication *application)
  {
    float x = in.x();
    float y = in.y();
    float width = application->GetWindow(true)->GetWidth();
    float height = application->GetWindow(true)->GetHeight();
    float widthOver2 = width / 2.0f;
    float heightOver2 = height / 2.0f;
    out = osg::Vec3(x/widthOver2, -(y/heightOver2), 0.0f);
  }
};

void PokerPlayer::UpdateSitIn()
{
  // be sure to have seat down finished !!!!!!!!!!!!!
  int idSeatDown=GetBody()->GetCoreAnimationId("seatDown");
  if (mMouseState.mLeftButtonClicked && mClickOnChair && !mBody->GetScheduler()->isAnimationActive(idSeatDown))
    {
      mMoveTimeout = 0.0f;
      PokerPlayerUtil::ScreenPosNormalize(mBeginPos, osg::Vec3(mMouseState.mX, mMouseState.mY, 0.0f), mGame);
      mCurrentPos = mBeginPos;
      mDeltaPos = osg::Vec3(0.0f, 0.0f, 0.0f);
      mDeltaSet = false;
      osg::ref_ptr<MAFPacket> packet = mGame->GetPacketsModule()->Create("PacketPokerId");
      packet->SetMember("serial", mGame->GetPoker()->GetModel()->mMe);
      packet->SetMember("game_id", mGame->GetPoker()->GetModel()->mGameSerial);
      mGame->PythonCall(mGame->GetPyScheduler(), "suspend", packet.get());
      mStandUpAnimationId = mBody->PlayStandUp();
      mMouseState.WarpMouse(true);
      mCamera->GetCameraController()->SetRespondToEvents(false);
    }
}

void PokerPlayer::UpdateSitOut()
{
  // Warning we spam the server with SendPacketSit !!!
  if (mMouseState.mLeftButtonClicked && mClickOnChair) {
    SendPacketSit();
  }
}

void PokerPlayer::UpdateSittingOut()
{
  float delta = 0.0f;
  osg::Vec3 deltaPos;
  PokerPlayerUtil::ScreenDirNormalize(deltaPos, osg::Vec3(mMouseState.mRX, mMouseState.mRY, 0.0f), mGame);
  
  mCurrentPos += deltaPos;

  if (!mDeltaSet)
    {
      osg::Vec3 diffPos = mCurrentPos - mBeginPos;
      float diffPosLengthLimit = mDiffPosLengthLimit;
      if (diffPos.length() > diffPosLengthLimit)
	{
	  mDeltaPos = diffPos;
	  mDeltaSet = true;
	}
      else
	{
	}
    }
  else
    {
      float angle = deltaPos * mDeltaPos;
      float deltaIncrement = mDeltaIncrement;
      if (angle > 0.0f)
	delta += deltaIncrement;	
      if (angle < 0.0f)
	delta -= deltaIncrement;
    }
  
  float newTimeout = mMoveTimeout + delta;
  const float timeoutLimit = 0.5f;
  
  if (newTimeout <= 0.0f)
    {
      newTimeout = 0.0f;
      if (!mMouseState.mLeftButtonClicked)
	{
	  mMouseState.WarpMouse(false);
	  mBody->AddTimeSitOut(newTimeout - mMoveTimeout);
	  mCamera->GetCameraController()->SetRespondToEvents(true);
	  mBody->GetScheduler()->stop(mStandUpAnimationId);
	  mStandUpAnimationId = -1;
	  osg::ref_ptr<MAFPacket> packet = mGame->GetPacketsModule()->Create("PacketPokerId");
	  packet->SetMember("serial", mGame->GetPoker()->GetModel()->mMe);
	  packet->SetMember("game_id", mGame->GetPoker()->GetModel()->mGameSerial);
	  mGame->PythonCall(mGame->GetPyScheduler(), "resume", packet.get());
	}
    }
  else if (newTimeout >= timeoutLimit)
    {
      newTimeout = newTimeout < 1.0f ? newTimeout : 1.0f;
      if (!mMouseState.mLeftButtonClicked)
	{
	  //newTimeout = 1.0f;
	  mMouseState.WarpMouse(false);	  
	  mBody->AddTimeSitOut(newTimeout - mMoveTimeout);
	  mCamera->GetCameraController()->SetRespondToEvents(true);
	  mBody->GetScheduler()->stop(mStandUpAnimationId);
	  mStandUpAnimationId = -1;
	  SendPacketSitOut();
	}
      else
	mBody->AddTimeSitOut(newTimeout - mMoveTimeout);
    }
  else if (newTimeout < timeoutLimit)
    {    
      if (!mMouseState.mLeftButtonClicked)
	{ 
	  mMouseState.WarpMouse(false);
	  mBody->AddTimeSitOut(newTimeout - mMoveTimeout);
	  mBody->GetScheduler()->stop(mStandUpAnimationId);
	  mStandUpAnimationId = -1;
	  osg::ref_ptr<MAFPacket> packet = mGame->GetPacketsModule()->Create("PacketPokerId");
	  packet->SetMember("serial", mGame->GetPoker()->GetModel()->mMe);
	  packet->SetMember("game_id", mGame->GetPoker()->GetModel()->mGameSerial);
	  mGame->PythonCall(mGame->GetPyScheduler(), "resume", packet.get());
	  mCamera->GetCameraController()->SetRespondToEvents(true);	  
	}
      else
	mBody->AddTimeSitOut(newTimeout - mMoveTimeout);
    }

  mMoveTimeout = newTimeout;
}

void PokerPlayer::SendPacketSitOut()
{
  osg::ref_ptr<MAFPacket> packet = mGame->GetPacketsModule()->Create("PacketPokerSitOut");
  packet->SetMember("serial", mGame->GetPoker()->GetModel()->mMe);
  packet->SetMember("game_id", mGame->GetPoker()->GetModel()->mGameSerial);
  mGame->PythonCall("sendPacketSitOut", packet.get());
  mGame->PythonCall(mGame->GetPyScheduler(), "resume", packet.get());
}

void PokerPlayer::SendPacketSit()
{
  osg::ref_ptr<MAFPacket> packet = mGame->GetPacketsModule()->Create("PacketPokerSit");
  packet->SetMember("serial", mGame->GetPoker()->GetModel()->mMe);
  packet->SetMember("game_id", mGame->GetPoker()->GetModel()->mGameSerial);
  mGame->PythonCall("sendPacketSit", packet.get());
  mGame->PythonCall(mGame->GetPyScheduler(), "resume", packet.get());
}

PokerMoveChipsBet2PotController* PokerPlayer::GetFreeAnimationBet2Pot()
{
  int e=(int)mMoveChipsFromBetToPot.size();
  for (int i=0;i<e;i++)
    if (mMoveChipsFromBetToPot[i]->IsFinished())
      return mMoveChipsFromBetToPot[i].get();
  
  g_error("Warning no enough animation for Bet2Pot");
  return 0;
}


PokerMoveChipsPot2PlayerController* PokerPlayer::GetFreeAnimationPot2Player()
{
  int e=(int)mMoveChipsFromPotToPlayer.size();
  for (int i=0;i<e;i++)
    if (mMoveChipsFromPotToPlayer[i]->IsFinished())
      return mMoveChipsFromPotToPlayer[i].get();
  
  g_error("Warning no enough animation for Pot2Player");
  return 0;
}


int PokerPlayer::GetMaxCardsAnimation() { return MAX_ANIMATION_CARD_TO_PLAY;}

void PokerPlayer::AnimateCard(int i)
{
  mAnimate1Card[i]->setNodeMask(MAF_VISIBLE_MASK);
  osg::MultipleAnimationPathCallback* cb=dynamic_cast<osg::MultipleAnimationPathCallback*>(mAnimate1Card[i]->getUpdateCallback());
  cb->setPause(false);
  cb->reset();
  
  // play a card
  if (mSounds["dealCard"].get())
    mSounds["dealCard"]->Play();
}


osg::MatrixTransform* PokerPlayer::GetAnimationCard(int i) 
{
  return mAnimate1Card[i].get();
}



void PokerPlayer::HideAnimateCard(int i)
{
  mAnimate1Card[i]->setNodeMask(~(MAF_VISIBLE_MASK|MAF_COLLISION_MASK));
  osg::MultipleAnimationPathCallback* cb=dynamic_cast<osg::MultipleAnimationPathCallback*>(mAnimate1Card[i]->getUpdateCallback());
  cb->reset();
  cb->setPause(true);
}

int PokerPlayer::GetNbCardsDisplayed()
{
  return mBody->GetModel()->GetNbCardsDisplayed();
}

void PokerPlayer::ShowCard(int i)
{
  mBody->GetModel()->ShowCard(i);
}

void PokerPlayer::HideCard(int i)
{
  mBody->GetModel()->HideCard(i);
}
