/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *o
 * Authors:
 * Cedric Pinson cpinson@freesheep.org
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#include <osg/Geode>

#include <maf/maferror.h>
#include <maf/utils.h>

#include <PokerPot.h>
#include <PokerApplication.h>
#include <PokerMoveChips.h>
#include <PokerChipsStack.h>

PokerPotController::PokerPotController(PokerApplication* game) :
  mSidePotTimer(0.f)
{
  mGame=game;

  std::string tablecenter_string = game->HeaderGet("sequence", "/sequence/tablecenter/@url");
  if(tablecenter_string.empty()) 
    g_error("PokerPotController::PokerPotController: failed to read /sequence/tablecenter/@url");

  std::string tablecenter_anchor = game->HeaderGet("sequence", "/sequence/tablecenter/@anchor");
  if(tablecenter_anchor.empty()) 
    g_error("PokerPotController::PokerPotController: failed to read /sequence/tablecenter/@anchor");

  assert(game->mSetData->GetAnchor(tablecenter_anchor));
  mCenterTransform = game->mSetData->GetAnchor(tablecenter_anchor)->asTransform()->asMatrixTransform();
  if (!mCenterTransform)
    g_error("PokerPotController::PokerPotController: failed to convert node to transform maybe a bad file item in escn");



  /// get all side pot anchor and attach a chipstack on it
  std::string sidepot_anchor = game->HeaderGet("sequence", "/sequence/tablecenter/@sidepot");
  if(sidepot_anchor.empty()) 
    g_error("PokerPotController::PokerPotController: failed to read /sequence/tablecenter/@sidepot");

  std::string nbpot_anchor = game->HeaderGet("sequence", "/sequence/tablecenter/@nbpots");
  if(nbpot_anchor.empty()) 
    g_error("PokerPotController::PokerPotController: failed to read /sequence/tablecenter/@nbpots");
  int nbPots=atoi(nbpot_anchor.c_str());
  char sp[128];
  std::vector<int> defaultTest(13,0);
  for (int i=0;i<nbPots;i++) {
    sprintf(sp,sidepot_anchor.c_str(),i+1);
    osg::MatrixTransform* mt=game->mSetData->GetAnchor(std::string(sp))->asTransform()->asMatrixTransform();
    mPots.push_back(mt);

    PokerChipsStackController* chips=new PokerChipsStackController(game);
    MAFAnchor* anchor = game->mSetData->GetAnchor(sp);
    if(anchor == 0)
      g_error("PokerPotController::PokerPotController: anchor %s not found", sp);
    else
      chips->Anchor(anchor);
    game->AddController(chips);

    mChips.push_back(chips);
  }

  
  // angular speed
  const std::string &as = game->HeaderGet("sequence", "/sequence/tablecenter/@angular_speed");
  if (as.empty())
    g_error("PokerPotController::PokerPotController tablecenter/@angular_speed not found");
  mAngularSpeed = atof(as.c_str()); //rad/s
  

  const std::string &url = game->HeaderGet("sequence", "/sequence/tablecenter/@spline");
  if (url.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@spline not found");
  MAFXmlData *data = game->mDatas->GetXml(url);
  if (!data)
    g_error("PokerPotController::PokerPotController url %s not found",url.c_str());


  const std::string &crack_spline=game->HeaderGet("sequence", "/sequence/tablecenter/@crack_spline");
  if (crack_spline.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@crack_spline not found");
  LoadSpline(mInterpolator, data, crack_spline.c_str());

  const std::string& spline_amplitude=game->HeaderGet("sequence", "/sequence/tablecenter/@spline_amplitude");
  if (spline_amplitude.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@spline_amplitude not found");
  float amp = osg::PI*atof(spline_amplitude.c_str());
  mCrackAmplitude = amp;
  mInterpolator.Init(0,amp); // osg::PI*3/4


  const std::string& mininc_angle=game->HeaderGet("sequence", "/sequence/tablecenter/@mininc_angle");
  if (mininc_angle.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@mininc_angle not found");
  mParamMinIncToMove=osg::PI/180.0*atof(mininc_angle.c_str());
  if (mParamMinIncToMove<=mCrackAmplitude) // should be cool with the delay
    mParamMinIncToMove=mCrackAmplitude; //*1.5

  const std::string& delay=game->HeaderGet("sequence", "/sequence/tablecenter/@delay");
  if (delay.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@delay not found");
  mParamTimeBeforeMoving=atof(delay.c_str());

  const std::string& stop_threshold=game->HeaderGet("sequence", "/sequence/tablecenter/@stop_threshold");
  if (stop_threshold.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@stop_threshold not found");
  mParamStopThreshold=osg::PI/180.0*atof(stop_threshold.c_str());
//   mParamStopThreshold=mParamMinIncToMove;

  const std::string& freeze_angle=game->HeaderGet("sequence", "/sequence/tablecenter/@freeze_angle");
  if (freeze_angle.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@freeze_angle not found");
  mParamMaxAngleToFreezeCenter = osg::PI/180.0*atof(freeze_angle.c_str());

  const std::string& radius=game->HeaderGet("sequence", "/sequence/tablecenter/@radius");
  if (radius.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@radius not found");
  mParamRadiusCenter = atof(radius.c_str());


  const std::string& crackDuration=game->HeaderGet("sequence", "/sequence/tablecenter/@crack_duration");
  if (crackDuration.empty())
    g_error("PokerPotController::PokerPotController /sequence/tablecenter/@crack_duration not found");
  mCrackDuration = atof(crackDuration.c_str());

//   mParamStartCrack=mCrackAmplitude;
  mParamStartCrack=mCrackAmplitude*0.5;

  mCanModifyTarget=false;
  mStartCrack=0;
  mCrackAngle=0;
  mInitialMatrix=mCenterTransform->getMatrix()*osg::Matrix::translate(0,0,0);
  mCurrentAngle=0;
  mInitialDirection=osg::Vec3(mInitialMatrix(0,2),0,mInitialMatrix(2,2));
  mInitialDirection.normalize();
  mCenterIsMoving=false;
  mTimeBeforeMoving=-1;
  mTargetAngle=0;
  mTimeLastMove=0;
  SetCurrentPot(0);
  mFreezeCenter=false;
  mTimeCrack=1e3;
  mCrackAnim=0;
  mCrackSign=1.0;
	mCurrentSidePot=0;
  mTargetSidePot=0;
//   g_debug("nb Count Center %d",mCenterTransform->referenceCount());

}


void PokerPotController::SetCurrentPot(int pot)
{
//   if (mCurrentPotIndex==pot)
//     return;
  mCurrentPotIndex=pot;
  osg::Vec3 center=mInitialMatrix.getTrans();
  osg::Vec3 worldPot=mPots[pot]->getMatrix().getTrans();
  mCurrentPotDirection=worldPot-mInitialMatrix.getTrans();
  mCurrentPotDirection[1]=0;
  mCurrentPotDirection.normalize();
}


PokerPotController::~PokerPotController()
{
//   g_debug("nb Count Center %d",mCenterTransform->referenceCount());
  RecursiveClearUserData(mCenterTransform.get());

  int end=(int)mChips.size();
  for (int i=0;i<end;i++) {
    //g_debug("PokerPotController::mChips[%d] -- %d",i,mChips[i]->referenceCount());
    //RecursiveClearUserData(mPots[i].get());
    mPots[i]->removeChild(mChips[i]->GetModel()->GetArtefact());
    mPots[i]=0;
    mGame->RemoveController(mChips[i].get());
//     g_debug("PokerPotController::mChips[%d] -- %d",i,mChips[i]->referenceCount());
    mChips[i]=0;
  }
//   osg::NodeVisitor* leakNodes = RecursiveLeakCollect(mCenterTransform.get());
//   RecursiveLeakCheck(leakNodes);
  mCenterTransform=0;
}



void PokerPotController::SetPotValue(const std::vector<int>& value,int index)
{
  mChips[index]->SetChips(value);
}

void PokerPotController::LoadKeys(std::vector<osg::Vec2> &keys, MAFXmlData *data, const std::string &name)
{
  if (data != NULL) {
    const std::list<std::string> &xResultList = data->GetList("/splines/" + name + "/list/entry/@xvalue");
    const std::list<std::string> &yResultList = data->GetList("/splines/" + name + "/list/entry/@yvalue");
    
    g_assert(xResultList.size() == yResultList.size());
    typedef std::list<std::string>::const_iterator It;
    It xbegin = xResultList.begin();
    It xend = xResultList.end();
    
    It ybegin = yResultList.begin();
    It yend = yResultList.end();
    
    while(xbegin != xend) {
      if ((*xbegin).empty() || (*ybegin).empty() )
	g_error("PokerPotController::LoadKeys spline %s seems broken",name.c_str());
      keys.push_back(osg::Vec2(atof((*xbegin).c_str()), atof((*ybegin).c_str())));
      xbegin++;
      ybegin++;
    }
  }
}


void PokerPotController::ResetPots()
{
  std::vector<int> nullValue(0);
  int e=(int)mChips.size();
  for (int i=0;i<e;i++)
    mChips[i]->SetChips(nullValue);
}


void PokerPotController::FreezeCenter()
{
  if (mFreezeCenter)
    return;

  mFreezeCenter=true;
  if (fabs(mTargetAngle-mCurrentAngle)>mParamMaxAngleToFreezeCenter)
    if (mTargetAngle-mCurrentAngle>0)
      mTargetAngle=mCurrentAngle+mParamMaxAngleToFreezeCenter;
    else
      mTargetAngle=mCurrentAngle-mParamMaxAngleToFreezeCenter;

//   g_debug("current angle %f target angle %f new target %f",mCurrentAngle,oldTarget,mTargetAngle);
  mTimeLastMove=10; // needed to validate the stop condition
}

void PokerPotController::UnFreezeCenter()
{
  mFreezeCenter=false;
}


bool PokerPotController::IsTargetReach(float threshold)
{
  if (fabs(mTargetAngle-mCurrentAngle)<threshold)
    return true;
  return false;
}


osg::Vec3 PokerPotController::GetSidePotDirection(int i)
{
  osg::Vec3 center=mInitialMatrix.getTrans();
  osg::Vec3 worldPot=mPots[i]->getMatrix().getTrans();
  osg::Vec3 res=worldPot-mInitialMatrix.getTrans();
  res[1]=0;
  res.normalize();
  return res;
}

void PokerPotController::UpdateSidePotDirection(float dt)
{
  int e=(int)mChips.size();
  int index=0;
  for (int i=0;i<e;i++)
    if(!mChips[i]->GetChips().empty())
      index=i;

  mTargetSidePot=index;
  if (mTargetSidePot!=mCurrentSidePot && mSidePotTimer<=0) {

    osg::Vec3 curDir=GetSidePotDirection(mCurrentSidePot);
    osg::Vec3 targetDir=GetSidePotDirection(mTargetSidePot);

    float a=fabs(acos(targetDir*curDir));
    mSidePotTimer=a/mAngularSpeed;
  }

  if (mSidePotTimer>0) {

    osg::Vec3 curDir=GetSidePotDirection(mCurrentSidePot);
    osg::Vec3 targetDir=GetSidePotDirection(mTargetSidePot);
    float sign=(osg::Vec3(0,1,0)^curDir)*targetDir;
    if (sign>0)
      sign=-1;
    else
      sign=1;
    
    osg::Matrix mat=osg::Matrix::identity();
    mat.makeRotate(mAngularSpeed*dt,osg::Vec3(0,sign,0));
    mCurrentPotDirection=osg::Matrix::transform3x3(mat,mCurrentPotDirection);
    mSidePotTimer-=dt;
    if (mSidePotTimer<0) {
      mCurrentSidePot=mTargetSidePot;
      mCurrentPotDirection=GetSidePotDirection(mCurrentSidePot);
    }
  }
}



static void ComputeSignAndAngle(const osg::Vec3& base,const osg::Vec3& vec2,float& sign, float& angle)
{
  sign=-((osg::Vec3(0,1,0)^base)*vec2>0?1:-1); // get the current signe to go to target
  float dirdotcdir=vec2*base;
  if (dirdotcdir>=1.0)
    dirdotcdir=0.9999999;
  if (dirdotcdir<=-1.0)
    dirdotcdir=-0.9999999;
  angle=sign*acos(dirdotcdir);
}

bool PokerPotController::Update(MAFApplication* application)
{
  PokerApplication* game = dynamic_cast<PokerApplication*>(application);
  if(game->HasEvent())
    return true;

  osg::Matrix tmat;
  PokerCameraModel* camera=(dynamic_cast<PokerCameraController*>(mGame->GetScene()->GetModel()->mCamera.get()))->GetModel();

  float dt=GetDeltaFrame()*1.0/1000;
//   float dt=GetDelta()*1.0/1000;
  // align current pot to the last sidepot non null if needed
  UpdateSidePotDirection(dt);

  osg::Vec3 camDirection=camera->GetPosition()-mInitialMatrix.getTrans(); camDirection[1]=0; // lock in y orientation
  camDirection.normalize();


  if (mFreezeCenter) { // we want to freeze the center so we override direction
    tmat.makeRotate(mTargetAngle,osg::Vec3(0,1,0));
    camDirection=tmat*mInitialDirection;
  }

  // make the current direction from angle
  tmat.makeRotate(mCurrentAngle,osg::Vec3(0,1,0));
  osg::Vec3 currentDirection=tmat*mInitialDirection;
  currentDirection.normalize();

  float sign,diffAngle;
  ComputeSignAndAngle(currentDirection,camDirection,sign,diffAngle);

  float targetSign,targetDiffAngle;
  ComputeSignAndAngle(currentDirection,osg::Matrix::rotate(mTargetAngle,osg::Vec3(0,1,0))*mInitialDirection,targetSign,targetDiffAngle);
  sign=targetSign;

  float angleBetweenTargetAndCamera=diffAngle-(mTargetAngle-mCurrentAngle);
  
  
  // if can modify target and the target angle is significative, ok we change it
  //  if (mCanModifyTarget && fabs(angleBetweenTargetAndCamera) > mParamMinIncToMove) { // pb here
  if (mCanModifyTarget && fabs(angleBetweenTargetAndCamera) > mParamMinIncToMove) { // pb here
    if (mStartCrack==2) // means that it take effect only if we reach the end of a sequence
      mTimeBeforeMoving=mParamTimeBeforeMoving;

    mTargetAngle=mCurrentAngle+diffAngle;
    mCanModifyTarget=false;
    mStartCrack=0;
  }


  // manage notion when we are in first view
  // manage the delay before starting animation
  if (!mStartCrack && !mCenterIsMoving && (mTimeBeforeMoving<0 || camera->GetMode()!=PokerCameraModel::CAMERA_FREE_MODE && !mFreezeCenter) ) {
    mCenterIsMoving=true;
  }
  mTimeBeforeMoving-=dt;


  if (mCenterIsMoving) {

    if (fabs(mTargetAngle-mCurrentAngle) <= mParamStartCrack && !mStartCrack) {
      mStartCrack=1; // active crack sequence
      mTimeCrack=0;
    }

    if (mStartCrack==1) {
      // crack sequence
      float val=mTimeCrack/mCrackDuration;
      if (val>1)
	val=1;
      else if (val<0)
	val=0;
      mInterpolator.Get(mCrackAngle,val);
      mCrackAngle*=-sign;
      if (mTimeCrack<mCrackDuration) {
	mTimeCrack+=dt;
      } else {
	// sequence is finished
	mCurrentAngle-=mCrackAngle;
	mTargetAngle=mCurrentAngle;
	mCenterIsMoving=false; // reset flag pot should be stopped
	mCrackAngle=0;
	mStartCrack=2;
      }
    } else {
      // classic path before crack sequence
      float res=sign*mAngularSpeed*dt;
//       printf("resangle %f\n",res);
      mCurrentAngle+=res;
    }
  }

//   printf("dt %f _ mTimeBeforeMoving %f _ mTimeCrack %f _ mStartCrack %d _ mCurrentAngle %f\n",
// 	 dt,
// 	 mTimeBeforeMoving,
// 	 mTimeCrack,
// 	 mStartCrack,
// 	 mCurrentAngle);

  // manage flag mCanModifyTarget
  // end of crack sequence or motion before a crack sequence is valid to modify target
  if (mStartCrack==2 || mStartCrack==0) { 
    mCanModifyTarget=true;
  } else
    mCanModifyTarget=false;


  float angleToUse=-mCurrentAngle+mCrackAngle;

  // update is ok put result in matrix
  osg::Matrix mat;
  mat.makeRotate(angleToUse,osg::Vec3(0,1,0));
  mat.setTrans(mInitialMatrix.getTrans());

  // add an offset matrix to be aligned with the current pot
  osg::Vec3 side=osg::Vec3(0,1,0)^mCurrentPotDirection;
  osg::Matrix offset;
  offset.makeIdentity();
  offset(0,0)=side[0];
  offset(1,0)=side[1];
  offset(2,0)=side[2];
  offset(0,1)=0;
  offset(1,1)=1;
  offset(2,1)=0;
  offset(0,2)=mCurrentPotDirection[0];
  offset(1,2)=mCurrentPotDirection[1];
  offset(2,2)=mCurrentPotDirection[2];
  mCenterTransform->setMatrix(offset*mat);
  return true;
}


float PokerPotController::BuildAnimationBetToPot(PokerMoveChipsBet2PotController* animation,int pot)
{
  assert(animation && "passing null node" );

  osg::Vec3 srcInWorldSpace=MAFComputeLocalToWorld(animation->mNode.get()).getTrans();
  osg::Matrix worldCenterTable=MAFComputeLocalToWorld(mCenterTransform.get());
  osg::Matrix worldPot=MAFComputeLocalToWorld(mPots[pot].get());

  osg::Vec3 dirCenter2Pot=worldPot.getTrans()-worldCenterTable.getTrans();
  osg::Vec3 dirSrc2Center2=srcInWorldSpace-worldCenterTable.getTrans();

  float distFromCenter=dirSrc2Center2.length();
  dirSrc2Center2.normalize();
  dirCenter2Pot.normalize();

  // compute angle relative to src
  float cosAngle=dirCenter2Pot*dirSrc2Center2;
  float angle=acos(cosAngle);

  if ((osg::Vec3(0,1,0)^dirSrc2Center2)*dirCenter2Pot < 0 )
    angle=-angle;

  // compute baseMatrix
  osg::Matrix offset;
  offset.makeIdentity();
  osg::Vec3 side=osg::Vec3(0,1,0)^(dirSrc2Center2);
  offset(0,0)=side[0];
  offset(0,1)=side[1];
  offset(0,2)=side[2];
  offset(1,0)=0;
  offset(1,1)=1;
  offset(1,2)=0;
  offset(2,0)=dirSrc2Center2[0];
  offset(2,1)=dirSrc2Center2[1];
  offset(2,2)=dirSrc2Center2[2];
  offset.setTrans(worldCenterTable.getTrans());
  
  animation->mAngleToInterpolate=angle;
  animation->mBaseMatrix=offset;
  animation->mDistanceFromCenter=distFromCenter;
  animation->mDistanceToInterpolate=distFromCenter-(worldPot.getTrans()-worldCenterTable.getTrans()).length();
  animation->InitAnimation();
  return 0;
}



float PokerPotController::BuildAnimationPotToPlayer(PokerMoveChipsPot2PlayerController* animation,int pot)
{
  assert(animation && "passing null node" );

  osg::Vec3 srcInWorldSpace=MAFComputeLocalToWorld(mPots[pot].get()).getTrans();
  osg::Matrix worldCenterTable=MAFComputeLocalToWorld(mCenterTransform.get());
  osg::Matrix worldPot=MAFComputeLocalToWorld(animation->mNode.get());

  osg::Vec3 dirCenter2Pot=worldPot.getTrans()-worldCenterTable.getTrans();
  osg::Vec3 dirSrc2Center2=srcInWorldSpace-worldCenterTable.getTrans();

  float distFromCenter=dirCenter2Pot.length();
  dirSrc2Center2.normalize();
  dirCenter2Pot.normalize();

  // compute angle relative to src
  float cosAngle=dirCenter2Pot*dirSrc2Center2;
  float angle=acos(cosAngle);

  if ((osg::Vec3(0,1,0)^dirSrc2Center2)*dirCenter2Pot < 0 )
    angle=-angle;

  // compute baseMatrix
  osg::Matrix offset;
  offset.makeIdentity();
  osg::Vec3 side=osg::Vec3(0,1,0)^(dirSrc2Center2);
  offset(0,0)=side[0];
  offset(0,1)=side[1];
  offset(0,2)=side[2];
  offset(1,0)=0;
  offset(1,1)=1;
  offset(1,2)=0;
  offset(2,0)=dirSrc2Center2[0];
  offset(2,1)=dirSrc2Center2[1];
  offset(2,2)=dirSrc2Center2[2];
  offset.setTrans(worldCenterTable.getTrans());
  
  animation->mAngleToInterpolate=angle;
  animation->mBaseMatrix=offset;
  animation->mDistanceFromCenter=distFromCenter;
  animation->mDistanceToInterpolate=distFromCenter-(srcInWorldSpace-worldCenterTable.getTrans()).length();
  animation->InitAnimation();
  return 0;
}
