/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 * Henry Precheur	<henry@precheur.org>
 * Igor Kravtchenko <igor@obraz.net>
 *
 */

#include "pokerStdAfx.h"

#include <maf/maferror.h>
#include <maf/window.h>
#include <maf/hit.h>
#include <ugame/text.h> // FIXME < > or " " ?

#ifndef WIN32
#include "PokerApplication.h"
#include "PokerToolTip.h"
#endif

static const float	pre_fade_in_time = 1500.0f;
static const float	fade_in_time = 400.0f;
static const float	fade_out_time = 200.0f;

// #define _DEBUG_TOOP_TIP_

#ifdef	_DEBUG_TOOP_TIP_
#	define DEBUG_CHANGE_STATE(s)	g_debug("state => " #s)
#else	// !_DEBUG_TOOP_TIP_
#	define DEBUG_CHANGE_STATE(s)
#endif	// _DEBUG_TOOP_TIP_

PokerToolTipController::PokerToolTipController(PokerApplication* app)
  : mScreenWidth(app->GetWindow(true)->GetWidth()),
    mScreenHeight(app->GetWindow(true)->GetHeight()),
    mState(NONE), mTime(0)
{
  // UGAMEArtefact Init
  MAFVisionModel*	model = new MAFVisionModel();
  SetModel(model);
  Init();

  // Create Tooltip 3D model
  std::string	data_path = app->HeaderGet("settings", "/settings/data/@path");
  mPat = new osg::PositionAttitudeTransform;
  mText =
    new UGAMEFramedText("WWW",
		      MAFLoadImage(data_path + "/tooltip.tga", app->GetOptions()),
		      MAFLoadFont(data_path + "/FreeSans.ttf", app->GetOptions()));
  mText->getText()->setColor(osg::Vec4(0, 0, 0, 0));
  mText->getText()->setCharacterSize(10);
  mPat->setNodeMask(0);
  mPat->addChild(mText);
  model->SetNode(mPat.get());

  // fade in / out color
  osg::Vec4	tmp_colors[1] = { osg::Vec4(1, 1, 1, 0) };
  mColor = new osg::Vec4Array(tmp_colors, tmp_colors + 1);
  osg::Geometry*	geometry = mText->getGeometry();
  geometry->setColorArray(mColor);
  geometry->setColorBinding(osg::Geometry::BIND_OVERALL);

  // create path => tooltip map
  std::list<std::string> tooltip_path_list =
    app->HeaderGetList("sequence", "/sequence/tooltips/tooltip/@path");
  std::list<std::string> tooltip_list =
    app->HeaderGetList("sequence", "/sequence/tooltips/tooltip/@text");

  std::list<std::string>::const_iterator i = tooltip_path_list.begin();
  std::list<std::string>::const_iterator j = tooltip_list.begin();
  while (i != tooltip_path_list.end())
    mPath2Tooltip[*i++] = *j++;
}

PokerToolTipController::~PokerToolTipController()
{
  mPat = 0;
}

bool	PokerToolTipController::Update(MAFApplication* app)
{
  if (app->HasEvent())
    return true;

  switch (mState)
    {
    case NONE:
      break;
    case PREFADEIN:
      mTime += GetDeltaFrame();
      if (mTime > pre_fade_in_time)
	{
	  mTime = 0;
	  mState = FADEIN;
	  DEBUG_CHANGE_STATE(FADEIN);
	  mPat->setNodeMask(MAF_VISIBLE_MASK);
	}
      break;
    case FADEIN:
      {
	mTime += GetDeltaFrame();
	float	scale = mTime / fade_in_time;
	Fade(scale);
	// mPat->setScale(osg::Vec3(scale, scale, scale));
	if (mTime > fade_in_time)
	  {
	    mTime = 0;
	    mState = DISPLAYED;
	    DEBUG_CHANGE_STATE(DISPLAYED);
	  }
      }
      break;
    case DISPLAYED:
      break;
    case FADEOUT:
      mTime += GetDeltaFrame();
      float	scale = (fade_out_time - mTime) / fade_out_time;
      Fade(scale);
      // mPat->setScale(osg::Vec3(scale, scale, scale));
      if (mTime > fade_out_time)
	{
	  mTime = 0;
	  mState = NONE;
	  DEBUG_CHANGE_STATE(NONE);
	  mPat->setNodeMask(0);
	}
      break;
    }

  return true;
}

void	PokerToolTipController::HandleHit(MAFHit& hit)
{
  g_assert(mPath2Tooltip.find(hit.mPath) != mPath2Tooltip.end());

  if (mText->getStringUTF8() !=
      mPath2Tooltip.find(hit.mPath)->second)
    {
      Reset();
    }

  switch (mState)
    {
    case NONE:
      mState = PREFADEIN;
      DEBUG_CHANGE_STATE(PREFADEIN);
      mText->setStringUTF8(mPath2Tooltip.find(hit.mPath)->second);
      break;
    case PREFADEIN:
      {
	// show & place tooltip
	const osg::BoundingBox&	bbox = mText->getBoundingBox();
	float	x_len = bbox.xMax() - bbox.xMin();
	mPat->setPosition(osg::Vec3(hit.mMouseX - x_len, mScreenHeight - hit.mMouseY, 0));
      }
      break;
    case FADEOUT:
      Fade(0);
      mTime = 0;
      mState = FADEIN;
      DEBUG_CHANGE_STATE(FADEIN);
      break;
    case FADEIN:
    case DISPLAYED:
      break;
    };
}

void	PokerToolTipController::HandleNotHit()
{
  switch (mState)
    {
    case NONE:
      g_assert(false && "state = none");
      break;
    case PREFADEIN:
      mState = NONE;
      DEBUG_CHANGE_STATE(NONE);
      break;
    case FADEIN:
      mState = FADEOUT;
      DEBUG_CHANGE_STATE(FADEOUT);
      mTime = fade_in_time - mTime;
      break;
    case DISPLAYED:
      mState = FADEOUT;
      DEBUG_CHANGE_STATE(FADEOUT);
      mTime = 0;
      break;
    case FADEOUT:
      g_assert(false && "state = fade out");
      break;
    }
}

osg::Node*	PokerToolTipController::GetNode()
{
  return mPat.get();
}

void	PokerToolTipController::Fade(float f)
{
  (*mColor)[0] = osg::Vec4(1, 1, 1, f);
  osgText::Text*	text = mText->getText();
  osg::Vec4	text_color = text->getColor();
  text_color.w() = f;
  text->setColor(text_color);
}

void	PokerToolTipController::Reset()
{
  mTime = 0;
  mState = NONE;
  DEBUG_CHANGE_STATE(NONE);
  Fade(0.0f);
}
