#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Cedric Pinson <cpinson@freesheep.org>
#  Loic Dachary <loic@gnu.org>
#
"""\
Animation manager for poker3d. It handles all animations, except the ones
for which python would be too slow (dynamically generated facial noise
animations for instance).

Can be reloaded dynamically: while running the game, F11 will reload this
file and rebind the relevant python objects to the new code. This mandatory
rebinding is not intuitive but must be done. Part of the code of this file
deal with this issue (stealFrom methods for instance). Most of the time
it should not be an issue. However it MUST be taken in account when adding
new classes or when dealing with any kind of object that might not be
explicitly rebound by the stealFrom functions.

The function registered with setStopCallback won't be re-assigned to the
code reloaded with F11. When their time comes they will execute the code
that existed before F11 was hit. This loophole is considered acceptable
because such callbacks are not currently used to launch animations that
will last forever (or at least for a long time). It should be fixed but
there is not urgency in doing so (2004/10/31).

"""
from twisted.internet import reactor
from twisted.internet import defer

from underware import animated
from poker.pokerpackets import *
from random import choice, uniform
from pokerchat import PokerChat

class PokerPlayer:
    """A poker player seated at a poker table"""
    
    def __init__(self, *args, **kwargs):
        self.myself = kwargs.get('myself', False)
        self.parent = kwargs.get('parent', None)
        self.application = kwargs.get('application', None)
        self.config = kwargs.get('config', None)
        self.serial = kwargs.get('serial', 0)
        self.sitin = False
        self.animation = kwargs.get('animation', None)
        self.timers = {}
        self.animationRunning = False
        self.waitIdleAnimation = ["idle01","idle02","idle03","idle04","idle05","idle06","idle07","idle08"]
        self.lookingCardsAnimations = ["lookA","lookB","lookC"]
        self.interactorsAnimations = [ "bet","fold","check"]
        self.interactorExcludeList = self.waitIdleAnimation[:]+self.lookingCardsAnimations[:]

    def destroy(self):
        "Avoid circular references"
        self.stopAll()
        del self.animation
        del self.parent

    def stealFrom(self, other):
        "Helper to implement the dynamic code reload facility"
        self.myself = other.myself
        self.parent = other.parent
        self.application = other.application
        self.config = other.config
        self.serial = other.serial
        self.sitin = other.sitin
        self.animation = other.animation
        for (function, timer) in other.timers.iteritems():
            self.timers[function] = timer
            timer.func = eval('self.' + function)
        return self

    def suspend(self):
        """Another part of the program wants to play with animations
        and asks us to suspend all running animations. The "resume"
        function may be called at a later time if we are allowed to
        keep handling animations."""
        self.suspend=True
        self.stopAll()

    def resume(self):
        """Run animations according to the state of the game. It is
        designed to be called after a suspend but will also work
        otherwise."""
        self.suspend=False
        self.stopAll()
        if self.sitin:
            self.waitSit(0)
        else:
            self.waitStand(0)
        
    def callLater(self, delay, function, *args, **kwargs):
        """Remember each scheduled call to this instance member functions.
        It is necessary to properly implement the reload facility. The
        stealFrom method uses this map to re-assign the scheduled calls
        to a new object based on the newly loaded code. If this is not
        done, the scheduled call will execute the code from the class
        as it was before it was reloaded. While this is an understandable
        behaviour when you think about it, the casual user is only
        disturbed and do not want to hear about it.         
        """
        timer = reactor.callLater(delay, function, *args, **kwargs)
        self.timers[function.__name__] = timer

    def stopAll(self):
        """Stop the currently running animations and the scheduled animations.
        """
        self.animation.stop(animated.ALL)
        for (function, timer) in self.timers.iteritems():
            if timer and timer.active():
                timer.cancel()
        self.timers = {}
        self.stopNoises()



    def stopAnimationList(self,animations,fade):
        """Stop all animation in the specified list
        """
        for anim in animations:
            if self.animation.isAnimationActive(anim):
                self.animation.stop(anim,fade_out=fade)

    def isPlayingAnimationList(self,animations):
        for anim in animations:
            if self.animation.isAnimationActive(anim):
                return True
        return False
        
    def stopAllWaitIdle(self,fade):
        """Stop all idle animations.
        """
        self.stopAnimationList(self.waitIdleAnimation,fade)

    def isInPosition(self):
        return self.serial == self.parent.game.getSerialInPosition()
    
    def scheduleWaitIdle(self, delay):
        min_random = float(self.config.headerGet("/sequence/animation/player/@min_random"))
        max_random = float(self.config.headerGet("/sequence/animation/player/@max_random"))
        when = uniform(min_random + delay, max_random + delay)
        self.callLater(when, self.waitIdle)


    def stopNoises(self):
        self.parent.application.PythonAccept(PacketPokerAnimationPlayerNoise(serial=self.serial,action="stop"))

    def playNoises(self):
        self.parent.application.PythonAccept(PacketPokerAnimationPlayerNoise(serial=self.serial,action="start"))

    def waitIdle(self):
        if not self.sitin:
            return

        # do not play idle animation if an animtion in the list conflict with but
        # but we want to be called later
        excludeList=self.waitIdleAnimation[:] + self.interactorsAnimations[:] + self.lookingCardsAnimations[:] + ["win"] + ["lookat"]
        if self.isPlayingAnimationList(excludeList) == False:
            self.stopNoises()
            animationSelected=choice(self.waitIdleAnimation)
            timeOfIdle=self.animation.getDuration(animationSelected)
            self.animation.run(animationSelected,
                               fade_in = 0.3,
                               fade_out = 0.3)
            self.callLater(timeOfIdle, self.playNoises)
        self.scheduleWaitIdle(0)

    def playIdle(self):
        """ Now there is no playIdle so play a wait idle instead
        """
        self.waitIdle()
        return
        
        """ Call when a player does not play and is in position.
        """
        if self.suspend == False:
            self.animation.run("idle04",
                               fade_in = 0.3,
                               fade_out = 0.3)
            
    # todo, insert condition to not conflict with waitIdle, interaction animation, and win
    def isValidToLookCard(self):
        excludeList=self.lookingCardsAnimations[:] + self.interactorsAnimations[:] + ["win"]
        return self.isPlayingAnimationList(excludeList)
        
    def startLookCards(self):
        if self.myself: return

        # check if it's not a problem to play LookCard
        if self.isValidToLookCard() is True:
            return

        # stop all and play fix, it avoids to stop each animation
        self.stopAll()
        self.waitSit(0)
        ID = self.animation.run("lookA",
                                channel = animated.FOREGROUND)
        self.animation.setStopCallback(ID,
                                       self.idleLookCards)

    def idleLookCards(self):
        # check if it's not a problem to play LookCard
        if self.isValidToLookCard() is True:
            return
        
        ID = self.animation.run("lookB",
                                channel = animated.FOREGROUND,
                                length = animated.FOREVER)
        self.callLater(3, self.stopLookCards, ID)

    def stopLookCards(self, idle_id):
        self.animation.stop(idle_id,
                            fade_out = 0.2)
        # check if it's not a problem to play LookCard
        if self.isValidToLookCard() is True:
            return
        ID = self.animation.run("lookC",
                                channel = animated.FOREGROUND,
                                fade_in = 0.2)

    def sitin(self):
        if self.sitin is True:
            return
        self.sitin = True
        #
        # Here we should take in account the case when the
        # player is in the middle of the sitout animation as
        # a result of the PokerPlayer::UpdateSittingOut interaction.
        #
        self.stopAll()
        if self.parent.stream:
            #
            # The player is siting down because he was previously standing
            #
            self.animation.run("seatDown")
            delay = self.animation.getDuration("seatDown") - 0.3
        else:
            #
            # The player was already seated when we arrived at the table
            #
            delay = 0
        self.waitSit(0)

    def waitSit(self, delay):
        self.animation.run("fix",
                           channel = animated.BACKGROUND,
                           length = animated.FOREVER,
                           delay = delay)
        self.animation.run("breath",
                           channel = animated.BACKGROUND,
                           length = animated.FOREVER,
                           fade_in = 0.3,
                           fade_out = 0.3)
        self.scheduleWaitIdle(delay)
        self.playNoises()

        
    def check(self):
        self.stopAnimationList(self.interactorExcludeList,0.2)
        self.animation.run("check",
                           fade_in = 0.2,
                           fade_out = 0.3)

    def bet(self):
        self.stopAnimationList(self.interactorExcludeList,0.2)
        self.animation.run("bet",
                           fade_in = 0.2,
                           fade_out = 0.3,
                           weight = 10000)


    def fold(self, game_id):
        self.stopAnimationList(self.interactorExcludeList+["lookat"],0.2)
        packet = PacketPokerAnimationPlayerFold(game_id = game_id,
                                                serial = self.serial,
                                                animation="fold")
        self.application.PythonAccept(packet)
        return
        self.stopAnimationList(self.interactorExcludeList+["lookat"],0.2)
        ID = self.animation.run("fold",
                                fade_in = 0.3)
        self.animation.setStopCallback(ID,
                                       self.foldEndCallback,
                                       game_id)

    def foldEndCallback(self, game_id):
        self.animation.run("",
                           channel = animated.FOREGROUND,
                           length = animated.ONCE,
                           fade_in = 0.1,
                           fade_out = 0.8)

        #
        # WATCH OUT : this function may be run after the application
        # is stopped or after the table for which the serial was valid
        # is gone. Make sure the functions called are properly dealing
        # with these border cases.
        #
        packet = PacketPokerDisplayCard(game_id = game_id,
                                        serial = self.serial,
                                        index = [0,1,2,3],
                                        display = 0)
        self.application.PythonAccept(packet)

    def win(self):
        #check before to stop animation that could be conflict with win animation
        exclude=self.interactorsAnimations[:] + self.lookingCardsAnimations[:] +self.waitIdleAnimation[:]
        self.stopAnimationList(self.interactorExcludeList,0.1)
        self.animation.run("win",
                           channel = animated.FOREGROUND,
                           length = animated.ONCE,
                           fade_in = 0.1,
                           fade_out = 0.8)
        

    def sitout(self):
        if self.sitin is False:
            return
        
        self.sitin = False
        self.stopAll()

        if self.parent.stream:
            #
            # The player is siting up because he was previously seated
            #
            self.animation.run("seatUp")
            delay = self.animation.getDuration("seatUp")
        else:
            #
            # The player was already standing up when we arrived at the table
            #
            delay = 0
        self.waitStand(0)

    def waitStand(self, delay):
        self.animation.run("upIdle01",
                           channel = animated.BACKGROUND,
                           length = animated.FOREVER,
                           delay = delay)
        self.stopNoises()

    def emote(self, packet):
        if self.sitin == False:
            return
        chatCommands = packet.message.split()
        for chatCommand in chatCommands:
            packetType = PokerChat.getChatType(chatCommand)
            if packetType != None:
                animationsNotPlaying = self.isPlayingAnimationList(PokerChat.emoteAnimation[:] + self.interactorsAnimations[:] + self.lookingCardsAnimations[:]) == False
                if animationsNotPlaying:
                    self.stopAll()
                    self.waitSit(0)
                    self.parent.received2function[packetType](self.parent.protocol, packet)
                else:
                    print "skipped"
                return
                
class PokerAnimationScheduler:
    """Packet receiver (see pokerpackets.py for the list of packets
    and their documentation). It is registered by
    poker3d.py:PokerClientFactory on a
    pokerclient.py:PokerClientProtocol instance talking to the server.
    """
    def __init__(self, *args, **kwargs):
        self.game = None
        self.stream = True # see PACKET_POKER_STREAM_MODE documentation
        def setStream(state): self.stream = state
        self.received2function = {
            PACKET_POKER_STREAM_MODE: lambda protocol, packet: setStream(True),

            PACKET_POKER_BATCH_MODE: lambda protocol, packet: setStream(False),

            PACKET_POKER_TABLE: self.table,

            PACKET_POKER_TABLE_QUIT: self.tableQuit,

            PACKET_POKER_TABLE_DESTROY: self.tableQuit,

            PACKET_POKER_PLAYER_ARRIVE: self.playerArrive,

            PACKET_POKER_PLAYER_LEAVE: self.playerLeave,

            PACKET_POKER_LOOK_CARDS: lambda protocol, packet:
            self.toPlayer(PokerPlayer.startLookCards, packet.serial),

            PACKET_POKER_SIT: lambda protocol, packet:
            self.toPlayer(PokerPlayer.sitin, packet.serial),

            PACKET_POKER_SIT_OUT: lambda protocol, packet:
            self.toPlayer(PokerPlayer.sitout, packet.serial),

            PACKET_POKER_FOLD: lambda protocol, packet:
            self.toPlayer(PokerPlayer.fold, packet.serial, packet.game_id),

            PACKET_POKER_CHECK: lambda protocol, packet:
            self.toPlayer(PokerPlayer.check, packet.serial),

            PACKET_POKER_CALL: lambda protocol, packet:
            self.toPlayer(PokerPlayer.bet, packet.serial),

            PACKET_POKER_RAISE: lambda protocol, packet:
            self.toPlayer(PokerPlayer.bet, packet.serial),

            PACKET_POKER_TIMEOUT_WARNING: lambda protocol, packet:
            self.toPlayer(PokerPlayer.playIdle, packet.serial),

            PACKET_POKER_PLAYER_WIN: lambda protocol, packet:
            self.toPlayer(PokerPlayer.win, packet.serial),

            PACKET_POKER_CHAT: lambda protocol, packet:
            self.toPlayer(PokerPlayer.emote, packet.serial, packet),
            }

        self.application = kwargs.get("application", None)
        self.application.SetPyScheduler(self)
	self.config = kwargs.get("config", None)
	self.settings = kwargs.get("settings", None)
	self.verbose = self.settings and int(self.settings.headerGet("/settings/@verbose")) or 0
        self.protocol = None
        self.game = None
        self.serial2player = {}
        self.steal_count = 0

    def stealFrom(self, other):
        self.steal_count = other.steal_count + 1
        self.stream = other.stream
        self.application = other.application
        self.application.SetPyScheduler(self)
        self.config = other.config
        self.settings = other.settings
        self.verbose = other.verbose
        self.game = other.game
        other.unsetProtocol()
        self.setProtocol(other.protocol)
        for (serial, player) in other.serial2player.iteritems():
            self.serial2player[serial] = PokerPlayer().stealFrom(player)

    def stealCount(self):
        return self.steal_count

    def playerArrive(self, protocol, packet):
        if packet.game_id != self.game.id:
            print "PokerAnimationScheduler::playerArrive unexpected packet.game_id (%d) != self.game.id (%d) " % ( packet.game_id, self.game.id )
        player = PokerPlayer(parent = self,
                             myself = packet.serial == protocol.getSerial(),
                             game_id = packet.game_id,
                             serial = packet.serial,
                             animation = animated.Animated(self.application, packet.serial),
                             config = self.config,
                             application = self.application)

        self.serial2player[packet.serial] = player

    def playerLeave(self, protocol, packet):
        self.serial2player[packet.serial].destroy()
        del self.serial2player[packet.serial]
        
    def playersRemove(self):
	for serial in self.serial2player.keys():
            self.serial2player[serial].destroy()
            del self.serial2player[serial]
	self.serial2player = {}

    def table(self, protocol, packet):
        if self.verbose > 1:
            print "PokerAnimationScheduler: observing table %d" % packet.id
        self.game = self.protocol.getGame(packet.id)

    def tableQuit(self, protocol, packet):
        if self.verbose > 1:
            print "PokerAnimationScheduler: quit/destroy table %d" % packet.game_id
        self.game = None
        self.playersRemove()

    def suspend(self, packet):
        self.toPlayer(PokerPlayer.suspend, packet.serial)
        
    def resume(self, packet):
        self.toPlayer(PokerPlayer.resume, packet.serial)

    def toPlayer(self, function, serial, *args):
        """Call the "function" of the player object with serial equal
        to "serial" using the "*args" arguments. """
        player = self.serial2player.get(serial, None)
        if player:
            return function(player, *args)
        else:
            return None
        
    def setProtocol(self, protocol):
        self.protocol = protocol
        for (type, function) in self.received2function.iteritems():
            protocol.registerHandler('current', type, function)
        #
        # In order to ease debugging we want the pokeranimation
        # callback to be called first so that the C++ side can
        # assume that there are no pending pointers to the animated
        # object.
        #
        tmp = protocol.callbacks['current'][PACKET_POKER_PLAYER_LEAVE]
        tmp.remove(self.playerLeave)
        tmp.insert(0, self.playerLeave)
        tmp = protocol.callbacks['current'][PACKET_POKER_TABLE]
        tmp.remove(self.table)
        tmp.insert(0, self.table)
        tmp = protocol.callbacks['current'][PACKET_POKER_TABLE_DESTROY]
        tmp.remove(self.tableQuit)
        tmp.insert(0, self.tableQuit)

    def unsetProtocol(self):
        for (type, function) in self.received2function.iteritems():
            self.protocol.unregisterHandler('current', type, function)
            
def create(application, config, settings):
    return PokerAnimationScheduler(application = application,
                                   config = config,
                                   settings = settings)

