#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#  Henry Precheur <henry@precheur.org>
#
import sys
sys.path.insert(0, "..")
sys.path.insert(0, "../../python")

from string import split, lower
from random import randint
from time import sleep

import pycurl
import libxml2
from twisted.python import components
from twisted.persisted import sob
from twisted.application import internet, service, app
from twisted.internet import pollreactor
import sys
if not sys.modules.has_key('twisted.internet.reactor'):
    print "installing poll reactor"
    pollreactor.install()
else:
    print "poll reactor already installed"
from twisted.internet import reactor, error

from twibber.elements import ChatPresence
from underware.client import UGAMEClientProtocol, UGAMEClientFactory
from underware.config import Config
from poker.pokergame import PokerGame, PokerGameClient, PokerPlayer
from poker.pokercards import PokerCards
from poker.pokerchips import PokerChips
from poker.pokerpackets import *
from poker import pokerjabber

LEVEL2ITERATIONS = {
    0: 10,
    1: 1000,
    2: 10000,
    3: 50000,
    4: 100000,
    5: 200000
    }

class PokerBotClient(UGAMEClientProtocol):
    """Poker client"""

    def __init__(self):
        UGAMEClientProtocol.__init__(self)
        self.user.name = "BOT%09d" % randint(0, 200000000)
        self._prefix = self.user.name + " "
        self.user.password = "fakefake"

        self.jabber = None
        self.actionDone = False
        self.level = 1
        self.ev = None
        self.broke = False

    def _handleConnection(self, packet):
        if packet.type == PACKET_SERIAL:
            self.user.serial = packet.serial
            outfit = "Male #%d" % randint(2,9)
            self.sendPacket(PacketPokerPlayerInfo(serial = packet.serial,
                                                  name = self.getName(),
                                                  outfit = outfit))
            return
        
        game = self.factory.game
        
        if packet.type == PACKET_POKER_START:
            if game.isRunning():
                raise UserWarning, "you should not be here (state: %s)" % game.state
            elif packet.hand_serial == 0:
                print "*ERROR* game start was refused"
            else:
                self.old_state = game.state
                game.setTime(packet.time)
                game.setHandsCount(packet.hands_count)
                game.setLevel(packet.level)
                game.beginTurn(packet.hand_serial)

                if self.factory.watch == False:
                    #
                    # The robot has not interest to watch a game, it wants to play.
                    # That may happen if the server refuses to sit the bot (lack of
                    # seats, most likely).
                    #
                    if self.getSerial() not in game.player_list:
                        self.transport.loseConnection()

        elif packet.type == PACKET_POKER_CANCELED:
            game.canceled(packet.serial, packet.amount)
            
        elif packet.type == PACKET_POKER_TABLE_DESTROY:
            self.transport.loseConnection()
            
        elif packet.type == PACKET_POKER_TABLE_LIST:
            found = False
            table_info = self.factory.table
            for table in packet.packets:
                if table_info.has_key("name"):
                    found = table.name == table_info["name"]
                else:
                    found = ( table.betting_structure == table_info["betting_structure"] and
                              table.variant == table_info["variant"] )
                if found:
                    self.sendPacket(PacketPokerTableJoin(game_id = table.id,
                                                         serial = self.getSerial()))
                    self.sendPacket(PacketPokerSeat(game_id = table.id,
                                                    serial = self.getSerial()))
                    self.sendPacket(PacketPokerBuyIn(game_id = table.id,
                                                     serial = self.getSerial()))
                    self.sendPacket(PacketPokerSit(game_id = table.id,
                                                   serial = self.getSerial()))
                    self.sendPacket(PacketPokerAutoBlindAnte(game_id = table.id,
                                                             serial = self.getSerial()))
                    break

            if not found:
                print "Unable to find table %s " % table_info["name"]
                self.transport.loseConnection()

        elif packet.type == PACKET_POKER_TABLE:
            game.id = packet.id
            game.name = packet.name
            game.setVariant(packet.variant)
            game.setBettingStructure(packet.betting_structure)
            game.setMaxPlayers(packet.seats)
            game.reset()
            if self.factory.start_game:
                self.startGame()

        elif packet.type == PACKET_POKER_PLAYER_ARRIVE:
            game.addPlayer(packet.serial)

        elif packet.type == PACKET_POKER_TABLE_MOVE:
            game.removePlayer(packet.serial)

        elif packet.type == PACKET_POKER_PLAYER_LEAVE:
            game.removePlayer(packet.serial)
            if self.getSerial() == packet.serial:
                self.transport.loseConnection()

        elif packet.type == PACKET_POKER_POSITION:
            game.setPosition(packet.serial)
            
        elif packet.type == PACKET_POKER_SEAT:
            if packet.seat == 255:
                print "failed to get a seat"
            
        elif packet.type == PACKET_POKER_SEATS:
	    game.setSeats(packet.seats)
	    
        elif packet.type == PACKET_POKER_SIT:
            game.sit(packet.serial)

        elif packet.type == PACKET_POKER_SIT_OUT:
            game.sitOut(packet.serial)

        elif packet.type == PACKET_POKER_PLAYER_CARDS:
	    player = game.serial2player[packet.serial]
	    player.hand.set(packet.cards)
            self.ev = None

        elif packet.type == PACKET_POKER_BOARD_CARDS:
	    game.board.set(packet.cards)
            self.ev = None

        elif packet.type == PACKET_POKER_DEALER:
            game.setDealer(packet.serial)

        elif packet.type == PACKET_POKER_IN_GAME:
            for serial in game.serialsAll():
                if serial in packet.players:
                    game.sit(serial)
                else:
                    game.sitOut(serial)
        
        elif packet.type == PACKET_POKER_WIN:
            if not game.moneyDistributed():
                game.distributeMoney()
                if game.winners != packet.serials:
                    raise UserWarning, "game.winners %s != packet.serials %s" % (game.winners, packet.serials)
                game.endTurn()
            #
            # Give up if we're broke
            #
            if ( game.serial2player.has_key(self.getSerial()) and
                 game.isBroke(self.getSerial()) ):
                self.broke = True
                self.transport.loseConnection()
            elif self.factory.start_game:
                reactor.callLater(self.factory.delay, self.startGame)
            
        elif packet.type == PACKET_POKER_PLAYER_CHIPS:
            player = game.serial2player[packet.serial]
            player.bet.set(packet.bet)
            player.money.set(packet.money)

        elif packet.type == PACKET_POKER_FOLD:
            game.fold(packet.serial)
            
        elif packet.type == PACKET_POKER_CALL:
            game.call(packet.serial)

        elif packet.type == PACKET_POKER_RAISE:
            game.callNraise(packet.serial, packet.amount)

        elif packet.type == PACKET_POKER_CHECK:
            game.check(packet.serial)

        elif packet.type == PACKET_POKER_BLIND:
            game.blind(packet.serial, packet.amount, packet.dead)

        elif packet.type == PACKET_POKER_ANTE:
            game.ante(packet.serial, packet.amount)

        elif packet.type == PACKET_POKER_STATE:
            if packet.string == "end":
                game.endState()

            if game.state == "blindAnte":
                game.nextRound()

            if game.state != "end":
                game.initRound()

            if game.state != packet.string:
                print " *ERROR* state = %s, expected %s instead " % ( game.state, packet.string )

        elif packet.type == PACKET_POKER_BLIND_REQUEST:
            if ( game.getSerialInPosition() == self.getSerial() ):
                self.sendPacket(PacketPokerBlind(game_id = game.id,
                                                 serial = self.getSerial()))
            
        elif packet.type == PACKET_POKER_ANTE_REQUEST:
            if ( game.getSerialInPosition() == self.getSerial() ):
                self.sendPacket(PacketPokerAnte(game_id = game.id,
                                                serial = self.getSerial()))
            
        elif packet.type == PACKET_POKER_TIMEOUT_WARNING:
            if packet.serial == self.getSerial() and self.factory.verbose:
                print "timeout warning for %d" % packet.serial
                
        self.schedulePlay(packet)

    def startGame(self):
        game = self.factory.game
        self.sendPacket(PacketPokerStart(game_id = game.id,
                                         serial = self.getSerial()))

    def quit(self):
        self.sendPacket(PacketQuit())
        
    def eval(self):
        if self.level == 0:
            actions = ("check", "call", "raise", "fold")
            return actions[randint(0, 3)]

        game = self.factory.game
        if not self.ev:
            self.ev = game.handEV(self.getSerial(), LEVEL2ITERATIONS[self.level])
        ev = self.ev
        
        if game.state == "pre-flop":
            if ev < 100:
                action = "check"
            elif ev < 500:
                action = "call"
            else:
                action = "raise"
        elif game.state == "flop" or game.state == "third":
            if ev < 200:
                action = "check"
            elif ev < 600:
                action = "call"
            else:
                action = "raise"
        elif game.state == "turn" or game.state == "fourth":
            if ev < 300:
                action = "check"
            elif ev < 700:
                action = "call"
            else:
                action = "raise"
        else:
            if ev < 400:
                action = "check"
            elif ev < 800:
                action = "call"
            else:
                action = "raise"
            
        return (action, ev)
        
    def schedulePlay(self, packet):
        if hasattr(self, "playTimer"):
            if self.factory.verbose:
                print "*CRITICAL* schedulePlay: will not schedule twice"
            return
        
        game = self.factory.game
        if ( packet.type == PACKET_POKER_CHECK or 
             packet.type == PACKET_POKER_CALL or
             packet.type == PACKET_POKER_RAISE or
             packet.type == PACKET_POKER_FOLD ) and packet.serial == self.getSerial():
            self.actionDone = False

        if not game.isBlindAnteRound() and game.canPlay(self.getSerial()):
            if not self.actionDone:
                self.playTimer = reactor.callLater(self.factory.delay, self.play)

    def play(self):
        if self.playTimer.active():
            self.playTimer.cancel()
        del self.playTimer
        
        game = self.factory.game
        (desired_action, ev) = self.eval()
        actions = game.possibleActions(self.getSerial())
        if self.factory.verbose:
            print "%s serial = %d, hand = %s, board = %s" % (self.getName(), self.getSerial(), game.getHandAsString(self.getSerial()), game.getBoardAsString())
            print "%s wants to %s (ev = %d)" % (self.getName(), desired_action, ev)
        while not desired_action in actions:
            if desired_action == "check":
                desired_action = "fold"
            elif desired_action == "call":
                desired_action = "check"
            elif desired_action == "raise":
                desired_action = "call"

        if desired_action == "fold":
            self.sendPacket(PacketPokerFold(game_id = game.id,
                                            serial = self.getSerial()))
        elif desired_action == "check":
            self.sendPacket(PacketPokerCheck(game_id = game.id,
                                             serial = self.getSerial()))
        elif desired_action == "call":
            self.sendPacket(PacketPokerCall(game_id = game.id,
                                            serial = self.getSerial()))
        elif desired_action == "raise":
            minimum = PokerChips(game.chips_values)
            self.sendPacket(PacketPokerRaise(game_id = game.id,
                                             serial = self.getSerial(),
                                             amount = minimum.chips))
        else:
            print "=> unexpected actions = %s" % actions
        self.actionDone = True

    def jabberSetup(self):
        if not self.factory.jabber:
            if self.factory.verbose > 1:
                print "Jabber not specified"
            return

        auth = pokerjabber.PokerJabberAuthOrCreate(
            host = self.factory.jabber["host"],
            port = int(self.factory.jabber["port"]),
            name = self.user.name,
            password = self.user.password,
            verbose = self.factory.verbose
            )
        auth.registerHandler(pokerjabber.JABBER_AUTHORCREATE_OK, self.jabberAuthOk)
        auth.registerHandler(pokerjabber.JABBER_AUTHORCREATE_FAILED, self.jabberAuthFailed)
        auth.jabberAuth()
        self.auth = auth

    def jabberMessage(self, message):
        to = lower(self.factory.table["name"]) + "@" + self.factory.jabber["chat"]
        self.jabber.sendMessage(to = to, body = message, ttype = 'groupchat')
        reactor.callLater(5 * 60, self.jabberMessage, message)
        
    def jabberAuthFailed(self):
        del self.auth

    def jabberAuthOk(self, jabber):
        del self.auth
        self.jabber = jabber
        to = lower(self.factory.table["name"]) + "@" + self.factory.jabber["chat"] + "/" + self.user.name
        jabber.sendStanza(ChatPresence(to = to))
#        reactor.callLater(randint(0, 10 * 60), self.jabberMessage, "Hi there")
        
    def protocolEstablished(self):
        self.sendPacket(PacketLogin(name = self.user.name,
                                    password = self.user.password))
        self.sendPacket(PacketPokerTableSelect())
        self.jabberSetup()

    def connectionLost(self, reason):
        print "%s lost connection" % self.getName()

class PokerBotFactory(UGAMEClientFactory):

    def __init__(self, *args, **kwargs):
        UGAMEClientFactory.__init__(self, *args, **kwargs)
        config = kwargs["config"]
        self.config = config
        self.table = kwargs["table"]
        
        self.level = config.headerGet("/bot/@level")
        self.reconnect = config.headerGet("/bot/@reconnect") == "yes"
        self.watch = config.headerGet("/bot/@watch") == "yes"
        self.delay = config.headerGetInt("/bot/@delay")
        self.verbose = config.headerGetInt("/bot/@verbose")
        self.jabber = config.headerGetProperties("/bot/jabber")
        if self.jabber:
            self.jabber = self.jabber[0]
        self.start_game = config.headerGet("/bot/@start_game") == "yes"

        dirs = split(config.headerGet("/bot/path"))
        self.game = PokerGameClient("file:poker.%s.xml", dirs)
        self.game.verbose = self.verbose

        self.protocol = PokerBotClient
        self.bot = None
        
    def clientConnectionFailed(self, connector, reason):
        print "Failed to establish connection to table %s" % self.table["name"]
        print reason
        self.bot.parent.removeService(self.bot)
        
    def clientConnectionLost(self, connector, reason):
        reconnect = False
        if self.reconnect:
            if self.protocol_instance.broke == True:
                print "Re-establishing (get more money)."
                reactor.callLater(2, connector.connect)
                reconnect = True
        else:
            print "The poker3d server connection to %s was closed" % self.table["name"]
            if not reason.check(error.ConnectionDone):
                print reason
        if not reconnect:
            self.bot.parent.removeService(self.bot)

class Bots(service.MultiService):

    def setConfig(self, config):
        self.count = 0
        self.config = config
        self.verbose = config.headerGetInt("/bot/@verbose")

    def addService(self, _service):
        service.MultiService.addService(self, _service)
        self.check()

    def removeService(self, _service):
        service.MultiService.removeService(self, _service)
        self.check()

    def check(self):
        if self.verbose > 1:
            print "%d bots currently active" % len(self.services)
        if len(self.services) <= 0:
            reactor.callLater(0, reactor.stop)

def Application(name, uid=None, gid=None):
    """Return a compound class.

    Return an object supporting the C{IService}, C{IServiceCollection},
    C{IProcess} and C{sob.IPersistable} interfaces, with the given
    parameters. Always access the return value by explicit casting to
    one of the interfaces.
    """
    ret = components.Componentized()
    for comp in (Bots(), sob.Persistent(ret, name), service.Process(uid, gid)):
        ret.addComponent(comp, ignoreClass=1)
    service.IService(ret).setName(name)
    return ret

class Bot(internet.TCPClient):

    def stopService(self):
        #
        # If the connection is still available (i.e. the bots
        # were stopped because of a SIGINT signal), properly
        # close it before exiting.
        #
        if(hasattr(self._connection.transport, "protocol")):
            protocol = self._connection.transport.protocol
            #
            # If the connection fails, the transport exists but
            # the protocol is not set
            #
            if protocol:
                self._connection.transport.protocol.quit()
        return internet.TCPClient.stopService(self)
    
def run(argv):
    configuration = sys.argv[-1][:5] == "file:" and sys.argv[-1] or "file:/etc/poker3d/bot/poker.bot.xml"

    config = Config([''])
    config.loadHeader(configuration)

    host = config.headerGet("/bot/host")
    port = config.headerGetInt("/bot/port")

    bots = Application('pokerbot')
    services = service.IServiceCollection(bots)
    services.setConfig(config)
    
    for table in config.headerGetProperties("/bot/table"):
        for i in range(0, int(table["count"])):
            factory = PokerBotFactory(config = config,
                                      table = table)
            bot = Bot(host, port, factory)
            factory.bot = bot
            bot.setServiceParent(services)
    return bots

application = run(sys.argv[1:])

if __name__ == '__main__':
    try:
        app.startApplication(application, None)
        reactor.run()
    except:
        if application.verbose:
            print_exc()
        else:
            print sys.exc_value
