#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#  Henry Precheur <henry@precheur.org>
#

from underware.client import UGAMEClientProtocol

from pokereval import PokerEval
from poker.pokergame import PokerGameClient, PokerPlayer
from poker.pokercards import PokerCards
from poker.pokerchips import PokerChips
from poker.pokerpackets import *

class PokerClientProtocol(UGAMEClientProtocol):
    """Poker client"""

    def __init__(self):
        UGAMEClientProtocol.__init__(self)
        self.callbacks = {
            'sent': {},
            'current': {},
            'not_current': {}
            }
        self.currentGameId = None
        self.pending_auth_request = False
        self.serial_in_position = 0

    def setCurrentGameId(self, game_id):
        self.currentGameId = game_id

    def getCurrentGameId(self):
        return self.currentGameId
        
    def connectionMade(self):
        "connectionMade"
        self._lagmax = self.factory.delays.get("lag", 15)
        self._hold_enable = self.factory.delays_enable
        UGAMEClientProtocol.connectionMade(self)

    def registerHandler(self, what, name, meth):
        if name:
            names = [ name ]
        else:
            names = PacketNames.keys()
        callbacks = self.callbacks[what]
        for name in names:
            callbacks.setdefault(name, []).append(meth)
        
    def unregisterHandler(self, what, name, meth):
        if name:
            names = [ name ]
        else:
            names = PacketNames.keys()
        callbacks = self.callbacks[what]
        for name in names:
            callbacks[name].remove(meth)
        
    def normalizeChips(self, game, chips):
        chips = chips[:]
        underware_chips = []
        for chip_value in game.chips_values:
            count = chips.pop(0)
            if count > 0:
                underware_chips.append(chip_value)
                underware_chips.append(count)

        return underware_chips
            
    def updatePlayerChips(self, game, player):
        packet = PacketPokerPlayerChips(game_id = game.id,
                                        serial = player.serial,
                                        bet = self.normalizeChips(game, player.bet.chips),
                                        money = self.normalizeChips(game, player.money.chips))
        if self.factory.verbose > 2:
            print "updatePlayerChips: %s" % packet
        return packet

    def updatePotsChips(self, game, side_pots):
        packets = []
        
        if not side_pots:
            packet = PacketPokerChipsPotReset(game_id = game.id)
            return [ packet ]
        
        index = 0
        for (amount, total) in side_pots['pots']:
            chips = PokerChips(game.chips_values, amount)
            bet = self.normalizeChips(game, chips.chips)
            pot = PacketPokerPotChips(game_id = game.id,
                                      index = index,
                                      bet = bet)
            if self.factory.verbose > 1:
                print "updatePotsChips: %s " % pot
            packets.append(pot)
            index += 1
        return packets

    def chipsPlayer2Bet(self, game, player, chips):
        packets = []
        money = PokerChips(game.chips_values, chips)
        chips = self.normalizeChips(game, money.chips)
        packet = PacketPokerChipsPlayer2Bet(game_id = game.id,
                                            serial = player.serial,
                                            chips = chips)
        if self.factory.verbose > 2:
            print "chipsPlayer2Bet: %s" % packet
        packets.append(packet)
        packets.append(self.updatePlayerChips(game, player))
        return packets

    def chipsBet2Pot(self, game, player, bet, pot_index):
        packets = []
        if ( pot_index == 0 and
             player.dead.toint() > 0 and
             game.isSecondRound() ):
            #
            # The ante or the dead are already in the pot
            #
            bet -= player.dead.toint()
        bet = PokerChips(game.chips_values, bet)
        chips = self.normalizeChips(game, bet.chips)
        packet = PacketPokerChipsBet2Pot(game_id = game.id,
                                         serial = player.serial,
                                         chips = chips,
                                         pot = pot_index)
        if self.factory.verbose > 2:
            print "chipsBet2Pot: %s" % packet
        packets.append(packet)
        packets.append(self.updatePlayerChips(game, player))
        return packets
        
    def chipsPot2Player(self, game, player, bet, pot_index, reason):
        bet = PokerChips(game.chips_values, bet)
        chips = self.normalizeChips(game, bet.chips)
        packet = PacketPokerChipsPot2Player(game_id = game.id,
                                            serial = player.serial,
                                            chips = chips,
                                            pot = pot_index,
                                            reason = reason)
        if self.factory.verbose > 2:
            print "chipsPot2Player: %s" % packet
        return packet
        
    def handleUserInfo(self, packet):
        self.play_money = packet.play_money

    def handleSerial(self, packet):
        self.user.serial = packet.serial
        self.sendPacket(PacketPokerGetUserInfo(serial = packet.serial))

    def logout(self):
        self.sendPacket(PacketLogout())
        self.user.logout()
        
    def _handleConnection(self, packet):
        forward_packets = [ packet ]
        
        game = self.factory.packet2game(packet)
        
        if packet.type == PACKET_POKER_USER_INFO:
            self.handleUserInfo(packet)

        elif packet.type == PACKET_POKER_TABLE_DESTROY:
            self.factory.deleteGame(packet.serial)
            if self.currentGameId == packet.serial:
                self.currentGameId = None
            
        elif packet.type == PACKET_POKER_TABLE:
            if packet.id == 0:
                print "*CRITICAL* server refused our request"
            else:
                new_game = self.factory.getOrCreateGame(packet.id)
                new_game.name = packet.name
                new_game.setTime(0)
                new_game.setVariant(packet.variant)
                new_game.setBettingStructure(packet.betting_structure)
                new_game.setMaxPlayers(packet.seats)
                new_game.reset()
                self.currentGameId = new_game.id
                self.updatePotsChips(new_game, [])

        elif packet.type == PACKET_AUTH_REQUEST:
            self.pending_auth_request = True

        elif packet.type == PACKET_AUTH_EXPIRES:
            self.pending_auth_request = False

        elif packet.type == PACKET_AUTH_REFUSED:
            self.pending_auth_request = False

        elif packet.type == PACKET_AUTH_CANCEL:
            self.pending_auth_request = False

        elif packet.type == PACKET_AUTH_OK:
            pass
        
        elif packet.type == PACKET_SERIAL:
            self.handleSerial(packet)
            if not self.pending_auth_request:
                self.sendPacket(PacketPokerPlayerInfo(serial = self.getSerial(),
                                                      name = self.getName(),
                                                      outfit = self.factory.getOutfit()))
                self.deleteGames()
            self.pending_auth_request = False

        elif packet.type == PACKET_POKER_START:
            if packet.hand_serial == 0:
                print "*CRITICAL* game start was refused"
                forward_packets.remove(packet)
            elif game.isRunning():
                raise UserWarning, "you should not be here (state: %s)" % game.state
            else:
                game.setTime(packet.time)
                game.setHandsCount(packet.hands_count)
                game.setLevel(packet.level)
                game.beginTurn(packet.hand_serial)
                forward_packets.append(PacketPokerBoardCards(game_id = game.id, serial = self.getSerial()))
                for serial in game.player_list:
                    forward_packets.append(self.updatePlayerChips(game, game.serial2player[serial]))
                forward_packets.extend(self.updatePotsChips(game, []))

        elif packet.type == PACKET_POKER_CANCELED:
            if packet.amount > 0:
                player = game.getPlayer(packet.serial)
                bet = player.bet.toint()
                if bet > 0:
                    forward_packets.extend(self.chipsBet2Pot(game, player, player.bet, 0))
                if packet.amount > 0:
                    forward_packets.append(self.chipsPot2Player(game, player, packet.amount, 0, "canceled"))
            game.canceled(packet.serial, packet.amount)

        elif packet.type == PACKET_POKER_PLAYER_INFO:
            game.getPlayer(packet.serial).name = packet.name
            self.factory.serial2player_info[packet.serial] = packet

        elif packet.type == PACKET_POKER_PLAYER_ARRIVE:
            game.addPlayer(packet.serial)
            game.getPlayer(packet.serial).name = packet.name
            self.factory.serial2player_info[packet.serial] = packet

        elif ( packet.type == PACKET_POKER_PLAYER_LEAVE or
               packet.type == PACKET_POKER_TABLE_MOVE ) :
            game.removePlayer(packet.serial)

        elif packet.type == PACKET_POKER_POSITION:
            game.setPosition(packet.serial)
            forward_packets.remove(packet)

        elif packet.type == PACKET_POKER_SEAT:
            if packet.seat == 255:
                print "*CRITICAL* This seat is busy"
            else:
                if game.isTournament():
                    self.sendPacket(PacketPokerSit(serial = self.getSerial(),
                                                   game_id = game.id))
            
        elif packet.type == PACKET_POKER_LOOK_CARDS:
            pass
            
        elif packet.type == PACKET_POKER_SEATS:
            game.setSeats(packet.seats)
            
        elif packet.type == PACKET_POKER_PLAYER_CARDS:
            player = game.getPlayer(packet.serial)
            player.hand.set(packet.cards)
            forward_packets.remove(packet)

        elif packet.type == PACKET_POKER_BOARD_CARDS:
            game.board.set(packet.cards)

        elif packet.type == PACKET_POKER_DEALER:
            game.setDealer(packet.serial)
            
        elif packet.type == PACKET_POKER_SIT_OUT:
            game.sitOut(packet.serial)

        elif packet.type == PACKET_POKER_SIT:
            game.sit(packet.serial)
            
        elif packet.type == PACKET_POKER_WAIT_FOR:
            game.getPlayer(packet.serial).wait_for = packet.reason
            forward_packets.remove(packet)
            
        elif packet.type == PACKET_POKER_IN_GAME:
            for serial in game.serialsAll():
                if serial in packet.players:
                    if not game.isSit(serial):
                        game.sit(serial)
                        forward_packets.append(PacketPokerSit(game_id = game.id,
                                                              serial = serial))
                else:
                    if game.isSit(serial):
                        wait_for = game.getPlayer(serial).wait_for
                        game.sitOut(serial)
                        if wait_for:
                            forward_packets.append(PacketPokerWaitFor(game_id = game.id,
                                                                      serial = serial,
                                                                      reason = wait_for))
                        else:
                            forward_packets.append(PacketPokerSitOut(game_id = game.id,
                                                                     serial = serial))
            
        elif packet.type == PACKET_POKER_WIN:
            for ser in packet.serials:
                forward_packets.append(PacketPokerPlayerWin(serial=ser,game_id=game.id))
            
            if game.winners:
                forward_packets.extend(self.packetsPot2Player(game))
            else:
                game.distributeMoney()

                winners = game.winners[:]
                winners.sort()
                packet.serials.sort()
                if winners != packet.serials:
                    raise UserWarning, "game.winners %s != packet.serials %s" % (winners, packet.serials)
                forward_packets.extend(self.packetsShowdown(game))
                forward_packets.extend(self.packetsPot2Player(game))
                game.endTurn()
                forward_packets.append(PacketPokerPosition(game_id = game.id,
                                                           serial = 0))
            
        elif packet.type == PACKET_POKER_PLAYER_CHIPS:
            forward_packets.remove(packet)
            player = game.serial2player[packet.serial]
            if game.isRunning():
                #
                # Sanity checks. Immediately after the game begins the server
                # sends the amount of chips for each player so that the client
                # can resynchronize and issue a warning.
                #
                bet = PokerChips(game.chips_values, packet.bet)
                if player.bet.toint() != bet.toint():
                    if self.factory.verbose > 1:
                        print "server says player %d has a bet of %d chips and client thinks it has %d" % ( packet.serial, bet.toint(), player.bet.toint())
                    player.bet.set(packet.bet)
                money = PokerChips(game.chips_values, packet.money)
                if player.money.toint() != money.toint():
                    if self.factory.verbose > 1:
                        print "server says player %d has a money of %d chips and client thinks it has %d" % ( packet.serial, money.toint(), player.money.toint())
                    player.money.set(packet.money)
            else:
                #
                # If server sends chips amount for a player while not in game,
                # 
                #
                player.bet.set(packet.bet)
                player.money.set(packet.money)

            chips = PacketPokerPlayerChips(game_id = game.id,
                                           serial = packet.serial,
                                           money = self.normalizeChips(game, packet.money),
                                           bet = self.normalizeChips(game, packet.bet))
            forward_packets.append(chips)

        elif packet.type == PACKET_POKER_FOLD:
            game.fold(packet.serial)
            if game.isSitOut(packet.serial):
                forward_packets.append(PacketPokerSitOut(game_id = game.id,
                                                         serial = packet.serial))
            
        elif packet.type == PACKET_POKER_CALL:
            player = game.getPlayer(packet.serial)
            transfered = player.money.copy()
            game.call(packet.serial)
            transfered.subtract(player.money)
            forward_packets.extend(self.chipsPlayer2Bet(game, player, transfered.chips))

        elif packet.type == PACKET_POKER_RAISE:
            game.callNraise(packet.serial, packet.amount)
            player = game.getPlayer(packet.serial)
            forward_packets.append(PacketPokerHighestBetIncrease(game_id = game.id))
            forward_packets.extend(self.chipsPlayer2Bet(game, player, packet.amount))

        elif packet.type == PACKET_POKER_CHECK:
            game.check(packet.serial)

        elif packet.type == PACKET_POKER_BLIND:
            player = game.getPlayer(packet.serial)
            highest_bet = game.highestBetNotFold()
            game.blind(packet.serial, packet.amount, packet.dead)
            if ( highest_bet < game.blind_info["big"] and
                 packet.amount >= game.blind_info["big"] ):
                forward_packets.append(PacketPokerHighestBetIncrease(game_id = game.id))
            if packet.dead > 0:
                forward_packets.extend(self.chipsPlayer2Bet(game, player, packet.dead))
                forward_packets.extend(self.chipsBet2Pot(game, player, packet.dead, 0))
            forward_packets.extend(self.chipsPlayer2Bet(game, player, packet.amount))

        elif packet.type == PACKET_POKER_ANTE:
            player = game.getPlayer(packet.serial)
            game.ante(packet.serial, packet.amount)
            forward_packets.extend(self.chipsPlayer2Bet(game, player, packet.amount))
            forward_packets.extend(self.chipsBet2Pot(game, player, packet.amount, 0))

        elif packet.type == PACKET_POKER_STATE:
            if ( packet.string != "end" and not game.isBlindAnteRound() ):
                forward_packets.insert(-1, PacketPokerEndRound(game_id = game.id))
                
            if packet.string == "end":
                game.endState()
                
            if game.isBlindAnteRound():
                game.nextRound()
            #
            # A state change is received at the begining of each
            # betting round. No state change is received when
            # reaching showdown or otherwise terminating the hand.
            #
            if game.isFirstRound():
                game.initRound()
            else:
                if ( packet.string == "end" and
                     game.isSingleUncalledBet(game.side_pots) ):
                    forward_packets.extend(self.moveBet2Player(game))
                else:
                    forward_packets.extend(self.moveBet2Pot(game))
                    
                if packet.string != "end":
                    game.initRound()

            if game.isRunning() and game.cardsDealt() and game.downCardsDealtThisRoundCount() > 0:
                forward_packets.append(PacketPokerDealCards(game_id = game.id,
                                                            numberOfCards = game.downCardsDealtThisRoundCount(),
                                                            serials = game.serialsNotFold()))
            if game.isRunning() and game.cardsDealt() and game.cardsDealtThisRoundCount() :
                for player in game.playersNotFold():
                    cards = player.hand.toRawList()
                    forward_packets.append(PacketPokerPlayerCards(game_id = game.id,
                                                                  serial = player.serial,
                                                                  cards = cards))

            if game.state != packet.string:
                print " *CRITICAL* state = %s, expected %s instead " % ( game.state, packet.string )

        if ( game and
             game.id == self.currentGameId and
             game.isRunning() ):
            position_changed = self.serial_in_position != game.getSerialInPosition()
            if position_changed:
                self_was_in_position = self.serial_in_position == self.getSerial()
                self.serial_in_position = game.getSerialInPosition()
                self_in_position = self.serial_in_position == self.getSerial()
                if self.serial_in_position > 0:
                    forward_packets.extend(self.updateBetLimit(game))
                    forward_packets.append(PacketPokerPosition(game_id = game.id,
                                                               serial = self.serial_in_position))
                    if ( self_was_in_position and not self_in_position ):
                        forward_packets.append(PacketPokerSelfLostPosition(game_id = game.id,
                                                                           serial = self.serial_in_position))
                    if ( not self_was_in_position and self_in_position ):
                        forward_packets.append(PacketPokerSelfInPosition(game_id = game.id,
                                                                         serial = self.serial_in_position))
                elif self_was_in_position:
                    forward_packets.append(PacketPokerSelfLostPosition(game_id = game.id,
                                                                       serial = self.getSerial()))

        elif ( game and
               game.id == self.currentGameId and
               not game.isRunning() and
               self.serial_in_position > 0 ):
            forward_packets.append(PacketPokerSelfLostPosition(game_id = game.id,
                                                               serial = self.getSerial()))
            self.serial_in_position = 0
            
            
                

        for forward_packet in forward_packets:
            self.publishPacket(forward_packet)

    def moveBet2Pot(self, game):
        packets = []
        contributions = game.getPots()['contributions']
        last_round = max(contributions.keys())
        round_contributions = contributions[last_round]
        for (pot_index, pot_contribution) in round_contributions.iteritems():
            for (serial, amount) in pot_contribution.iteritems():
                player = game.getPlayer(serial)
                packets.extend(self.chipsBet2Pot(game, player, amount, pot_index))

        packets.extend(self.updatePotsChips(game, game.getPots()))
        return packets
        
    #
    # Should be move all bets back to players (for uncalled bets)
    # This is a border case we don't want to handle right now
    #
    moveBet2Player = moveBet2Pot
        
    def updateBetLimit(self, game):
        if not self.getSerial() in game.serialsPlaying():
            return []
            
        packets = []
        (min_bet, max_bet, to_call) = game.betLimits(self.getSerial())
        found = None
        steps = game.chips_values[:]
        steps.reverse()
        for step in steps:
            if min_bet % step == 0 and max_bet % step == 0:
                found = step
                break
        if found:
            if self.factory.verbose:
                print " => bet min=%d, max=%d, step=%d" % ( min_bet, max_bet, found)
            packets.append(PacketPokerBetLimit(game_id = game.id,
                                               min = min_bet,
                                               max = max_bet,
                                               step = found,
                                               call = to_call))
        else:
            print "*CRITICAL* no chip value (%s) is suitable to step from min_bet = %d to max_bet = %d" % ( game.chips_values, min_bet, max_bet )
        return packets

    def packetsPot2Player(self, game):
        packets = []
        current_pot = 0
        game_state = game.showdown_stack[0]
        pots = game_state['side_pots']['pots']
        frame_count = len(game.showdown_stack) - 1
        for frame in game.showdown_stack:
            if frame['type'] == 'left_over':
                player = game.getPlayer(frame['serial'])
                packets.append(self.chipsPot2Player(game, player, frame['chips_left'], len(pots) - 1, "left_over"))
            elif frame['type'] == 'uncalled':
                player = game.getPlayer(frame['serial'])
                packets.append(self.chipsPot2Player(game, player, frame['uncalled'], len(pots) - 1, "uncalled"))
            elif frame['type'] == 'resolve':
                cumulated_pot_size = 0
                next_pot = current_pot
                for (pot_size, pot_total) in pots[current_pot:]:
                    cumulated_pot_size += pot_size
                    next_pot += 1
                    if cumulated_pot_size >= frame['pot']:
                        break
                if cumulated_pot_size != frame['pot']:
                    print "*CRITICAL* pot %d, total size = %d, expected %d" % ( current_pot, cumulated_pot_size, frame['pot'] )
                merged_pot = next_pot - 1
                if merged_pot > current_pot:
                    merge = PacketPokerChipsPotMerge(game_id = game.id,
                                                     sources = range(current_pot, merged_pot),
                                                     destination = merged_pot)
                    if self.factory.verbose > 2:
                        print "packetsPot2Player: %s" % merge
                    packets.append(merge)
                if frame_count == 1 and len(frame['serial2share']) == 1:
                    #
                    # Happens quite often : single winner. Special case where
                    # we use the exact chips layout saved in game_state.
                    #
                    serial = frame['serial2share'].keys()[0]
                    packets.append(self.chipsPot2Player(game, game.getPlayer(serial), game_state['pot'], merged_pot, "win"))
                else:
                    #
                    # Possibly complex showdown, cannot avoid breaking chip stacks
                    #
                    for (serial, share) in frame['serial2share'].iteritems():
                        packets.append(self.chipsPot2Player(game, game.getPlayer(serial), share, merged_pot, "win"))
                current_pot = next_pot
            else:
                pass
                
        for player in game.serial2player.itervalues():
            packets.append(self.updatePlayerChips(game, player))
        packets.extend(self.updatePotsChips(game, []))
        return packets
        
    def packetsShowdown(self, game):
        packets = []
        if game.variant == "7stud":
            for player in game.playersAll():
                packets.append(PacketPokerPlayerNoCards(game_id = game.id,
                                                        serial = player.serial))
                if player.hand.areVisible():
                    packet = PacketPokerPlayerCards(game_id = game.id,
                                                    serial = player.serial,
                                                    cards = player.hand.tolist(True))
                    packet.visibles = "hole"
                    packets.append(packet)

        if game.winners:
            #
            # If some winners left the game, we can't accurately
            # display the showdown.
            #
            for serial in game.winners:
                if serial not in game.serial2player.keys():
                    return packets

            for (serial, best) in game.serial2best.iteritems():
                for (side, (value, cards)) in best.iteritems():
                    if serial in game.side2winners[side]:
                        if len(cards) > 1:
                            side = game.isHighLow() and side or ""
                            if side == "low":
                                hand = ""
                            else:
                                hand = game.readableHandValueShort(side, cards[0], cards[1:])
                            packets.append(PacketPokerBestCards(game_id = game.id,
                                                                serial = serial,
                                                                side = side,
                                                                cards = cards[1:],
                                                                board = game.board.tolist(True),
                                                                hand = hand))
        return packets

    def publishQuit(self):
        for game in self.factory.games.values():
            self.publishTableQuit(game)

    def publishTableQuit(self, game):
        for serial in game.serialsAll():
            packet = PacketPokerPlayerLeave(game_id = game.id,
                                            serial = serial)
            self.publishPacket(packet)
        self.publishPacket(PacketPokerTableQuit(game_id = game.id,
                                                serial = self.getSerial()))
        
    def resendPackets(self, game_id):
        """ ex redisplay  / refresh """
        game = self.getGame(game_id)
        packets = []
        packet = PacketPokerTable(id = game.id,
                                  name = game.name,
                                  variant = game.variant,
                                  seats = game.max_players,
                                  betting_structure = game.betting_structure)
        packets.append(PacketPokerBatchMode(game_id = game.id))
        packet.seats_all = game.seats_all
        packets.append(packet)
        for player in game.playersAll():
            player_info = self.factory.serial2player_info[player.serial]
            player_info.game_id = game.id
            packets.append(player_info)
            if player.isSit():
                packets.append(PacketPokerSit(game_id = game.id,
                                              serial = player.serial))
            else:
                packets.append(PacketPokerSitOut(game_id = game.id,
                                                 serial = player.serial))
            packets.append(self.updatePlayerChips(game, player))
        packets.append(PacketPokerSeats(game_id = game.id,
                                        seats = game.seats()))
        packets.append(PacketPokerStart(game_id = game.id,
                                        hand_serial = game.hand_serial))
        for player in game.playersAll():
            packet = PacketPokerPlayerCards(game_id = game.id,
                                            serial = player.serial,
                                            cards = player.hand.toRawList())
            packets.append(packet)
        packets.append(PacketPokerBoardCards(game_id = game.id,
                                             cards = game.board.tolist(False)))
        if game.isRunning():
            packets.extend(self.updatePotsChips(game, game.getPots()))
            packets.append(PacketPokerPosition(game_id = game.id,
                                               serial = game.getSerialInPosition()))
            packets.extend(self.updateBetLimit(game))
        else:
            packets.extend(self.packetsShowdown(game))
        packets.append(PacketPokerStreamMode(game_id = game.id))

        for packet in packets:
            self.publishPacket(packet)

    def deleteGames(self):
        self.currentGameId = None
        for game_id in self.factory.games.keys():
            self.deleteGame(game_id)
        
    def deleteGame(self, game_id):
        return self.factory.deleteGame(game_id)

    def getGame(self, game_id):
        return self.factory.getGame(game_id)
    
    def sendPacket(self, packet):
        if packet.type == PACKET_POKER_TABLE_QUIT:
            self.publishTableQuit(self.getGame(packet.game_id))
        elif packet.type == PACKET_QUIT:
            self.ignoreIncomingData()
            self.publishQuit()
        UGAMEClientProtocol.sendPacket(self, packet)

    def protocolEstablished(self):
        self.user.name = self.factory.name
        self.user.password = self.factory.password
        self.publishPacket(PacketBootstrap())

    def protocolInvalid(self, server, client):
        self.publishPacket(PacketProtocolError(message = "Upgrade the client from\nhttp://mekensleep.org/\nSever version is %s\nClient version is %s" % ( server, client ) ))
                        
    def publishPacket(self, packet, what = None):
        if self.factory.verbose > 2:
            print "publishPacket: %s" % packet
        if not what:
            what = 'not_current'
            if hasattr(packet, "game_id") and packet.game_id == self.currentGameId:
                what = 'current'
            if packet.type == PACKET_POKER_TABLE:
                what = 'current'
        if self.callbacks[what].has_key(packet.type):
            callbacks = self.callbacks[what][packet.type]
            for callback in callbacks:
                callback(self, packet)
