#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#  Henry Precheur <henry@precheur.org>
#
#

import sys
sys.path.insert(0, ".")
sys.path.insert(0, "..")
sys.path.insert(0, "../../python")

from re import match
from types import *
from string import split, lower, join
from pprint import pprint, pformat
from time import time
import os
from traceback import print_exc

from twisted.application import internet, service, app
from twisted.internet import pollreactor
import sys
if not sys.modules.has_key('twisted.internet.reactor'):
    print "installing poll reactor"
    pollreactor.install()
else:
    print "poll reactor already installed"
from twisted.internet import reactor, defer

from twibber.client import XMPPClient, XMPPClientFactory
from twibber.elements import ChatPresence, ChatInvite, ChatKick

from underware.server import UGAMEServer, UGAMEServerFactory
from underware.user import User
from underware.config import Config, loadURL

from poker.pokergame import PokerGameServer, PokerPlayer, PokerChips, breakGames, equalizeGames
from poker.pokercards import PokerCards
from poker.pokerpackets import *

class PokerServer(UGAMEServer):
    """Poker server"""

    def __init__(self):
        self.tables = {}
        self.player_info = PacketPokerPlayerInfo(name = "anonymous",
                                                 outfit = "default")
        UGAMEServer.__init__(self)

    def __del__(self):
        print "PokerServer instance deleted"

    def packet2table(self, packet):
        if hasattr(packet, "game_id") and self.tables.has_key(packet.game_id):
            return self.tables[packet.game_id]
        else:
            return False

    def _handleConnection(self, packet):
        if not self.isAuthorized(packet.type):
            self.askAuth(packet)
            return

        if packet.type == PACKET_LOGIN:
            for table in self.tables.values():
                table.quitPlayer(self, self.getSerial())
            self.auth(packet)
            return

        elif packet.type == PACKET_POKER_GET_USER_INFO:
            if self.getSerial() == packet.serial:
                self.getUserInfo(packet.serial)
            else:
                print "attempt to get user info for user %d by user %d" % ( packet.serial, self.getSerial() )
            return

        elif packet.type == PACKET_POKER_PLAYER_INFO:
            if self.getSerial() == packet.serial:
                self.setPlayerInfo(packet)
            else:
                print "attempt to set player info for player %d by player %d" % ( packet.serial, self.getSerial() )
            return
                
        table = self.packet2table(packet)
            
        if table:
            game = table.game

            if packet.type == PACKET_POKER_START:
                if game.isRunning():
                    print "player %d tried to start a new game while in game " % self.getSerial()
                    self.sendPacketVerbose(PacketPokerStart(game_id = game.id))
                elif self.factory.shutting_down:
                    print "server shutting down"
                elif table.owner != 0 and self.getSerial() != table.owner:
                    print "player %d tried to start a new game but is not the owner of the table" % self.getSerial()
                    self.sendPacketVerbose(PacketPokerStart(game_id = game.id))
                else:
                    hand_serial = table.beginTurn()

            elif packet.type == PACKET_POKER_SEAT:
                if ( self.getSerial() == packet.serial or
                     self.getSerial() == table.owner ):
                    if not table.seatPlayer(self, packet.serial, packet.seat):
                        packet.seat = 255
                    else:
                        packet.seat = game.getPlayer(packet.serial).seat
                    self.getUserInfo(packet.serial)
                    self.sendPacketVerbose(packet)
                else:
                    print "attempt to get seat for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )
                
            elif packet.type == PACKET_POKER_BUY_IN:
                if self.getSerial() == packet.serial:
                    table.buyInPlayer(self, packet.amount)
                else:
                    print "attempt to bring money for player %d by player %d" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_CHAT:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    #
                    # Hard wired 128 because jabber will take over at some point
                    #
                    table.chatPlayer(self, packet.serial, packet.message[:128])
                else:
                    print "attempt chat for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_PLAYER_LEAVE:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    table.leavePlayer(self, packet.serial)
                else:
                    print "attempt to leave for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_SIT:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    table.sitPlayer(self, packet.serial)
                else:
                    print "attempt to sit back for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )
                
            elif packet.type == PACKET_POKER_SIT_OUT:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:

                    table.sitOutPlayer(self, packet.serial)
                else:
                    print "attempt to sit out for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )
                
            elif packet.type == PACKET_POKER_AUTO_BLIND_ANTE:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    table.autoBlindAnte(self, packet.serial, True)
                else:
                    print "attempt to set auto blind/ante for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )
                
            elif packet.type == PACKET_POKER_NOAUTO_BLIND_ANTE:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    table.autoBlindAnte(self, packet.serial, False)
                else:
                    print "attempt to set auto blind/ante for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )
                
            elif packet.type == PACKET_POKER_BLIND:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    
                    game.blind(packet.serial)
                else:
                    print "attempt to pay the blind of player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_WAIT_BIG_BLIND:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    
                    game.waitBigBlind(packet.serial)
                else:
                    print "attempt to wait for big blind of player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_ANTE:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    
                    game.ante(packet.serial)
                else:
                    print "attempt to pay the ante of player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_LOOK_CARDS:
                table.broadcast(packet)
                
            elif packet.type == PACKET_POKER_FOLD:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    
                    game.fold(packet.serial)
                else:
                    print "attempt to fold player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_CALL:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    
                    game.call(packet.serial)
                else:
                    print "attempt to call for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_RAISE:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    
                    game.callNraise(packet.serial, packet.amount)
                else:
                    print "attempt to raise for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_CHECK:
                if self.getSerial() == packet.serial or self.getSerial() == table.owner:
                    
                    game.check(packet.serial)
                else:
                    print "attempt to check for player %d by player %d that is not the owner of the game" % ( packet.serial, self.getSerial() )

            elif packet.type == PACKET_POKER_TABLE_QUIT:
                table.quitPlayer(self, self.getSerial())

            elif packet.type == PACKET_POKER_HAND_REPLAY:
                table.handReplay(self, packet.serial)

            elif self.getSerial() == table.owner:

                if packet.type == PACKET_POKER_TABLE_CLOSE:
                    game.close()

                elif packet.type == PACKET_POKER_TABLE_OPEN:
                    game.open()

                elif packet.type == PACKET_POKER_TIMEOUT_POLICY:
                    table.timeout_policy = packet.string

                elif packet.type == PACKET_POKER_TABLE_DESTROY:
                    table.destroy()

                elif packet.type == PACKET_POKER_TABLE_KICK:
                    if table.serial2client.has_key(packet.serial):
                        table.kickPlayer(table.serial2client[packet.serial], packet.serial)
                    else:
                        print "%d can't kick player %d (unknown to table %d)" % ( self.getSerial(), packet.serial, game.id )
                
            table.update()

        if packet.type == PACKET_POKER_TABLE_GROUP:
            serial = self.factory.setTableGroup(self.getSerial(), packet.game_ids)
            self.sendPacketVerbose(PacketPokerTableGroupSerial(serial = serial))

        elif packet.type == PACKET_POKER_TABLE_UNGROUP:
            serial = self.factory.unsetTableGroup(self.getSerial(), packet.serial)
            self.sendPacketVerbose(PacketPokerTableUngroup(serial = serial))
            
        elif packet.type == PACKET_POKER_TABLE_GROUP_BALANCE:
            result = self.factory.balance(self.getSerial(), packet.serial)
            self.sendPacketVerbose(PacketPokerTableGroupBalance(serial = result))
            
        elif packet.type == PACKET_POKER_TABLE_JOIN:
            table = self.factory.getTable(packet.game_id)
            if table:
                if not table.joinPlayer(self, self.getSerial()):
                    self.sendPacketVerbose(PacketPokerTable())

        elif not table and packet.type == PACKET_POKER_HAND_REPLAY:
            table = self.createTable(PacketPokerTable())
            if table:
                table.handReplay(self, packet.serial)

        elif packet.type == PACKET_POKER_TABLE:
            table = self.createTable(packet)
            if table:
                table.joinPlayer(self, self.getSerial())
            
        elif packet.type == PACKET_POKER_TABLE_SELECT:
            self.listTables(packet)

        elif packet.type == PACKET_POKER_HAND_SELECT:
            self.listHands(packet, self.getSerial())

        elif packet.type == PACKET_POKER_HAND_SELECT_ALL:
            self.listHands(packet, None)

        elif packet.type == PACKET_QUIT:
            for table in self.tables.values():
                table.quitPlayer(self, self.getSerial())

        elif packet.type == PACKET_LOGOUT:
            for table in self.tables.values():
                table.quitPlayer(self, self.getSerial())
            self.logout()
            
        self.expect(self._handleConnection)

    def setPlayerInfo(self, packet):
        self.player_info = packet
        self.factory.setPlayerInfo(packet)

    def getPlayerInfo(self):
        return self.player_info
    
    def listTables(self, packet):
        packets = []
        for table in self.factory.listTables(packet.string, self.getSerial()):
            game = table.game
            packet = PacketPokerTable(id = game.id,
                                      name = game.name,
                                      variant = game.variant,
                                      betting_structure = game.betting_structure,
                                      seats = game.max_players,
                                      players = game.allCount(),
                                      hands_per_hour = game.stats["hands_per_hour"],
                                      average_pot = game.stats["average_pot"],
                                      percent_flop = game.stats["percent_flop"],
                                      timeout = table.playerTimeout,
                                      observers = len(table.observers),
                                      waiting = len(table.waiting))
            packets.append(packet)
        self.sendPacketVerbose(PacketPokerTableList(packets = packets))

    def listHands(self, packet, serial):
        if serial != None:
            sql = "select distinct hands.serial from hands,user2hand "
            sql += " where user2hand.hand_serial = hands.serial "
            sql += " and user2hand.user_serial = %d " % serial
            if packet.string:
                sql += " and " + packet.string
            sql += " order by hands.serial desc limit 50 "
        else:
            sql = "select serial from hands "
            if packet.string:
                sql += "where " + packet.string
        hands = self.factory.listHands(sql)
        self.sendPacketVerbose(PacketPokerHandList(hands = hands))

    def createTable(self, packet):
        table = self.factory.createTable(self.getSerial(), {
            "seats": packet.seats,
            "name": packet.name,
            "variant": packet.variant,
            "betting_structure": packet.betting_structure,
            "timeout": packet.timeout,
            "transient": True })
        if not table:
            self.sendPacket(PacketPokerTable())
        return table            

    def join(self, table):
        game = table.game
        
        self.tables[game.id] = table

        self.sendPacketVerbose(PacketPokerTable(id = game.id,
                                                name = game.name,
                                                variant = game.variant,
                                                seats = game.max_players,
                                                betting_structure = game.betting_structure))
        self.sendPacketVerbose(PacketPokerBatchMode(game_id = game.id))
        nochips = PokerChips(game.chips_values).chips
        for player in game.serial2player.values():
            player_info = table.getPlayerInfo(player.serial)
            self.sendPacketVerbose(PacketPokerPlayerArrive(game_id = game.id,
                                                           serial = player.serial,
                                                           name = player_info.name,
                                                           outfit = player_info.outfit))
            if not game.isPlaying(player.serial):
                self.sendPacketVerbose(PacketPokerPlayerChips(game_id = game.id,
                                                              serial = player.serial,
                                                              bet = nochips,
                                                              money = player.money.chips))
                if game.isSit(player.serial):
                    self.sendPacketVerbose(PacketPokerSit(game_id = game.id,
                                                          serial = player.serial))

        self.sendPacketVerbose(PacketPokerSeats(game_id = game.id,
                                                seats = game.seats()))
        cache = table.createCache()
        if game.isRunning():
            #
            # If a game is running, replay it.
            #
            # If a player reconnects, his serial number will match
            # the serial of some packets, for instance the cards
            # of his hand. We rely on private2public to turn the
            # packet containing cards real cards into placeholders
            # in this case.
            #
            for past_packet in table.history2packets(game.historyGet(), game.id, table.createCache()):
                self.sendPacketVerbose(table.private2public(past_packet, self.getSerial()))
        self.sendPacketVerbose(PacketPokerStreamMode(game_id = game.id))

    def addPlayer(self, table, seat):
        table.game.addPlayer(self.getSerial(), seat)
        self.sendNewPlayerInformation(table)
        
    def connectionLost(self, reason):
        if self.factory.verbose:
            print "Connection lost for %s/%d" % ( self.getName(), self.getSerial() )
        for table in self.tables.values():

            table.disconnectPlayer(self, self.getSerial())
        UGAMEServer.connectionLost(self, reason)

    def getUserInfo(self, serial):
        self.sendPacketVerbose(self.factory.getUserInfo(serial))

    def movePlayerTo(self, table, money):
        self.join(table)
        table.game.open()
        table.game.addPlayer(self.getSerial())
        table.game.close()

        table.game.serial2player[self.getSerial()].money = money

        self.sendNewPlayerInformation(table)

    def movePlayerFrom(self, table, to_game_id):
        game = table.game
        table.broadcast(PacketPokerTableMove(game_id = game.id,
                                             serial = self.getSerial(),
                                             to_game_id = to_game_id))
        game.removePlayer(self.getSerial())
    
    def removePlayer(self, table, serial):
        game = table.game
        playing = game.isPlaying(serial)
        if not playing:
            #
            # If the player is not in a game, the removal will be effective
            # immediately and can be announced to all players, including
            # the one that will be removed.
            #
            table.broadcast(PacketPokerPlayerLeave(game_id = game.id,
                                                   serial = serial))
            
        game.removePlayer(serial)

        return not playing

    def sitPlayer(self, table, serial):
        game = table.game
        #
        # It does not harm to sit if already sit and it
        # resets the autoPlayer/wait_for flag.
        #
        if game.sit(serial):
            table.broadcast(PacketPokerSit(game_id = game.id,
                                           serial = serial))

    def sitOutPlayer(self, table, serial):
        game = table.game
        if table.isOpen():
            if game.sitOutNextTurn(serial):
                table.broadcast(PacketPokerSitOut(game_id = game.id,
                                                  serial = serial))
        else:
            game.autoPlayer(serial)
            table.broadcast(PacketPokerAutoFold(game_id = game.id,
                                                serial = serial))

    def autoBlindAnte(self, table, serial, auto):
        game = table.game
        if game.isTournament():
            return
        game.getPlayer(serial).auto_blind_ante = auto
        if auto:
            self.sendPacketVerbose(PacketPokerAutoBlindAnte(game_id = game.id,
                                                            serial = serial))
        else:
            self.sendPacketVerbose(PacketPokerNoautoBlindAnte(game_id = game.id,
                                                              serial = serial))
        
    def setMoney(self, table, amount):
        game = table.game

        game.payBuyIn(self.getSerial(), amount)
        player = game.getPlayer(self.getSerial())
        nochips = PokerChips(game.chips_values).chips
        table.broadcast(PacketPokerPlayerChips(game_id = game.id,
                                               serial = self.getSerial(),
                                               bet = nochips,
                                               money = player.money.chips))
        
    def sendNewPlayerInformation(self, table):
        game = table.game
        if self.factory.verbose > 1:
            print "about player %d" % self.getSerial()
        nochips = PokerChips(game.chips_values).chips
        serial = self.getSerial()
        player_info = self.getPlayerInfo()
        packets = [
            PacketPokerPlayerArrive(game_id = game.id,
                                    serial = serial,
                                    name = player_info.name,
                                    outfit = player_info.outfit),
            PacketPokerSeats(game_id = game.id, seats = game.seats()),
            PacketPokerPlayerChips(game_id = game.id,
                                   serial = serial,
                                   bet = nochips,
                                   money = game.getPlayer(serial).money.chips)
            ]
        table.broadcast(packets)

    def error(self, message):
        print message

class PokerTable:
    def __init__(self, factory, id, description):
        self.factory = factory
        self.game = PokerGameServer("file:poker.%s.xml", factory.dirs)
        self.game.verbose = factory.verbose
        self.observers = []
        self.waiting = []
        game = self.game
        game.id = id
        game.name = description["name"]
        game.setVariant(description["variant"])
        game.setBettingStructure(description["betting_structure"])
        game.setMaxPlayers(int(description["seats"]))
        self.playerTimeout = int(description["timeout"])
        self.transient = description.has_key("transient")
        settings = self.factory.settings
        self.delays = settings.headerGetProperties("/server/delays")[0]
        self.autodeal = settings.headerGet("/server/@autodeal") == "yes"
        self.temporaryPlayersPattern = settings.headerGet("/server/users/@temporary")
        self.history_index = 0
        self.cache = self.createCache()
        self.owner = 0
        self.serial2client = {}
        self.timer_info = {
            "playerTimeout": None,
            "playerTimeoutSerial": 0
            }
        self.timeout_policy = "sitOut"

    def isValid(self):
        return hasattr(self, "factory")
    
    def destroy(self):
        if self.transient:
            self.factory.destroyTable(self.game.id)
            
        self.broadcast(PacketPokerTableDestroy(serial = self.game.id))
        for client in self.serial2client.values() + self.observers:
            del client.tables[self.game.id]
            
        self.factory.deleteTable(self)
        del self.factory

    def getName(self, serial):
        if self.serial2client.has_key(serial):
            name = self.serial2client[serial].getName()
        else:
            name = self.factory.getName(serial)
        return name

    def getPlayerInfo(self, serial):
        if self.serial2client.has_key(serial):
            info = self.serial2client[serial].getPlayerInfo()
        else:
            info = self.factory.getPlayerInfo(serial)
        return info
        
    def createCache(self):
        return { "board": PokerCards(), "pockets": {} }
    
    def beginTurn(self):
        info = self.timer_info
        if info.has_key("dealTimeout"):
            if info["dealTimeout"].active():
                info["dealTimeout"].cancel()
            del info["dealTimeout"]

        if not self.isRunning():
            self.historyReset()
            hand_serial = self.factory.getHandSerial()
            game = self.game
            print "Dealing hand %s/%d" % ( game.name, hand_serial )
            game.setTime(time())
            game.beginTurn(hand_serial)
        
    def historyReset(self):
        self.history_index = 0
        self.cache = self.createCache()

    def cards2packets(self, game_id, board, pockets, cache):
        packets = []
        #
        # If no pockets or board specified (different from empty pockets),
        # ignore and keep the cached values
        #
        if board != None:
            if board != cache["board"]:
                packets.append(PacketPokerBoardCards(game_id = game_id,
                                                     cards = board.tolist(False)))
                cache["board"] = board.copy()

        if pockets != None:
            #
            # Send new pockets or pockets that changed
            #
            for (serial, pocket) in pockets.iteritems():
                if not cache["pockets"].has_key(serial) or cache["pockets"][serial] != pocket:
                    packets.append(PacketPokerPlayerCards(game_id = game_id,
                                                          serial = serial,
                                                          cards = pocket.toRawList()))
                if not cache["pockets"].has_key(serial):
                    cache["pockets"][serial] = pocket.copy()
        return packets

    def broadcast(self, packets):
        game = self.game

        if not type(packets) is ListType:
            packets = ( packets, )
            
        for packet in packets:
            if self.factory.verbose > 1:
                print "broadcast %s " % packet
            for serial in game.serial2player.keys():
                #
                # Player may be in game but disconnected.
                #
                if self.serial2client.has_key(serial):
                    client = self.serial2client[serial]
                    client.sendPacket(self.private2public(packet, serial))
            for client in self.observers:
                client.sendPacket(packet)

    def private2public(self, packet, serial):
        game = self.game
        #
        # Cards private to each player are shown only to the player
        #
        if packet.type == PACKET_POKER_PLAYER_CARDS and packet.serial != serial:
            private = PacketPokerPlayerCards(game_id = packet.game_id,
                                             serial = packet.serial,
                                             cards = PokerCards(packet.cards).tolist(False))
            return private
        else:
            return packet
        
    def history2packets(self, history, game_id, cache):
        game_index = 0
        player_list_index = 7
        packets = []
        for event in history:
            type = event[0]
            if type == "game":
                (type, level, hand_serial, hands_count, time, variant, betting_structure, player_list, dealer, serial2chips) = event
                if len(serial2chips) > 1:
                    chips_values = serial2chips['values']
                    nochips = PokerChips(chips_values).chips
                    for (serial, chips) in serial2chips.iteritems():
                        if serial == 'values':
                            continue
                        packets.append(PacketPokerPlayerChips(game_id = game_id,
                                                              serial = serial,
                                                              bet = nochips,
                                                              money = chips))
                packets.append(PacketPokerInGame(game_id = game_id,
                                                 players = player_list))
                packets.append(PacketPokerDealer(game_id = game_id,
                                                 serial = dealer))
                packets.append(PacketPokerStart(game_id = game_id,
                                                hand_serial = hand_serial,
                                                hands_count = hands_count,
                                                time = time,
                                                level = level))
                
            elif type == "wait_for":
                (type, serial, reason) = event
                packets.append(PacketPokerWaitFor(game_id = game_id,
                                                  serial = serial,
                                                  reason = reason))
                
            elif type == "player_list":
                (type, player_list) = event
                packets.append(PacketPokerInGame(game_id = game_id,
                                                 players = player_list))

            elif type == "round":
                (type, name, board, pockets) = event
                packets.extend(self.cards2packets(game_id, board, pockets, cache))
                packets.append(PacketPokerState(game_id = game_id,
                                                string = name))

            elif type == "position":
                (type, position) = event
                packets.append(PacketPokerPosition(game_id = game_id,
                                                   serial = position))
                
            elif type == "showdown":
                (type, board, pockets) = event
                packets.extend(self.cards2packets(game_id, board, pockets, cache))
                
            elif type == "blind_request":
                (type, serial, amount, dead, is_late) = event
                packets.append(PacketPokerBlindRequest(game_id = game_id,
                                                       serial = serial,
                                                       amount = amount,
                                                       dead = dead,
                                                       is_late = is_late))

            elif type == "wait_blind":
                (type, serial) = event
                pass
                    
            elif type == "blind":
                (type, serial, amount, dead) = event
                packets.append(PacketPokerBlind(game_id = game_id,
                                                serial = serial,
                                                amount = amount,
                                                dead = dead))

            elif type == "ante_request":
                (type, serial, amount) = event
                packets.append(PacketPokerAnteRequest(game_id = game_id,
                                                      serial = serial,
                                                      amount = amount))

            elif type == "ante":
                (type, serial, amount) = event
                packets.append(PacketPokerAnte(game_id = game_id,
                                               serial = serial,
                                               amount = amount))

            elif type == "all-in":
                pass
            
            elif type == "call":
                (type, serial, amount) = event
                packets.append(PacketPokerCall(game_id = game_id,
                                               serial = serial))
                
            elif type == "check":
                (type, serial) = event
                packets.append(PacketPokerCheck(game_id = game_id,
                                                serial = serial))
                
            elif type == "fold":
                (type, serial) = event
                packets.append(PacketPokerFold(game_id = game_id,
                                               serial = serial))

            elif type == "raise":
                (type, serial, amount) = event
                packets.append(PacketPokerRaise(game_id = game_id,
                                                serial = serial,
                                                amount = amount))

            elif type == "canceled":
                (type, serial, amount) = event
                packets.append(PacketPokerCanceled(game_id = game_id,
                                                   serial = serial,
                                                   amount = amount))
                
            elif type == "end":
                (type, winners, winner2share, showdown_stack) = event
                packets.append(PacketPokerState(game_id = game_id,
                                                string = "end"))
                packets.append(PacketPokerWin(game_id = game_id,
                                              serials = winners))

            elif type == "sitOut":
                (type, serial) = event
                packets.append(PacketPokerSitOut(game_id = game_id,
                                                 serial = serial))
                    
            elif type == "leave":
                (type, quitters) = event
                for serial in quitters:
                    packets.append(PacketPokerPlayerLeave(game_id = game_id,
                                                          serial = serial))
                
            elif type == "finish":
                pass
            
            else:
                print "*ERROR* unknown history type %s " % type
        return packets

    def syncDatabase(self):
        game = self.game
        updates = {}
        reset_bet = False
        for event in game.historyGet()[self.history_index:]:
            type = event[0]
            if type == "game":
                pass
            
            elif type == "wait_for":
                pass
            
            elif type == "player_list":
                pass
            
            elif type == "round":
                pass
            
            elif type == "showdown":
                pass
                
            elif type == "position":
                pass
                
            elif type == "blind_request":
                pass
            
            elif type == "wait_blind":
                pass
            
            elif type == "blind":
                (type, serial, amount, dead) = event
                if not updates.has_key(serial):
                    updates[serial] = 0
                updates[serial] -= amount + dead

            elif type == "ante_request":
                pass
            
            elif type == "ante":
                (type, serial, amount) = event
                if not updates.has_key(serial):
                    updates[serial] = 0
                updates[serial] -= amount

            elif type == "all-in":
                pass
            
            elif type == "call":
                (type, serial, amount) = event
                if not updates.has_key(serial):
                    updates[serial] = 0
                updates[serial] -= amount
                
            elif type == "check":
                pass
                
            elif type == "fold":
                pass
            
            elif type == "raise":
                (type, serial, amount) = event
                amount = PokerChips(game.chips_values, amount).toint()
                if not updates.has_key(serial):
                    updates[serial] = 0
                updates[serial] -= amount

            elif type == "canceled":
                (type, serial, amount) = event
                if serial > 0 and amount > 0:
                    if not updates.has_key(serial):
                        updates[serial] = 0
                    updates[serial] += amount
                
            elif type == "end":
                (type, winners, winner2share, showdown_stack) = event
                for (serial, share) in winner2share.iteritems():
                    if not updates.has_key(serial):
                        updates[serial] = 0
                    updates[serial] += share
                reset_bet = True
                if len(showdown_stack) > 0:
                    self.factory.setRating(winners, showdown_stack[0]['player_list']);

            elif type == "sitOut":
                pass

            elif type == "leave":
                pass
            
            elif type == "finish":
                (type, hand_serial) = event
                self.factory.saveHand(self.compressedHistory(game.historyGet()), hand_serial)
            
            else:
                print "*ERROR* unknown history type %s " % type

        for (serial, amount) in updates.iteritems():
            self.factory.updatePlayerMoney(serial, game.id, amount)

        if reset_bet:
            self.factory.resetBet(game.id)
        elif hasattr(self, "factory") and self.factory.verbose > 2:
            (money, bet) = self.factory.tableMoneyAndBet(game.id)
            if bet and game.potAndBetsAmount() != bet:
                print "table %d bet mismatch %d in memory versus %d in database" % ( game.id, game.potAndBetsAmount(), bet)

    def historyReduce(self):
        game = self.game
        if self.history_index < len(game.historyGet()):
            game.historyReduce()
            self.history_index = len(game.historyGet())
            
    def compressedHistory(self, history):
        new_history = []
        cached_pockets = None
        cached_board = None
        for event in history:
            type = event[0]
            if ( type == "all-in" or
                 type == "wait_for" ) :
                pass
            
            elif type == "game":
                new_history.append(event)
                
            elif type == "round":
                (type, name, board, pockets) = event

                if pockets != cached_pockets:
                    cached_pockets = pockets
                else:
                    pockets = None

                if board != cached_board:
                    cached_board = board
                else:
                    board = None

                new_history.append((type, name, board, pockets))

            elif type == "showdown":
                (type, board, pockets) = event
                if pockets != cached_pockets:
                    cached_pockets = pockets
                else:
                    pockets = None

                if board != cached_board:
                    cached_board = board
                else:
                    board = None

                new_history.append((type, board, pockets))
            
            elif ( type == "call" or
                   type == "check" or
                   type == "fold" or
                   type == "raise" or
                   type == "canceled" or
                   type == "position" or
                   type == "blind" or
                   type == "ante" or
                   type == "player_list" ):
                new_history.append(event)

            elif type == "end":
                (type, winners, winner2share, showdown_stack) = event
                new_history.append((type, winners, winner2share, []))

            elif type == "sitOut":
                new_history.append(event)
                    
            elif type == "leave":
                pass
                
            elif type == "finish":
                pass
            
            else:
                print "*ERROR* unknown history type %s " % type

        return new_history

    def gameResolve2messages(self, hands, serial2name, serial2displayed, frame):
        def card2string(hand):
            eval = self.game.eval
            return join(eval.card2string(hand[1][1:]))
        
        messages = []
        best = { 'hi': 0,
                 'low': 0x0FFFFFFF }
        for serial in frame['serials']:
            for side in ('hi', 'low'):
                hand = hands[serial]
                if not hand.has_key(side):
                    continue
                if hand[side][1][0] == 'Nothing':
                    continue

                hand = hand[side]
                show = False
                if ( ( side == 'hi' and best['hi'] <= hand[0] ) or
                     ( side == 'low' and best['low'] >= hand[0] ) ):
                    best[side] = hand[0]
                    show = True

                if serial2displayed.has_key(serial) and not serial in frame[side]:
                    #
                    # If the player already exposed the hand and is not going
                    # to win this side of the pot, there is no need to issue
                    # a message.
                    #
                    continue
                
                if show:
                    serial2displayed[serial] = True
                    value = self.game.readableHandValueLong(side, hand[1][0], hand[1][1:])
                    messages.append("%s shows %s for %s " % ( serial2name[serial], value, side ))
                else:
                    messages.append("%s mucks loosing hand" % ( serial2name[serial] ))

        for side in ('hi', 'low'):
            if not frame.has_key(side):
                continue
            message = join([ serial2name[serial] for serial in frame[side] ])
            if len(frame[side]) > 1:
                message += " tie for %s " % side
            else:
                message += " wins %s " % side
            messages.append(message)

        if len(frame['serial2share']) > 1:
            message = "winners share a pot of %d" % frame['pot']
            if frame.has_key('chips_left'):
                message += " (minus %d odd chips)" % frame['chips_left']
            messages.append(message)
            
        for (serial, share) in frame['serial2share'].iteritems():
            messages.append("%s receives %d" % ( serial2name[serial], share ))

        return messages
            
    def game2messages(self):
        game = self.game
        messages = []
        subject = ''
        for event in game.historyGet()[self.history_index:]:
            type = event[0]
            if type == "game":
                (type, level, hand_serial, hands_count, time, variant, betting_structure, player_list, dealer, serial2chips) = event
                subject = "hand #%d, %s, %s" % ( hand_serial, variant, betting_structure )
                
            elif type == "wait_for":
                (type, serial, reason) = event
                messages.append("%s waiting for " % self.getName(serial))
                messages.append("%s" % ( reason == "late" and "late blind" or "big blind"))
            
            elif type == "player_list":
                pass
            
            elif type == "round":
                (type, name, board, pockets) = event
                messages.append("%s, %d players" % ( name, len(pockets) ))
                if not board.isEmpty():
                    messages.append("Board: %s" % game.cards2string(board))

            elif type == "showdown":
                (type, board, pockets) = event
                
            elif type == "position":
                pass
            
            elif type == "blind_request":
                pass
            
            elif type == "wait_blind":
                pass
            
            elif type == "blind":
                (type, serial, amount, dead) = event
                if dead > 0:
                    dead_message = " and %d dead man" % dead
                else:
                    dead_message = ""
                messages.append("%s pays %d blind%s" % ( self.getName(serial), amount, dead_message ))

            elif type == "ante_request":
                pass
            
            elif type == "ante":
                (type, serial, amount) = event
                messages.append("%s pays %d ante" % ( self.getName(serial), amount ))

            elif type == "all-in":
                (type, serial) = event
                messages.append("%s is all in" % self.getName(serial))
            
            elif type == "call":
                (type, serial, amount) = event
                messages.append("%s calls %d" % ( self.getName(serial), amount ))
                
            elif type == "check":
                (type, serial) = event
                messages.append("%s checks" % self.getName(serial))
                
            elif type == "fold":
                (type, serial) = event
                messages.append("%s folds" % self.getName(serial))

            elif type == "raise":
                (type, serial, amount) = event
                amount = PokerChips(game.chips_values, amount).toint()
                messages.append("%s raise %d" % ( self.getName(serial), amount ) )

            elif type == "canceled":
                (type, serial, amount) = event
                if serial > 0 and amount > 0:
                    returned_message = " (%d returned to %s)" % ( amount, self.getName(serial) )
                else:
                    returned_message = ""
                messages.append("turn canceled%s" % returned_message)
                
            elif type == "end":
                (type, winners, winner2share, showdown_stack) = event
                if not showdown_stack[0].has_key('serial2best'):
                    serial = winners[0]
                    messages.append("%s receives %d (everyone else folded)" % ( self.getName(serial), winner2share[serial] ))
                else:
                    serial2displayed = {}
                    hands = showdown_stack[0]['serial2best']
                    serial2name = dict(zip(hands.keys(), [ self.getName(serial) for serial in hands.keys() ]))
                    for frame in showdown_stack[1:]:
                        message = None
                        if frame['type'] == 'left_over':
                            message = "%s receives %d odd chips" % ( self.getName(frame['serial']), frame['chips_left'])
                        elif frame['type'] == 'uncalled':
                            message = "returning uncalled bet %d to %s" % ( frame['uncalled'], serial2name[frame['serial']] )
                        elif frame['type'] == 'resolve':
                            messages.extend(self.gameResolve2messages(hands, serial2name, serial2displayed, frame))
                        else:
                            print "*ERROR* unexpected showdown_stack frame type %s" % frame['type']
                        if message:
                            messages.append(message)
            elif type == "sitOut":
                (type, serial) = event
                messages.append("%s sits out" % ( self.getName(serial) ))

            elif type == "leave":
                pass

            elif type == "finish":
                pass
            
            else:
                print "*ERROR* unknown history type %s " % type

        return (subject, messages)

    def syncChat(self):
        (subject, messages) = self.game2messages()
        if messages or subject:
            if self.factory.jabber:
                self.factory.jabberMessage(self.game.name, subject, messages)
            if self.factory.chat:
                if messages:
                    message = "".join(map(lambda line: "Dealer: " + line + "\n", messages))
                    self.broadcast(PacketPokerChat(game_id = self.game.id,
                                                   serial = 0,
                                                   message = message))

    def syncJabber(self):
        (subject, messages) = self.game2messages()
        if messages or subject:
            self.factory.jabberMessage(self.game.name, subject, messages)

    def delayedActions(self):
        game = self.game
        for event in game.historyGet()[self.history_index:]:
            type = event[0]
            if type == "leave":
                (type, quitters) = event
                for serial in quitters:
                    if not self.transient:
                        self.factory.leavePlayer(serial, game.id)
                    if self.serial2client.has_key(serial):
                        self.seated2observer(self.serial2client[serial])

    def autoDeal(self):
        self.beginTurn()
        self.update()
        
    def scheduleAutoDeal(self):
        info = self.timer_info
        if info.has_key("dealTimeout") and info["dealTimeout"].active():
            info["dealTimeout"].cancel()
        if self.factory.shutting_down:
            return
        if not self.autodeal:
            if self.factory.verbose > 2:
                print "No autodeal"
            return
        if self.transient:
            if self.factory.verbose > 2:
                print "No autodeal on transient tables"
            return
        if self.isRunning():
            if self.factory.verbose > 2:
                print "Not autodealing because game is running"
            return
        game = self.game
        if game.sitCount() < 2:
            if self.factory.verbose > 2:
                print "Not autodealing because less than 2 players willing to play"
            return
        #
        # Do not auto deal a table where there are only temporary
        # users (i.e. bots)
        #
        onlyTemporaryPlayers = True
        for serial in game.serialsSit():
            if not match("^" + self.temporaryPlayersPattern, self.getName(serial)):
                onlyTemporaryPlayers = False
                break
        if onlyTemporaryPlayers:
            if self.factory.verbose > 2:
                print "Not autodealing because player names sit in match %s" % self.temporaryPlayersPattern
            return
        if self.factory.verbose > 2:
            print "Autodeal scheduled in %f seconds" % float(self.delays["autodeal"])
        info["dealTimeout"] = reactor.callLater(float(self.delays["autodeal"]), self.autoDeal)
        
    def update(self):
        if not self.isValid():
            return
        
        self.updateTimers()
        game = self.game
        packets = self.history2packets(game.historyGet()[self.history_index:], game.id, self.cache);
        self.syncDatabase()
        self.syncChat()
        self.delayedActions()
        if len(packets) > 0:
            self.broadcast(packets)
        self.historyReduce()
        self.scheduleAutoDeal()

    def handReplay(self, client, hand):
        history = self.factory.loadHand(hand)
        if not history:
            return
        print "handReplay"
        pprint(history)
        (type, level, hand_serial, hands_count, time, variant, betting_structure, player_list, dealer, serial2chips) = history[0]
        game = self.game
        for serial in game.serialsAll():
            client.sendPacketVerbose(PacketPokerPlayerLeave(game_id = game.id,
                                                            serial = serial))
        game.reset()
        game.name = "*REPLAY*"
        game.setVariant(variant)
        game.setBettingStructure(betting_structure)
        game.setTime(time)
        game.setHandsCount(hands_count)
        game.setLevel(level)
        game.hand_serial = hand
        for serial in player_list:
            game.addPlayer(serial)
            game.getPlayer(serial).money.chips = serial2chips[serial]
            game.sit(serial)
        if self.isJoined(client):
            client.join(self)
        else:
            self.joinPlayer(client, client.getSerial())
        serial = client.getSerial()
        cache = self.createCache()
        for packet in self.history2packets(history, game.id, cache):
            if packet.type == PACKET_POKER_PLAYER_CARDS and packet.serial == serial:
                packet.cards = cache["pockets"][serial].toRawList()
            if packet.type == PACKET_POKER_PLAYER_LEAVE:
                continue
            client.sendPacketVerbose(packet)
        
    def isJoined(self, client):
        return client in self.observers or self.serial2client.has_key(client.getSerial())

    def isSeated(self, client):
        return self.isJoined(client) and self.game.isSeated(client.getSerial())

    def isSit(self, client):
        return self.isSeated(client) and self.game.isSit(client.getSerial())

    def isSerialObserver(self, serial):
        return serial in [ client.getSerial() for client in self.observers ]
    
    def isOpen(self):
        return self.game.is_open

    def isRunning(self):
        return self.game.isRunning()

    def seated2observer(self, client):
        del self.serial2client[client.getSerial()]
        self.observers.append(client)

    def observer2seated(self, client):
        self.observers.remove(client)
        self.serial2client[client.getSerial()] = client
        
    def quitPlayer(self, client, serial):
        game = self.game
        if self.isSit(client):
            if self.isOpen():
                game.sitOutNextTurn(serial)
            game.autoPlayer(serial)
        self.update()
        if self.isSeated(client):
            #
            # If not on a closed table, stand up
            #
            if self.isOpen():
                if client.removePlayer(self, serial):
                    self.factory.leavePlayer(serial, game.id)
                    self.seated2observer(client)
                else:
                    self.update()
            else:
                client.error("cannot quit a closed table")
                return False

        if self.isJoined(client):
            #
            # The player is no longer connected to the table
            #
            self.destroyPlayer(client, serial)

        return True

    def kickPlayer(self, client, serial):
        game = self.game

        if self.isSeated(client):
            if client.removePlayer(self, serial):
                self.factory.leavePlayer(serial, game.id)
                self.seated2observer(client)
            else:
                self.update()

        return True

    def disconnectPlayer(self, client, serial):
        game = self.game

        if self.isSeated(client):
            if self.isOpen():
                #
                # If not on a closed table, stand up.
                #
                if client.removePlayer(self, serial):
                    self.factory.leavePlayer(serial, game.id)
                    self.seated2observer(client)
                else:
                    self.update()
            else:
                #
                # If on a closed table, the player
                # will stay at the table, he does not
                # have the option to leave.
                #
                pass
                
        if self.isJoined(client):
            #
            # The player is no longer connected to the table
            #
            self.destroyPlayer(client, serial)

        return True

    def leavePlayer(self, client, serial):
        game = self.game
        if self.isSit(client):
            if self.isOpen():
                game.sitOutNextTurn(serial)
            game.autoPlayer(serial)
        self.update()
        if self.isSeated(client):
            #
            # If not on a closed table, stand up
            #
            if self.isOpen():
                if client.removePlayer(self, serial):
                    self.factory.leavePlayer(serial, game.id)
                    self.seated2observer(client)
                else:
                    self.update()
            else:
                client.error("cannot leave a closed table")
                return False

        return True
        
    def movePlayer(self, client, serial, to_game_id):
        game = self.game
        #
        # We are safe because called from within the server under
        # controlled circumstances.
        #

        money = game.serial2player[serial].money

        client.movePlayerFrom(self, to_game_id)
        self.destroyPlayer(client, serial)

        other_table = self.factory.getTable(to_game_id)
        other_table.serial2client[serial] = client

        money_check = self.factory.movePlayer(serial, game.id, to_game_id)
        if money_check != money.toint():
            print " *ERROR* movePlayer: player %d money %d in database, %d in memory" % ( serial, money_check, money.toint() )
        
        client.movePlayerTo(other_table, money)
        if self.factory.verbose:
            print "player %d moved from table %d to table %d" % ( serial, game.id, to_game_id )

    def joinPlayer(self, client, serial):
        game = self.game
        #
        # Silently do nothing if already joined
        #
        if self.isJoined(client):
            return True

        if len(client.tables) >= self.factory.simultaneous:
            if self.factory.verbose:
                print " *ERROR* joinPlayer: %d seated at %d tables (max %d)" % ( serial, len(client.tables), self.factory.simultaneous )
            return False
        
        #
        # Player is now an observer, unless he is seated
        # at the table.
        #
        client.join(self)
        if not self.game.isSeated(client.getSerial()):
            self.observers.append(client)
        else:
            self.serial2client[serial] = client
        #
        # If it turns out that the player is seated
        # at the table already, presumably because he
        # was previously disconnected from a tournament
        # or an ongoing game.
        #
        if self.isSeated(client):
            #
            # Sit back immediately, as if we just seated
            #
            game.comeBack(serial)
            
        return True

    def seatPlayer(self, client, serial, seat):
        game = self.game
        if not self.isJoined(client):
            client.error("player %d can't seat before joining" % serial)
            return False
        #
        # Do nothing if already seated
        #
        if self.isSeated(client):
            print "player %d is already seated" % serial
            return False

        if not game.canAddPlayer(serial):
            client.error("table refuses to seat player %d" % serial)
            return False

        amount = 0
        if self.transient:
            amount = game.buyIn()
            
        if not self.factory.seatPlayer(serial, game.id, amount):
            return False

        self.observer2seated(client)

        client.addPlayer(self, seat)
        if amount > 0:
            client.setMoney(self, amount)

        # inviting seated player to the jabber room
        self.factory.jabberInvitePlayer(game.name, self.getName(serial))
        
        return True

    def sitOutPlayer(self, client, serial):
        game = self.game
        if not self.isSeated(client):
            client.error("player %d can't sit out before getting a seat" % serial)
            return False
        #
        # Silently do nothing if already sit out
        #
        if not self.isSit(client):
            return True

        client.sitOutPlayer(self, serial)
        return True

    def chatPlayer(self, client, serial, message):
        self.broadcast(PacketPokerChat(game_id = self.game.id,
                                       serial = serial,
                                       message = message + "\n"))

    def autoBlindAnte(self, client, serial, auto):
        game = self.game
        if not self.isSeated(client):
            client.error("player %d can't set auto blind/ante before getting a seat" % serial)
            return False
        client.autoBlindAnte(self, serial, auto)
        
    def sitPlayer(self, client, serial):
        game = self.game
        if not self.isSeated(client):
            client.error("player %d can't sit before getting a seat" % serial)
            return False

        client.sitPlayer(self, serial)
        return True
        
    def destroyPlayer(self, client, serial):
        game = self.game
        if serial:
            self.factory.jabberKickPlayer(game.name, self.getName(serial))
        if client in self.observers:
            self.observers.remove(client)
        else:
            del self.serial2client[serial]
        del client.tables[self.game.id]
        #
        # If there is no-one left and the table is transient
        # and the owner is no longer watching the table,
        # get rid of it. 
        #
        if ( self.transient and
             len(self.serial2client.keys()) == 0 and
             not self.isSerialObserver(self.owner) ):
            self.destroy()

    def buyInPlayer(self, client, amount):
        game = self.game
        if not self.isSeated(client):
            client.error("player %d can't bring money to a table before getting a seat" % client.getSerial())
            return False

        if client.getSerial() in game.serialsPlaying():
            client.error("player %d can't bring money while participating in a hand" % client.getSerial())
            return False

        if self.transient:
            client.error("player %d can't bring money to a transient table" % client.getSerial())
            return False

        amount = self.factory.buyInPlayer(client.getSerial(), game.id, max(amount, game.buyIn()))
        client.setMoney(self, amount)
        return True
        
    def playerWarningTimer(self, serial):
        game = self.game
        info = self.timer_info
        if game.isRunning() and serial == game.getSerialInPosition():
            timeout = self.playerTimeout / 2;
            self.broadcast(PacketPokerTimeoutWarning(game_id = game.id,
                                                     serial = serial,
                                                     timeout = timeout))
            self.factory.jabberMessage(game.name, '', [ "%s has %d seconds to respond" % ( self.getName(serial), timeout ) ])
            info["playerTimeout"] = reactor.callLater(timeout, self.playerTimeoutTimer, serial)
        else:
            self.updateTimers()

    def playerTimeoutTimer(self, serial):
        if self.factory.verbose:
            print "player %d times out" % serial
        game = self.game
        if game.isRunning() and serial == game.getSerialInPosition():
            if self.timeout_policy == "sitOut":
                game.sitOutNextTurn(serial)
                game.autoPlayer(serial)
            elif self.timeout_policy == "fold":
                game.autoPlayer(serial)
                self.broadcast(PacketPokerAutoFold(game_id = game.id,
                                                   serial = serial))
            else:
                print " *ERROR* unknown timeout_policy %s" % self.timeout_policy
            self.broadcast(PacketPokerTimeoutNotice(game_id = game.id,
                                                    serial = serial))
            self.factory.jabberMessage(game.name, '', [ "%s did not act in time" % self.getName(serial) ])
            self.update()
        else:
            self.updateTimers()
        
    def cancelTimers(self):
        info = self.timer_info

        timer = info["playerTimeout"]
        if timer != None:
            if timer.active():
                timer.cancel()
            info["playerTimeout"] = None
        info["playerTimeoutSerial"] = 0
        
    def updateTimers(self):
        game = self.game
        info = self.timer_info

        timer = info["playerTimeout"]
        if game.isRunning():
            serial = game.getSerialInPosition()
            #
            # Any event in the game resets the player timeout
            #
            if ( info["playerTimeoutSerial"] != serial or
                 len(game.historyGet()) > self.history_index ):
                if timer != None and timer.active():
                    timer.cancel()

                timer = reactor.callLater(self.playerTimeout / 2, self.playerWarningTimer, serial)
                info["playerTimeout"] = timer
                info["playerTimeoutSerial"] = serial
        else:
            #
            # If the game is not running, cancel the previous timeout
            #
            self.cancelTimers()
        
class PokerServerFactory(UGAMEServerFactory):
    """server factory"""

    def __init__(self, settings):
        database = settings.headerGetProperties("/server/database")[0]
        UGAMEServerFactory.__init__(self, database = database)
        self.jabber = None
        self.protocol = PokerServer
        self.settings = settings
        self.dirs = split(settings.headerGet("/server/path"))
        self.tables = []
        self.groups = {}
        self.table_serial = 100
        self.shutting_down = False
        self.simultaneous = self.settings.headerGetInt("/server/@simultaneous")
        self.verbose = self.settings.headerGetInt("/server/@verbose")
        self.chat = self.settings.headerGet("/server/@chat") == "yes"
        for description in self.settings.headerGetProperties("/server/table"):
            self.createTable(0, description)
        self.authSetLevel(PACKET_POKER_SEAT, User.REGULAR)
        self.authSetLevel(PACKET_POKER_HAND_SELECT_ALL, User.ADMIN)

    def shutdown(self):
        self.shutting_down = True
        self.shutdown_deferred = defer.Deferred()
        reactor.callLater(0, self.shutdownCheck)
        return self.shutdown_deferred
        
    def shutdownCheck(self):
        playing = 0
        for table in self.tables:
            if table.game.isRunning():
                playing += 1
        if self.verbose and playing > 0:
            print "Shutting down %d games still running" % playing
        if playing <= 0:
            if self.verbose:
                print "Shutdown immediately"
            self.shutdown_deferred.callback(True)
        else:
            reactor.callLater(10, self.shutdownCheck)
        
    def isShuttingDown(self):
        return self.shutting_down
    
    def stopFactory(self):
        pass
        
    def jabberMessage(self, chat, subject, messages):
        if self.jabber:
            self.jabber.message(chat, subject, messages)
            
    def jabberJoinChat(self, name):
        if self.jabber:
            self.jabber.joinChat(name)

    def jabberInvitePlayer(self, chat, name):
        if self.jabber and name != "anonymous":
            self.jabber.inviteChat(chat, name)
            
    def jabberKickPlayer(self, chat, name):
        if self.jabber and name != "anonymous":
            self.jabber.kickChat(chat, name)

    def setJabber(self, jabber):
        self.jabber = jabber
        for table in self.tables:
            self.jabberJoinChat(table.game.name)

    def unsetJabber(self):
        self.jabber = None

    def getHandSerial(self):
        cursor = self.db.cursor()
        cursor.execute("insert into hands (description) values ('[]')")
        #
        # Accomodate for MySQLdb versions < 1.1
        #
        if hasattr(cursor, "lastrowid"):
            serial = cursor.lastrowid
        else:
            serial = cursor.insert_id()
        cursor.close()
        return int(serial)

    def loadHand(self, serial):
        cursor = self.db.cursor()
        sql = ( "select description from hands where serial = " + str(serial) )
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* loadHand(%d) expected one row got %d" % ( serial, cursor.rowcount )
            cursor.close()            
            return None
        (description,) = cursor.fetchone()
        cursor.close()
        try:
            history = eval(description.replace("\r",""))
            return history
        except:
            print " *ERROR* loadHand(%d) eval failed for %s" % ( serial, description )
            print_exc()
            return None
            
    def saveHand(self, description, hand_serial):
        (type, level, hand_serial, hands_count, time, variant, betting_structure, player_list, dealer, serial2chips) = description[0]
        cursor = self.db.cursor()

        sql = ( "update hands set " + 
                " description = %s "
                " where serial = " + str(hand_serial) )
        if self.verbose > 1:
            print "saveHand: %s" % sql
        cursor.execute(sql, pformat(description))
        if cursor.rowcount != 1 and cursor.rowcount != 0:
            print " *ERROR* modified %d rows (expected 1 or 0): %s " % ( cursor.rowcount, sql )
            cursor.close()
            return

        sql = "insert into user2hand values "
        sql += ", ".join(map(lambda player_serial: "(%d, %d)" % ( player_serial, hand_serial ), player_list))
        if self.verbose > 1:
            print "saveHand: %s" % sql
        cursor.execute(sql)
        if cursor.rowcount != len(player_list):
            print " *ERROR* inserted %d rows (expected exactly %d): %s " % ( cursor.rowcount, len(player_list), sql )

       
        cursor.close()
        
    def listHands(self, sql):
        cursor = self.db.cursor()
        if self.verbose > 1:
            print "listHands: %s" % sql
        cursor.execute(sql)
        hands = cursor.fetchmany()
        cursor.close()
        return map(lambda x: x[0], hands)

    def listTables(self, select, serial):
        tables = []
        for table in self.tables:
            game = table.game
            if select == "my":
                if serial in game.serialsAll():
                    tables.append(table)
            else:
                tables.append(table)
        return tables

    def cleanUp(self, temporary_users = ''):
        cursor = self.db.cursor()
        sql = "delete from user2table"
        cursor.execute(sql)
        cursor.close()

        if temporary_users:
            cursor = self.db.cursor()
            sql = "delete from users where name like '" + temporary_users + "%'"
            cursor.execute(sql)
            cursor.close()
        
    def getMoney(self, serial, base):
        cursor = self.db.cursor()
        sql = ( "select " + base + "_money from users where serial = " + str(serial) )
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* getMoney(%d) expected one row got %d" % ( serial, cursor.rowcount )
            cursor.close()            
            return 0
        (money,) = cursor.fetchone()
        cursor.close()
        money = int(money)
        if money < 0:
            print " *ERROR* getMoney(%d) found %d" % ( serial, money)
            money = 0
        return money

    def getPlayerInfo(self, serial):
        placeholder = PacketPokerPlayerInfo(serial = serial,
                                            name = "anonymous",
                                            outfit = "default")
        if serial == 0:
            return placeholder
        
        cursor = self.db.cursor()
        sql = ( "select name,outfit from users where serial = " + str(serial) )
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* getPlayerInfo(%d) expected one row got %d" % ( serial, cursor.rowcount )
            return placeholder
        (name,outfit) = cursor.fetchone()
        cursor.close()
        return PacketPokerPlayerInfo(serial = serial,
                                     name = name,
                                     outfit = outfit)

    def getUserInfo(self, serial):
        cursor = self.db.cursor()
        sql = ( "select play_money,point_money,rating from users where serial = " + str(serial) )
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* getUserInfo(%d) expected one row got %d" % ( serial, cursor.rowcount )
            return PacketPokerUserInfo(serial = serial)
        (play_money,point_money,rating) = cursor.fetchone()
        cursor.close()
        return PacketPokerUserInfo(serial = serial,
                                   play_money = play_money,
                                   point_money = point_money,
                                   rating = rating)

    def setPlayerInfo(self, player_info):
        cursor = self.db.cursor()
        sql = ( "update users set "
                " name = '" + player_info.name + "', "
                " outfit = '" + player_info.outfit + "' "
                " where serial = " + str(player_info.serial) )
        if self.verbose > 1:
            print "setPlayerInfo: %s" % sql
        cursor.execute(sql)
        if cursor.rowcount != 1 and cursor.rowcount != 0:
            print " *ERROR* modified %d rows (expected 1 or 0): %s " % ( cursor.rowcount, sql )
        
        
    def getName(self, serial):
        if serial == 0:
            return "anonymous"
        
        cursor = self.db.cursor()
        sql = ( "select name from users where serial = " + str(serial) )
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* getName(%d) expected one row got %d" % ( serial, cursor.rowcount )
            return "ERROR"
        (name,) = cursor.fetchone()
        cursor.close()
        return name

    def buyInPlayer(self, serial, table_id, amount):
        withdraw = min(self.getMoney(serial, "play"), amount)
        cursor = self.db.cursor()
        sql = ( "update users,user2table set "
                " user2table.money = " + str(withdraw) + ", "
                " users.play_money = users.play_money - " + str(withdraw) + " "
                " where users.serial = " + str(serial) + " and "
                "       user2table.user_serial = " + str(serial) + " and "
                "       user2table.table_serial = " + str(table_id) )
        if self.verbose > 1:
            print "buyInPlayer: %s" % sql
        cursor.execute(sql)
        if cursor.rowcount != 0 and cursor.rowcount != 2:
            print " *ERROR* modified %d rows (expected 0 or 2): %s " % ( cursor.rowcount, sql )
        return withdraw

    def seatPlayer(self, serial, table_id, amount):
        status = True
        cursor = self.db.cursor()
        sql = ( "insert user2table ( user_serial, table_serial, money) values "
                " ( " + str(serial) + ", " + str(table_id) + ", " + str(amount) + " )" )
        if self.verbose > 1:
            print "seatPlayer: %s" % sql
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* modified %d rows (expected 1): %s " % ( cursor.rowcount, sql )
            status = False
        cursor.close()
        return status

    def movePlayer(self, serial, from_table_id, to_table_id):
        money = -1
        cursor = self.db.cursor()
        sql = ( "select money from user2table "
                "  where user_serial = " + str(serial) + " and"
                "        table_serial = " + str(from_table_id) )
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* movePlayer(%d) expected one row got %d" % ( serial, cursor.rowcount )
        (money,) = cursor.fetchone()
        cursor.close()

        if money > 0:
            cursor = self.db.cursor()
            sql = ( "update user2table "
                    "  set table_serial = " + str(to_table_id) +
                    "  where user_serial = " + str(serial) + " and"
                    "        table_serial = " + str(from_table_id) )
            if self.verbose > 1:
                print "movePlayer: %s" % sql
            cursor.execute(sql)
            if cursor.rowcount != 1:
                print " *ERROR* modified %d rows (expected 1): %s " % ( cursor.rowcount, sql )
                money = -1
            cursor.close()

        # HACK CHECK
#        cursor = self.db.cursor()
#        sql = ( "select sum(money), sum(bet) from user2table" )
#        cursor.execute(sql)
#        (total_money,bet) = cursor.fetchone()
#        if total_money + bet != 120000:
#            print "BUG(6) %d" % (total_money + bet)
#            os.abort()
#        cursor.close()
        # END HACK CHECK
        
        return money
        
    def leavePlayer(self, serial, table_id):
        status = True
        cursor = self.db.cursor()
        sql = ( "update users,user2table set "
                " users.play_money = users.play_money + user2table.money "
                " where users.serial = " + str(serial) + " and "
                "       user2table.user_serial = " + str(serial) + " and "
                "       user2table.table_serial = " + str(table_id) )
        if self.verbose > 1:
            print "leavePlayer %s" % sql
        cursor.execute(sql)
        if cursor.rowcount > 1:
            print " *ERROR* modified %d rows (expected 0 or 1): %s " % ( cursor.rowcount, sql )
            status = False
        sql = ( "delete from user2table "
                " where user_serial = " + str(serial) + " and "
                "       table_serial = " + str(table_id) )
        if self.verbose > 1:
            print "leavePlayer %s" % sql
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* modified %d rows (expected 1): %s " % ( cursor.rowcount, sql )
        cursor.close()
        return status

    def updatePlayerMoney(self, serial, table_id, amount):
        status = True
        cursor = self.db.cursor()
        sql = ( "update user2table set "
                " money = money + " + str(amount) + ", "
                " bet = bet - " + str(amount) +
                " where user_serial = " + str(serial) + " and "
                "       table_serial = " + str(table_id) )
        if self.verbose > 1:
            print "updatePlayerMoney: %s" % sql
        cursor.execute(sql)
        if cursor.rowcount != 1:
            print " *ERROR* modified %d rows (expected 1): %s " % ( cursor.rowcount, sql )
            status = False
        cursor.close()
        
#         # HACK CHECK
#         cursor = self.db.cursor()
#         sql = ( "select sum(money), sum(bet) from user2table" )
#         cursor.execute(sql)
#         (money,bet) = cursor.fetchone()
#         if money + bet != 120000:
#             print "BUG(4) %d" % (money + bet)
#             os.abort()
#         cursor.close()

#         cursor = self.db.cursor()
#         sql = ( "select user_serial,table_serial,money from user2table where money < 0" )
#         cursor.execute(sql)
#         if cursor.rowcount >= 1:
#             (user_serial, table_serial, money) = cursor.fetchone()
#             print "BUG(11) %d/%d/%d" % (user_serial, table_serial, money)
#             os.abort()
#         cursor.close()
#         # END HACK CHECK
        
        return status

    def tableMoneyAndBet(self, table_id):
        cursor = self.db.cursor()
        sql = ( "select sum(money), sum(bet) from user2table where table_serial = " + str(table_id) )
        cursor.execute(sql)
        status = cursor.fetchone()
        cursor.close()
        return  status
        
    def destroyTable(self, table_id):

#         # HACK CHECK
#         cursor = self.db.cursor()
#         sql = ( "select * from user2table where money != 0 and bet != 0 and table_serial = " + str(table_id) )
#         cursor.execute(sql)
#         if cursor.rowcount != 0:
#             print "BUG(10)"
#             os.abort()
#         cursor.close()
#         # END HACK CHECK
        
        cursor = self.db.cursor()
        sql = ( "delete from user2table "
                "  where table_serial = " + str(table_id) )
        if self.verbose > 1:
            print "destroy: %s" % sql
        cursor.execute(sql)

    def setRating(self, winners, serials):
        url = self.settings.headerGet("/server/@rating")
        if url == "":
            return
        
        params = []
        for first in range(0, len(serials) - 1):
            for second in range(first + 1, len(serials)):
                first_wins = serials[first] in winners
                second_wins = serials[second] in winners
                if first_wins or second_wins:
                    param = "a=%d&b=%d&c=" % ( serials[first], serials[second] )
                    if first_wins and second_wins:
                        param += "2"
                    elif first_wins:
                        param += "0"
                    else:
                        param += "1"
                    params.append(param)

        params = join(params, '&')
        if self.verbose > 2:
            print "setRating: url = %s" % url + params
        content = loadURL(url + params)
        if self.verbose > 2:
            print "setRating: %s" % content
        
    def resetBet(self, table_id):
        status = True
        cursor = self.db.cursor()
        sql = ( "update user2table set bet = 0 "
                " where table_serial = " + str(table_id) )
        if self.verbose > 1:
            print "resetBet: %s" % sql
        cursor.execute(sql)
        cursor.close()

#         # HACK CHECK
#         cursor = self.db.cursor()
#         sql = ( "select sum(money), sum(bet) from user2table" )
#         cursor.execute(sql)
#         (money,bet) = cursor.fetchone()
#         if money + bet != 120000:
#             print "BUG(2) %d" % (money + bet)
#             os.abort()
#         cursor.close()
#         # END HACK CHECK
        
        return status
        
    def getTable(self, game_id):
        for table in self.tables:
            if game_id == table.game.id:
                game = table.game
                return table
        return False

    def createTable(self, owner, description):
        #
        # Do not create two tables by the same name
        #
        if filter(lambda table: table.game.name == description["name"], self.tables):
            return False
        
        id = self.table_serial
        table = PokerTable(self, id, description)
        table.owner = owner
        self.tables.append(table)
        self.table_serial += 1
        self.jabberJoinChat(table.game.name)

        if self.verbose:
            print "table created : %s" % table.game.name

        return table

    def deleteTable(self, table):
        if self.verbose:
            print "table %s/%d removed from server" % ( table.game.name, table.game.id )
        self.tables.remove(table)
        for groups in self.groups.itervalues():
            if table in groups:
                groups.remove(table)

    def setTableGroup(self, owner, table_ids):
        tables = []
        for table in self.tables:
            if table.game.id in table_ids:
                if table.owner != owner:
                    print "*ERROR* player %d attempted to balance table %d but is not the owner" % ( owner, table.game.id )
                    return 0
                tables.append(table)

        serial = self.table_serial
        self.groups[serial] = tables
        self.table_serial += 1
        return serial

    def unsetTableGroup(self, owner, group_id):
        if not group_id in self.groups.keys():
            print "*ERROR* not group id %d" % group_id
            return 0
        
        for table in self.groups[group_id]:
            if table.owner != owner:
                print "*ERROR* player %d attempted to unset table group %d but is not the owner of the tables it contains" % ( owner, group_id )
                return 0
        del self.groups[group_id]
        return group_id
        
    def balance(self, owner, group_id):
        for table in self.groups[group_id]:
            if table.owner != owner:
                print "*ERROR* player %d attempted to balance table group %d but is not the owner of all the tables it contains" % ( owner, group_id )
                return 0

        balance_packet = PacketPokerTableGroupBalance(serial = group_id)
        tables = self.groups[group_id]

        games = [ table.game for table in tables ]
        id2table = dict(zip([ game.id for game in games ], tables))
        
        to_break = breakGames(games)
        tables_broken = {}
        for (from_id, to_id, serials) in to_break:
            for serial in serials:
                table = id2table[from_id]
                table.movePlayer(table.serial2client[serial], serial, to_id)
            tables_broken[from_id] = True

        if len(to_break) > 0:
            for table in self.groups[group_id]:
                table.broadcast(balance_packet)
            for table_id in tables_broken.keys():
                table = id2table[table_id]
                table.destroy()
            return group_id
        
        to_equalize = equalizeGames(games)
        for (from_id, to_id, serial) in to_equalize:
            table = id2table[from_id]
            table.movePlayer(table.serial2client[serial], serial, to_id)
            table.broadcast(PacketPokerTableGroupBalance(serial = group_id))
        if len(to_equalize) > 0:
            for table in self.groups[group_id]:
                table.broadcast(balance_packet)
            return group_id
        else:
            return 0

class PokerJabberClient(XMPPClient):

    def authSuccess(self):
        (familly, host, port) = self.transport.getPeer()
        print "Dealer %s logged in jabber at %s:%d" % ( self.factory.username, host, port )
        self.sendPresence()
        self.factory.poker.setJabber(self)
        
    def authFailed(self, error):
        """ """
        (familly, host, port) = self.transport.getPeer()
        print """
Failed to login jabber server %s:%d for user %s
With a regular jabber client, check that the configuration file parameters
are correct and that the %s user already exists.
Jabber is now disabled for this session.
""" % ( host, port, self.factory.username, self.factory.username )
        self.closeStream()

    def message(self, chat, subject, messages):
        if subject:
            self.sendMessage(to = lower(chat) + "@" + self.factory.chat,
                             subject = subject,
                             ttype = 'groupchat')
        subject = subject and [ subject ] or []
        message = join(messages + subject, "\n" + self.factory.settings["user"] + ": ")
        self.sendMessage(to = lower(chat) + "@" + self.factory.chat,
                         body = message,
                         ttype = 'groupchat')
        
    def joinChat(self, chat):
        if self.factory.verbose > 1:
            print "PokerJabberClient: join groupchat %s" % chat
        presence = ChatPresence(to = lower(chat) + "@" + self.factory.chat + "/" + self.factory.username)
        self.sendStanza(presence)

    def inviteChat(self, chat, name):
        if self.factory.verbose > 1:
            print "PokerJabberClient: invite: %s/%s" % (chat, name)
        invitation = ChatInvite(to = name + "@" + self.factory.server,
                                chat = lower(chat) + "@" + self.factory.chat)
        self.sendStanza(invitation)

    def kickChat(self, chat, name, reason = ''):
        kick = ChatKick(kicker = self.factory.settings["user"], # + "@" + self.factory.server,
                        kicked = name, # + "@" + self.factory.server,
                        chat = lower(chat) + "@" + self.factory.chat,
                        reason = reason)
        self.sendStanza(kick)
    
    def connectionLost(self, reason):
        self.factory.poker.unsetJabber()
        XMPPClient.connectionLost(self, reason)

class PokerJabberClientFactory(XMPPClientFactory):

    protocol = PokerJabberClient

    def __init__(self, *args, **kwargs):
        self.poker = kwargs["poker"]
        del kwargs["poker"]
        self.chat = kwargs["chat"]
        del kwargs["chat"]
        self.port = kwargs["port"]
        del kwargs["port"]
        self.verbose = kwargs["verbose"]
        del kwargs["verbose"]
        XMPPClientFactory.__init__(self, *args, **kwargs)
        self.settings = self.poker.settings.headerGetProperties("/server/jabber")[0]

    def clientConnectionFailed(self, connector, reason):
        print """
Unable to connect to jabber server %s:%d.
Jabber disabled for this session.""" % ( self.server, self.port )
        print reason
        XMPPClientFactory.clientConnectionFailed(self, connector, reason)

class PokerService(internet.TCPServer):

    def stopService(self):
        deferred = self._port.factory.shutdown()
        deferred.addCallback(lambda x: internet.TCPServer.stopService(self))
        return deferred
    
def run(argv):
    configuration = sys.argv[-1][:5] == "file:" and sys.argv[-1] or "file:/etc/poker3d/server/poker.server.xml"
    settings = Config([''])
    settings.loadHeader(configuration)
    if not settings.header:
        sys.exit(1)

    application = service.Application('poker')
    serviceCollection = service.IServiceCollection(application)

    poker_factory = PokerServerFactory(settings)
    poker_factory.cleanUp(temporary_users = settings.headerGet("/server/users/@temporary"))
    poker_port = settings.headerGetInt("/server/port")
    PokerService(poker_port, poker_factory).setServiceParent(serviceCollection)
    if settings.headerGetList("/server/jabber"):
        jabber_host = settings.headerGet("/server/jabber/@host")
        jabber_port = settings.headerGetInt("/server/jabber/@port")
        jabber_factory = PokerJabberClientFactory(jabber_host,
                                                  settings.headerGet("/server/jabber/@user"),
                                                  settings.headerGet("/server/jabber/@password"),
                                                  poker = poker_factory,
                                                  port = jabber_port,
                                                  verbose = poker_factory.verbose,
                                                  chat = settings.headerGet("/server/jabber/@chat"))
    
        from twibber import xmppstream
        if poker_factory.verbose > 3:
            xmppstream.debug = poker_factory.verbose
        else:
            xmppstream.debug = 0

        from twibber import xmlstream
        if poker_factory.verbose > 3:
            xmlstream.debug = poker_factory.verbose
        else:
            xmlstream.debug = 0

        internet.TCPClient(jabber_host, jabber_port, jabber_factory).setServiceParent(serviceCollection)

    else:
        print "Jabber dialog not requested in %s" % settings.url

    application.verbose = poker_factory.verbose
    return application
        
        
application = run(sys.argv)

if __name__ == '__main__':
    try:
        app.startApplication(application, None)
        reactor.run()
    except:
        if application.verbose:
            print_exc()
        else:
            print sys.exc_value
