#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#  Henry Precheur <henry@precheur.org>
#
import sys
sys.path.insert(0, "../../python")
sys.path.insert(0, "..")

from string import split, find
from pprint import pprint
from random import random
from time import sleep
import signal

import pycurl
import libxml2

from twisted.internet import reactor, error

from underware.client import UGAMEClientProtocol, UGAMEClientFactory
from underware.config import Config

from poker.pokergame import PokerGameClient
from poker.pokercards import PokerCards
from poker.pokerchips import PokerChips
from poker.pokerpackets import *

class PokerTournamentClient(UGAMEClientProtocol):
    """Poker client"""

    def __init__(self):
        UGAMEClientProtocol.__init__(self)
        self.actionDone = False
        
    def _handleConnection(self, packet):
        game = self.factory.game

        if packet.type == PACKET_POKER_START:
            if game.isRunning():
                raise UserWarning, "you should not be here (state: %s)" % game.state
            elif packet.hand_serial == 0:
                print "*ERROR* game start was refused, try again later"
                tournament = self.factory.tournament
                reactor.callLater(tournament.showdown_delay, tournament.startGame, self, self.factory)
            else:
                game.setTime(packet.time)
                game.setHandsCount(packet.hands_count)
                game.setLevel(packet.level)
                game.beginTurn(packet.hand_serial)

        elif packet.type == PACKET_POKER_CANCELED:
            game.canceled(packet.serial, packet.amount)

        if packet.type == PACKET_POKER_TABLE:
            if packet.id == 0:
                tournament = self.factory.tournament
                tournament.loop = False
                print "server refuses to create table"
                reactor.stop()
            else:
                game.id = packet.id
                game.name = packet.name
                game.setVariant(packet.variant)
                game.setBettingStructure(packet.betting_structure)
                game.setMaxPlayers(packet.seats)

        if packet.type == PACKET_POKER_PLAYER_ARRIVE:
            game.addPlayer(packet.serial)

        elif packet.type == PACKET_SERIAL:
            self.user.serial = packet.serial
            
        elif packet.type == PACKET_POKER_POSITION:
            game.setPosition(packet.serial)

        elif packet.type == PACKET_POKER_PLAYER_LEAVE:
            game.removePlayer(packet.serial)

        elif packet.type == PACKET_POKER_SEATS:
	    game.setSeats(packet.seats)
	    
        elif packet.type == PACKET_POKER_PLAYER_CARDS:
	    player = game.serial2player[packet.serial]
	    player.hand.set(packet.cards)

        elif packet.type == PACKET_POKER_BOARD_CARDS:
	    game.board.set(packet.cards)

        elif packet.type == PACKET_POKER_DEALER:
            game.setDealer(packet.serial)

        elif packet.type == PACKET_POKER_SIT_OUT:
            game.sitOut(packet.serial)

        elif packet.type == PACKET_POKER_SIT:
            game.sit(packet.serial)
            
        elif packet.type == PACKET_POKER_IN_GAME:
            for serial in game.serialsAll():
                if serial in packet.players:
                    game.sit(serial)
                else:
                    game.sitOut(serial)
            
        elif packet.type == PACKET_POKER_WIN:
            if not game.winners:
                game.distributeMoney()

                winners = game.winners[:]
                winners.sort()
                packet.serials.sort()
                if winners != packet.serials:
                    raise UserWarning, "game.winners %s != packet.serials %s" % (winners, packet.serials)

                game.endTurn()
            
        elif packet.type == PACKET_POKER_PLAYER_CHIPS:
            player = game.serial2player[packet.serial]
            if game.isRunning():
                #
                # Sanity checks. Immediately after the game begins the server
                # sends the amount of chips for each player so that the client
                # can resynchronize and issue a warning.
                #
                bet = PokerChips(game.chips_values, packet.bet)
                if player.bet.toint() != bet.toint():
                    print "server says player %d has a bet of %d chips and client thinks it has %d" % ( packet.serial, bet.toint(), player.bet.toint())
                    player.bet.set(packet.bet)
                money = PokerChips(game.chips_values, packet.money)
                if player.money.toint() != money.toint():
                    print "server says player %d has a money of %d chips and client thinks it has %d" % ( packet.serial, money.toint(), player.money.toint())
                    player.money.set(packet.money)
            else:
                #
                # If server sends chips amount for a player while not in game,
                # 
                #
                player.bet.set(packet.bet)
                player.money.set(packet.money)

        elif packet.type == PACKET_POKER_FOLD:
            game.fold(packet.serial)
            
        elif packet.type == PACKET_POKER_CALL:
            game.call(packet.serial)

        elif packet.type == PACKET_POKER_RAISE:
            game.callNraise(packet.serial, packet.amount)

        elif packet.type == PACKET_POKER_CHECK:
            game.check(packet.serial)

        elif packet.type == PACKET_POKER_STATE:
            if packet.string == "end":
                game.endState()
            if game.isBlindAnteRound():
                game.nextRound()
            if packet.string != "end":
                game.initRound()
            if game.state != packet.string:
                print " *ERROR* game %d state = %s, server says %s " % ( game.id, game.state, packet.string )

        elif packet.type == PACKET_POKER_BLIND:
            game.blind(packet.serial, packet.amount, packet.dead)

        elif packet.type == PACKET_POKER_ANTE:
            game.ante(packet.serial, packet.amount)

        elif packet.type == PACKET_POKER_TABLE_DESTROY:
            if game.id == packet.serial:
                tournament = self.factory.tournament
                tournament.tables.remove(self.factory)
                self.transport.loseConnection()
                
        self.factory.tournament.manage(self, self.factory, packet)
        
    def protocolEstablished(self):
        table_info = self.factory.table_info
        self.sendPacket(PacketLogin(name = self.factory.name))
        self.sendPacket(PacketPokerTable(name = table_info["name"],
                                         seats = int(table_info["seats"]),
                                         variant = table_info["variant"],
                                         betting_structure = table_info["betting_structure"],
                                         timeout = int(table_info["timeout"])))

class PokerTournamentFactory(UGAMEClientFactory):
    def __init__(self, *args, **kwargs):
        UGAMEClientFactory.__init__(self, *args, **kwargs)
        self.tournament = kwargs["tournament"]
        self.table_info = kwargs["table_info"]
        self.dirs = kwargs["dirs"]
        self.verbose = self.tournament.verbose
        self.protocol = PokerTournamentClient
        
    def buildProtocol(self, addr):
	UGAMEClientFactory.buildProtocol(self, addr)
        self.game = PokerGameClient("file:poker.%s.xml", self.dirs)
        self.game.verbose = self.verbose
	self.protocol_instance.name = self.tournament.name
	return self.protocol_instance

    def clientConnectionFailed(self, connector, reason):
        print "reason: %s" % reason
        reactor.stop()
        
    def clientConnectionLost(self, connector, reason):
        if not reason.check(error.ConnectionDone):
            print reason

STARTING = 1
RUNNING = 2
BALANCING = 3

class PokerTournament:

    def __init__(self, configfile):
        self.configfile = configfile
        self.run()

    def reset(self):
        self.config = False
        self.players_count = 0
        self.tables = []
        self.state = STARTING
        self.rank = []
        self.serial2info = {}
        self.group_id = 0
        self.verbose = 0

    def manage(self, client, table, packet):
        if packet.type == PACKET_POKER_TABLE_GROUP_SERIAL:
            if packet.serial == 0:
                print "*ERROR* server refused to create the requested table group"
            self.group_id = packet.serial
            
        if packet.type == PACKET_POKER_TABLE_GROUP_BALANCE:
            if packet.serial == 0:
                if self.verbose > 1:
                    print "no need to balance"
            self.state = RUNNING
            
        if packet.type == PACKET_POKER_TABLE_UNGROUP:
            if packet.serial == 0:
                print "*ERROR* server refused to destroy table group"
            reactor.stop()
            
        elif packet.type == PACKET_POKER_SIT:
            if self.state == STARTING:
                self.players_count += 1
                print "players %d (%d seated)" % ( packet.serial, self.players_count )
                if self.players_quota == self.players_count:
                    table_ids = [ table.game.id for table in self.tables ]
                    group = PacketPokerTableGroup(game_ids = table_ids)
                    self.tables[0].protocol_instance.sendPacket(group)
                    self.broadcast(PacketPokerTableClose())
                    self.broadcast(PacketPokerTimeoutPolicy(string = "fold"))
                    self.broadcast(PacketPokerStart())
                    self.state = RUNNING
            
        elif packet.type == PACKET_POKER_PLAYER_ARRIVE:
            if self.state == STARTING:
                print "players %s %d/%d" % ( packet.name, self.players_count, self.players_quota )
                    
            self.serial2info[packet.serial] = {
                "name": packet.name,
                "client": client,
                }

        elif packet.type == PACKET_POKER_PLAYER_LEAVE:
            if self.state == STARTING:
                self.players_count -= 1
                del self.serial2info[packet.serial]
            else:
                self.eliminate(packet.serial)
                
        elif packet.type == PACKET_POKER_WIN:
            #
            # Kick players who do not have any chips left
            #
            to_leave = 0
            for player in table.game.playersAll():
                if player.money.toint() <= 0:
                    to_leave += 1
                    client.sendPacket(PacketPokerTableKick(game_id = table.game.id,
                                                           serial = player.serial))
            #
            # Balance tables if not already balancing them and if 
            # some players lost and if there will be more than one
            # player at the tables when they left.
            #
            if ( self.state == RUNNING and
                 to_leave > 0 and
                 self.players_count - to_leave > 1 ):
                self.state = BALANCING
                balance = PacketPokerTableGroupBalance(serial = self.group_id)
                table.protocol_instance.sendPacket(balance)

            if self.players_count - to_leave > 1:
                reactor.callLater(self.showdown_delay, self.startGame, client, table)

    def eliminate(self, serial):
        info = self.serial2info[serial]
        self.rank.append((self.players_count, info["name"], serial))
        del self.serial2info[serial]
        self.players_count -= 1
        print "%d remaining players" % self.players_count
        if self.players_count == 1:
            for (serial, info) in self.serial2info.iteritems():
                self.rank.append((1, info["name"], serial))
            pprint(self.rank)
            client = info["client"]
            game = client.factory.game
            print "The winner is player %d with %d chips" % ( serial, game.serial2player[serial].money.toint() )
            reactor.callLater(20, self.cleanUp, client)

    def cleanUp(self, client):
        game = client.factory.game
        client.sendPacket(PacketPokerTableDestroy(game_id = game.id))
        client.sendPacket(PacketPokerTableUngroup(serial = self.group_id))
        self.stop()
        
    def startGame(self, client, table):
        #
        # If the table was broken, give up
        #
        if not table in self.tables:
            return

        if self.state == RUNNING:
            client.sendPacket(PacketPokerStart(game_id = table.game.id))
        else:
            print "delay startGame for table %d" % table.game.id
            reactor.callLater(self.showdown_delay, self.startGame, client, table)
            
    def broadcast(self, packet):
        for table in self.tables:
            if hasattr(packet, "game_id"):
                packet.game_id = table.game.id
            table.protocol_instance.sendPacket(packet)

    def run(self):
        self.reset()
        config = Config([''])
        config.loadHeader(self.configfile)

        self.loop = config.headerGet("/tournament/@loop") == "yes"

        host = config.headerGet("/tournament/host")
        port = config.headerGetInt("/tournament/port")
        self.name = config.headerGet("/tournament/@name")
        self.verbose = config.headerGetInt("/tournament/@verbose")
        self.players_quota = config.headerGetInt("/tournament/@players")
        self.showdown_delay = config.headerGetInt("/tournament/@showdown_delay")
        dirs = split(config.headerGet("/tournament/path"))
        for table_info in config.headerGetProperties("/tournament/table"):
            table = PokerTournamentFactory(tournament = self,
                                           table_info = table_info,
                                           dirs = dirs)
            table.name = self.name
            reactor.connectTCP(host, port, table)
            self.tables.append(table)

    def stop(self):
        reactor.disconnectAll()
        if self.loop:
            print "respawning tournament in 10 seconds"
            reactor.callLater(10, self.run)

if __name__ == '__main__':
    tournament = PokerTournament(sys.argv[1])
    reactor.run()
