/* -*-c++-*- 
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */
#ifndef _EXG_BASETYPES_H
#define _EXG_BASETYPES_H

#include <exg/exg_vector3.h>
#include <exg/exg_vector4.h>
#include <exg/exg_object.h>
#include <exg/exg_visitor.h>

namespace exg {

  class EXG_EXPORT MapObjectPointer : public std::map<std::string, Pointer<Object> >, public Object {
  private:
    typedef std::map<std::string, Pointer<Object> > Base;


    template <class T> T* AddProperty(const std::string& nameProperty) {
      if (find(nameProperty)!=end())
        return 0;
      T* a=new T;
      (*this)[nameProperty]=a;
      return a;
    }

  protected:
    ~MapObjectPointer() {}

  public:
    COMMON_FUNCTIONS(MapObjectPointer)

    typedef Base::const_iterator const_iterator;
    typedef Base::iterator iterator;

    virtual MapObjectPointer* AsMapObjectPointer(void) { return this; }
    virtual const MapObjectPointer* AsMapObjectPointer(void) const { return this; }

    inline virtual void OSave(std::ostream& out, IOContext* context = 0) const {
      int length = size();
      Save(length, out, context);
      for(const_iterator i = begin(); i != end(); i++) {
	Save(i->first, out, context);
	Save(i->second, out, context);
      }
    }
    inline virtual void OLoad(std::istream& in, IOContext* context = 0) {
      int length;
      Load(length, in, context);
      clear();
      for(int i = 0; i < length; i++) {
	std::string key;
	Load(key, in, context);
	Load((*this)[key], in, context);
      }
    }

    virtual int GetType(void) const { return MAP_OBJECT_POINTER; }
    inline static Object* Create() { return new MapObjectPointer; }

    Object* GetProperty(const std::string& nameProperty) {
      iterator a=find(nameProperty);
      if (a==end())
        return 0;
      return (*a).second.Get();
    }

    const Object* GetProperty(const std::string& nameProperty) const {
      const_iterator a=find(nameProperty);
      if (a==end())
        return 0;
      return (*a).second.Get();
    }

    VectorFloat* AddPropertyVectorFloat(const std::string& name);

    VectorString* AddPropertyVectorString(const std::string& name);

    VectorObjectPointer* AddPropertyVectorObjectPointer(const std::string& name);

    MapObjectPointer* AddPropertyMapObjectPointer(const std::string& name);

    std::ostream& operator<<(std::ostream& o);

    Object* Clone() {
      MapObjectPointer* n=new MapObjectPointer;
      for (iterator i=begin();i!=end();i++)
        if ((*i).second.Get())
          (*n)[(*i).first]=(*i).second->Clone();
      return n;
    }

    virtual bool CompareLessThan(Object* _compare) {
      MapObjectPointer* _c=_compare->AsMapObjectPointer();
      assert(_c);
      for (MapObjectPointer::iterator i1=begin();i1!=end();i1++) {
	MapObjectPointer::iterator found=_c->find((*i1).first); // get key of s1 in s2
	if (found==_c->end())
	  assert(0 && "vertex has not the same key");

	//std::cout << "Key " << (*i1).first << std::endl;
	assert((*i1).second.Get() && (*found).second.Get());
	if ((*i1).second->CompareLessThan((*found).second.Get()))
	  return true;
	else if ((*found).second->CompareLessThan((*i1).second.Get()))
	  return false;
      }
      return false;
    }

  };

  template<typename Tp>
  class EXG_EXPORT VectorObject : public std::vector<Tp>, public Object {
  private:
    typedef std::vector<Tp> Base;

  protected:
    ~VectorObject() {}

  public:
    typedef typename Base::const_iterator const_iterator;
    typedef typename Base::iterator iterator;

    VectorObject() { }
    VectorObject(size_t n, const Tp& value) : Base(n, value) {}
    explicit VectorObject(size_t n) : Base(n) {}
    template<typename InputIterator>
    VectorObject(InputIterator first, InputIterator last) : Base(first, last) {}

    inline virtual void OSave(std::ostream& out, IOContext* context = 0) const {
      int length = this->size();
      Save(length, out, context);
      for(const_iterator i = this->begin(); i != this->end(); i++) {
	const Tp& element = *i;
	Save(element, out, context);
      }
    }

    inline virtual void OLoad(std::istream& in, IOContext* context = 0) {
      int length;
      this->clear();
      Load(length, in, context);
      for(int i = 0; i < length; i++) {
	Tp element;
	Load(element, in, context);
	push_back(element);
      }
    }

    virtual bool CompareLessThan(Object* _compare) = 0;


  };

  class EXG_EXPORT VectorInt : public VectorObject<int> {
  private:
    typedef VectorObject<int> Base;
    typedef int Tp;

  protected:
    virtual ~VectorInt() {}

  public:
    COMMON_FUNCTIONS(VectorInt)

    VectorInt() { }
    VectorInt(size_t n, const Tp& value) : Base(n, value) {}
    explicit VectorInt(size_t n) : Base(n) {}
    template<typename InputIterator>
    VectorInt(InputIterator first, InputIterator last) : Base(first, last) {}

    virtual VectorInt* AsVectorInt(void) { return this; }
    virtual const VectorInt* AsVectorInt(void) const { return this; }

    virtual int GetType(void) const { return VECTOR_INT; }
    inline static Object* Create() { return new VectorInt; }

    virtual bool CompareLessThan(Object* _compare) {
      VectorInt* _c=_compare->AsVectorInt();
      assert(_c);
      VectorInt::size_type nb=size();
      for (VectorInt::size_type i=0;i<nb;i++)
	if ((*this)[i]<(*_c)[i])
	  return true;
	else if ((*this)[i]>(*_c)[i])
	  return false;
      return false;;
    }

  };
  
  class EXG_EXPORT VectorFloat : public VectorObject<float> {
  private:
    typedef VectorObject<float> Base;
    typedef float Tp;

  protected:
    virtual ~VectorFloat() {}

  public:
    COMMON_FUNCTIONS(VectorFloat)

    VectorFloat() { }
    VectorFloat(size_t n, const Tp& value) : Base(n, value) {}
    explicit VectorFloat(size_t n) : Base(n) {}
    template<typename InputIterator>
    VectorFloat(InputIterator first, InputIterator last) : Base(first, last) {}
    explicit VectorFloat(const Vector3f& v3) : Base(&v3[0], &v3[3]) {}
    explicit VectorFloat(const Vector3d& v3) : Base(&v3[0], &v3[3]) {}
    explicit VectorFloat(const Vector4f& v3) : Base(&v3[0], &v3[4]) {}
    explicit VectorFloat(const Vector4d& v3) : Base(&v3[0], &v3[4]) {}

    virtual VectorFloat* AsVectorFloat(void) { return this; }
    virtual const VectorFloat* AsVectorFloat(void) const { return this; }

    virtual int GetType(void) const { return VECTOR_FLOAT; }
    inline static Object* Create() { return new VectorFloat; }

    std::ostream& operator<<(std::ostream& o);

    Object* Clone() {
      VectorFloat* n=new VectorFloat(begin(),end());
      return n;
    }


    virtual bool CompareLessThan(Object* _compare) {
      VectorFloat* _c=_compare->AsVectorFloat();
      assert(_c);
      VectorFloat::size_type nb=size();
      assert(nb && _c->size());
      for (VectorFloat::size_type i=0;i<nb;i++) {
	//	std::cout << "Compare " <<(*this)[i] << " with " << (*_c)[i] << std::endl;
	if ((*this)[i]<(*_c)[i])
	  return true;
	else if ((*this)[i]>(*_c)[i])
	  return false;
      }
      return false;
    }


  };
  
  class EXG_EXPORT VectorString : public VectorObject<std::string> {
  private:
    typedef VectorObject<std::string> Base;
    typedef std::string Tp;

  protected:
    virtual ~VectorString() {}

  public:
    COMMON_FUNCTIONS(VectorString)

    VectorString() { }
    VectorString(size_t n, const Tp& value) : Base(n, value) {}
    explicit VectorString(size_t n) : Base(n) {}
    template<typename InputIterator>
    VectorString(InputIterator first, InputIterator last) : Base(first, last) {}

    virtual VectorString* AsVectorString(void) { return this; }
    virtual const VectorString* AsVectorString(void) const { return this; }

    virtual int GetType(void) const { return VECTOR_STRING; }
    inline static Object* Create() { return new VectorString; }


    std::ostream& operator<<(std::ostream& o);

    Object* Clone() {
      VectorString* n=new VectorString(begin(),end());
      return n;
    }


    virtual bool CompareLessThan(Object* _compare) {
      VectorString* _c=_compare->AsVectorString();
      assert(_c);
      VectorString::size_type nb=size();
      for (VectorString::size_type i=0;i<nb;i++)
	if ((*this)[i]<(*_c)[i])
	  return true;
	else if ((*this)[i]>(*_c)[i])
	  return false;
      return false;
    }


  };
  
  class EXG_EXPORT VectorObjectPointer : public VectorObject<Pointer<Object> > {
  private:
    typedef VectorObject<Pointer<Object> > Base;
    typedef Pointer<Object> Tp;

  protected:
    virtual ~VectorObjectPointer() {}

  public:
    COMMON_FUNCTIONS(VectorObjectPointer)

    VectorObjectPointer() { }
    VectorObjectPointer(size_t n, const Tp& value) : Base(n, value) {}
    explicit VectorObjectPointer(size_t n) : Base(n) {}
    template<typename InputIterator>
    VectorObjectPointer(InputIterator first, InputIterator last) : Base(first, last) {}

    virtual VectorObjectPointer* AsVectorObjectPointer(void) { return this; }
    virtual const VectorObjectPointer* AsVectorObjectPointer(void) const { return this; }

    virtual int GetType(void) const { return VECTOR_OBJECT_POINTER; }
    inline static Object* Create() { return new VectorObjectPointer; }

    std::ostream& operator<<(std::ostream& o);

    Object* Clone() {
      VectorObjectPointer* n=new VectorObjectPointer;
      for (iterator i=begin();i!=end();i++)
        n->push_back((*i)->Clone());
      return n;
    }

    virtual bool CompareLessThan(Object* _compare) {
      VectorObjectPointer* _c=_compare->AsVectorObjectPointer();
      assert(_c);
      VectorObjectPointer::size_type nb=size();
      for (VectorObjectPointer::size_type i=0;i<nb;i++) {
	assert((*this)[i].Get() && (*_c)[i].Get());
	if ((*this)[i]->CompareLessThan((*_c)[i].Get()))
	  return true;
	else if ((*_c)[i]->CompareLessThan((*this)[i].Get()))
	  return false;
      }
      return false;
    }


  };
};

#endif // _EXG_BASETYPES_H 
