/**
   Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   ****************************************************************************
   * @file   exg_material.h
   *
   * @brief   base for material exchange
   *
   *****************************************************************************
   *
   * @author  Cedric Pinson
   *
   * @date    Created 2001/04
   *
   * @version $Id: exg_material.h,v 1.25 2004/10/27 20:37:54 loic Exp $
   *
   ****************************************************************************/

#ifndef exg_material_h
#define exg_material_h

#include <exg/exg_basetypes.h>
#include <exg/exg_file.h>
#include <vector>
#include <algorithm>

namespace exg
{
  class Polygon;
  class Visitor;

  /**
   *  Class that describes a material.
   *
   *  reservered keyword:
   *
   *  "name"     File() identify the material's name
   *  "ambient"  VectorFloat(4) identify a color 
   *  "diffuse"  VectorFloat(4) identify a color 
   *  "specular" VectorFloat(4) identify a color 
   *  "normal"   if exist means that vertexes have a normal
   *  "maps"  VectorObjectPointer() contains pointer of MapObjectPointer() used
   *          to describe a channel that have a consequence on vertex data. 
   *          Typically when using texture mapping, vertexes have additionnal
   *          data like uv. So entries in "maps" give the description of this.
   *      "name"     File() identify a channel by a name. This channel name must be
   *                 used to find the good channel in vertexes @see Vertex.
   *      "file"     PointerObject() identify a pointer of File() in Mesh["files"]
   *                 @see Mesh.
   *      "element"  VectorFloat(1) contains the number of coordinates in vertexes
   *                 @see Vertex. this number is casted to int.
   *      "mode"     File() contains a string we use it like a additive to make a 
   *                 map in additive mode. this field can be omitted.
   *
   *
   */
  class EXG_EXPORT Material : public MapObjectPointer
  {
  protected:
    virtual ~Material() {}

  public:
    friend class Polygon;

    inline static MapObjectPointer* CreateVertexData() {
      MapObjectPointer* vd=new MapObjectPointer;
      (*vd)["file"]=0;
      (*vd)["name"]=new VectorString(1);
      (*vd)["element"]=new VectorFloat(1);
      return vd;
    }

   public:
    COMMON_FUNCTIONS(Material)

    /// Default constructor
    Material() {
      (*this)["name"]=new File;
    }

    /// @return Name of material
    const std::string& GetName() const { 
      File* a=(File*)(*find("name")).second.Get();
      return a->GetName();
    }

    /// @param const std::string& name to use for material
    void SetName(const std::string& name) { 
      File* a=(File*)(*find("name")).second.Get();
      a->SetName(name);
    }

    /// @return the number of polygon using this material
    int GetNbPolygons() const { return mPolygons.size();}


    /**
     * Return the ith polygon that uses this material
     *
     * @param int ith polygon wanted
     * @return Polygon* polygon
     */
    Polygon* GetPolygon(int i) { return mPolygons[i].Get();}



    /**
     * Register a polygon to a material.
     *
     * @param polygon Polygon* to add to this material
     */
    void AddPolygon(Polygon* polygon);

    /**
     * Return the ith polygon that uses this material
     *
     * @param i th polygon wanted
     * @return const Polygon* polygon
     */
    const Polygon* GetPolygon(int i) const { return mPolygons[i].Get();}

    void Traverse(Visitor& visitor);

    virtual Material* AsMaterial(void) { return this; }
    virtual const Material* AsMaterial(void) const { return this; }
    virtual int GetType(void) const { return MATERIAL; }
    inline static Object* Create() { return new Material; }

    std::ostream& operator<<(std::ostream& o);

    virtual void ORepair(IOContext* context);

    Object* Clone() {
      Material* n=new Material;
      for (iterator i=begin();i!=end();i++)
        if ((*i).second.Get())
          (*n)[(*i).first]=(*i).second->Clone();
      return n;
    }

   private:
    
    /// Polygons that use this material
    std::vector<Pointer<Polygon> > mPolygons;
  };

}

#endif
