/**
    Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 ****************************************************************************
 * @file   mth_matrix3.h
 *
 * @brief   matrix 3x3
 *
 *****************************************************************************
 *
 * @author  psc80
 *
 * @date    Created 2001/08
 *
 * @version $Id: exg_matrix3.h,v 1.5 2004/10/31 21:28:45 loic Exp $
 *
 ****************************************************************************/

#ifndef EXG_MATRIX3_H
#define EXG_MATRIX3_H

#include <exg/exg_vector3.h>
#include <exg/exg_plane3.h>

namespace exg {


  /**
   *  Classe for matrix 3x3 
   */
  template <class T> class Matrix3_T {

    template <class X> friend class Vector3_T;


    /** Representation
     *
     *  v[0][0] v[0][1] v[0][2]
     *  v[1][0] v[1][1] v[1][2]
     *  v[2][0] v[2][1] v[2][2]
     *
     */
    Vector3_T<T> v[3];

   public:


    /// do nothing
    Matrix3_T() {}


    /// Constructor with a matrix */
    template <class C> explicit Matrix3_T(const Matrix3_T<C>& _m) { 
      v[0].Init((T)_m[0][0],(T)_m[0][1],(T)_m[0][2]);
      v[1].Init((T)_m[1][0],(T)_m[1][1],(T)_m[1][2]);
      v[2].Init((T)_m[2][0],(T)_m[2][1],(T)_m[2][2]);
    }


    /// Constructor for a rotation with respect to a (normalized) axis
    Matrix3_T(T _angle, const Vector3_T<T>& _axis) {
      SetRotationFromAxis(_angle,_axis); }


    /// Constructor for euler angles yaw/pict/roll
    Matrix3_T( T _y, T _p, T _r) {
      SetRotationFromEuler(_y, _p, _r); }


    /// Constructor with three vectors
    Matrix3_T(const Vector3_T<T>& _v1, const Vector3_T<T>& _v2, const Vector3_T<T>& _v3) {
      Init(_v1, _v2, _v3); }


    /// Initialize a matrix from three vectors
    void Init(const Vector3_T<T>& _v1, const Vector3_T<T>& _v2, const Vector3_T<T>& _v3) {
      v[0]=_v1; v[1]=_v2; v[2]=_v3;}



    /// Set the matrix to zero
    void Init() { v[0].Init(); v[1].Init(); v[2].Init(); }



    /// Set the matrix to identity
    inline static Matrix3_T Identity() { 
      Matrix3_T a;
      a.SetIdentity();
      return a;}


    /// Return matrix rotation
    inline static Matrix3_T Rotation(T angle, const Vector3_T<T>& ax) { 
      Matrix3_T a;
      a.SetRotationFromAxis(angle,ax);
      return a;
    }


    /// Return matrix rotation from euler
    inline static Matrix3_T Rotation(const T h, const T p, const T r) { 
      Matrix3_T a;
      a.SetRotationFromEuler(h,p,r);
      return a;
    }


    inline bool IsIdentity() const {
      Matrix3_T m=Identity();
      return !memcmp(&v,&m.v,sizeof(m));
    }


    /// Set the matrix to identity
    inline void SetIdentity() {
      v[0].Init(1,0,0); 
      v[1].Init(0,1,0); 
      v[2].Init(0,0,1);
    }

    /// Return the transposed matrix
    void Transpose() { 
      T t;
      t = v[0][1]; v[0][1] = v[1][0]; v[1][0] = t;
      t = v[0][2]; v[0][2] = v[2][0]; v[2][0] = t;
      t = v[2][1]; v[2][1] = v[1][2]; v[1][2] = t;}
    


    /** Calculate the determinant of matrix
     *  http://mathworld.wolfram.com/Determinant.html
     *
     *	a0 a1 a2  v0
     *	b0 b1 b2  v1
     *	c0 c1 c2  v2
     *
     *	Det(M)=a0.b1.c2-a0.b2.c1 + a1.b2.c0-a1.b0.c2 + a2.b0.c1-a2.b1.c0
     */
    inline T Determinant() {
      return 
        // each diagonal +
        v[0][0]*v[1][1]*v[2][2] + v[0][1]*v[1][2]*v[2][0] + v[0][2]*v[1][0]*v[2][1] -
        // each diagonal -
        v[0][2]*v[1][1]*v[2][0] - v[0][1]*v[1][0]*v[2][2] - v[0][0]*v[1][2]*v[2][1];}



    /** Calculate the Inverse of matrix
     *  http://mathworld.wolfram.com/Determinant.html
     */
    inline int Inverse() {
      T a=Determinant();
      if (!a)
        return -1;
      a=(T)1.0/a ;

      /// From calc of xemacs
      Vector3_T<T> nv0(v[2][2]*v[1][1] - v[1][2]*v[2][1], 
                    v[0][2]*v[2][1] - v[2][2]*v[0][1], 
                    v[1][2]*v[0][1] - v[0][2]*v[1][1]);
      Vector3_T<T> nv1(v[1][2]*v[2][0] - v[2][2]*v[1][0],
                    v[2][2]*v[0][0] - v[0][2]*v[2][0],
                    v[0][2]*v[1][0] - v[1][2]*v[0][0]);
      Vector3_T<T> nv2(v[2][1]*v[1][0] - v[1][1]*v[2][0], 
                    v[0][1]*v[2][0] - v[2][1]*v[0][0],
                    v[1][1]*v[0][0] - v[0][1]*v[1][0]);
      nv0*=a;
      nv1*=a;
      nv2*=a;
      Init(nv0,nv1,nv2);
      return 0;
    }

    /// Build a rotation with respect to a (normalized !) axis
    void SetRotationFromAxis(T _angle, const Vector3_T<T>& _ax) {
      T s = (T)sin ((double)_angle);
      T c = (T)cos ((double)_angle);
      T t = (T)1.0 - c ;

      v[0][0] = t * _ax[0] * _ax[0] + c;
      v[1][0] = t * _ax[0] * _ax[1] + s * _ax[2];
      v[2][0] = t * _ax[0] * _ax[2] - s * _ax[1];

      v[0][1] = t * _ax[1] * _ax[0] - s * _ax[2];
      v[1][1] = t * _ax[1] * _ax[1] + c;
      v[2][1] = t * _ax[1] * _ax[2] + s * _ax[0];

      v[0][2] = t * _ax[2] * _ax[0] + s * _ax[1];
      v[1][2] = t * _ax[2] * _ax[1] - s * _ax[0];
      v[2][2] = t * _ax[2] * _ax[2] + c;
    }



    /// Build a rotation from euler angles*/
    void SetRotationFromEuler(T _h, T _p, T _r) {
      /*
        {ch*cr - sh*(sp*sr), ch*sr + sh*(sp*cr), -(sh*cp)} 
        {-(cp*sr), cp*cr, sp} 
        {sh*cr + ch*(sp*sr), sh*sr - ch*(sp*cr), ch*cp}
      */
      T sh,ch,sp,cp,sr,cr;

      if (_h == 0.0) {
        sh=0.0;
        ch=1.0;
      } else {
        sh = (T)sin ((double)_h);
        ch = (T)cos ((double)_h);
      }

      if (_p == 0.0) {
        sp=0.0;
        cp=1.0;
      } else {
        sp = (T)sin ((double)_p);
        cp = (T)cos ((double)_p);
      }

      if (_r == 0.0) {
        sr=0.0;
        cr=1.0;
      } else {
        sr = (T)sin ((double)_r);
        cr = (T)cos ((double)_r);
      }

      v[0][0]=ch*cr - sh*(sp*sr);
      v[0][1]=ch*sr + sh*(sp*cr);
      v[0][2]=-(sh*cp);
      v[1][0]=-(cp*sr);
      v[1][1]=cp*cr;
      v[1][2]=sp;
      v[2][0]=sh*cr + ch*(sp*sr);
      v[2][1]=sh*sr - ch*(sp*cr);
      v[2][2]=ch*cp;
    }



    /// Extract euler angle from matrix in order yaw,pitch roll
    inline void GetEulerAngles(Vector3_T<T>& _angle) const {
      _angle.Init(atan(v[1][0]/v[0][0]),asin(v[2][0]),atan(v[2][1]/v[2][2]));}




    /// Coordinates accessor operator
    inline Vector3_T<T>& operator [] (int _i) { return v[_i]; }


    /// const coordinates accessor operator
    inline const Vector3_T<T>& operator [] (int _i) const { return v[_i]; }


    /// Addtion of two matrixes
    inline Matrix3_T operator + (const Matrix3_T& _m) const {
      return Matrix3_T(v[0]+_m[0], v[1]+_m[1], v[2]+_m[2]); }



    /// Division by a scalar*/
    inline Matrix3_T operator / (T _a) const {
      T a=(T)1.0/_a;
      return Matrix3_T(v[0]*a, v[1]*a, v[2]*a);}



    /// Mutliplication by a salar value
    inline Matrix3_T operator * (T _a) const {
      return Matrix3_T(v[0]*_a, v[1]*_a, v[2]*_a);}



    /// Mutliplication by a vector (vector to the right)
    inline Vector3_T<T> operator * (const Vector3_T<T>& _v) const {
      return Vector3_T<T>(v[0].Dot(_v),v[1].Dot(_v),v[2].Dot(_v));}



    /// Mutliplication by a matrix
    inline Matrix3_T operator * (const Matrix3_T& _m) const {
      Matrix3_T col(_m);
      col.Transpose();
      return Matrix3_T(Vector3_T<T>(v[0].Dot(col[0]), v[0].Dot(col[1]), v[0].Dot(col[2])),
                       Vector3_T<T>(v[1].Dot(col[0]), v[1].Dot(col[1]), v[1].Dot(col[2])),
                       Vector3_T<T>(v[2].Dot(col[0]), v[2].Dot(col[1]), v[2].Dot(col[2])));
    }



    /// Operator +=
    inline const Matrix3_T& operator += (const Matrix3_T& _m) {
      v[0]+=_m[0]; v[1] += _m[1]; v[2] += _m[2]; return *this;}


  };



  /// Mutliplication by a plane
  template <class T> inline Plane3_T<T> operator * ( const Matrix3_T<T>& _m, const Plane3_T<T>& _p) {
    return Plane3_T<T>(_m*_p.Direction(),_p.Position());
  }


  typedef Matrix3_T<float>  Matrix3f;
  typedef Matrix3_T<double> Matrix3d;

}


#endif
