/**
   Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   ****************************************************************************
   * @file   exg_mesh.h
   *
   * @brief   base for mesh exchange
   *
   *****************************************************************************
   *
   * @author  Cedric Pinson
   *
   * @date    Created 2001/04
   *
   * @version $Id: exg_mesh.h,v 1.19 2004/07/10 07:34:30 loic Exp $
   *
   ****************************************************************************/


#ifndef exg_mesh_h
#define exg_mesh_h

#include <string>
#include "exg/exg_vector3.h"
#include <exg/exg_basetypes.h>
#include "exg/exg_point.h"
#include "exg/exg_polygon.h"
#include "exg/exg_file.h"
#include "exg/exg_material.h"

namespace exg
{
  
  class Visitor;

  class EXG_EXPORT Mesh : public MapObjectPointer
  {

    /**
     *  Set number of elements for a given container
     */
    template <class T> void SetNbElements(int n,VectorObjectPointer& container) {
      int nb=container.size();
      container.resize(n);
      if (n<=nb)
        return;
      generate(container.begin()+nb,container.end(),T::Create);
    }


   protected:
    virtual ~Mesh();

   public:
    COMMON_FUNCTIONS(Mesh)

    /// Default constructor
    Mesh() {
      (*this)["name"]=new File;
      (*this)["center"]=new VectorFloat(3);
      (*this)["points"]=new VectorObjectPointer;
      (*this)["polygons"]=new VectorObjectPointer;
      (*this)["materials"]=new VectorObjectPointer;
      (*this)["files"]=new VectorObjectPointer;
    }

    /// Set name of mesh
    void SetName(const std::string& name) { 
      File* s=(File*)(*this)["name"].Get();
      s->SetName(name);
    }

    const std::string& GetName() const {
      File* s=(File*)(*find("name")).second.Get();
      return s->GetName();
    }

    /**
     *  Set number of points
     *  
     *  @param int number of points wanted
     */
    void SetNbPoints(int i) {
      VectorObjectPointer* a=(*find("points")).second->AsVectorObjectPointer();
      SetNbElements<Point>(i,*a);
    }

    /**
     *  Set number of polygon
     *  
     *  @param int number of polygons wanted
     */
    void SetNbPolygons(int i) {
      VectorObjectPointer* a=(*find("polygons")).second->AsVectorObjectPointer();
      SetNbElements<Polygon>(i,*a);
    }

    /**
     *  Set number of materials
     *  
     *  @param int number of materials wanted
     */
    void SetNbMaterials(int i) {
      VectorObjectPointer* a=(*find("materials")).second->AsVectorObjectPointer();
      SetNbElements<Material>(i,*a);
    }

    /**
     *  Set number of files, @see File
     *
     *  @param int number file wanted
     */
    void SetNbFiles(int i) {
      VectorObjectPointer* a=(*find("files")).second->AsVectorObjectPointer();
      SetNbElements<File>(i,*a);
    }

    /// Accessors
    int GetNbPoints() const {
      return (*find("points")).second->AsVectorObjectPointer()->size();
    }

    /// Accessors
    int GetNbPolygons() const {
      return (*find("polygons")).second->AsVectorObjectPointer()->size();
    }

    /// Accessors
    int GetNbMaterials() const {
      return (*find("materials")).second->AsVectorObjectPointer()->size();
    }

    /// Accessors
    int GetNbFiles() const {
      return (*find("files")).second->AsVectorObjectPointer()->size();
    }

    /**
     *  Return the ith polygon
     *
     *  @param int polygon wanted
     *  @return Polygon*
     */
    Polygon* GetPolygon(int i) {
      VectorObjectPointer* a;
      a=(*find("polygons")).second->AsVectorObjectPointer();
      return (*a)[i]->AsPolygon();
    }

    /**
     *  Return the ith Material
     *
     *  @param int material wanted
     *  @return Material*
     */
    Material* GetMaterial(int i) {
      VectorObjectPointer* a;
      a=(*find("materials")).second->AsVectorObjectPointer();
      return (*a)[i]->AsMaterial();
    }

    /**
     *  Return the ith File
     *
     *  @param int file wanted
     *  @return File*
     */
    File* GetFile(int i) {
      VectorObjectPointer* a;
      a=(*find("files")).second->AsVectorObjectPointer();
      return (*a)[i]->AsFile();
    }

    /**
     *  Return the ith point
     *
     *  @param int point wanted
     *  @return Point*
     */
    Point* GetPoint(int i) {
      VectorObjectPointer* a;
      a=(*find("points")).second->AsVectorObjectPointer();
      return (*a)[i]->AsPoint();
    }

    void Traverse(Visitor& visitor);

    virtual Mesh* AsMesh(void) { return this; }
    virtual const Mesh* AsMesh(void) const { return this; }
    virtual int GetType(void) const { return MESH; }
    inline static Object* Create() { return new Mesh; }

    std::ostream& operator<<(std::ostream& o);

    virtual void ORepair(IOContext* context);
  };

}

#endif         
