/* -*-c++-*- 
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */
/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2003 Robert Osfield 
 *
 * This library is open source and may be redistributed and/or modified under  
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or 
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * OpenSceneGraph Public License for more details.
*/

#ifndef _EXG_OBJECT_H
#define _EXG_OBJECT_H

#include <cassert>
#include <iostream>
#include <string>
#include <map>
#include <vector>

#include <exg/exg_export.h>

namespace exg {

  class Visitor;

  template<class T> class Pointer {
  public:
    typedef T ElementType;

    Pointer() :mPtr(0L) {}
    Pointer(T* t):mPtr(t)              { if (mPtr) mPtr->Ref(); }
    Pointer(const Pointer& rp):mPtr(rp.mPtr)  { if (mPtr) mPtr->Ref(); }
    ~Pointer()                           { if (mPtr) mPtr->Unref(); mPtr=0; }

    inline Pointer& operator = (const Pointer& rp)
      {
	if (mPtr==rp.mPtr) return *this;
	T* tmpPtr = mPtr;
	mPtr = rp.mPtr;
	if (mPtr) mPtr->Ref();
	// Unref second to prevent any deletion of any object which might
	// be referenced by the other object. i.e rp is child of the
	// original mPtr.
	if (tmpPtr) tmpPtr->Unref();
	return *this;
      }

    inline Pointer& operator = (T* ptr)
      {
	if (mPtr==ptr) return *this;
	T* tmpPtr = mPtr;
	mPtr = ptr;
	if (mPtr) mPtr->Ref();
	// Unref second to prevent any deletion of any object which might
	// be referenced by the other object. i.e rp is child of the
	// original mPtr.
	if (tmpPtr) tmpPtr->Unref();
	return *this;
      }

    // comparison operators for Pointer.
    inline bool operator == (const Pointer& rp) const { return (mPtr==rp.mPtr); }
    inline bool operator != (const Pointer& rp) const { return (mPtr!=rp.mPtr); }
    inline bool operator < (const Pointer& rp) const { return (mPtr<rp.mPtr); }
    inline bool operator > (const Pointer& rp) const { return (mPtr>rp.mPtr); }

    // comparion operator for const T*.
    inline bool operator == (const T* ptr) const { return (mPtr==ptr); }
    inline bool operator != (const T* ptr) const { return (mPtr!=ptr); }
    inline bool operator < (const T* ptr) const { return (mPtr<ptr); }
    inline bool operator > (const T* ptr) const { return (mPtr>ptr); }


    inline T& operator*()  { return *mPtr; }

    inline const T& operator*() const { return *mPtr; }

    inline T* operator->() { return mPtr; }

    inline const T* operator->() const   { return mPtr; }

    inline bool operator!() const	{ return mPtr==0L; }

    inline bool Valid() const	{ return mPtr!=0L; }
        
    inline T* Get() { return mPtr; }

    inline const T* Get() const { return mPtr; }

    inline T* Take() { return this->release();}

    inline T* Release() { T* tmp=mPtr; if (mPtr) mPtr->UnrefNoDelete(); mPtr=0; return tmp;}


  private:
    T* mPtr;
  };

  class DeleteHandler;
  class IOContext;
  class VectorInt;
  class VectorFloat;
  class VectorString;
  class VectorObjectPointer;
  class MapObjectPointer;
  class Point;
  class Vertex;
  class Polygon;
  class Material;
  class File;
  class Mesh;

#define COMMON_FUNCTIONS(name) \
        virtual void Accept(Visitor& visitor) { if(visitor.CheckAndTagNode(*this)) { visitor.PushOntoNodePath(this); visitor.Apply(*this); visitor.PopFromNodePath(); } } \
        virtual char* ClassName(void) { return  #name; }
	
  class EXG_EXPORT Object {
  public:
    Object() 
      {
        indent=0;
	mRefCount=0;
      }
    Object(const Object&) {
      mRefCount=0;
    }

    enum {
      NONE = 0,
      MAP_OBJECT_POINTER,
      VECTOR_INT,
      VECTOR_FLOAT,
      VECTOR_STRING,
      VECTOR_OBJECT_POINTER,
      POINT,
      VERTEX,
      POLYGON,
      MESH,
      FILE,
      MATERIAL
    };

    typedef std::map<int,Object* (*)(void)> Factory;

    inline Object& operator = (const Object&) { return *this; }

    friend class DeleteHandler;

    /** Set a DeleteHandler to which deletion of all referenced counted objects
     * will be delegated to.*/
    static void setDeleteHandler(DeleteHandler* handler);

    /** Get a DeleteHandler.*/
    static DeleteHandler* GetDeleteHandler();


    /** increment the reference count by one, indicating that 
	this object has another pointer which is referencing it.*/
    inline void Ref() const { ++mRefCount; }
        
    /** decrement the reference count by one, indicating that 
	a pointer to this object is referencing it.  If the
	reference count goes to zero, it is assumed that this object
	is no longer referenced and is automatically deleted.*/
    inline void Unref() const;
        
    /** decrement the reference count by one, indicating that 
	a pointer to this object is referencing it.  However, do
	not delete it, even if ref count goes to 0.  Warning, UnrefNoDelete() 
	should only be called if the user knows exactly who will
	be resonsible for, one should prefer Unref() over UnrefNoDelete() 
	as the later can lead to memory leaks.*/
    inline void UnrefNoDelete() const { --mRefCount; }
        
    /** return the number pointers currently referencing this object. */
    inline int ReferenceCount() const { return mRefCount; }

    virtual char* ClassName(void) { return "Object"; }

    virtual VectorInt* AsVectorInt(void) { return 0; }
    virtual const VectorInt* AsVectorInt(void) const { return 0; }
    virtual VectorFloat* AsVectorFloat(void) { return 0; }
    virtual const VectorFloat* AsVectorFloat(void) const { return 0; }
    virtual VectorString* AsVectorString(void) { return 0; }
    virtual const VectorString* AsVectorString(void) const { return 0; }
    virtual VectorObjectPointer* AsVectorObjectPointer(void) { return 0; }
    virtual const VectorObjectPointer* AsVectorObjectPointer(void) const { return 0; }
    virtual MapObjectPointer* AsMapObjectPointer(void) { return 0; }
    virtual const MapObjectPointer* AsMapObjectPointer(void) const { return 0; }
    virtual Point* AsPoint(void) { return 0; }
    virtual const Point* AsPoint(void) const { return 0; }
    virtual Vertex* AsVertex(void) { return 0; }
    virtual const Vertex* AsVertex(void) const { return 0; }
    virtual Polygon* AsPolygon(void) { return 0; }
    virtual const Polygon* AsPolygon(void) const { return 0; }
    virtual Material* AsMaterial(void) { return 0; }
    virtual const Material* AsMaterial(void) const { return 0; }
    virtual Mesh* AsMesh(void) { return 0; }
    virtual const Mesh* AsMesh(void) const { return 0; }
    virtual File* AsFile(void) { return 0; }
    virtual const File* AsFile(void) const { return 0; }

    inline virtual void OSave(std::ostream& out, IOContext* context = 0) const { }
    inline virtual void OLoad(std::istream& in, IOContext* context = 0) { }
    inline virtual void ORepair(IOContext* context) { }

    virtual int GetType(void) const = 0;

    static Factory& GetFactory(void);


    virtual void Accept(Visitor& visitor);

    virtual void Traverse(Visitor& visitor) {}

    virtual std::ostream& operator<<(std::ostream& o) {return o;}

    void Indent(int nb,std::ostream& o);

    virtual Object* Clone() { assert(0); return 0;}

    virtual bool CompareLessThan(Object* _compare) = 0;

  protected:
    virtual ~Object();
    mutable int mRefCount;
    static Factory* mFactory;
    static int indent;
  };

  void Save(const float& what, std::ostream& out, IOContext* context);
  void Save(const int& what, std::ostream& out, IOContext* context);
  void Save(const std::string& what, std::ostream& out, IOContext* context);
  void EXG_EXPORT Save(const Object* what, std::ostream& out, IOContext* context = 0);
  inline void EXG_EXPORT Save(const Object* what) {
    Save(what, std::cout);
  }
  inline void EXG_EXPORT Save(const Pointer<Object>& what, std::ostream& out, IOContext* context = 0) { Save(((Pointer<Object>&)what).Get(), out, context); }
  inline void EXG_EXPORT Save(const Object& what, std::ostream& out, IOContext* context = 0) {
    const Object* tmp = &what;
    Save(tmp, out, context);
  }

  void Load(float& what, std::istream& in, IOContext* context);
  void Load(int& what, std::istream& in, IOContext* context);
  void Load(std::string& what, std::istream& in, IOContext* context);
  void EXG_EXPORT Load(Pointer<Mesh>& what, std::istream& in, IOContext* context = 0);
  void EXG_EXPORT Load(Pointer<Object>& what, std::istream& in, IOContext* context = 0);
  inline void EXG_EXPORT Load(Pointer<Object>& what) {
    Load(what, std::cin);
  }
 
  void Repair(Object* what, IOContext* context);

  class EXG_EXPORT IOContext {
  public:
    IOContext() : mSerial(1) {}

    std::map<const Object*,int> mObject2Serial;
    std::map<int,Pointer<Object> > mSerial2Object;
    int mSerial;
  };
    

  /** Class for override the default delete behavior so that users can
   * implment their own object deletion schemes.  This might be done to
   * help implement protection of multiple threads from deleting
   * objects unintentionally.  Note, the DeleteHandler cannot itself be
   * reference counted, otherwise it would be responsible for deleting
   * itself!  An static auto_ptr<> is used internally in Object.cpp to
   * manage the DeleteHandler's memory.*/
  class DeleteHandler {
  public:

    virtual ~DeleteHandler() {}

    /** Flush any cache of objects that need to be deleted by doing an actual delete.*/
    virtual void Flush() {}
        
    inline void DoDelete(const Object* object) { delete object; }
         
    /** Request the deletion of an object. 
     * Depending on users implementation of DeleteHandler, the delete of the object may occur 
     * straight away or be delayed until DoDelete is called.
     * The default implementation does a delete straight away.*/
    virtual void RequestDelete(const Object* object) { DoDelete(object); }
  };

  inline void Object::Unref() const {
    --mRefCount;
    if (mRefCount==0) {
      if (GetDeleteHandler()) GetDeleteHandler()->RequestDelete(this);
      else delete this;
    } else if (mRefCount<0) {
      throw 2325;
    }
  }
};

#endif // _EXG_OBJECT_H 
