/**
   Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   ****************************************************************************
   * @file   mth_plane3.h
   *
   * @brief   Oriented plane
   *
   *****************************************************************************
   *
   * @author  Cedric Pinson
   *
   * @date    Created 2001/08
   *
   * @version $Id: exg_plane3.h,v 1.4 2004/05/02 12:55:42 cpinson Exp $
   *
   ****************************************************************************/

#ifndef exg_plane3_h
#define exg_plane3_h

#include <exg/exg_vector3.h>

namespace exg
{

  /** Oriented plane
   *
   * An oriented plane is described by a normal and a signed distance to the
   * origin with respect to this normal. 
   *
   */
  template <class T> class Plane3_T {

    /// Normal of plane
    Vector3_T<T> normal;

    /// position on axis
    T pos;

   public:


    /** Default constructor*/
    Plane3_T() { }


    /** Initialize a plane with a normal and a signed position*/
    Plane3_T(const Vector3_T<T>& n, const T p) : normal(n), pos(p) { }



    template <class C> Plane3_T(const Plane3_T<C>& v)
      { normal=(Vector3_T<T>)v.normal; pos=(T)v.pos; }


    /** Initialise a plane passing through three points v, v1 and v2
     *
     * @param v  point to pass through
     * @param v1 point to pass through
     * @param v2 point to pass through
     *
     */
    void Init(const Vector3_T<T>& v, const Vector3_T<T>& v1, const Vector3_T<T>& v2) {
      normal = (v1-v).Cross((v2-v));
      normal.Normalize();
      pos = v.Dot(normal);
    }


    /** Initialize a plane with a normal and a signed position*/
    void Init(const Vector3_T<T>& n, const T p) { normal=n; pos=p; }


    /** Initialize a plane with a normal and a signed position*/
    void Init(const Plane3_T<T>& n) { normal=n.normal; pos=n.pos; }



    /** Initialize a plane with a normal and a vector*/
    void Init(const Vector3_T<T>& n, const Vector3_T<T> &p) { normal=n; pos=n*p ; }



    /** Constructor from three points*/
    Plane3_T(const Vector3_T<T>& v, const Vector3_T<T>& v1, const Vector3_T<T>& v2) {
      Init(v, v1, v2);
    }


    /** Return the signed distance of a point from the plane.
     *
     * @param p Point to test
     *
     * @return positive if in the front half space, negative otherwise.
     *
     */
    T Dist(const Vector3_T<T>& p) const {
      return p.Dot(normal) - pos;  }



    /// Return normal
    const Vector3_T<T>& Direction() const { return normal;}

    /// Return position on axis
    T Position() const { return pos;}




       
    /// unari minus operator
    Plane3_T operator - () const {
      return Plane3_T(-normal, -pos);  }


    /// operator translation
    Plane3_T operator + (const Vector3_T<T> & v) const {
      return Plane3_T(normal, pos + v.Dot(normal));
    }


    /// operator translation inverse
    Plane3_T operator - (const Vector3_T<T> & v) const {
      return Plane3_T(normal, pos - v.Dot(normal));
    }


  };

  typedef Plane3_T<float> Plane3f;
  typedef Plane3_T<double> Plane3d;

}

#endif
