/**
   Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   ****************************************************************************
   * @file   exg_polygon.h
   *
   * @brief   Describe polygon
   *
   *****************************************************************************
   *
   * @author  Cedric Pinson
   *
   * @date    Created 2001/04
   *
   * @version $Id: exg_polygon.h,v 1.24 2004/11/10 10:39:25 izidor79 Exp $
   *
   ****************************************************************************/

#ifndef exg_polygon_h
#define exg_polygon_h

#include <exg/exg_plane3.h>
#include <exg/exg_basetypes.h>
#include <exg/exg_vertex.h>
#include <algorithm>

namespace exg
{
  class Material;
  class Visitor;

  /**
   *  Polygon class
   */
  class EXG_EXPORT Polygon : public MapObjectPointer
  {
   protected:
    virtual ~Polygon() {}

   public:
    COMMON_FUNCTIONS(Polygon)

    /**
     *  Default constructor
     */
    Polygon() {
      (*this)["vertexes"]=new VectorObjectPointer;
      (*this)["material"]=0;
    }


    /// Return number of vertexes for this polygon
    int GetNbVertexes() const { 
      return (*find("vertexes")).second->AsVectorObjectPointer()->size();}


    /**
     *  Set the number of vertex
     *
     *  @param int number of vertexes to create (it create physically all vertexes)
     *  @return int 0 if ok else -1
     */
    void SetNbVertexes(int n, bool bGenerate = true) {
      int nb=GetNbVertexes();
      VectorObjectPointer* a=(*find("vertexes")).second->AsVectorObjectPointer();
      a->resize(n);
      if (n<=nb)
        return;
	  if (bGenerate)
		generate(a->begin()+nb,a->end(),Vertex::Create);
    }


    /**
     *  Return the ith vertex of polygon
     *
     *  @param int ith vertex to get
     *  @return Vertex* vertex to return
     */
    Vertex* GetVertex(int i);


    /**
     *  Return the ith vertex of polygon
     *
     *  @param int ith vertex to get
     *  @return Vertex* vertex to return
     */
    const Vertex* GetVertex(int i) const { 
      const VectorObjectPointer* v=(*find("vertexes")).second->AsVectorObjectPointer();
      return (*v)[i]->AsVertex();
    }



    /**
     *  Return the material of polygon
     *
     *  @return Material* material of polygon
     */
    Material* GetMaterial() { 
      return (*this)["material"].Get()->AsMaterial();
    }



    /**
     *  Assign the material and init the polygon with the properties of material
     *
     *  @param Material* material to assign
     */
    void SetMaterial(Material* material);


    /**
     *  Return material of polygon
     *
     *  @return const Material*
     */
    const Material* GetMaterial() const {
      return GetProperty("material")->AsMaterial();
    }


    /**
     *  Return the plane of polygon
     *
     *  @return Plane3 plane of polygon
     */
    Plane3f GetPlane() const {
      return Plane3f(GetVertex(2)->GetPoint()->GetPosition(),
                     GetVertex(1)->GetPoint()->GetPosition(),
                     GetVertex(0)->GetPoint()->GetPosition());
    }

    void Traverse(Visitor& visitor);

    virtual Polygon* AsPolygon(void) { return this; }
    virtual const Polygon* AsPolygon(void) const { return this; }
    virtual int GetType(void) const { return POLYGON; }
    inline static Object* Create() { return new Polygon; }


    std::ostream& operator<<(std::ostream& o);

  };

}

#endif
