/**
    Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 ****************************************************************************
 * @file   mth_vector3.h
 *
 * @brief   3 dimensional vectors
 *
 *****************************************************************************
 *
 * @author  psc80
 *
 * @date    Created 2001/05
 *
 * @version $Id: exg_vector3.h,v 1.5 2004/04/28 10:37:19 loic Exp $
 *
 ****************************************************************************/

#ifndef exg_vector3_h
#define exg_vector3_h


#include <cmath>

namespace exg {

  
  template <class T> class Vector3_T {

   protected:

    T c[3];
   
   public:


    /// do nothing
    Vector3_T() {}


    /// constructor with scalar
    inline Vector3_T( T _a, T _b, T _c) {
      Init(_a,_b,_c);
    }



    /// Initialize the vector with another vector
    template <class C> explicit Vector3_T(const Vector3_T<C>& _v) {
      c[0]=(T)_v[0]; c[1]=(T)_v[1]; c[2]=(T)_v[2];
    }


    /// Set the vector to 0
    inline void Init() {
      c[0]=0; c[1]=0; c[2]=0;
    }


    /// Set the vector to 0
    inline void Init( T _a, T _b, T _c) {
      c[0]=_a; c[1]=_b; c[2]=_c;
    }


    /// Initialize the vector with another vector
    template <class C> void Init(const Vector3_T<C>& _v) {
      c[0]=(T)_v.c[0]; c[1]=(T)_v.c[1]; c[2]=(T)_v.c[2];
    }



    /// Compute the dot product against another vector
    inline T Dot(const Vector3_T& _v) const {
      return c[0]*_v.c[0]+c[1]*_v.c[1]+c[2]*_v.c[2];
    }



    /// Multiply by row
    inline Vector3_T Multiply(const Vector3_T& _v) const {
      return Vector3_T(c[0]*_v.c[0],c[1]*_v.c[1],c[2]*_v.c[2]);
    }



    /// Return the squared magnitude (length) of the vector
    inline T MagSqr() const { 
      return c[0]*c[0]+c[1]*c[1]+c[2]*c[2];
    }


    /// Return the magnitude (length) of the vector
    inline T Mag() const { return (T)sqrt(MagSqr()); }


    
    /// Return the squared distance between two vectors
    inline T DistSqr(const Vector3_T& _v) const {
      return (*this-_v).MagSqr();}


    /// Return the distance between two vectors
    inline T Dist(const Vector3_T& _v) const {
      return (*this-_v).Mag(); }


    
    /** Normalize this vector
     *
     */
    inline void Normalize() {
      T a = MagSqr();
      *this *= (T)(1./sqrt(a));
    }


    /// Cross product
    inline Vector3_T Cross( const Vector3_T& _v ) const {
      return Vector3_T(c[1]*_v.c[2] - c[2]*_v.c[1], 
                       c[2]*_v.c[0] - c[0]*_v.c[2], 
                       c[0]*_v.c[1] - c[1]*_v.c[0]);
    }






    inline Vector3_T operator ^ (const Vector3_T& _v) const {
      return this->Cross(_v); }


    /// Coordinates accessor operator
    inline T& operator [] (int _i) { return c[_i]; }

    
    /// Const coordinates accessor operator
    inline const T& operator [] (int _i) const { return c[_i]; }


    /// Operator * (dot product)
    inline T operator * (const Vector3_T& _v) const {
      return this->Dot(_v);}


    /// Addition of two vectors
    inline Vector3_T operator + (const Vector3_T& _v) const {
      return Vector3_T(c[0]+_v.c[0],c[1]+_v.c[1],c[2]+_v.c[2]);
    }


    /// Substract of two vectors
    inline Vector3_T operator - (const Vector3_T& _v) const {
      return Vector3_T(c[0]-_v.c[0],c[1]-_v.c[1],c[2]-_v.c[2]);
    }


    /// Comparaison operator
    bool operator == ( const Vector3_T& _v ) const {
      return c[0]==_v.c[0] && c[1]==_v.c[1] && c[2]==_v.c[2];
    }


    /// Comparaison operator
    bool operator != ( const Vector3_T& _v ) const {
      return !(*this==_v);}


    /// += operator
    inline const Vector3_T& operator += ( const Vector3_T& _v ) {
      c[0]+=_v.c[0]; c[1]+=_v.c[1]; c[2]+=_v.c[2];
      return *this;}


    /// -= operator
    inline const Vector3_T& operator -= ( const Vector3_T& _v ) {
      c[0]-=_v.c[0]; c[1]-=_v.c[1]; c[2]-=_v.c[2];
      return *this;}


    /// *= operator
    inline const Vector3_T& operator *= ( T _v ) {
      c[0]*=_v; c[1]*=_v; c[2]*=_v;
      return *this;}


    /// /= operator
    inline const Vector3_T& operator /= ( T _v ) {
      T a=(T)1.0/_v;
      c[0]*=a; c[1]*=a; c[2]*=a;
      return *this;}



    /// Operator -
    inline Vector3_T operator - () const {
      return Vector3_T(-c[0],-c[1],-c[2]);
    }


    /// Mutliplication by a scalar value (to the right)
    inline Vector3_T operator * ( T _a) const {
      return Vector3_T(c[0]*_a,c[1]*_a,c[2]*_a);
    }


    /// Division by a scalar value
    inline Vector3_T operator / (T _v) const {
      T a=(T)1.0/_v;
      return Vector3_T(c[0]*a,c[1]*a,c[2]*a);
    }

  };


  template <class T> inline Vector3_T<T> operator * (T _a, const Vector3_T<T>& _v) {
    return Vector3_T<T>(_a*_v[0], _a*_v[1], _a*_v[2]); }


  typedef Vector3_T<float> Vector3f;
  typedef Vector3_T<double> Vector3d;

}
#endif
