/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Igor kravtchenko <igor@obraz.net>
 *
 */

#ifndef maf_data_h 
#define maf_data_h

#ifndef MAF_USE_VS_PCH

#include <string>
#include <vector>

#include <glib.h>
#include <vector>
#ifdef WIN32
#include <tinyxml/tinyxml.h>
#else
#include <cal3d/tinyxml.h>
#endif

#include <exg/exg_matrix4.h>
#include <exg/exg_vector3.h>

#include <osg/ref_ptr>
#include <osg/Matrix>
#include <osg/LightSource>
#include <osgDB/ReaderWriter>

#include <libxml/tree.h>
#include <libxml/xpath.h>

#include <AL/al.h>
#include <openalpp/alpp.h>

#include <maf/mafexport.h>

#endif

#include <osgCal/CoreModel>

#include <maf/utils.h>
#include <maf/camera.h>
#include <maf/anchor.h>

namespace osgText { class Font; }

class MAF_EXPORT MAFData {
public:
  MAFData() { }
  virtual ~MAFData() {}

  virtual bool Load(const std::string& path, osgDB::ReaderWriter::Options* options) = 0;
  virtual MAFData* Clone(void) { abort(); }
};

class MAF_EXPORT MAFVisionData : public MAFData {
public:
  MAFVisionData() {}
  virtual ~MAFVisionData() {}

  MAFAnchor* GetAnchor(const std::string& __name) { return 0; }

  MAFCameraController* GetCamera(const std::string& name);
  MAFCameraController* getFirstCamera();

  std::vector<osg::ref_ptr<osg::LightSource> >& GetLights() { return mLights; }

  osg::ref_ptr<osg::LightSource>		getLightByIndex(int) const;
  
protected:
  std::vector<osg::ref_ptr<osg::LightSource> > mLights;
  std::map<std::string, osg::ref_ptr<MAFCameraController> > mCameras;
};

class MAF_EXPORT MAFOSGData : public MAFVisionData {
public:
  MAFOSGData() {}
  virtual ~MAFOSGData() {}

  virtual bool Load(const std::string& path, osgDB::ReaderWriter::Options* options);
  virtual MAFData* Clone(void);

  void SetDescription(const std::string& name);
  MAFAnchor* GetAnchor(const std::string& name);
  void GroupAnchors(osg::Group* group, const std::vector<std::string>& names);
  osg::BoundingBox GetBound();
  osg::Node* GetNode(const std::string& name);

  osg::Group* GetGroup(void) { return mGroup.get(); }

protected:
  osg::ref_ptr<osg::Group> mGroup;
};

class MAF_EXPORT MAFESCNData : public MAFOSGData
{
public:

	explicit		MAFESCNData		(void);
	virtual			~MAFESCNData	(void);

	bool			Load(const std::string& path, const std::string& dir, const std::string& file, osgDB::ReaderWriter::Options* options);

	osg::Node *		Convert(TiXmlElement * pElt, osg::Group* parent, const std::string& parentPath, osgDB::ReaderWriter::Options* options);


	bool			ParseAnimation	(TiXmlElement * pElt, osg::Group* parent);
	void			getAttribute	(TiXmlElement * pElt, const std::string & strName, exg::Matrix4f & m);
	void			getAttribute	(TiXmlElement * pElt, const std::string & strName, exg::Vector4f & v);

	exg::Vector4f	getAmbientColor() const { return mAmbientColor; }
	exg::Vector4f	getBackGroundColor() const { return mBackGroundColor; }
	exg::Vector4f	getFogColor() const { return mFogColor; }
	bool			isUseFog() const { return mbUseFog; }

	// set the global states set of the scene
	// it includes ambient, lights, fog, etc.
	void			setGlobalStateSet(osg::StateSet *);

private:

	std::string		mPath;
	std::string		mDir;
	std::string		mFile;
	int				mCurrentLight;
	exg::Vector4f	mAmbientColor;
	exg::Vector4f	mBackGroundColor;
	exg::Vector4f	mFogColor;
	bool			mbUseFog;
};

class MAFCal3dData : public MAFVisionData {
public:
  typedef std::map<std::string,std::string> MeshDescription;
  typedef std::vector<MeshDescription> Outfit;
  MAF_EXPORT MAFCal3dData();
  MAF_EXPORT virtual ~MAFCal3dData();

  MAF_EXPORT virtual bool Load(const std::string& dir, osgDB::ReaderWriter::Options* options);

  osgCal::CoreModel* GetCoreModel(void) { return mCoreModel.get(); }
  CalCoreModel* GetCalCoreModel(void) { return mCoreModel->getCalCoreModel(); }

  bool GetConfigurable(void) { return mConfigurable; }
  MAF_EXPORT MAFCal3dData::Outfit* GetOutfit(const std::string& outfitName);

  MAF_EXPORT int getNbOutfits() const;
  MAF_EXPORT const char* getOutfitNameByIndex(int) const;

  // public to hack it called from pokerbody
  MAF_EXPORT bool GetOutfit(Outfit* outfit, const std::string& xpath);
private:
  MAF_EXPORT void LoadCFG(const std::string& dir, const std::string& path);
  MAF_EXPORT void LoadXML(const std::string& dir, const std::string& path);

  osg::ref_ptr<osgCal::CoreModel> mCoreModel;
  bool mConfigurable;
  xmlDocPtr mDocument;
  xmlXPathContextPtr mXpath;
  std::string mPath;
};

class MAF_EXPORT MAFAudioData : public MAFData {
 public:
  MAFAudioData():mSoundData(0) {}
  virtual ~MAFAudioData();
  
  virtual bool Load(const std::string& path, osgDB::ReaderWriter::Options* options) = 0;

  openalpp::SoundData* GetSoundData() { return mSoundData.get();}

 protected:
  void MAFAudioData::Error(const openalpp::Error& error);

  osg::ref_ptr<openalpp::SoundData> mSoundData;
};

class MAF_EXPORT MAFAudioDataWAV : public MAFAudioData {
 public:
  MAFAudioDataWAV(){}
  virtual ~MAFAudioDataWAV(){}
  
  virtual bool Load(const std::string& path, osgDB::ReaderWriter::Options* options);
};


class MAF_EXPORT MAFAudioDataOGG : public MAFAudioData {
 public:
  MAFAudioDataOGG() {}
  virtual ~MAFAudioDataOGG() {}
  
  virtual bool Load(const std::string& path, osgDB::ReaderWriter::Options* options);
};

class MAF_EXPORT MAFXmlData : public MAFData {
 public:
  MAFXmlData();
  virtual ~MAFXmlData();
  virtual bool Load(const std::string &path, osgDB::ReaderWriter::Options* options);


  std::string MAFXmlData::Get(const std::string& path);
  std::list<std::string> MAFXmlData::GetList(const std::string& path);
 private:
  xmlDocPtr mDocument;
};


class MAF_EXPORT MAFCursorData : public MAFData {
 public:
  MAFCursorData();
  virtual ~MAFCursorData();
  virtual bool Load(const std::string &path, osgDB::ReaderWriter::Options* options);
  
  SDL_Cursor *GetCursor();
  SDL_Cursor *CreateCursor();
  SDL_Cursor *GetOrCreateCursor();
 private:  
  SDL_Cursor *mCursor;
  Uint8 mData[256];
};

class XwncDesktop;

class MAF_EXPORT MAFRepositoryData {
public:
  MAFRepositoryData()
	  : mDesktop(0)
  {
	
  }
  ~MAFRepositoryData();

  void Load(const std::string& dir);
  
  MAFVisionData* GetVision(const std::string& name);
  MAFAudioData* GetAudio(const std::string& name) { return mAudioMap[name]; }
  MAFXmlData* GetXml(const std::string &name) { return mXmlMap[name]; }
  MAFCursorData *GetCursor(const std::string &name) {return mCursorMap[name]; }
  
  XwncDesktop*	GetDesktop() const
  {
    return mDesktop;
  }

  void XwncConnect(const std::string& url);

private:  
  std::map<std::string, MAFVisionData*> mVisionMap;
  std::map<std::string, MAFAudioData*> mAudioMap;
  std::map<std::string, MAFXmlData*> mXmlMap;
  std::map<std::string, MAFCursorData*> mCursorMap;
  XwncDesktop* mDesktop;
};

MAF_EXPORT osgText::Font* MAFLoadFont(const std::string& font_file, osgDB::ReaderWriter::Options* options);
MAF_EXPORT osg::Image* MAFLoadImage(const std::string& image_file, osgDB::ReaderWriter::Options* options);

#endif // maf_data_h
