#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#
# 
import sys
from time import time

from twisted.internet import reactor, protocol

from underware.packets import Packet, PacketFactory, PacketNames

PROTOCOL_MAJOR = 0
PROTOCOL_MINOR = 2

class UGAMEProtocol(protocol.Protocol):
    """UGAMEProtocol"""

    def __init__(self):
        self._packet = []
        self._packet_len = 0
        self._handler = self._handleVersion
        self._already_expecting = 0
        self._hold_enable = True
        self._hold_packets = False
        self._hold_predicate = None
        self._packets = []
        self._timer = False
        self._lagmax = 0
        self._prefix = ""
        self.established = 0

    def connectionMade(self):
        "connectionMade"
        self._sendVersion()

    def connectionLost(self, reason):
        self.established = 0
        
    def _sendVersion(self):
        self.transport.write('CGI %03d.%03d\n' % ( PROTOCOL_MAJOR, PROTOCOL_MINOR ) )

    def _handleConnection(self):
        pass

    def ignoreIncomingData(self):
        self._handler = self._voidHandler

    def _voidHandler(self):
        self._packets = []
        self._packet_len = 0
        
    def _handleVersion(self):
        buffer = ''.join(self._packet)
        if '\n' in buffer:
            if buffer[:3] == 'CGI':
                major, minor = [int(x) for x in buffer[4:11].split('.')]
                if (major, minor) != ( PROTOCOL_MAJOR, PROTOCOL_MINOR ):
                    self.protocolInvalid("%03d.%03d" % ( major, minor ), "%03d.%03d" % ( PROTOCOL_MAJOR, PROTOCOL_MINOR ))
                    self.transport.loseConnection()
                    return
            buffer = buffer[12:]
            self.established = 1
            self._packet[:] = [buffer]
            self._packet_len = len(buffer)
            self._handler = self._handleExpected
            self._expected_len = Packet.format_size
            self.expect(self._handleConnection)
            if self.factory.verbose > 1:
                print "protocol established"
            self.protocolEstablished()
        else:
            self._packet[:] = [buffer]
            self._packet_len = len(buffer)

    def protocolEstablished(self):
        pass

    def protocolInvalid(self, server, client):
        pass
    
    def expect(self, handler, *args, **kwargs):
        self._expected_handler = handler
        self._expected_args = args
        self._expected_kwargs = kwargs
        self._handleExpected()

    def hold(self, delay, predicate):
        if not self._hold_enable:
            return
        
        if self._timer and self._timer.active():
            self._timer.cancel()

        if delay > 0:
            if ( hasattr(self, "_lagstart") and
                 self._lagmax > 0 and
                 time() - self._lagstart > self._lagmax ):
                if self.factory.verbose > 1:
                    print " => lagging %f seconds, hold not honored" % (time() - self._lagstart)
                self._handleExpected()
            else:
                self._hold_predicate = predicate
                self._hold_packets = True
                self._timer = reactor.callLater(delay, self.hold, 0, None)
        else:
            self._hold_packets = False
            self._hold_predicate = None
            self._timer = False
            self._handleExpected()
                              
    def _handleExpected(self):
        if self._already_expecting:
            return
        
        self._already_expecting = 1

        if not self._hold_predicate:
            while len(self._packets) > 0:
                packet = self._packets.pop(0)
                self._expected_handler(packet, *self._expected_args, **self._expected_kwargs)
        
        if self._packet_len >= self._expected_len:
            type = Packet()
            buffer = ''.join(self._packet)
            while len(buffer) >= self._expected_len:
                type.unpack(buffer)
                if type.length <= len(buffer):
                    
                    if PacketFactory.has_key(type.type):
                        packet = PacketFactory[type.type]()
                        buffer = packet.unpack(buffer)
                        if self.factory.verbose > 1:
                            print "%s(%d bytes) => %s" % ( self._prefix, type.length, packet )
                        if ( self._hold_predicate and
                             self._hold_predicate(packet) ):
                            self._packets.append(packet)
                        else:
                            self._expected_handler(packet, *self._expected_args, **self._expected_kwargs)
                            
                    else:
                        if self.factory.verbose > 1:
                            print "unknown message received (id %d, length %d)\n" % ( type.type, type.length )
                        if self.factory.verbose > 2:
                            print "known types are %s " % PacketNames
                        buffer = buffer[type.length:]
                    self._expected_len = Packet.format_size
                else:
                    self._expected_len = type.length
            # print "expecting %d bytes" % self._expected_len
            self._packet[:] = [buffer]
            self._packet_len = len(buffer)

        if len(self._packets) > 0:
            if self._hold_packets:
                #
                # The packets are being buffered, we should check if
                # the lag introduced is not growing above an acceptable
                # threshold.
                #
                if not hasattr(self, "_lagstart"):
                    #
                    # The lagstart timestamp marks the moment we started
                    # to bufferize packets. It will be deleted only
                    # when this function is called and no packet awaits
                    # (see below)
                    #
                    self._lagstart = time()
                elif self._lagmax > 0:
                    #
                    # If the lagging time is too high, cancel the
                    # packet bufferization.
                    #
                    if self.factory.verbose > 1:
                        print " => lagging %f seconds behind" % (time() - self._lagstart)
                    if time() - self._lagstart > self._lagmax:
                        self.hold(0, None)
        else:
            #
            # If there are no pending packets it means that
            # we are waiting for the network to feed us with
            # data, therefore we are not lagging behind
            #
            if hasattr(self, "_lagstart"):
                del self._lagstart

        while not self._hold_packets and len(self._packets) > 0:
            packet = self._packets.pop(0)
            self._expected_handler(packet, *self._expected_args, **self._expected_kwargs)
        self._already_expecting = 0

    def dataReceived(self, data):
        self._packet.append(data)
        self._packet_len += len(data)
        self._handler()

