/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *  Henry Precheur <henry@precheur.org>
 */

#include "mafStdAfx.h"

#ifdef _DEBUG // for Windows python23_d.lib is not in distribution... ugly but works
#	undef _DEBUG
#	include <Python.h>
#	define _DEBUG
#else
#	include <Python.h>
#endif

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstring>
#include <iomanip>
#include <iostream>
#include <sstream>

#include <maf/maferror.h>
#include <maf/application.h>
#include "maf/packets.h"
#include <maf/profile.h>
#include <maf/timer.h>
#include <maf/window.h>

#ifdef WIN32
#include <winsock.h>
#else
#include <sys/select.h>
#endif
#include <time.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <SDL.h>

#endif

MAFApplication::MAFApplication()
{
  mRunning = true;
  mExitCode = 0;
  mWindow = NULL;
  mWindowWidth = 1024;
  mWindowHeight = 768;
  mWindowFullScreen = false;
  mFrameRateLimit = 300;
  //  mFrameRateLimit = 6;
  mLastEvent = NULL;
  mFocus = NULL;
  mFd = -1;
  mReactor = NULL;
  mClient = NULL;
  mPacketsModule = NULL;
  mReactorLastCall = 0;
  mFPSLastUpdate = GetRealTimeInMS();
  mFPSCount = 0;
  mFPS = 0;
  mRunningControllers = false;
  mAmetista = false;
  mMouseX = mMouseY = 0;
  memset(mEventLockMap, 0, sizeof (mEventLockMap));
  mOptions = new osgDB::ReaderWriter::Options;
  mCursor=0;
}

MAFApplication::~MAFApplication()
{
  mRunning = false;
  mControllers.clear();
  if(mPacketsModule)
    delete mPacketsModule;
}

void MAFApplication::SetClient( PyObject* client )
{
  mClient = client;
}

void MAFApplication::SetReactor( PyObject* reactor )
{
  mReactor = reactor;
  PyObject* waker = PyObject_GetAttrString(reactor, "waker");
  if(waker) {
    PyObject* fileno = PyObject_CallMethod(waker, "fileno", NULL);
    if(fileno) {
      mFd = PyInt_AsLong(fileno);
    } else {
      g_critical("MAFApplication::SetReactor: fileno method not found (ignored)");
    }
  } else {
      g_critical("MAFApplication::SetReactor: waker attribute not found (ignored)");
  }
  PyErr_Clear();
}

void MAFApplication::SetPacketsModule(const std::string& module) {
  mPacketsModule = new MAFPacketsModule(module);
}

MAFPacketsModule* MAFApplication::GetPacketsModule(void) {
  if(mPacketsModule == 0)
    g_critical("MAFApplication::GetPacketsModule: module is NULL");
  return mPacketsModule;
}

void MAFApplication::PythonCall(PyObject* instance, const std::string& method, MAFPacket* packet)
{
  if(instance) {
    PyObject* result = 0;
    if(packet)
      result = PyObject_CallMethod(instance, (char*)method.c_str(), "O", packet->GetPacket());
    else
      result = PyObject_CallMethod(instance, (char*)method.c_str(), NULL);
    if(result == 0)
      throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFApplication::PythonCall: failed to call %s method for python object 0x%08x", method.c_str(), instance);
    else
      Py_DECREF(result);
  } else {
    g_critical("MAFApplication::PythonCall: python interface not set, cannot call method");
  }
}

void MAFApplication::CheckReactor( void )
{
  if(PyObject_CallMethod(mReactor, "iterate", NULL) == 0)
    throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFApplication::PythonCall: failed to call iterate method for python object 0x%08x", mReactor);
}

int MAFApplication::Run()
{
  double last_update = GetRealTimeInMS();
//   double delta_min = 1000 / mFrameRateLimit;
  double delta_min = 1.0/mFrameRateLimit*1000;
  double now =last_update;
  unsigned int frameNumber=0;
  while (mRunning)
  {
    now = GetRealTimeInMS();
    double delta = now - last_update;
    mDeltaFrame=delta;
    if (delta<delta_min)
      continue;

    frameNumber++;
      {
      PROFILE_SAMPLE("MAFApplication::Run");

      /* GetLastEvent() only reports an event if it occured in this loop,
       * otherwise it is set to NULL (polling APIs are of course still available)
       */
      
      mScene->DoIntersection(this, mMouseX, mMouseY);

      while((mLastEvent = SDL_PollEvent (&mEvent) ? &mEvent : NULL)) {
	// update mouse position for intersection on the HUD
	if (mLastEvent && mLastEvent->type == SDL_MOUSEMOTION)
	  mMouseX = mLastEvent->motion.x, mMouseY = mLastEvent->motion.y;
	RunControllers();
	CheckReactor();
      } 

      // Call once with no pending event
      RunControllers();
      CheckReactor();

      /* Limit frame rate : wait if rendering took less than 1/mFrameRate
       */
      if (1 || delta >= delta_min) { // cedric: i do that in order to profile an entire frame (with update and draw)
	/* Then call view renders. If any, they are registered to the current window.
	 */
	if (mWindow)
	  mWindow->Render();

	last_update = GetRealTimeInMS();
	mFPSCount++;
      }

      if(GetRealTimeInMS() - mFPSLastUpdate > 1000) {
	mFPS = (mFPSCount * 1000.f) / (GetRealTimeInMS() - mFPSLastUpdate);
	mFPSLastUpdate = GetRealTimeInMS();
	mFPSCount = 0;
      }
    }
    last_update=now;
  }
  Uninit();

#ifdef USE_PROFILE
  std::stringstream op;
  GetProfiler()->GetRootNode()->DisplayFlatStats(op);
  g_debug("Profiling:%s", op.str().c_str());
#endif // USE_PROFILE

  return mExitCode;
}

void MAFApplication::RunControllers(void) {
  
  // it takes too much time
//   mScene->DoIntersection(this, mMouseX, mMouseY);

  mRunningControllers = true;

  ControllerList::iterator i;

  for (i = mControllers.begin(); i != mControllers.end(); i++)
    (*i)->DoUpdate(this);

  mRunningControllers = false;

  for (i = mControllersToRemove.begin(); i != mControllersToRemove.end(); i++)
    RemoveController(i->get());
  mControllersToRemove.clear();

  for (i = mControllersToAdd.begin(); i != mControllersToAdd.end(); i++)
    AddController(i->get());
  mControllersToAdd.clear();
}

void MAFApplication::Quit(int Code)
{
  mRunning = false;
  mExitCode = Code;
}


MAFWindow* MAFApplication::GetWindow(bool opengl)
{
  if(!mWindow || mWindow->GetOpenGL() != opengl)
  {
    if(mWindow) delete mWindow;
    MAFWindow* window = new MAFWindow();

    std::string param = HeaderGet("settings", "/settings/screen/@fullscreen");
    window->SetFullScreen(param == "yes");
    param = HeaderGet("settings", "/settings/screen/@width");
    if(param != "")
      window->SetWidth(atoi(param.c_str()));
    param = HeaderGet("settings", "/settings/screen/@height");
    if(param != "")
      window->SetHeight(atoi(param.c_str()));
    window->SetOpenGL(opengl);
    if(window->Init()) {
      char tmp[32];
      mWindow = window;
      snprintf(tmp, 32, "%d", window->GetWidth());
      HeaderSet("settings", "/settings/screen/@width", tmp);
      snprintf(tmp, 32, "%d", window->GetHeight());
      HeaderSet("settings", "/settings/screen/@height", tmp);
    } else {
      g_error("MAFApplication::GetWindow: failed to create window");
      delete window;
    }
  }

  return mWindow;
}

void MAFApplication::AddController(MAFController* controller)
{
  if(mRunning) {
    if(mRunningControllers)
      mControllersToAdd.push_back(controller);
    else
      mControllers.push_front(controller);
  }
}

void MAFApplication::RemoveController(MAFController* controller)
{
  if(mRunning) {
    if(mRunningControllers) {
      // g_debug("MAFApplication::RemoveController delayed");
      mControllersToRemove.push_back(controller);
    } else {
      // g_debug("MAFApplication::RemoveController");
      mControllers.remove(controller);
    }
  }
}

SDL_Event*	MAFApplication::GetLastEventIgnoreLocking() const
{
  return mLastEvent;
}

SDL_Event*	MAFApplication::GetLastEvent(MAFController* controller) const
{
  if (mLastEvent == 0)
    return 0;
  else
    {
      if (mEventLockMap[mLastEvent->type] != 0 && 
	  mEventLockMap[mLastEvent->type] != controller)
	return 0;
      else
	return mLastEvent;
    }
}

bool	MAFApplication::HasEvent() const
{
  return mLastEvent != 0;
}

bool	MAFApplication::LockEvent(Uint8 type, MAFController* controller)
{
  if (mEventLockMap[type] == 0)
    {
      mEventLockMap[type] = controller;
      return true;
    }
  else
    {
      g_critical("%s lock request on event %X, already locked controller %p",
		 __FUNCTION__, type, mEventLockMap[type]);
      return false;
    }
}

void	MAFApplication::UnlockEvent(Uint8 type, MAFController* controller)
{
  if (mEventLockMap[type] != controller)
    g_critical("%s event %X was not locked by %p but %p",
	       __FUNCTION__, type, controller, mEventLockMap[type]);
  mEventLockMap[type] = 0;
}

bool	MAFApplication::IsLocked(Uint8 type) const
{
  return mEventLockMap[type] != 0;
}

bool	MAFApplication::LockMouse(MAFController* controller)
{
  bool	is_ok = true;

  is_ok &= LockEvent(SDL_MOUSEMOTION, controller);
  is_ok &= LockEvent(SDL_MOUSEBUTTONUP, controller);
  is_ok &= LockEvent(SDL_MOUSEBUTTONDOWN, controller);

  return is_ok;
}

void	MAFApplication::UnlockMouse(MAFController* controller)
{
  UnlockEvent(SDL_MOUSEMOTION, controller);
  UnlockEvent(SDL_MOUSEBUTTONUP, controller);
  UnlockEvent(SDL_MOUSEBUTTONDOWN, controller);
}

bool	MAFApplication::IsLockedMouse() const
{
  bool is_locked_motion = IsLocked(SDL_MOUSEMOTION);
  bool is_locked_button_up = IsLocked(SDL_MOUSEBUTTONUP);
  bool is_locked_button_down = IsLocked(SDL_MOUSEBUTTONDOWN);

  if ((is_locked_motion != is_locked_button_up) ||
      (is_locked_motion != is_locked_button_down))
    g_warning("%s non consistent mouse locking",
	      __FUNCTION__);

  return (is_locked_motion &&
	  is_locked_button_up &&
	  is_locked_button_down);
}

bool	MAFApplication::LockKeyboard(MAFController* controller)
{
  bool	is_ok = true;

  is_ok &= LockEvent(SDL_KEYUP, controller);
  is_ok &= LockEvent(SDL_KEYDOWN, controller);

  return is_ok;
}

void	MAFApplication::UnlockKeyboard(MAFController* controller)
{
  UnlockEvent(SDL_KEYUP, controller);
  UnlockEvent(SDL_KEYDOWN, controller);
}

bool	MAFApplication::IsLockedKeyboard() const
{
  bool is_locked_key_up = IsLocked(SDL_KEYUP);
  bool is_locked_key_down = IsLocked(SDL_KEYDOWN);

  if (is_locked_key_up != is_locked_key_down)
    g_warning("%s non consistent mouse locking",
	      __FUNCTION__);

  return (is_locked_key_up &&
	  is_locked_key_down);
}

MAFController* MAFApplication::GetFocus(void)
{
  return mFocus;
}

void MAFApplication::SetFocus(MAFController* controller)
{
  mFocus = controller;
}

std::string MAFApplication::HeaderGet(const std::string& name, const std::string& path)
{
  std::list<std::string> result = HeaderGetList(name, path);
  return result.size() > 0 ? result.front() : "";
}

std::map<std::string,std::string> MAFApplication::HeaderGetProperties(const std::string& name, const std::string& path)
{
  std::map<std::string,std::string> result;
  std::list<std::map<std::string,std::string> > results = HeaderGetPropertiesList(name, path);
  if(results.empty())
    return result;
  else
    return results.front();
}

std::list<std::map<std::string,std::string> > MAFApplication::HeaderGetPropertiesList(const std::string& name, const std::string& path)
{
  std::list<std::map<std::string,std::string> > results;

  if(mHeaders.find(name) != mHeaders.end()) {
    xmlDocPtr header = mHeaders[name];
    /* Create xpath evaluation context */
    xmlXPathContextPtr xpathCtx = xmlXPathNewContext(header);
    if(xpathCtx == NULL)
      g_error("MAFApplication::HeaderGetList: unable to create new XPath context");

    /* Evaluate xpath expression */
    xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)path.c_str(), xpathCtx);
    if(xpathObj == NULL) {
      g_error("Error: unable to evaluate xpath expression %s", path.c_str());
      xmlXPathFreeContext(xpathCtx); 
      throw this;
    }

    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    if(nodes && nodes->nodeNr >= 1) {
      for(int i = 0; i < nodes->nodeNr; i++) {
	xmlNodePtr node = nodes->nodeTab[i];
	switch(node->type) {
	case XML_ELEMENT_NODE:
	  {
	    std::map<std::string,std::string> result;
	    xmlAttr* attribute;
	    for(attribute = node->properties; attribute; attribute = attribute->next) {
	      const char* content = (const char*)xmlNodeGetContent((xmlNode*)attribute);	      
	      result[(const char*)attribute->name] = content;
	      xmlFree((void*)content);
	    }
	    results.push_back(result);
	  }
	  break;
	default:
	  break;
	}
      }
    }
    xmlXPathFreeObject(xpathObj);
    xmlXPathFreeContext(xpathCtx); 
  } else {
    throw new MAFError(UNDERWARE_MAF_ERROR_XML, "MAFApplication::HeaderGetListProperties: %s is not a know XML document", name.c_str());
  }

  return results;
}

std::list<std::string> MAFApplication::HeaderGetList(const std::string& name, const std::string& path)
{
  std::list<std::string> result;
  
  if(mHeaders.find(name) != mHeaders.end()) {
    xmlDocPtr header = mHeaders[name];
    /* Create xpath evaluation context */
    xmlXPathContextPtr xpathCtx = xmlXPathNewContext(header);
    if(xpathCtx == NULL)
      g_error("MAFApplication::HeaderGetList: unable to create new XPath context");

    /* Evaluate xpath expression */
    xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)path.c_str(), xpathCtx);
    if(xpathObj == NULL) {
      g_error("Error: unable to evaluate xpath expression %s", path.c_str());
      xmlXPathFreeContext(xpathCtx); 
      throw this;
    }

    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    if(nodes && nodes->nodeNr >= 1) {
      for(int i = 0; i < nodes->nodeNr; i++) {
	xmlNodePtr node = nodes->nodeTab[i];
	switch(node->type) {
	case XML_ELEMENT_NODE:
	case XML_ATTRIBUTE_NODE:
	  {
	    const char* content = (const char*)xmlNodeGetContent(node);
	    result.push_back(content);
	    xmlFree((void*)content);
	  }
	  break;
	default:
	  break;
	}
      }
    }
    xmlXPathFreeObject(xpathObj);
    xmlXPathFreeContext(xpathCtx); 
  } else {
    throw new MAFError(UNDERWARE_MAF_ERROR_XML, "MAFApplication::HeaderGetList: %s is not a know XML document", name.c_str());
  }

  return result;
}

void MAFApplication::HeaderSet(const std::string& name, const std::string& path, const std::string& value)
{
  if(mHeaders.find(name) != mHeaders.end()) {
    xmlDocPtr header = mHeaders[name];
    /* Create xpath evaluation context */
    xmlXPathContextPtr xpathCtx = xmlXPathNewContext(header);
    if(xpathCtx == NULL)
      g_error("MAFApplication::HeaderSet: unable to create new XPath context");

    /* Evaluate xpath expression */
    xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)path.c_str(), xpathCtx);
    if(xpathObj == NULL) {
      g_error("Error: unable to evaluate xpath expression %s", path.c_str());
      xmlXPathFreeContext(xpathCtx); 
      throw this;
    }

    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    if(nodes && nodes->nodeNr >= 1) {
      for(int i = 0; i < nodes->nodeNr; i++) {
	xmlNodePtr node = nodes->nodeTab[i];
	switch(node->type) {
	case XML_ELEMENT_NODE:
	case XML_ATTRIBUTE_NODE:
	  {
	    xmlNodeSetContent(node, (const xmlChar*)value.c_str());
	  }
	  break;
	default:
	  break;
	}
      }
    }
    xmlXPathFreeObject(xpathObj);
    xmlXPathFreeContext(xpathCtx); 
  } else {
    throw new MAFError(UNDERWARE_MAF_ERROR_XML, "MAFApplication::HeaderSet: %s is not a know XML document", name.c_str());
  }

}

/*
 * Private methods
 */

void MAFApplication::ParseDefaultOpts(void)
{
  mWindowFullScreen = HeaderGet("settings", "/settings/fullscreen") == "yes";
}

void MAFApplication::Uninit(void)
{
  mControllers.clear();

  OnExit(mExitCode);

  if(mWindow) {
    delete mWindow;
    mWindow = NULL;
  }
}


void MAFApplication::SetCursor(MAFCursorController* cursor)
{
  if (mCursor.get()) {
    mCursor->ReleaseCursor();
    RemoveController(mCursor.get());
  }

  if (cursor) {
    mCursor=cursor;
    mCursor->InitCursor();
    AddController(mCursor.get());
  }
}
