/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Henry Precheur <henry@precheur.org>
 */

#include "mafStdAfx.h"

// #define _DEBUG_LOCK_EVENT_

#ifndef MAF_USE_VS_PCH
#include <maf/maferror.h>
#include <maf/data.h>
#include <maf/hit.h>
#include <maf/application.h>
#include <maf/application2d.h>
#include <maf/window.h>
#include <maf/wnc_window.h>
#include <maf/wnc_desktop.h>
#include <maf/wnc_source.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <osg/Transform>
#include <osg/Geode>

#include <maf/animate2d.h>

#endif

#ifdef WIN32
typedef unsigned long XID;
typedef XID KeySym;
#endif


// Model

void MAFApplication2DModel::Init(void)
{
  g_assert(mDesktop != 0);
}

// Controller
void MAFApplication2DController::Init( void )
{
  if(!GetModel())
    SetModel(new MAFApplication2DModel);
  MAFVisionController::Init();

  osg::Group*	hit = GetModel()->GetDesktop()->GetHitableWindows();
  osg::Group*	disp = GetModel()->GetDesktop()->GetMappedWindows();
  osg::Group*	group = new osg::Group();

  /*
   * we insert a collision only group in the scene
   */
  hit->setNodeMask(MAF_COLLISION_MASK);
  group->addChild(hit);

  /*
   * now we insert a display only group in the scene
   */
  disp->setNodeMask(MAF_VISIBLE_MASK);
  group->addChild(disp);

  GetModel()->SetNode(group);
  BindToNode(hit);

  mDeltaFlushMotion = 0;
  mNeedFlush = false;
  mIsFocused = false;
  mIsLocked = false;
}

#define XK_MISCELLANY
#ifdef WIN32
#include <maf/wnc_win32x.h>
#else
#include <X11/keysymdef.h>
#endif

static unsigned long table3_sdl2rfb[] = {
XK_Num_Lock, //	SDLK_NUMLOCK		= 300,
XK_Caps_Lock, //	SDLK_CAPSLOCK		= 301,
XK_Scroll_Lock, //	SDLK_SCROLLOCK		= 302,
XK_Shift_R, //	SDLK_RSHIFT		= 303,
XK_Shift_L, //	SDLK_LSHIFT		= 304,
XK_Control_R, //	SDLK_RCTRL		= 305,
XK_Control_L, //	SDLK_LCTRL		= 306,
XK_Alt_R, //	SDLK_RALT		= 307,
XK_Alt_L, //	SDLK_LALT		= 308,
XK_Meta_R, //	SDLK_RMETA		= 309,
XK_Meta_L, //	SDLK_LMETA		= 310,
XK_Super_R, //	SDLK_LSUPER		= 311,		/* Left "Windows" key */
XK_Super_L, //	SDLK_RSUPER		= 312,		/* Right "Windows" key */
XK_Mode_switch, //	SDLK_MODE		= 313,		/* "Alt Gr" key */
XK_Multi_key, //	SDLK_COMPOSE		= 314,		/* Multi-key compose key */
0
};

static unsigned long table2_sdl2rfb[] = {
XK_KP_0, //	SDLK_KP0		= 256,
XK_KP_1, //	SDLK_KP1		= 257,
XK_KP_2, //	SDLK_KP2		= 258,
XK_KP_3, //	SDLK_KP3		= 259,
XK_KP_4, //	SDLK_KP4		= 260,
XK_KP_5, //	SDLK_KP5		= 261,
XK_KP_6, //	SDLK_KP6		= 262,
XK_KP_7, //	SDLK_KP7		= 263,
XK_KP_8, //	SDLK_KP8		= 264,
XK_KP_9, //	SDLK_KP9		= 265,
XK_KP_Decimal, //	SDLK_KP_PERIOD		= 266,
XK_KP_Divide, //	SDLK_KP_DIVIDE		= 267,
XK_KP_Multiply, //	SDLK_KP_MULTIPLY	= 268,
XK_KP_Subtract, //	SDLK_KP_MINUS		= 269,
XK_KP_Add, //	SDLK_KP_PLUS		= 270,
XK_KP_Enter, //	SDLK_KP_ENTER		= 271,
XK_KP_Equal, //	SDLK_KP_EQUALS		= 272,
XK_Up, //	SDLK_UP			= 273,
XK_Down, //	SDLK_DOWN		= 274,
XK_Right, //	SDLK_RIGHT		= 275,
XK_Left, //	SDLK_LEFT		= 276,
XK_Insert, //	SDLK_INSERT		= 277,
XK_Home, //	SDLK_HOME		= 278,
XK_End, //	SDLK_END		= 279,
XK_Page_Up, //	SDLK_PAGEUP		= 280,
XK_Page_Down, //	SDLK_PAGEDOWN		= 281,
XK_F1, //	SDLK_F1			= 282,
XK_F2, //	SDLK_F2			= 283,
XK_F3, //	SDLK_F3			= 284,
XK_F4, //	SDLK_F4			= 285,
XK_F5, //	SDLK_F5			= 286,
XK_F6, //	SDLK_F6			= 287,
XK_F7, //	SDLK_F7			= 288,
XK_F8, //	SDLK_F8			= 289,
XK_F9, //	SDLK_F9			= 290,
XK_F10, //	SDLK_F10		= 291,
XK_F11, //	SDLK_F11		= 292,
XK_F12, //	SDLK_F12		= 293,
XK_F13, //	SDLK_F13		= 294,
XK_F14, //	SDLK_F14		= 295,
XK_F15, //	SDLK_F15		= 296,
0
};

static unsigned long table1_sdl2rfb[] = {
0, // 0 SDLK_
0, // 1 SDLK_
0, // 2 SDLK_
0, // 3 SDLK_
0, // 4 SDLK_
0, // 5 SDLK_
0, // 6 SDLK_
0, // 7 SDLK_
XK_BackSpace, // 8 SDLK_BACKSPACE
XK_Tab, // 9 SDLK_TAB
0, // 10 SDLK_
0, // 11 SDLK_
XK_Clear, // 12 SDLK_CLEAR
XK_Return, // 13 SDLK_RETURN
0, // 14 SDLK_
0, // 15 SDLK_
0, // 16 SDLK_
0, // 17 SDLK_
0, // 18 SDLK_
XK_Pause, // 19 SDLK_PAUSE
0, // 20 SDLK_
0, // 21 SDLK_
0, // 22 SDLK_
0, // 23 SDLK_
0, // 24 SDLK_
0, // 25 SDLK_
0, // 26 SDLK_
XK_Escape, // 27 SDLK_ESCAPE
0
};

unsigned long key_sdl2rfb(SDL_Event* event)
{
  Uint16 sdl_unicode = event->key.keysym.unicode;
  SDLKey sdl_keysym = event->key.keysym.sym;
  KeySym rfb_keysym = '?';
  if(sdl_keysym <= SDLK_ESCAPE)
    rfb_keysym = table1_sdl2rfb[sdl_keysym];
  else if(sdl_keysym >= SDLK_KP0 && sdl_keysym <= SDLK_F15)
    rfb_keysym = table2_sdl2rfb[sdl_keysym - SDLK_KP0];
  else if(sdl_keysym >= SDLK_NUMLOCK && sdl_keysym <= SDLK_COMPOSE)
    rfb_keysym = table3_sdl2rfb[sdl_keysym - SDLK_NUMLOCK];
  else if (event->key.keysym.mod & KMOD_CTRL ||
	   event->key.keysym.mod & KMOD_ALT ||
	   event->key.keysym.mod & KMOD_META)
    rfb_keysym = sdl_keysym;
  else
    rfb_keysym = sdl_unicode;

  return rfb_keysym;
}

CARD8	get_pointer_state(Uint8 state)
{
  CARD8	c = ((state & SDL_BUTTON(1) ? 1 : 0) << 0 |
	     (state & SDL_BUTTON(2) ? 1 : 0) << 1);
  return c;
}

// find the focused window
void MAFApplication2DController::HandleHit(MAFHit& hit)
{
  osg::Geode*	node = dynamic_cast<osg::Geode*>(hit.mHit._nodePath.back());
  osg::Group*	g = GetModel()->GetDesktop()->GetMappedWindows();
  
  mId = 0;
  mFocusedWindowName = "none";
  for (unsigned i = 0; i < g->getNumChildren(); i++)
    {
      XwncWindow*	win = dynamic_cast<XwncWindow*>(g->getChild(i));
      // g_assert(win->getNumChildren() == 1);
      // FIXME if rectangle toobig window wont have child
      if (win->getNumChildren() != 1)
	continue;
      osg::Geode*	geode = dynamic_cast<osg::Geode*>(win->getChild(0));
      if (geode == node)
	{
	  mId = win->getFrameID();
	  mFocusedWindowName = win->GetTitle();
	  break;
	}
    }
}

void MAFApplication2DController::HandleNotHit()
{
  mId = 0;
  mFocusedWindowName = "none";
}

void	MAFApplication2DController::
SetDefaultFocusedWindow(const std::string& window_name)
{
  mDefaultFocusedWindowName = window_name;
}

std::map<std::string, MAFApplication2DAnimate*>& MAFApplication2DController::GetName2Animate()
{
  return GetModel()->GetDesktop()->GetName2Animate();
}

void	MAFApplication2DController::HandleDefault(SDL_Event* event)
{
  switch (event->type)
    {
    case SDL_KEYUP:
    case SDL_KEYDOWN:
      {
	XwncWindow*	window = GetModel()->GetDesktop()->GetWindow(mId);
	if (!window)
	  window = GetModel()->GetDesktop()->GetWindow(mDefaultFocusedWindowName);
	if (!window)
	  window = GetModel()->GetDesktop()->GetWindow();
	
	if (window)
	  {
	    int	x, y;

	    window->getRealPosition(&x, &y);
	    GetModel()->GetDesktop()->getServer()
	      ->pointerEvent(window->getFrameID(),
			     x, y, get_pointer_state(0)); // FIXME
	    HandleKeyboard(event);
	  }
      }
      break;
    case SDL_MOUSEBUTTONUP:
    case SDL_MOUSEMOTION:
    case SDL_MOUSEBUTTONDOWN:
      if (mIsFocused)
	HandleMouse(event);
      break;
    default:
      break;
    }
}

void	MAFApplication2DController::HandleKeyboard(SDL_Event* event)
{
  switch (event->type)
    {
    case SDL_KEYDOWN:
    case SDL_KEYUP:
      GetModel()->GetDesktop()->getServer()
	->keyEvent(key_sdl2rfb(event), event->type == SDL_KEYDOWN);
      break;
    default:
      break;
    }
}

void	MAFApplication2DController::HandleMouse(SDL_Event* event)
{
  XwncWindow*	win = 	GetModel()->GetDesktop()->GetWindow(mId);
  
  if (!win)
    {
      g_critical("%s event for a non existing window %lX (%s)",
		 __FUNCTION__, mId, mFocusedWindowName.c_str());
      return;
    }

  switch (event->type)
    {
    case SDL_MOUSEMOTION:
      mLastMouseEvent = *event;
      mNeedFlush = true;
      break;
    case SDL_MOUSEBUTTONDOWN:
    case SDL_MOUSEBUTTONUP:
      FlushMotionEvent();
      win->pointerEvent(event->button.x, event->button.y,
			get_pointer_state(event->button.state));
      break;
    default:
      break;
    }
}

void	MAFApplication2DController::FlushMotionEvent()
{
  XwncWindow*	win = 	GetModel()->GetDesktop()->GetWindow(mId);
  SDL_Event *	event = &mLastMouseEvent;

  if (!win)
    g_warning("%s event for a non existing window %lX (%s)",
	      __FUNCTION__, mId, mFocusedWindowName.c_str());
  else {
    win->pointerEvent(event->motion.x, event->motion.y,
		      get_pointer_state(event->motion.state));

    // lock cursor
    if (mApplication->IsLockedMouse()) {
      mApplication->GetCursor()->WarpMouse(event->motion.x,event->motion.y);
    }
  }
  mDeltaFlushMotion = 0;
  mNeedFlush = false;
}

bool MAFApplication2DController::Update(class MAFApplication* App)
{
  SDL_Event* event = App->GetLastEvent(this);

  if (event)
    {
      if (App->GetFocus() == this)
	{
	  if (!GetModel()->GetDesktop()->GetWindow(mId))
	    {
	      g_critical("%s focused but no window with correct id %lX (%s)",
			 __FUNCTION__, mId, mFocusedWindowName.c_str());
	    }

	  HandleKeyboard(event);
	  HandleMouse(event);
	  if (!mIsLocked)
	    {
	      LOCK_KEYBOARD(App, this);
	      mIsLocked = true;
	    }
	  if (event->type == SDL_MOUSEBUTTONDOWN)
	    {
	      LOCK_MOUSE(App, this);
	      mIsFocused = true;
	    }
	  if (event->type == SDL_MOUSEBUTTONUP && mIsFocused)
	    {
	      UNLOCK_MOUSE(App, this);
	      mIsFocused = false;
	    }
	}
      else
	{
	  if (event->type == SDL_MOUSEBUTTONUP && mIsFocused)
	    {
	      UNLOCK_MOUSE(App, this);
	      mIsFocused = false;
	    }
	  if (mIsLocked)
	    {
	      UNLOCK_KEYBOARD(App, this);
	      mIsLocked = false;
	    }
	  HandleDefault(event);
	}
    }
  else if (!App->HasEvent())
    {
      // FIXME works fine on fast machine, but it might need
      // enhancement for slow machine ...
      const float	min_delta = 75;
      mDeltaFlushMotion += GetDeltaFrame();

      if (mNeedFlush && mDeltaFlushMotion > min_delta)
	FlushMotionEvent(); // flush motion event in the queue
    }
    
  GetModel()->GetDesktop()->getServer()->check();

  {
    std::map<std::string, MAFApplication2DAnimate*>& name2animate = GetModel()->GetDesktop()->GetName2Animate();
    for(std::map<std::string, MAFApplication2DAnimate*>::iterator i = name2animate.begin();
	i != name2animate.end();
	i++) {
      const std::string& name = i->first;
      MAFApplication2DAnimate* animate = i->second;
      XwncWindow* window = GetModel()->GetDesktop()->GetWindow(name);
      bool mouseOver = window != 0 ? (App->GetFocus() == this && window->getFrameID() == mId) : false;
      animate->Update(mouseOver, GetDelta());
    }
  }
  
  return true;
}
