/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/maferror.h>
#include <maf/packets.h>
#endif

#ifdef _DEBUG // for Windows python23_d.lib is not in distribution... ugly but works
#	undef _DEBUG
#	include <Python.h>
#	define _DEBUG
#else
#	include <Python.h>
#endif

MAFPacketsModule::~MAFPacketsModule() {
  if(mModule) Py_DECREF(mModule);
}

MAFPacketsModule::MAFPacketsModule(const std::string& module) {
  mModule = PyImport_ImportModule((char*)module.c_str());
  if(!mModule)
    throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::MAFPacket: import %s failed", module.c_str());
  LoadTypes();
}

MAFPacket* MAFPacketsModule::Create(const std::string& pyclass) {
  std::string constructor = pyclass + "()";
  PyObject* pypacket = PyRun_String(constructor.c_str(), Py_eval_input, PyModule_GetDict(mModule), PyModule_GetDict(mModule));
  if(!pypacket)
    throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacketsModule::create: %s()", pyclass.c_str());

  MAFPacket* packet = new MAFPacket(this, pypacket);
  Py_DECREF(pypacket);
  return packet;
}

long MAFPacketsModule::GetType(const std::string& name) const
{
  std::map<std::string,long>::const_iterator i = mTypes.find("PACKET_" + name);
  return i == mTypes.end() ? 0 : i->second;
}


void MAFPacketsModule::LoadTypes(void)
{
  int pos = 0;
  PyObject* key;
  PyObject* value;
  PyObject* dictionary = PyModule_GetDict(mModule);
  std::string prefix = "PACKET_";

  while(PyDict_Next(dictionary, &pos, &key, &value)) {
    std::string key_string = PyString_AsString(key);
    if(key_string.substr(0, prefix.size()) == prefix) {
      mTypes[key_string] = PyLong_AsLong(value);
    }
  }
}

MAFPacket::MAFPacket(MAFPacketsModule* module, PyObject* object)
{
  mPacketsModule = module;
  mPacket = object;
  Py_INCREF(mPacket);

  GetMember("type", mType);
}

MAFPacket::~MAFPacket()
{
  Py_DECREF(mPacket);
}

void MAFPacket::SetMember(const std::string& name, const std::string& value)
{
  SetMember(name, PyString_FromString(value.c_str()));
}

void MAFPacket::SetMember(const std::string& name, long value)
{
  SetMember(name, PyLong_FromLong(value));
}

void MAFPacket::SetMember(const std::string& name, double value)
{
  SetMember(name, PyFloat_FromDouble(value));
}

void MAFPacket::SetMember(const std::string& name, const std::vector<int>& values)
{
  PyObject* pyvalues = PyList_New(values.size());
  for(unsigned int i = 0; i < values.size(); i++) {
    PyObject* value = PyLong_FromLong((long)values[i]);
    if(!value)
      throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::SetMember PyLong_FromLong");
    if(PyList_SetItem(pyvalues, i, value) < 0) {
      throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::SetMember PyList_SetItem");
    }
  }
  SetMember(name, pyvalues);
}

void MAFPacket::SetMember(const std::string& name, PyObject* value)
{
  if(PyObject_SetAttrString(mPacket, (char*)name.c_str(), value) < 0) {
    Py_DECREF(value);
    throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::SetMember SetAttrString(%s)", name.c_str());
  }
  Py_DECREF(value);
}

void MAFPacket::GetMember(const std::string& name, std::string& value) const
{
  PyObject* pyvalue;
  GetMember(name, pyvalue);
  value = PyString_AsString(pyvalue);
  Py_DECREF(pyvalue);
}

void MAFPacket::GetMember(const std::string& name, long& value) const
{
  PyObject* pyvalue;
  GetMember(name, pyvalue);
  if(PyLong_Check(pyvalue)) {
      value = PyLong_AsLong(pyvalue);
  } else if(PyInt_Check(pyvalue)) {
      value = PyInt_AsLong(pyvalue);
  } else if(PyBool_Check(pyvalue)) {
      value = PyObject_IsTrue(pyvalue);
  } else {
    Py_DECREF(pyvalue);
    throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::GetMember attribute %s is neither a Long nor an Int", name.c_str());
  }
  Py_DECREF(pyvalue);
}

void MAFPacket::GetMember(const std::string& name, double& value) const
{
  PyObject* pyvalue;
  GetMember(name, pyvalue);
  value = PyFloat_AsDouble(pyvalue);
  Py_DECREF(pyvalue);
}

void MAFPacket::GetMember(const std::string& name, std::vector<int>& values) const
{
  values.clear();

  PyObject* attr;
  GetMember(name, attr);

  if(!PyList_Check(attr)) {
    Py_DECREF(attr);
    throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::GetMember attributed %s is not a List", name.c_str());
  }

  for(int i = 0; i < PyList_Size(attr); i++) {
    PyObject* value = PyList_GetItem(attr, i);
    if(PyLong_Check(value)) {
      values.push_back((int)PyLong_AsLong(value));
    } else if(PyInt_Check(value)) {
      values.push_back((int)PyInt_AsLong(value));
    } else {
      throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::GetMember element %d of attribute %s is neither a Long nor an Int", i, name.c_str());
    }
  }
  Py_DECREF(attr);
}

void MAFPacket::GetMember(const std::string& name, PyObject*& value) const
{
  if((value = PyObject_GetAttrString(mPacket, (char*)name.c_str())) == 0)
    throw new MAFError(UNDERWARE_MAF_ERROR_PYTHON, "MAFPacket::GetMember attribute %s is not found", name.c_str());
}

bool MAFPacket::IsType(const std::string& name) const
{
  return mPacketsModule->GetType(name) == mType;
}

