/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/maferror.h>
#include <maf/window.h>
#include <maf/scene.h>
#endif

MAFWindow::MAFWindow() : mFullScreen(true), mWidth(1024), mHeight(768), mSurface(0), mOpenGL(false)
{
}

MAFWindow::~MAFWindow()
{
  SDL_Quit();
}

void MAFWindow::SwapBuffers()
{
  if(mOpenGL)
    SDL_GL_SwapBuffers();
  else
    SDL_Flip(mSurface);
}

void MAFWindow::SetCaption(const std::string& Caption)
{
  SDL_WM_SetCaption(Caption.c_str(), NULL);
}


bool MAFWindow::AddView(MAFView* pView)
{
  mViews.push_front(pView);

  return true;
}

bool MAFWindow::DelView(MAFView* pView)
{
  mViews.remove(pView);

  return true;
}

void MAFWindow::Render()
{
  ViewList::iterator i;

  for (i = mViews.begin(); i != mViews.end(); i++)
    (*i)->Update(this);
}


bool MAFWindow::Init(void)
{
  if(mSurface == 0) {
    if(SDL_Init(SDL_INIT_VIDEO) != 0)
      throw new MAFError(UNDERWARE_MAF_ERROR_SDLINIT, "MAFWindow::Init: SDL_Init: %s\n", SDL_GetError ());

    SDL_EnableUNICODE(1);
    SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
  }

  /* Let's get some video information. */
  const SDL_VideoInfo* info = SDL_GetVideoInfo( );
  if(!info)
    throw new MAFError(UNDERWARE_MAF_ERROR_SDLINIT, "MAFWindow::Init: SDL_GetVideoInfo: %s\n", SDL_GetError ());

  /* If a dimension is given a zero size, we use either the screen
   * dimension if fullscreen is required (prevent nasty frequency
   * switch on CRTs), or either use half of the dimension screen in
   * window'ed mode.
   */

  /* We get the bpp we will request from
   * the display. On X11, VidMode can't change
   * resolution, so this is probably being overly
   * safe. Under Win32, ChangeDisplaySettings
   * can change the bpp.
   */
  int bpp = info->vfmt->BitsPerPixel;

  if(mWidth == 0 || mHeight == 0) {
    g_warning("width or height equal to zero, revert to default 1024x768");
    mWidth = 1024;
    mHeight = 768;
  }
  int flags = mFullScreen ? SDL_FULLSCREEN : 0;
  if(mOpenGL) {
    flags |= SDL_OPENGL;

    /* Now, we want to setup our requested
     * window attributes for our OpenGL window.
     * We want *at least* 5 bits of red, green
     * and blue. We also want at least a 16-bit
     * depth buffer.
     *
     * The last thing we do is request a double
     * buffered window. '1' turns on double
     * buffering, '0' turns it off.
     *
     * Note that we do not use SDL_DOUBLEBUF in
     * the flags to SDL_SetVideoMode. That does
     * not affect the GL attribute state, only
     * the standard 2D blitting setup.
     */
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, bpp );
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
  } else {
    flags |= SDL_DOUBLEBUF;
  }

  SDL_Surface* surface;

  {
    SDL_Rect **modes;
    int i;

    /* Get available fullscreen/hardware modes */
    modes = SDL_ListModes(info->vfmt, flags);

    /*
     * Check is there are any modes available
     * and set width x height to the maximum available.
     */
    int maxWidth = 0;
    int maxHeight = 0;
    bool found = mFullScreen ? false : true;
    
    if(modes != (SDL_Rect **)0 && modes != (SDL_Rect **)-1) {
      g_debug("Available video modes (%d bpp)", info->vfmt->BitsPerPixel);
      for(i = 0; modes[i] != 0; i++) {
	if(mWidth == modes[i]->w && mHeight == modes[i]->h)
	  found = true;
	g_debug("  %d x %d", modes[i]->w, modes[i]->h);
	if(maxWidth < modes[i]->w) {
	  maxWidth = modes[i]->w;
	  maxHeight = modes[i]->h;
	}
      }
    }

    /*
     * If there are no video modes matching the desired width /
     * height, chose the largest one.
     */
    if(!found) {
      g_debug("desired video mode %dx%d not available, revert to %dx%d", mWidth, mHeight, maxWidth, maxHeight);
      mWidth = maxWidth;
      mHeight = maxHeight;
    }
  }

  surface = SDL_SetVideoMode(mWidth, mHeight, bpp, flags);
  if(surface == 0 && bpp > 24 && (surface = SDL_SetVideoMode(mWidth, mHeight, bpp, flags)) == 0) {
    g_debug("Video with 32 bpp not available, revert to 24 bpp");
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 24 );
    surface = SDL_SetVideoMode(mWidth, mHeight, bpp, flags);
  }
  if(surface == 0 && bpp > 16 && (surface = SDL_SetVideoMode(mWidth, mHeight, bpp, flags)) == 0) {
    g_debug("Video with 24 bpp not available, revert to 16 bpp");
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
    surface = SDL_SetVideoMode(mWidth, mHeight, bpp, flags);
  }
  if(surface == 0)
    /* This could happen for a variety of reasons,
     * including DISPLAY not being set, the specified
     * resolution not being available, etc.
     */
    throw new MAFError(UNDERWARE_MAF_ERROR_SDLINIT, "MAFWindow::Init: SDL_SetVideoMode: %s\n", SDL_GetError ());

  mWidth = surface->w;
  mHeight = surface->h;

  mSurface = surface;
  return true;
}
